/* This file is part of GEGL editor -- a gtk frontend for GEGL
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (C) 2003, 2004, 2006 Øyvind Kolås
 */

#include "config.h"

#include <glib.h>
#include <gegl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "gegl-options.h"
#include "gegl-dot.h"

#ifdef G_OS_WIN32
#include <direct.h>
#define getcwd(b,n) _getcwd(b,n)
#ifndef PATH_MAX
#define PATH_MAX _MAX_PATH
#endif
#define realpath(a,b) _fullpath(b,a,_MAX_PATH)
#endif

#if HAVE_GTK
#include <gtk/gtk.h>
#include "gegl-bin-gui-types.h"
#include "editor.h"
#endif

#define DEFAULT_COMPOSITION \
"<gegl> <node operation='crop'> <params> <param name='x'>0</param> <param name='y'>0</param> <param name='width'>512</param> <param name='height'>216</param> </params> </node> <node operation='over'> <opacity value='1.0'/><node operation='shift'> <params> <param name='x'>80</param> <param name='y'>80</param> </params> </node> <node operation='text'> <params> <param name='string'>GEGL - A graph based image processing and compositing engine.</param> <param name='font'>Sans</param> <param name='size'>11</param> <param name='color'>#ffffaa</param> <param name='wrap'>500</param> <param name='alignment'>0</param> <param name='width'>402</param> <param name='height'>17</param> </params> </node> </node> <node operation='over'> <node operation='shift'> <params> <param name='x'>15</param> <param name='y'>160</param> </params> </node> <opacity value='0.4'/> <node name='text' operation='text'> <params> <param name='string'>2000-2008 © Calvin Williamson, Caroline Dahloff, Manish Singh, Jay Cox, Daniel Rogers, Sven Neumann, Michael Natterer,  Øyvind Kolås, Philip Lafleur, Dominik Ernst, Richard Kralovic, Kevin Cozens, Victor Bogado, Martin Nordholts, Geert Jordaens, Michael Schumacher, John Marshall, Étienne Bersac, Mark Probst, Håkon Hitland, Tor Lillqvist, Hans Breuer, Deji Akingunola, Bradley Broom, Hans Petter Jansson, Jan Heller, dmacks@netspace.org, Sven Anders, Hubert Figuière, Sam Hocevar, yahvuu at gmail.com, Nicolas Robidoux, Garry R. Osgood, Shlomi Fish and Jakub Steiner</param> <param name='font'>Sans</param> <param name='size'>8</param> <param name='color'>rgba(0.8313, 1.0000, 1.0000, 1.0000)</param> <param name='wrap'>485</param> <param name='alignment'>0</param> <param name='width'>355</param> <param name='height'>70</param> </params> </node> </node> <unsharp-mask std-dev='13' scale='9'/><node operation='lighten'> <opacity value='0.5'/> <gaussian-blur std-dev-x='7' std-dev-y='7' /> <translate x='-22' y='-21'/> <over><translate y='-39' x='-14'/> <fill vector='M 122.21636,80.498391 C 121.6567,80.495506 121.07513,80.509301 120.49761,80.529641 C 103.17133,81.148282 87.586977,88.3352 76.028858,99.592141 C 75.9866,99.633082 75.946008,99.676092 75.903858,99.717142 C 74.577223,101.01595 73.306942,102.37676 72.091358,103.77964 C 71.937069,103.9577 71.775043,104.13122 71.622608,104.31089 C 71.061068,104.96928 70.533738,105.63124 69.997608,106.31089 C 69.841516,106.50974 69.682739,106.70403 69.528858,106.90464 C 69.519846,106.91633 69.506612,106.92419 69.497608,106.93589 C 68.875009,107.74866 68.270704,108.56372 67.685108,109.40464 C 67.647233,109.45869 67.597827,109.50673 67.560108,109.56089 C 66.985093,110.39184 66.44167,111.23511 65.903858,112.09214 C 64.74877,113.92076 63.693571,115.80891 62.716358,117.74839 C 62.689029,117.80263 62.649793,117.85032 62.622608,117.90464 C 57.776081,127.58898 55.210085,138.57256 55.622608,150.12339 C 56.942678,187.08596 88.097363,215.78692 125.06011,214.46714 C 140.13944,213.92869 153.90199,208.4732 164.84136,199.71714 C 165.63518,199.08515 166.42157,198.44593 167.18511,197.77964 C 167.22947,197.74073 167.26586,197.69367 167.31011,197.65464 C 169.67769,195.56667 171.91735,193.33154 173.96636,190.93589 C 174.00008,190.89646 174.02648,190.8504 174.06011,190.81089 C 174.72216,190.03684 175.37126,189.23956 175.99761,188.43589 C 176.02972,188.39449 176.05935,188.35236 176.09136,188.31089 C 176.71068,187.51242 177.32087,186.69974 177.90386,185.87339 C 177.93861,185.82385 177.96299,185.76678 177.99761,185.71714 C 178.28783,185.30339 178.56044,184.88761 178.84136,184.46714 C 179.1525,183.99887 179.47944,183.53725 179.77886,183.06089 C 180.31984,182.20468 180.83918,181.31746 181.34136,180.43589 C 181.37107,180.3834 181.40554,180.33221 181.43511,180.27964 C 182.46479,178.4605 183.39043,176.57338 184.24761,174.65464 C 188.24044,165.71694 190.33994,155.77117 189.96636,145.31089 C 189.80172,140.70102 188.88514,136.09539 187.84136,131.77964 L 125.15386,131.77964 C 124.80033,131.82161 124.6719,131.78368 124.31011,131.77964 C 118.79418,131.77964 115.55086,133.161 113.18511,135.21714 C 112.45622,135.85064 111.779,136.53665 111.18511,137.27964 C 111.16173,137.30889 111.14577,137.34387 111.12261,137.37339 C 111.11335,137.38493 111.1006,137.39308 111.09136,137.40464 C 110.82495,137.74669 110.57822,138.09485 110.34136,138.46714 C 110.31249,138.51147 110.27606,138.54743 110.24761,138.59214 C 110.01349,138.96851 109.7962,139.35171 109.59136,139.74839 C 109.56326,139.80148 109.52518,139.85125 109.49761,139.90464 C 108.12506,142.56264 107.40529,145.55781 107.27886,147.56089 C 107.28644,147.70456 107.26666,147.85472 107.27886,147.99839 C 107.26219,148.52416 107.64888,150.25455 107.71636,150.52964 C 108.68033,154.45108 111.58124,158.1673 113.62261,159.84214 C 115.837,161.6579 119.88762,162.98031 123.77886,162.81089 C 126.9775,162.67275 130.06317,161.8094 133.18511,158.74839 C 133.38629,158.55438 133.60852,158.36742 133.81011,158.15464 C 133.84497,158.11726 133.86899,158.0676 133.90386,158.02964 C 133.99605,157.9308 134.09282,157.82006 134.18511,157.71714 C 134.33139,157.55149 134.47604,157.39335 134.62261,157.21714 C 134.82413,156.97845 135.01422,156.72637 135.21636,156.46714 C 135.25077,156.42104 135.30594,156.38754 135.34136,156.34214 C 135.52678,156.09376 135.72194,155.85036 135.93511,155.62339 C 137.71233,153.79965 140.32885,152.73955 142.84136,153.06089 C 145.71281,153.42669 148.12365,155.47306 149.21636,158.15464 C 150.17246,160.4999 149.85718,163.16134 148.56011,165.31089 C 148.52218,165.37374 148.47471,165.43644 148.43511,165.49839 C 148.30643,165.69155 148.17377,165.87782 148.02886,166.06089 C 147.6319,166.57724 147.22577,167.07786 146.81011,167.56089 C 146.77624,167.60061 146.75035,167.64639 146.71636,167.68589 C 146.33334,168.12694 145.92626,168.55385 145.52886,168.96714 C 145.49359,169.00416 145.47049,169.05534 145.43511,169.09214 C 145.03715,169.50222 144.62731,169.8969 144.21636,170.27964 C 138.38832,175.70762 131.32158,178.39763 124.74761,178.77964 C 124.62023,178.80538 124.43911,178.76067 124.31011,178.77964 C 123.88449,178.84332 120.57812,178.93412 119.24761,178.77964 C 118.66386,178.71192 114.44547,177.56358 112.90386,177.06089 C 111.15084,176.48902 109.36677,175.78558 107.81011,174.96714 C 104.25423,173.09721 100.84385,170.97725 98.497608,168.18589 C 94.944306,163.95685 93.141268,159.05188 92.153858,154.21714 C 92.02378,153.68863 91.816494,153.07568 91.716358,152.52964 C 91.512289,151.1461 91.291595,149.6455 91.278858,148.27964 C 91.262597,147.85283 91.276687,147.43069 91.278858,146.99839 C 91.326281,145.49434 91.428682,143.97356 91.716358,142.34214 C 91.955655,140.88677 92.536628,139.16152 92.997608,137.68589 C 93.572825,135.84302 94.579521,134.01924 95.528858,132.18589 C 95.633929,131.98306 95.831143,131.62324 96.060108,131.24839 C 96.078165,131.21913 96.104155,131.18442 96.122608,131.15464 C 96.159902,131.09399 96.177259,131.03017 96.216358,130.96714 C 97.067114,129.59561 98.335717,127.73153 98.497608,127.52964 C 98.571118,127.43768 98.641237,127.33991 98.716358,127.24839 C 103.44741,121.55913 111.28041,116.78149 120.96636,115.93589 C 121.58452,115.79497 122.28551,115.80145 122.93511,115.81089 L 181.81011,115.81089 C 181.7881,115.77025 181.7697,115.72648 181.74761,115.68589 L 181.90386,115.68589 C 170.67876,94.955244 148.77929,80.903591 123.90386,80.529641 C 123.33244,80.52084 122.77602,80.501276 122.21636,80.498391 z' color='#bb99cc44'/></over> <over><fill vector='M 226.60694,42.355325 C 225.46782,42.338235 224.32074,42.345455 223.16944,42.386575 C 208.13935,42.923365 194.47098,48.406794 183.60694,57.199075 C 181.94788,58.541745 180.34712,59.957281 178.82569,61.449075 C 177.33625,62.90951 175.89181,64.41913 174.54444,66.01158 C 174.51063,66.05136 174.4844,66.09672 174.45069,66.13658 C 173.77172,66.94284 173.12264,67.7675 172.48194,68.60533 C 172.473,68.61696 172.45961,68.62494 172.45069,68.63658 C 171.83306,69.44532 171.21899,70.2685 170.63819,71.10533 C 170.60846,71.14795 170.57407,71.18764 170.54444,71.23033 C 169.35811,72.93962 168.25723,74.72427 167.23194,76.54283 C 167.20209,76.59576 167.16788,76.64606 167.13819,76.69908 C 166.11737,78.52129 165.20637,80.40349 164.35694,82.32408 C 160.40023,91.27037 158.32833,101.21041 158.70069,111.63658 C 160.01647,148.47863 191.01487,177.29612 227.85694,175.98033 C 247.16433,175.29079 264.24303,166.44156 275.95069,152.88658 C 275.985,152.84685 276.01021,152.80139 276.04444,152.76158 C 276.51636,152.21259 276.96528,151.66948 277.41944,151.10533 C 277.62857,150.84695 277.8391,150.58559 278.04444,150.32408 C 278.24466,150.0677 278.44164,149.80216 278.63819,149.54283 C 279.07623,148.96794 279.49983,148.41324 279.91944,147.82408 C 279.94945,147.78173 279.98326,147.7415 280.01319,147.69908 C 280.39814,147.15588 280.76922,146.59772 281.13819,146.04283 C 281.16691,145.99941 281.2033,145.96132 281.23194,145.91783 C 282.42108,144.12047 283.52976,142.26696 284.54444,140.35533 L 225.66944,140.35533 C 222.78435,140.39613 220.09265,138.87856 218.63819,136.38658 C 217.36555,134.20609 217.21496,131.55433 218.16944,129.26158 C 218.30578,128.93404 218.45638,128.63558 218.63819,128.32408 C 218.78944,128.06493 218.96135,127.8115 219.13819,127.57408 C 219.35662,127.29417 219.60596,127.00761 219.85694,126.76158 C 220.10792,126.51555 220.35874,126.28396 220.63819,126.07408 C 222.03543,125.02467 223.77248,124.45483 225.57569,124.48033 L 290.48194,124.48033 C 290.49171,124.43857 290.50348,124.39712 290.51319,124.35533 L 290.57569,124.35533 C 291.89667,118.71049 292.51123,112.8004 292.29444,106.73033 C 292.13066,102.14476 291.48727,97.68074 290.45069,93.38658 L 225.66944,93.38658 C 222.78435,93.42738 220.09265,91.90981 218.63819,89.41783 C 217.36555,87.23734 217.21496,84.58559 218.16944,82.29283 C 218.30578,81.96529 218.45638,81.66683 218.63819,81.35533 C 218.78944,81.09618 218.96135,80.84275 219.13819,80.60533 C 219.35662,80.32542 219.60596,80.03886 219.85694,79.79283 C 220.10792,79.5468 220.35874,79.31521 220.63819,79.10533 C 222.03543,78.05592 223.77248,77.48608 225.57569,77.51158 L 284.20069,77.51158 C 284.17849,77.47061 284.16047,77.4275 284.13819,77.38658 L 284.29444,77.38658 C 273.10507,56.736848 251.3966,42.727245 226.60694,42.355325 z' color='#bb99cc44' /></over> <over><fill vector='M 345.13798,42.01115 C 344.57832,42.00827 343.99675,42.02206 343.41923,42.0424 C 326.09295,42.66104 310.5086,49.84796 298.95048,61.1049 C 298.90822,61.14584 298.86763,61.18885 298.82548,61.2299 C 297.49884,62.52871 296.22856,63.88952 295.01298,65.2924 C 294.85869,65.47046 294.69666,65.64398 294.54423,65.82365 C 293.98269,66.48204 293.45536,67.144 292.91923,67.82365 C 292.76314,68.0225 292.60436,68.21679 292.45048,68.4174 C 292.44147,68.42909 292.42823,68.43695 292.41923,68.44865 C 291.79663,69.26142 291.19233,70.07648 290.60673,70.9174 C 290.56885,70.97145 290.51945,71.01949 290.48173,71.07365 C 289.90671,71.9046 289.36329,72.74787 288.82548,73.6049 C 287.67039,75.43352 286.61519,77.32167 285.63798,79.26115 C 285.61065,79.31539 285.57141,79.36308 285.54423,79.4174 C 280.6977,89.10174 278.13171,100.08532 278.54423,111.63615 C 279.8643,148.59872 311.01898,177.29968 347.98173,175.9799 C 363.06106,175.44145 376.82361,169.98596 387.76298,161.2299 C 388.5568,160.59791 389.34319,159.95869 390.10673,159.2924 C 390.15109,159.25349 390.18748,159.20643 390.23173,159.1674 C 392.59931,157.07943 394.83897,154.8443 396.88798,152.44865 C 396.9217,152.40922 396.9481,152.36316 396.98173,152.32365 C 397.64378,151.5496 398.29288,150.75232 398.91923,149.94865 C 398.95134,149.90725 398.98097,149.86512 399.01298,149.82365 C 399.6323,149.02518 400.24249,148.2125 400.82548,147.38615 C 400.86023,147.33661 400.88461,147.27954 400.91923,147.2299 C 401.20945,146.81615 401.48206,146.40037 401.76298,145.9799 C 402.07412,145.51163 402.40106,145.05001 402.70048,144.57365 C 403.24146,143.71744 403.7608,142.83022 404.26298,141.94865 C 404.29269,141.89616 404.32716,141.84497 404.35673,141.7924 C 405.38641,139.97326 406.31205,138.08614 407.16923,136.1674 C 411.16206,127.2297 413.26156,117.28393 412.88798,106.82365 C 412.72334,102.21378 411.80676,97.60815 410.76298,93.2924 L 348.07548,93.2924 C 347.72195,93.33437 347.59352,93.29644 347.23173,93.2924 C 341.7158,93.2924 338.47248,94.67376 336.10673,96.7299 C 335.37784,97.3634 334.70062,98.04941 334.10673,98.7924 C 334.08335,98.82165 334.06739,98.85663 334.04423,98.88615 C 334.03497,98.89769 334.02222,98.90584 334.01298,98.9174 C 333.74657,99.25945 333.49984,99.60761 333.26298,99.9799 C 333.23411,100.02423 333.19768,100.06019 333.16923,100.1049 C 332.93511,100.48127 332.71782,100.86447 332.51298,101.26115 C 332.48488,101.31424 332.4468,101.36401 332.41923,101.4174 C 331.04668,104.0754 330.32691,107.07057 330.20048,109.07365 C 330.20806,109.21732 330.18828,109.36748 330.20048,109.51115 C 330.18381,110.03692 330.5705,111.76731 330.63798,112.0424 C 331.60195,115.96384 334.50286,119.68006 336.54423,121.3549 C 338.75862,123.17066 342.80924,124.49307 346.70048,124.32365 C 349.89912,124.18551 352.98479,123.32216 356.10673,120.26115 C 356.30791,120.06714 356.53014,119.88018 356.73173,119.6674 C 356.76659,119.63002 356.79061,119.58036 356.82548,119.5424 C 356.91767,119.44356 357.01444,119.33282 357.10673,119.2299 C 357.25301,119.06425 357.39766,118.90611 357.54423,118.7299 C 357.74575,118.49121 357.93584,118.23913 358.13798,117.9799 C 358.17239,117.9338 358.22756,117.9003 358.26298,117.8549 C 358.4484,117.60652 358.64356,117.36312 358.85673,117.13615 C 360.63395,115.31241 363.25047,114.25231 365.76298,114.57365 C 368.63443,114.93945 371.04527,116.98582 372.13798,119.6674 C 373.09408,122.01266 372.7788,124.6741 371.48173,126.82365 C 371.4438,126.8865 371.39633,126.9492 371.35673,127.01115 C 371.22805,127.20431 371.09539,127.39058 370.95048,127.57365 C 370.55352,128.09 370.14739,128.59062 369.73173,129.07365 C 369.69786,129.11337 369.67197,129.15915 369.63798,129.19865 C 369.25496,129.6397 368.84788,130.06661 368.45048,130.4799 C 368.41521,130.51692 368.39211,130.5681 368.35673,130.6049 C 367.95877,131.01498 367.54893,131.40966 367.13798,131.7924 C 361.30994,137.22038 354.2432,139.91039 347.66923,140.2924 C 347.54185,140.31814 347.36073,140.27343 347.23173,140.2924 C 346.80611,140.35608 343.49974,140.44688 342.16923,140.2924 C 341.58548,140.22468 337.36709,139.07634 335.82548,138.57365 C 334.07246,138.00178 332.28839,137.29834 330.73173,136.4799 C 327.17585,134.60997 323.76547,132.49001 321.41923,129.69865 C 317.86593,125.46961 316.06289,120.56464 315.07548,115.7299 C 314.9454,115.20139 314.73812,114.58844 314.63798,114.0424 C 314.43391,112.65886 314.21322,111.15826 314.20048,109.7924 C 314.18422,109.36559 314.19831,108.94345 314.20048,108.51115 C 314.2479,107.0071 314.3503,105.48632 314.63798,103.8549 C 314.87728,102.39953 315.45825,100.67428 315.91923,99.19865 C 316.49445,97.35578 317.50114,95.532 318.45048,93.69865 C 318.55555,93.49582 318.75276,93.136 318.98173,92.76115 C 318.99979,92.73189 319.02578,92.69718 319.04423,92.6674 C 319.08152,92.60675 319.09888,92.54293 319.13798,92.4799 C 319.98874,91.10837 321.25734,89.24429 321.41923,89.0424 C 321.49274,88.95044 321.56286,88.85267 321.63798,88.76115 C 326.36903,83.07189 334.20203,78.29425 343.88798,77.44865 C 344.50614,77.30773 345.20713,77.31421 345.85673,77.32365 L 404.73173,77.32365 C 404.70972,77.28301 404.69132,77.23924 404.66923,77.19865 L 404.82548,77.19865 C 393.60038,56.468 371.70091,42.41635 346.82548,42.0424 C 346.25406,42.0336 345.69764,42.01404 345.13798,42.01115 z' color='#bb99cc44'/></over> <over><fill vector='M 434.32043,50.261566 C 428.84905,53.205236 423.85935,56.918766 419.44543,61.199066 C 418.93593,61.69315 418.43893,62.15798 417.94543,62.66782 C 417.89944,62.71506 417.86628,62.7767 417.82043,62.82407 C 417.33803,63.3254 416.84953,63.83928 416.38293,64.35532 C 416.34316,64.39907 416.29758,64.43647 416.25793,64.48032 C 415.78786,65.00296 415.33646,65.53687 414.88293,66.07408 C 414.5,66.52525 414.12891,66.98783 413.75793,67.44908 C 413.64816,67.58623 413.52288,67.71731 413.41418,67.85533 C 413.38191,67.89611 413.3526,67.93948 413.32043,67.98033 C 412.87722,68.54579 412.46426,69.11934 412.03918,69.69908 C 412.01306,69.73453 411.97148,69.75733 411.94543,69.79283 C 411.07382,70.98062 410.23827,72.20411 409.44543,73.44908 C 409.41735,73.49317 409.37966,73.52992 409.35168,73.57408 C 408.97171,74.17691 408.61888,74.8024 408.25793,75.41783 C 408.23165,75.46239 408.19036,75.4982 408.16418,75.54283 C 407.80458,76.15923 407.47307,76.78927 407.13293,77.41783 C 401.6138,87.56023 398.62858,99.26477 399.07043,111.63658 C 400.38623,148.47862 431.35328,177.29612 468.19543,175.98033 C 486.01239,175.34401 501.94688,167.73673 513.50793,155.91783 C 513.97383,155.44374 514.43132,154.96799 514.88293,154.48033 C 514.91814,154.44213 514.94156,154.39361 514.97668,154.35533 C 515.42712,153.86668 515.88462,153.38842 516.32043,152.88658 C 516.3544,152.84727 516.3803,152.80097 516.41418,152.76158 C 516.84884,152.25872 517.27575,151.74601 517.69543,151.23033 C 517.7284,151.18962 517.7563,151.14612 517.78918,151.10533 C 517.84022,151.04232 517.89462,150.98103 517.94543,150.91783 C 518.30949,150.46283 518.65567,150.00734 519.00793,149.54283 C 519.4381,148.97863 519.87682,148.40201 520.28918,147.82408 C 520.32756,147.76995 520.37596,147.72208 520.41418,147.66783 C 520.79402,147.13212 521.14368,146.58991 521.50793,146.04283 C 521.54152,145.99208 521.56822,145.93743 521.60168,145.88658 C 521.67623,145.77395 521.74655,145.65593 521.82043,145.54283 C 522.10442,145.10543 522.39025,144.67461 522.66418,144.23033 C 523.05077,143.60746 523.42278,142.99142 523.78918,142.35533 C 524.14444,141.73858 524.48455,141.10908 524.82043,140.48033 C 524.84897,140.4269 524.88578,140.37759 524.91418,140.32408 L 467.50793,140.32408 C 466.97523,140.35108 466.45358,140.35533 465.91418,140.35533 C 449.44469,140.35534 435.93368,127.61155 434.72668,111.44908 C 434.38189,110.52981 434.17928,109.5553 434.19543,108.54283 L 434.32043,50.261566 z' color='#bb99cc44'/> </over> </node> <node operation='whitebalance'> <params> <param name='high-a-delta'>0.16500000000000001</param> <param name='high-b-delta'>0.20000000000000001</param> <param name='low-a-delta'>0</param> <param name='low-b-delta'>0</param> <param name='saturation'>0.29999999999999999</param> </params> </node> <node operation='translate'> <params> <param name='origin-x'>0</param> <param name='origin-y'>0</param> <param name='filter'>linear</param> <param name='hard-edges'>false</param> <param name='lanczos-width'>3</param> <param name='x'>-150</param> <param name='y'>1</param> </params> </node> <node operation='rotate'> <params> <param name='origin-x'>0</param> <param name='origin-y'>0</param> <param name='filter'>linear</param> <param name='hard-edges'>false</param> <param name='lanczos-width'>3</param> <param name='degrees'>20</param> </params> </node> <node operation='fractal-explorer'> <params> <param name='width'>650</param> <param name='height'>400</param> <param name='fractaltype'>1</param> <param name='xmin'>-0.5</param> <param name='xmax'>0.4</param> <param name='ymin'>-0.5</param> <param name='ymax'>1</param> <param name='iter'>20</param> <param name='cx'>-0.81999999999999995</param> <param name='cy'>0.19500000000000001</param> <param name='redstretch'>0.10000000000000001</param> <param name='greenstretch'>0.10000000000000001</param> <param name='bluestretch'>1</param> <param name='redmode'>0</param> <param name='greenmode'>1</param> <param name='bluemode'>0</param> <param name='redinvert'>false</param> <param name='greeninvert'>false</param> <param name='blueinvert'>true</param> <param name='ncolors'>8192</param> <param name='useloglog'>false</param> </params> </node> </gegl>"

/******************/

static gboolean file_is_gegl_xml (const gchar *path)
{
  gchar *extension;

  extension = strrchr (path, '.');
  if (!extension)
    return FALSE;
  extension++;
  if (extension[0]=='\0')
    return FALSE;
  if (!strcmp (extension, "xml")||
      !strcmp (extension, "XML"))
    return TRUE;
  return FALSE;
}

gint
main (gint    argc,
      gchar **argv)
{
  GeglOptions *o        = NULL;
  GeglNode    *gegl     = NULL;
  gchar       *script   = NULL;
  GError      *err      = NULL;
  gchar       *path_root = NULL;

  o = gegl_options_parse (argc, argv);
  gegl_init (&argc, &argv);

  if (o->xml)
    {
      gchar *tmp = g_malloc(PATH_MAX);
      tmp = getcwd (tmp, PATH_MAX);
      path_root = tmp;
    }
  else if (o->file)
    {
      if (!strcmp (o->file, "-"))  /* read XML from stdin */
        {
          gchar *tmp = g_malloc(PATH_MAX);
          tmp = getcwd (tmp, PATH_MAX);
          path_root = tmp;
        }
      else
        {
          gchar real_path[PATH_MAX];
          gchar *temp1 = g_strdup (o->file);
          gchar *temp2 = g_path_get_dirname (temp1);
          path_root = g_strdup (realpath (temp2, real_path));
          g_free (temp1);
          g_free (temp2);
        }
    }

  if (o->xml)
    {
      script = g_strdup (o->xml);
    }
  else if (o->file)
    {
      if (!strcmp (o->file, "-"))  /* read XML from stdin */
        {
#         define  buf_size 128
          gchar buf[buf_size];
          GString *acc = g_string_new ("");

          while (fgets (buf, buf_size, stdin))
            {
              g_string_append (acc, buf);
            }
          script = g_string_free (acc, FALSE);
        }
      else if (file_is_gegl_xml (o->file))
        {
          g_file_get_contents (o->file, &script, NULL, &err);
          if (err != NULL)
            {
              g_warning ("Unable to read file: %s", err->message);
            }
        }
      else
        {
          gchar *leaked_string = g_malloc (strlen (o->file) + 5);
          GString *acc = g_string_new ("");

            {gchar *file_basename;
             gchar *tmp;
             tmp=g_strdup (o->file);
             file_basename = g_path_get_basename (tmp);

             g_string_append (acc, "<gegl><load path='");
             g_string_append (acc, file_basename);
             g_string_append (acc, "'/></gegl>");

             g_free (tmp);
            }

          script = g_string_free (acc, FALSE);

          leaked_string[0]='\0';
          strcat (leaked_string, o->file);
          strcat (leaked_string, ".xml");
          /*o->file = leaked_string;*/
        }
    }
  else
    {
      if (o->rest)
        {
          script = g_strdup ("<gegl></gegl>");
        }
      else
        {
          script = g_strdup (DEFAULT_COMPOSITION);
        }
    }

  gegl = gegl_node_new_from_xml (script, path_root);

  if (o->rest)
    {
      GeglNode *proxy;
      GeglNode *iter;

      gchar **operation = o->rest;
      proxy = gegl_node_get_output_proxy (gegl, "output");
      iter = gegl_node_get_producer (proxy, "input", NULL);

      while (*operation)
        {
          GeglNode *new = gegl_node_new_child (gegl, "operation", *operation, NULL);
          if (iter)
            {
              gegl_node_link_many (iter, new, proxy, NULL);
            }
          else
            {
              gegl_node_link_many (new, proxy, NULL);
            }
          iter = new;
          operation++;
        }
    }

  switch (o->mode)
    {
      case GEGL_RUN_MODE_EDITOR:
#if HAVE_GTK
          gtk_init (&argc, &argv);
          editor_main (gegl, o);
#endif
          return 0;
        break;
      case GEGL_RUN_MODE_XML:
          g_print (gegl_node_to_xml (gegl, path_root));
          return 0;
        break;
      case GEGL_RUN_MODE_DOT:
          g_print (gegl_to_dot (gegl));
          return 0;
        break;
      case GEGL_RUN_MODE_PNG:
        {
          GeglNode *output = gegl_node_new_child (gegl,
                               "operation", "png-save",
                               "path", o->output,
                               NULL);
          gegl_node_connect_from (output, "input", gegl_node_get_output_proxy (gegl, "output"), "output");
          gegl_node_process (output);
          g_object_unref (output);
        }
        break;
      case GEGL_RUN_MODE_HELP:
        break;
      default:
        break;
    }

  g_free (o);
  g_object_unref (gegl);
  g_free (script);
  g_clear_error (&err);
  g_free (path_root);
  gegl_exit ();
  return 0;
}
