/* @(#)mtrees.c	1.2 1/16/90 */

#include <stdio.h>
#include "mtrees.h"

/* tree node allocator */
Mem treeAlloc = (Mem)NULL;

/* initialize tree memory allocator to malloc in blocks of num treenodes */
void treeInit(num)
     int num;
{
    treeAlloc = memNew(ALIGNSIZE(sizeof (struct treenode_s)), num);
}

static TreeNode NodeNew(stuff)
     Ptr stuff;
{
    TreeNode newnode;
    
    newnode = (TreeNode)memBlock(treeAlloc);
    newnode->node = stuff;
    newnode->left = newnode->right = (TreeNode)NULL;
    newnode->bal = EQ;
    return newnode;
}

static void NodeFree(node, fitem)
     TreeNode node;
     F_VOID fitem;
{
    if (fitem) (*fitem)(node->node);
    memFreeBlock((Ptr)node, treeAlloc);
}

/* return an empty binary tree. exit if malloc fails. */
Tree treeNew()
{
    Tree new;
  
    new = (Tree)memBlock(treeAlloc);
    *new = (TreeNode)NULL;
    return new;
}

/* when treeRight(root) is right high this will restore balance */
static void AVLRotateLeft(root)
     Tree root;
{
    TreeNode temp;
    
    temp = treeRight(root);
    treeRight(root) = temp->left;
    temp->left = *root;
    *root = temp;
}

/* when treeLeft(root) is left high this will restore balance */
static void AVLRotateRight(root)
     Tree root;
{
    TreeNode temp;

    temp = treeLeft(root);
    treeLeft(root) = temp->right;
    temp->right = *root;
    *root = temp;
}

/* when treeRight(root) is left high this will restore balance */
static void AVLDRRight(root)
     Tree root;
{
    TreeNode Rt, RLt;
    
    Rt = treeRight(root);
    RLt = treeLeft(&Rt);
    switch (RLt->bal) {
    case EQ:
	treeBal(root) = EQ;
	Rt->bal = EQ;
	break ;
    case LH:
	treeBal(root) = EQ;
	Rt->bal = RH;
	break ;
    case RH:
	treeBal(root) = LH;
	Rt->bal = EQ;
	break ;
    }
    RLt->bal = EQ;
    AVLRotateRight(&Rt);
    treeRight(root) = Rt;
    AVLRotateLeft(root);
}

/* when treeLeft(root) is right high this will restore balance */
static void AVLDRLeft(root)
     Tree root;
{
    TreeNode Lt, LRt;
    
    Lt = treeLeft(root);
    LRt = treeRight(&Lt);
    switch (LRt->bal) {
    case EQ:
	treeBal(root) = EQ;
	Lt->bal = EQ;
	break ;
    case RH:
	treeBal(root) = EQ;
	Lt->bal = LH;
	break ;
    case LH:
	treeBal(root) = LH;
	Lt->bal = EQ;
	break ;
    }
    LRt->bal = EQ;
    AVLRotateLeft(&Lt);
    treeLeft(root) = Lt;
    AVLRotateRight(root);
}

/* call this when the right subtree of root becomes unbalanced */
static void AVLRightBalance(root)
     Tree root;
{
    TreeNode Rt;
    
    Rt = treeRight(root);
    switch (Rt->bal) {
    case RH:
	treeBal(root) = EQ;
	Rt->bal = EQ;
	AVLRotateLeft(root);
	break ;
    case LH:
	AVLDRRight(root);
	break ;
    }
    return ;
}

/* call this when the left subtree of root becomes unbalanced */
static void AVLLeftBalance(root)
     Tree root;
{
    TreeNode Lt;
    
    Lt = treeLeft(root);
    switch (Lt->bal) {
    case LH:
	treeBal(root) = EQ;
	Lt->bal = EQ;
	AVLRotateRight(root);
	break ;
    case RH:
	AVLDRLeft(root);
	break ;
    }
    return ;
}

static Boolean AVLInsert(stuff, root, test, insert, placed)
     Ptr stuff;
     Tree root;
     F_INT test;
     F_PTR insert;
     Ptr *placed;
{
    int cmp;
    
    if (*root == (TreeNode)NULL) {
	*placed = (insert ? (*insert)(stuff, (Ptr)NULL) : stuff);
	*root = NodeNew(*placed);
	return TRUE;
    }
    else {
	/* replace contents */
	if ((cmp = (*test)(stuff, (*root)->node)) == 0) {
	    (*root)->node = 
	      *placed = (insert ? (*insert)(stuff, (*root)->node) :
			 (*root)->node);
	}
	/* switch to left branch */
	else if (cmp < 0) {
	    if (AVLInsert(stuff, &((*root)->left), test, insert, placed)) {
		switch ((*root)->bal) {
		case LH:
		    AVLLeftBalance(root);
		    return FALSE;
		case EQ:
		    (*root)->bal = LH;
		    return TRUE;
		default:
		    (*root)->bal = EQ;
		    return FALSE;
		}
	    }
	}
	/* switch to right branch */
	else {
	    if (AVLInsert(stuff, &((*root)->right), test, insert, placed)) {
		switch ((*root)->bal) {
		case LH:
		    (*root)->bal = EQ;
		    return FALSE;
		case EQ:
		    (*root)->bal = RH;
		    return TRUE;
		default:
		    AVLRightBalance(root);
		    return FALSE;
		}
	    }
	}
    }
    return FALSE;
}

/* insert stuff into tree
 *
 * test is a function of two args and orders the tree.  It should
 * return a positive, negative, or zero value depending on whether its
 * first argument is greater than, less than, or equal to its second
 * argument.
 *
 * If insert is not NULL then it is used to deterimine what to place in
 * the node.  The result of insert(stuff, previous contents) is stored
 * in the node and returned by treeInsert.  If the node is empty, the
 * previous contents parameter is NULL.
 *
 * If insert is NULL then stuff is inserted into the tree only if it is
 * not already in the tree. */
Ptr treeInsert(stuff, tree, test, insert)
     Ptr stuff;
     Tree tree;
     F_INT test;
     F_PTR insert;
{
    Ptr ret;
    
    (void)AVLInsert(stuff, tree, test, insert, &ret);
    return ret;
}

/* return a subtree of tree containing stuff in the root node
 *
 * If stuff is not in the tree NULL is returned.
 * test is the same function used to insert items. */
Tree treeFind(stuff, tree, test)
     Ptr stuff;
     Tree tree;
     F_INT test;
{
    Tree temp;
    int comp;
    
    if (temp = tree) {
	while (*temp) {
	    if ((comp = (*test)(stuff, (*temp)->node)) == 0) return temp;
	    temp = (comp < 0 ? &((*temp)->left) : &((*temp)->right));
	}
    }
    return (Tree)NULL;
}

/* visit the node of the tree in order (left, node, right)
 *
 * At each node, the call fcn(item at node, arg) is made. */
void treeInOrder(tree, fcn, arg)
     Tree tree;
     F_VOID fcn;
     Ptr arg;
{
    if (tree && *tree) {
	treeInOrder(&(*tree)->left, fcn, arg);
	(*fcn)((*tree)->node, arg);
	treeInOrder(&(*tree)->right, fcn, arg);
    }
}

void treePostOrder(tree, fcn, arg)
     Tree tree;
     F_VOID fcn;
     Ptr arg;
{
    if (tree && *tree) {
	treePostOrder(&(*tree)->left, fcn, arg);
	treePostOrder(&(*tree)->right, fcn, arg);
	if (fcn) (*fcn)((*tree)->node, arg);
    }
}

void treePreOrder(tree, fcn, arg)
     Tree tree;
     F_VOID fcn;
     Ptr arg;
{
    if (tree && *tree) {
	if (fcn) (*fcn)((*tree)->node, arg);
	treePreOrder(&(*tree)->left, fcn, arg);
	treePreOrder(&(*tree)->right, fcn, arg);
    }
}

static void treeIntClear(tree, func)
     Tree tree;
     F_VOID func;
{
    if (tree && *tree) {
	treeIntClear(&(*tree)->left, func);
	treeIntClear(&(*tree)->right, func);
	NodeFree(*tree, func);
    }
}

void treeClear(tree, fitem)
     Tree tree;
     F_VOID fitem;
{
    treeIntClear(tree, fitem);
    *tree = (TreeNode)NULL;
}

void treeFree(tree, fitem)
     Tree tree;
     F_VOID fitem;
{
    treeIntClear(tree, fitem);
    if (tree) memFreeBlock((Ptr)tree, treeAlloc);
}
