#ifndef _BUF_LLHASH_H
#define _BUF_LLHASH_H

/* Taken from http://www.purists.org/llhash/, which describes the file as
   Public Domain. */

#include <stdlib.h>

typedef struct _buf_hash_struct {
	void *glkobj;
	char *buf;
} buf_hash_struct;

#define BUF_LLHITEM buf_hash_struct
#define BUF_LLHKEY(p) ((int) (p).glkobj)
#define BUF_LLHSLOTS 19	/* I don't expect many entries */

int buf_hash(int p)
{
	/* swap lo and hi bytes to get rid of bunching due to word alignment */
	int lo = p << 16;
	int hi = p >> 16;
	int h = abs(lo + hi);
	return h;
}

/*
 *  Hash table item structure
 */
struct buf_item
{
        BUF_LLHITEM d;
        struct buf_item *next;
};

/*
 *  The hash table itself
 */
typedef struct buf_item *BUF_LLHTABLE[BUF_LLHSLOTS];

/*
 *  Hash function
 *
 *  Parameters:
 *
 *    key         Some hash table item's key
 *
 *  Return values:
 *
 *    >= 0        The item's hash value
 *
 *    <= -1       Error, the hash value cannot be calculated
 */
typedef int BUF_LLHFUNC(int key);

/*
 *  buf_llhinit: Initialize the hash table
 *
 *  Parameters:
 *
 *    t           Hash table (duh!)
 */
void
buf_llhinit(BUF_LLHTABLE t)
{
	int i = BUF_LLHSLOTS;

	while (i --) {
		t[i] = NULL;
	}
}

/*
 *  buf_llhlookup: Access an item
 *
 *  Parameters:
 *
 *    t           Hash table
 *
 *    key         Value uniquely identifying the item to be looked for
 *
 *  Return values:
 *
 *    non-NULL    The item in question was found; the return value is
 *                a pointer to the corresponding record in the hash table.
 *
 *    NULL        The item was not found in the hash table. Either it just
 *                isn't in there, or the hash function returned an error.
 */
BUF_LLHITEM *
buf_llhlookup(BUF_LLHTABLE t, int key)
{
	int slot;
	struct buf_item *tmp;

	if ((slot = buf_hash(key) % BUF_LLHSLOTS) < 0) {
		return NULL;
	};
	tmp = t[slot];
	while (tmp) {
		if (BUF_LLHKEY(tmp->d) == key) {
			return &tmp->d;
		}
		tmp = tmp->next;
	}
	return NULL;
}

/*
 *  buf_llhinsert: Insert an item into the hash table
 *
 *  Parameters:
 *
 *    t           Hash table
 *
 *    d           Item to be inserted
 *
 *  Return values:
 *
 *    nonzero     The item was inserted successfully
 *
 *    zero        The item could not be inserted. Either the function could
 *                not allocate the amount of memory necessary to store it,
 *                or the hash table already contains an item with the same
 *                key, or the hash function returned an error.
 *
 *  Note:
 *
 *    If you know for sure that key values are in fact unique identifiers,
 *    that is, that the calling functions will never try to make the hash
 *    table contain two items with the same key at the same time, you can
 *    speed up the function considerably by deleting the first statement.
 */
int
buf_llhinsert(BUF_LLHTABLE t, BUF_LLHITEM d)
{
	int slot;
	struct buf_item *tmp;

	/* delete this line to insert items in constant time: */
	if (buf_llhlookup(t, BUF_LLHKEY(d))) return 0;

	if ((slot = buf_hash(BUF_LLHKEY(d)) % BUF_LLHSLOTS) < 0 
		|| (tmp = malloc(sizeof(struct buf_item))) == NULL) {
		return 0;
	}
	tmp->d = d;
	tmp->next = t[slot];
	t[slot] = tmp;
	return 1;
}

/*
 *  buf_llhremove: Remove an item from the hash table
 *
 *  Parameters:
 *
 *    t           Hash table
 *
 *    key         Value uniquely identifying the item to be looked for
 *
 *  Return values:
 *
 *    nonzero     The item was removed successfully.
 *
 *    zero        The item could not be removed. Either it just wasn't
 *                found in the hash table, or the hash function returned
 *                an error.
 */
int
buf_llhremove(BUF_LLHTABLE t, int key)
{
	int slot;
	struct buf_item **tmp, *kill;

	if ((slot = buf_hash(key) % BUF_LLHSLOTS) < 0) {
		return 0;
	}
	tmp = &t[slot];
	while (*tmp) {
		if (BUF_LLHKEY((*tmp)->d) == key) {
			kill = *tmp;
			*tmp = (*tmp)->next;
			free(kill);
			return 1;
		}
		tmp = &((*tmp)->next);
	}
	return 0;
}

#endif // _BUF_LLHASH_H
