#ifndef _REF_LLHASH_H
#define _REF_LLHASH_H

/* Taken from http://www.purists.org/llhash/, which describes the file as
   Public Domain. */

#include <stdlib.h>

typedef struct _ref_hash_struct {
	char *buf;
	int refcount;
} ref_hash_struct;

#define REF_LLHITEM ref_hash_struct
#define REF_LLHKEY(p) ((int) (p).buf)
#define REF_LLHSLOTS 19	/* I don't expect many entries */

int ref_hash(int p)
{
	/* swap lo and hi bytes to get rid of bunching due to word alignment */
	int lo = p << 16;
	int hi = p >> 16;
	int h = abs(lo + hi);
	return h;
}

/*
 *  Hash table item structure
 */
struct ref_item
{
        REF_LLHITEM d;
        struct ref_item *next;
};

/*
 *  The hash table itself
 */
typedef struct ref_item *REF_LLHTABLE[REF_LLHSLOTS];

/*
 *  Hash function
 *
 *  Parameters:
 *
 *    key         Some hash table item's key
 *
 *  Return values:
 *
 *    >= 0        The item's hash value
 *
 *    <= -1       Error, the hash value cannot be calculated
 */
typedef int REF_LLHFUNC(int key);

/*
 *  ref_llhinit: Initialize the hash table
 *
 *  Parameters:
 *
 *    t           Hash table (duh!)
 */
void
ref_llhinit(REF_LLHTABLE t)
{
	int i = REF_LLHSLOTS;

	while (i --) {
		t[i] = NULL;
	}
}

/*
 *  ref_llhlookup: Access an item
 *
 *  Parameters:
 *
 *    t           Hash table
 *
 *    key         Value uniquely identifying the item to be looked for
 *
 *  Return values:
 *
 *    non-NULL    The item in question was found; the return value is
 *                a pointer to the corresponding record in the hash table.
 *
 *    NULL        The item was not found in the hash table. Either it just
 *                isn't in there, or the hash function returned an error.
 */
REF_LLHITEM *
ref_llhlookup(REF_LLHTABLE t, int key)
{
	int slot;
	struct ref_item *tmp;

	if ((slot = ref_hash(key) % REF_LLHSLOTS) < 0) {
		return NULL;
	};
	tmp = t[slot];
	while (tmp) {
		if (REF_LLHKEY(tmp->d) == key) {
			return &tmp->d;
		}
		tmp = tmp->next;
	}
	return NULL;
}

/*
 *  ref_llhinsert: Insert an item into the hash table
 *
 *  Parameters:
 *
 *    t           Hash table
 *
 *    d           Item to be inserted
 *
 *  Return values:
 *
 *    nonzero     The item was inserted successfully
 *
 *    zero        The item could not be inserted. Either the function could
 *                not allocate the amount of memory necessary to store it,
 *                or the hash table already contains an item with the same
 *                key, or the hash function returned an error.
 *
 *  Note:
 *
 *    If you know for sure that key values are in fact unique identifiers,
 *    that is, that the calling functions will never try to make the hash
 *    table contain two items with the same key at the same time, you can
 *    speed up the function considerably by deleting the first statement.
 */
int
ref_llhinsert(REF_LLHTABLE t, REF_LLHITEM d)
{
	int slot;
	struct ref_item *tmp;

	/* delete this line to insert items in constant time: */
	if (ref_llhlookup(t, REF_LLHKEY(d))) return 0;

	if ((slot = ref_hash(REF_LLHKEY(d)) % REF_LLHSLOTS) < 0 
		|| (tmp = malloc(sizeof(struct ref_item))) == NULL) {
		return 0;
	}
	tmp->d = d;
	tmp->next = t[slot];
	t[slot] = tmp;
	return 1;
}

/*
 *  ref_llhremove: Remove an item from the hash table
 *
 *  Parameters:
 *
 *    t           Hash table
 *
 *    key         Value uniquely identifying the item to be looked for
 *
 *  Return values:
 *
 *    nonzero     The item was removed successfully.
 *
 *    zero        The item could not be removed. Either it just wasn't
 *                found in the hash table, or the hash function returned
 *                an error.
 */
int
ref_llhremove(REF_LLHTABLE t, int key)
{
	int slot;
	struct ref_item **tmp, *kill;

	if ((slot = ref_hash(key) % REF_LLHSLOTS) < 0) {
		return 0;
	}
	tmp = &t[slot];
	while (*tmp) {
		if (REF_LLHKEY((*tmp)->d) == key) {
			kill = *tmp;
			*tmp = (*tmp)->next;
			free(kill);
			return 1;
		}
		tmp = &((*tmp)->next);
	}
	return 0;
}

#endif // _REF_LLHASH_H
