/*
	HDALLEG.C

	Non-portable functions specific to djgpp/Allegro (32-bit
	build)

	Copyright (c) 1995-2006 by Kent Tessman
*/


#include "heheader.h"
#include "hdheader.h"

#define line unused_line
#include "allegro.h"
#undef line

#include <bios.h>
//#include <signal.h>     /* interrupt processing */

#define window window_unused
#include <conio.h>
#undef window


/* Function prototypes: */
void debug_breakhandler(void);

/* From healleg.c: */
void UpdateScreen(void);
void ConstrainCursor(void);
void rectangle(int left, int top, int right, int bottom);

/* From hesound.c: */
void SuspendAudio(void);
void ResumeAudio(void);


struct old_window_data
{
	int top, bottom, left, right;
	int textcolor; long backcolor;
	int xpos, ypos;
};

/* the four different screens for flipping between: */
BITMAP *game_screen = NULL, *debugger_screen = NULL,
	 *help_screen = NULL, *auxiliary_screen = NULL;
BITMAP *active_screen_buffer;

/* from healleg.c */
extern BITMAP *screen_buffer;
extern int text_windowleft, text_windowtop, text_windowright,
	text_windowbottom;
extern int current_text_color, current_back_color,
	current_text_col, current_text_row;
extern int current_text_font;
extern int display_width, display_height;
extern int FONT_HEIGHT, FONT_WIDTH;
extern char screen_needs_repainting;
extern char override_latin1_conversion;

/* from hejpeg.c */
extern int HUGO_PALETTE[];

/* Because they couldn't possibly have come up with a more difficult
   way of reading ALT-key keystrokes from the PC keyboard:
*/
#define exttoascii(key) \
	switch (key) { \
		case 30:  {key = 'A'; break;}  case 48:  {key = 'B'; break;}\
		case 46:  {key = 'C'; break;}  case 32:  {key = 'D'; break;}\
		case 18:  {key = 'E'; break;}  case 33:  {key = 'F'; break;}\
		case 34:  {key = 'G'; break;}  case 35:  {key = 'H'; break;}\
		case 23:  {key = 'I'; break;}  case 36:  {key = 'J'; break;}\
		case 37:  {key = 'K'; break;}  case 38:  {key = 'L'; break;}\
		case 50:  {key = 'M'; break;}  case 49:  {key = 'N'; break;}\
		case 24:  {key = 'O'; break;}  case 25:  {key = 'P'; break;}\
		case 16:  {key = 'Q'; break;}  case 19:  {key = 'R'; break;}\
		case 31:  {key = 'S'; break;}  case 20:  {key = 'T'; break;}\
		case 22:  {key = 'U'; break;}  case 47:  {key = 'V'; break;}\
		case 17:  {key = 'W'; break;}  case 45:  {key = 'X'; break;}\
		case 21:  {key = 'Y'; break;}  case 44:  {key = 'Z'; break;}}


enum EXTENDED
{
	/* If a function key is pressed, the first byte of the two-byte
	 * code will return one of the following:
	 */
	extINTRO  = 0,		/* For keys common to all keyboards    */
	extINTRO2 = 0xE0,       /* For keypad and other keys unique to */
				/*   101-key keyboard		       */

	/* Second byte will be one of the following extended codes: */

	/* Numeric keypad */
	extUP    = 72,  extDOWN  = 80,  extLEFT  = 75,  extRIGHT = 77,
	extPGUP  = 73,  extPGDN  = 81,  extHOME  = 71,  extEND   = 79,
	extINS   = 82,  extDEL   = 83,

	extCTRL_PRTSC = 114,
	extCTRL_LEFT  = 115, extCTRL_RIGHT = 116,
	extCTRL_PGUP  = 132, extCTRL_PGDN  = 118,
	extCTRL_HOME  = 119, extCTRL_END   = 117,

	NullKey = 3,  extSH_Tab = 15,

	/* ALT+letter keys */
	extALT_A = 30, extALT_B = 48, extALT_C = 46, extALT_D = 32,
	extALT_E = 18, extALT_F = 33, extALT_G = 34, extALT_H = 35,
	extALT_I = 23, extALT_J = 36, extALT_K = 37, extALT_L = 38,
	extALT_M = 50, extALT_N = 49, extALT_O = 24, extALT_P = 25,
	extALT_Q = 16, extALT_R = 19, extALT_S = 31, extALT_T = 20,
	extALT_U = 22, extALT_V = 47, extALT_W = 17, extALT_X = 45,
	extALT_Y = 21, extALT_Z = 44,

	/* extALT+number keys */
	extALT_1 = 120,  extALT_2, extALT_3, extALT_4, extALT_5,
	extALT_6,        extALT_7, extALT_8, extALT_9, extALT_0,

	extALT_minus = 130,  extALT_equals,

	/* Function keys */
	extF1  = 59,  extF2, extF3, extF4, extF5,
	extF6,        extF7, extF8, extF9, extF10,
	extF11 = 133, extF12,

	/* SHIFT+function keys */
	extSH_F1 = 84,  extSH_F2, extSH_F3, extSH_F4, extSH_F5,
	extSH_F6,       extSH_F7, extSH_F8, extSH_F9, extSH_F10,
	extSH_F11= 137, extSH_F12,

	/* CTRL+function keys */
	extCTRL_F1 = 94, extCTRL_F2,  extCTRL_F3, extCTRL_F4,
	extCTRL_F5,      extCTRL_F6,  extCTRL_F7, extCTRL_F8,
	extCTRL_F9,      extCTRL_F10, extCTRL_F11 = 137,
	extCTRL_F12,

	/* ALT+function keys */
	extALT_F1 = 104,  extALT_F2, extALT_F3, extALT_F4, extALT_F5,
	extALT_F6,        extALT_F7, extALT_F8, extALT_F9, extALT_F10,
	extALT_F11 = 139, extALT_F12,
};


/* Keystroke data: */
#define UP_ARROW	11
#define	DOWN_ARROW      10
#define LEFT_ARROW	 8
#define RIGHT_ARROW	21
#define ENTER_KEY	13
#define ESCAPE_KEY	27
#define TAB_KEY		 9
#define DELETE_KEY     130
#define BACKSPACE_KEY  132

#define ALT_KEY		0x100
#define ALT_PRESSED	0x008
#define EXTENDED_KEY	0x100


/* Context help brings up the help topic for the currently active window,
   menu, input box, etc.
*/
#define FUNCTION_CONTEXTHELP	extF1

#define FUNCTION_TOPICHELP	extSH_F1
#define FUNCTION_SHORTCUTKEYS   extCTRL_F1

#define FUNCTION_OBJTREE	extF2
#define FUNCTION_MOVEOBJ	extSH_F2

#define FUNCTION_SEARCH		extF3

#define FUNCTION_EXIT		extALT_F4

#define FUNCTION_GO     	extF5
#define FUNCTION_FINISH		extSH_F5

#define FUNCTION_SWITCH_WINDOW	extF6

#define FUNCTION_SETVALUE	extF7

#define FUNCTION_STEP   	extF8
#define FUNCTION_STEPBACK	extSH_F8

#define FUNCTION_BREAKPOINT	extF9
#define FUNCTION_WATCHVALUE	extSH_F9

#define FUNCTION_STEPOVER	extF10
#define FUNCTION_SKIPNEXT	extSH_F10


/* debug_breakhandler

	Called whenever CTRL-C or CTRL-BREAK is pressed in order
	not to break out of the program, but only to return control to the
	Debugger.
*/

void debug_breakhandler(void)
{
	debugger_interrupt = true;
}


/* debug_getevent

	Is called in various debugger states to read an event.  (This
	djgpp version relays only keystrokes; a more complex port could
	report, for example, mouse events.)

	Most of these events should be self-explanatory.  The ALT key
	activates/deactivates the menubar.  A SHORTCUT_KEY action is
	a special keypress.  This is machine-specific:  for the PC,
	shortcut keys are ALT-key combinations.  (Other PC shortcuts
	are mapped to the function keys F1-F10.)
*/

void debug_getevent(void)
{
	int key = 0, old_key;

	if (screen_needs_repainting) UpdateScreen();

GetNewEvent:

	event.action = 0;
	event.object = 0;

	while (!kbhit())
	{
	        /* if ALT alone is pressed */
		if (_bios_keybrd(_KEYBRD_SHIFTSTATUS)&ALT_PRESSED && !kbhit())
		{
			/* wait for ALT to be released */
			while (_bios_keybrd(_KEYBRD_SHIFTSTATUS)&ALT_PRESSED);

			/* if released alone (i.e., not an ALT-key combo) */
			if (!kbhit()) key = ALT_KEY;
			break;
		}
	}

	if (!key) key = hugo_getkey();

	switch (key)
	{
		case ALT_KEY:
		{
			if (active_menu_heading==MENU_INACTIVE)
				event.action = OPEN_MENUBAR;
			else
				event.action = CANCEL;
			return;
		}
		case UP_ARROW:
			{event.action = MOVE; event.object = UP; break;}
		case DOWN_ARROW:
			{event.action = MOVE; event.object = DOWN; break;}
		case RIGHT_ARROW:
			{event.action = MOVE; event.object = RIGHT; break;}
		case LEFT_ARROW:
			{event.action = MOVE; event.object = LEFT; break;}
		case ENTER_KEY:
ReturnSelect:
			{event.action = SELECT; break;}
		case ESCAPE_KEY:
			{event.action = CANCEL; break;}
		case TAB_KEY:
			{event.action = TAB; break;}
		case DELETE_KEY:
			{event.action = DELETE; break;}

		default:
		{
			/* convert extended key to regular keypress */
                        if (key > EXTENDED_KEY)
			{
				/* A different conversion calculation than the QuickC
				   equivalent, which uses getch() instead of a BIOS
				   call (see healleg.c and hequickc.c)
				*/
				key/=EXTENDED_KEY+1;

				old_key = key;
				exttoascii(key);

				/* if not converted, i.e., a function key */
				if (key==old_key)
				{
					switch(key)
				  	{

				/* Extended-keyboard keys: */

						case extPGUP:
							{event.object = PAGEUP;
ReturnMove:
							event.action = MOVE;
							return;}
						case extPGDN:
							{event.object = PAGEDOWN;
							goto ReturnMove;}
						case extHOME:
							{event.object = HOME;
							goto ReturnMove;}
						case extCTRL_HOME:
							{event.object = START;
							goto ReturnMove;}
						case extEND:
							{event.object = END;
							goto ReturnMove;}
						case extCTRL_END:
							{event.object = FINISH;
							goto ReturnMove;}

				/* Various function shortcut keys: */

                                                case FUNCTION_EXIT:
							{event.object = MENU_FILE + FILE_EXIT;
							goto ReturnSelect;}
						case FUNCTION_GO:
							{event.object = MENU_RUN + RUN_GO;
							goto ReturnSelect;}
						case FUNCTION_FINISH:
							{event.object = MENU_RUN + RUN_FINISH;
							goto ReturnSelect;}
						case FUNCTION_SEARCH:
							{event.object = MENU_DEBUG + DEBUG_SEARCH;
							goto ReturnSelect;}
						case FUNCTION_WATCHVALUE:
							{event.object = MENU_DEBUG + DEBUG_WATCH;
							goto ReturnSelect;}
						case FUNCTION_SETVALUE:
							{event.object = MENU_DEBUG + DEBUG_SET;
							goto ReturnSelect;}
						case FUNCTION_BREAKPOINT:
							{event.object = MENU_DEBUG + DEBUG_BREAKPOINT;
							goto ReturnSelect;}
						case FUNCTION_OBJTREE:
							{event.object = MENU_DEBUG + DEBUG_OBJTREE;
							goto ReturnSelect;}
						case FUNCTION_MOVEOBJ:
							{event.object = MENU_DEBUG + DEBUG_MOVEOBJ;
							goto ReturnSelect;}
						case FUNCTION_STEP:
							{event.object = MENU_RUN + RUN_STEP;
							goto ReturnSelect;}
						case FUNCTION_STEPOVER:
							{event.object = MENU_RUN + RUN_STEPOVER;
							goto ReturnSelect;}
						case FUNCTION_SKIPNEXT:
							{event.object = MENU_RUN + RUN_SKIP;
							goto ReturnSelect;}
						case FUNCTION_STEPBACK:
							{event.object = MENU_RUN + RUN_STEPBACK;
							goto ReturnSelect;}
						case FUNCTION_SHORTCUTKEYS:
							{event.object = MENU_HELP + HELP_KEYS;
							goto ReturnSelect;}
						case FUNCTION_TOPICHELP:
							{event.object = MENU_HELP + HELP_TOPIC;
							goto ReturnSelect;}

						case FUNCTION_SWITCH_WINDOW:
							{event.action = SWITCH_WINDOW;
							return;}

						case FUNCTION_CONTEXTHELP:
							{SearchHelp(context_help);
							goto GetNewEvent;}

						default:  key = 0;
				  	}
				}
				event.action = SHORTCUT_KEY;
			}
			else event.action = KEYPRESS;

			event.object = toupper(key);

			break;
	       	}
	}
}


/* debug_input

	Called by InputBox to get an input of maxlen characters.  Must
	return an empty string ("") if the input is cancelled by pressing
	Escape.
*/

char *debug_input(int x, int y, int maxlen, char *def)
{
	char blank_entry = 0;
	static char i[255];
	int pos, k;
	int text_color = current_text_color;

	if (maxlen>251) maxlen = 251;

	/* print the default entry */
ReprintEntry:
	if (!blank_entry)
	{
		pos = strlen(def);
		strcpy(i, def);
	}
	else
		pos = 0;

	memset(i+pos, ' ', maxlen-pos);
	i[maxlen] = '\0';
	debug_settextpos(x, y);
	debug_print(i);

	event.action = 0;

	/* endless loop */
	while (true)
	{
		/* Print a cursor */
		debug_settextpos(x+pos, y);
		debug_settextcolor(15);
		debug_print("_");
		debug_settextcolor(text_color);

		debug_settextpos(x+pos, y);

		switch (k = hugo_getkey())
		{
			case LEFT_ARROW:
			case BACKSPACE_KEY:
			case DELETE_KEY:
			{
				if (pos>0)
				{
					debug_settextpos(x+pos-1, y);
					/* Print two spaces to erase last character
					   and cursor
					*/
					debug_print("  ");
					i[--pos] = '\0';
				}
				break;
			}
			case ENTER_KEY:
			{
				event.action = SELECT;
				i[pos] = '\0';
				return i;
			}
			case ESCAPE_KEY:
			{
				if (pos>0)
				{
					blank_entry = true;
					goto ReprintEntry;
				}
				event.action = CANCEL;
				return "";
			}
			default:
			{
				if (pos+1<maxlen && k>=' ' && k<128)
				{
					i[pos++] = (char)k;
					sprintf(line, "%c", k);
					debug_print(line);
				}
			}
		}
	}
}


/* debug_getinvocationpath

	Systems for which main()'s argv[0] does not contain the full
	path of the debugger executable will have to add the current
	path here.
*/

void debug_getinvocationpath(char *argv)
{
	strcpy(invocation_path, argv);
}


/* debug_print

	Prints the supplied string at the current cursor position
	on the debugger screen in the current color.
*/

void debug_print(char *a)
{
	override_latin1_conversion = true;	/* to allow symbols */
	hugo_print(a);
	override_latin1_conversion = false;
	screen_needs_repainting = true;
}


/* debug_settextcolor and debug_setbackcolor */

void debug_settextcolor(int c)
{
	hugo_settextcolor(c);
}

void debug_setbackcolor(int c)
{
	hugo_setbackcolor(c);
}


/* debug_hardcopy

	Sends the supplied string to the printer as a single line.
	The string does not end with a newline character.
*/

void debug_hardcopy(FILE *printer, char *a)
{
	if (fprintf(printer, a)<0) device_error = true;
	if (fprintf(printer, "\n")<0) device_error = true;
}


/* debug_switchscreen

	Switches the text display/output to either DEBUGGER, GAME, 
	HELP, or AUXILIARY, as supplied.  The cursor should also 
	be switched off or on (if applicable).
*/

extern int forbid, mp_volume;	/* see healleg.c */
extern char audio_suspended;

void debug_switchscreen(int a)
{
	int last_active_screen;
	static struct old_window_data debug_win, game_win;
	static int last_text_font;

	last_active_screen = active_screen;

	/* If there's something onscreen to save, save it */
	if (active_screen!=INACTIVE_SCREEN)
		blit(screen_buffer, active_screen_buffer, 0, 0, 0, 0,
			display_width, display_height);

	switch (active_screen = a)
	{
                case DEBUGGER:
		{
			/* First save current (game) window info */
			if (last_active_screen==GAME)
			{
				game_win.top = text_windowtop;
				game_win.left = text_windowleft;
				game_win.bottom = text_windowbottom;
				game_win.right = text_windowright;

				game_win.textcolor = current_text_color;
				game_win.backcolor = current_back_color;
				game_win.xpos = current_text_col;
				game_win.ypos = current_text_row;

				/* In case the printing is in italic, etc. */
				last_text_font = current_text_font;
				current_text_font = 0;
			}

			/* Save sound/music data; turn off sound when
			   switching to the debugger
			*/
			SuspendAudio();

			/* Switch to the debugger screen and copy it to
			   the memory bitmap
			*/
			blit(active_screen_buffer = debugger_screen, screen,
				0, 0, 0, 0, display_width, display_height);
			blit(screen, screen_buffer,
				0, 0, 0, 0, display_width, display_height);
			debug_cursor(0);

			/* Then restore previous debug window info */
			hugo_settextwindow(1, 1, D_SCREENWIDTH, D_SCREENHEIGHT);

			current_text_color = debug_win.textcolor;
			current_back_color = debug_win.backcolor;
			current_text_col = debug_win.xpos;
			current_text_row = debug_win.ypos;

			break;
		}
		case GAME:
		{
			/* First save current (debug) window info */
			if (last_active_screen==DEBUGGER)
			{
				debug_win.top = text_windowtop;
				debug_win.left = text_windowleft;
				debug_win.bottom = text_windowbottom;
				debug_win.right = text_windowright;

				debug_win.textcolor = current_text_color;
				debug_win.backcolor = current_back_color;
				debug_win.xpos = current_text_col;
				debug_win.ypos = current_text_row;

				current_text_font = last_text_font;
			}

			/* Restore sound/music data when switching back
			   to the game
			*/
			ResumeAudio();

			/* Switch to the game screen and copy it to the
			   memory bitmap
			*/
			blit(active_screen_buffer = game_screen, screen,
				0, 0, 0, 0, display_width, display_height);
			blit(screen, screen_buffer,
				0, 0, 0, 0, display_width, display_height);
			debug_cursor(1);

			/* Then restore previous game window info */
			hugo_settextwindow(game_win.left,
					game_win.top,
					game_win.right,
					game_win.bottom);

			current_text_color = game_win.textcolor;
			current_back_color = game_win.backcolor;
			current_text_row = game_win.ypos;
			current_text_col = game_win.xpos;

			break;
		}
		case HELP:
		case AUXILIARY:
		{
			if (active_screen==HELP)
			{
				/* Switch to the help screen */
				blit(active_screen_buffer = help_screen, screen,
					0, 0, 0, 0, display_width, display_height);
				blit(screen, screen_buffer,
					0, 0, 0, 0, display_width, display_height);
			}
			else
			{
				/* Switch to the auxiliary screen */
				blit(active_screen_buffer = auxiliary_screen, screen,
					0, 0, 0, 0, display_width, display_height);
				blit(screen, screen_buffer,
					0, 0, 0, 0, display_width, display_height);
			}
			debug_cursor(0);
			debug_settextpos(1, D_SCREENHEIGHT);
			break;
		}
	}
}


#define CREATE_BITMAP(a) (a = create_bitmap(display_width, display_height))

void hugo_init_screen(void)		/* from healleg.c */
{
	int i;

	hugo_settextmode();

	if (CREATE_BITMAP(game_screen)==NULL ||
	    CREATE_BITMAP(debugger_screen)==NULL ||
	    CREATE_BITMAP(help_screen)==NULL ||
	    CREATE_BITMAP(auxiliary_screen)==NULL)
	{
		hugo_clearfullscreen();
		printf("Fatal Error:  Debugger cannot display with existing video configuration\n");
	}

	debug_switchscreen(HELP);
	hugo_clearfullscreen();

	debug_switchscreen(AUXILIARY);
	hugo_clearfullscreen();

	debug_switchscreen(DEBUGGER);

	D_SCREENHEIGHT = display_height/FONT_HEIGHT;
	D_SCREENWIDTH =  display_width/FONT_WIDTH;

	window[CODE_WINDOW].top = D_SEPARATOR+1;
	window[CODE_WINDOW].height = D_SCREENHEIGHT-2-D_SEPARATOR;
	window[CODE_WINDOW].width = D_SCREENWIDTH-2;

	/* All others are the same height and width */
	for (i=VIEW_WATCH; i<WINDOW_COUNT; i++)
	{
		window[i].top = 3;
		window[i].height = D_SEPARATOR - 3;
		window[i].width = D_SCREENWIDTH - 2;
	}

	/* While we're at it, initialize the CTRL-C and CTRL-BREAK
	   handler:
	*/
/*
	signal(SIGINT, debug_breakhandler);
*/
}


void hugo_cleanup_screen(void)		/* from healleg.c */
{
	allegro_exit();
}


/* debug_cursor

	Turns the cursor on (1) or off (0)--if necessary.
*/

/* Blank because Allegro doesn't display a cursor unless we make one */
void debug_cursor(int n)
{}


/* debug_settextpos

	Different from hugo_settextpos() in that it doesn't do any
	accounting for proportional printing.
*/

void debug_settextpos(int x, int y)
{
	current_text_col = text_windowleft-1+x;
	current_text_row = text_windowtop-1+y;
	ConstrainCursor();

	gotoxy(current_text_col, current_text_row);

}


/* debug_windowsave and debug_windowrestore

   	Before drawing a menu or other box, it is necessary to save what
	was behind it to be restored when the menu or box is closed.
*/

void *debug_windowsave(int left, int top, int right, int bottom)
{
	int width, height;
	BITMAP *buf;

	width = (right-left+1)*FIXEDCHARWIDTH;
	height = (bottom-top+1)*FIXEDLINEHEIGHT;

	if ((buf = create_bitmap(width, height))==NULL)
		DebuggerFatal(D_MEMORY_ERROR);

	blit(screen, buf, (left-1)*FIXEDCHARWIDTH, (top-1)*FIXEDLINEHEIGHT,
		0, 0, width, height);

	debug_windowshadow(left, top, right, bottom);

	return buf;
}

void debug_windowrestore(void *b, int xpos, int ypos)	/* note:  (x, y) */
{
	BITMAP *buf = b;

	blit(buf, screen, 0, 0, 
		(xpos-1)*FIXEDCHARWIDTH, (ypos-1)*FIXEDLINEHEIGHT, buf->w, buf->h);
	blit(buf, screen_buffer,
		0, 0,
		(xpos-1)*FIXEDCHARWIDTH, (ypos-1)*FIXEDLINEHEIGHT, buf->w, buf->h);

	destroy_bitmap(buf);
}


/* debug_windowshadow

	Called, simply for convenience, by debug_windowsave in order to
	reuse its parameters.  Draws a shadow along the right and bottom
	of the just-saved and about-to-be-printed menu space.
*/

void debug_windowshadow(int left, int top, int right, int bottom)
{
	/* right side */
	rectfill(screen_buffer, (right-2)*FIXEDCHARWIDTH, top*FIXEDLINEHEIGHT,
		right*FIXEDCHARWIDTH-1, bottom*FIXEDLINEHEIGHT-1,
		HUGO_PALETTE[0]);

	/* bottom */
	rectfill(screen_buffer, (left+1)*FIXEDCHARWIDTH+1, (bottom-1)*FIXEDLINEHEIGHT,
		right*FIXEDCHARWIDTH-1, bottom*FIXEDLINEHEIGHT-1,
		HUGO_PALETTE[0]);
}


/* debug_windowbottomrow

	Necessary in case the OS in question forces a linefeed when a
	character is output to the bottom-right corner of the screen.
	This prints a caption at the left side of the screen, and the
	current filename at the right.
*/

void debug_windowbottomrow(char *caption)
{
	char b[256];
	int r;

	debug_settextpos(1, D_SCREENHEIGHT);
	debug_settextcolor(color[MENU_TEXT]);
	debug_setbackcolor(color[MENU_BACK]);

	strcpy(b, caption);     /* in case line[] is the source */

	strcpy(line, b);
	memset(line+strlen(b), ' ', D_SCREENWIDTH-strlen(b)-strlen(gamefile)-1);
	sprintf(line+D_SCREENWIDTH-strlen(gamefile)-1, "%s", gamefile);
	debug_print(line);

	/* print to the last screen position */
	
	/* r is needed in case display_height is not evenly divisible */
	r = display_height / FIXEDLINEHEIGHT;
	r *= FIXEDLINEHEIGHT;
	
	rectfill(screen_buffer,
		display_width-FIXEDCHARWIDTH, r-FIXEDLINEHEIGHT,
		display_width-1, r-1, HUGO_PALETTE[current_back_color]);
}


/* debug_windowscroll

	Scrolls the defined window in a direction given by the param
	constant.
*/

void debug_windowscroll(int left, int top, int right, int bottom, int param, int lines)
{
	debug_settextcolor(color[NORMAL_TEXT]);
	debug_setbackcolor(color[NORMAL_BACK]);

	switch (param)
	{
		case UP:
		{
			while (lines--)
			{
				blit(screen_buffer, screen_buffer,
				  (left-1)*FIXEDCHARWIDTH, top*FIXEDLINEHEIGHT,
				  (left-1)*FIXEDCHARWIDTH, (top-1)*FIXEDLINEHEIGHT,
				  (right-left+1)*FIXEDCHARWIDTH,
				  (bottom-top)*FIXEDLINEHEIGHT);

				rectfill(screen_buffer,
				  (left-1)*FIXEDCHARWIDTH, (bottom-1)*FIXEDLINEHEIGHT,
				  right*FIXEDCHARWIDTH-1, bottom*FIXEDLINEHEIGHT-1,
				  HUGO_PALETTE[current_back_color]);
			}
			break;
		}
		case DOWN:
		{
			while (lines--)
			{
				blit(screen_buffer, screen_buffer,
				  (left-1)*FIXEDCHARWIDTH, (top-1)*FIXEDLINEHEIGHT,
				  (left-1)*FIXEDCHARWIDTH, (top)*FIXEDLINEHEIGHT,
				  (right-left+1)*FIXEDCHARWIDTH,
				  (bottom-top)*FIXEDLINEHEIGHT);

				rectfill(screen_buffer,
				  (left-1)*FIXEDCHARWIDTH, (top-1)*FIXEDLINEHEIGHT,
				  right*FIXEDCHARWIDTH-1, top*FIXEDLINEHEIGHT-1,
				  HUGO_PALETTE[current_back_color]);
			}
			break;
		}
	}

	UpdateScreen();
}


/* debug_clearview

        Clears the selected window given by n.  (Basically, however,
	all the windows except Code share the same space.)
*/

void debug_clearview(int n)
{
	/* These are entirely separate full screens in this port */
	if (n==VIEW_HELP || n==VIEW_AUXILIARY)
	{
		hugo_clearfullscreen();
		return;
	}

	debug_settextcolor(color[NORMAL_TEXT]);
	debug_setbackcolor(color[NORMAL_BACK]);

	rectfill(screen_buffer,
		FIXEDCHARWIDTH, (window[n].top-1)*FIXEDLINEHEIGHT,
		(D_SCREENWIDTH-1)*FIXEDCHARWIDTH-1,
		(window[n].top+window[n].height-1)*FIXEDLINEHEIGHT-1,
		HUGO_PALETTE[current_back_color]);

	UpdateScreen();
}


/* debug_shortcutkeys

	Switches to the Auxiliary window/page and prints the shortcut
	keys available to this port/system.
*/

void debug_shortcutkeys(void)
{
	debug_switchscreen(HELP);
	debug_settextcolor(color[NORMAL_TEXT]);
	debug_setbackcolor(color[NORMAL_BACK]);
	hugo_clearfullscreen();

	sprintf(line, "Hugo Debugger v%d.%d%s Shortcut Keys\n\n", HEVERSION, HEREVISION, HEINTERIM);
	debug_settextpos(Center(line), 1);
	debug_settextcolor(color[SELECT_TEXT]);
	debug_setbackcolor(color[SELECT_BACK]);
	debug_print(line);

	debug_settextcolor(color[NORMAL_TEXT]);
	debug_setbackcolor(color[NORMAL_BACK]);

	debug_print("ALT    Activate menubar\n");
	debug_print("TAB    Toggle screen\n\n");

	debug_print("F1      Context Help   SHIFT-F1  Topic help      CTRL-F1 Shortcut Keys\n");
	debug_print("F2      Object Tree    SHIFT-F2  Move Object\n");
	debug_print("F3      Search\n");
	debug_print("ALT-F4  Exit\n");
	debug_print("F5      Go             SHIFT-F5  Finish Routine\n");
	debug_print("F6      Switch Window\n");
	debug_print("F7      Set Value\n");
	debug_print("F8      Step           SHIFT-F8  Step Back\n");
	debug_print("F9      Breakpoint     SHIFT-F9  Watch Value\n");
	debug_print("F10     Step Over      SHIFT-F10 Skip Next\n");

	debug_print("\nUse ARROW KEYS, HOME, END, PAGE UP, and PAGE DOWN to navigate.\n");
	debug_print("\nPress ENTER or DELETE to select or delete, where appropriate.");

	debug_windowbottomrow("Press any key...");
	hugo_waitforkey();

	debug_windowbottomrow("");
	debug_switchscreen(DEBUGGER);
}
