#charset "us-ascii"

/*
 *   TADS 3 Hypothetical Action Tester. This extension makes it possible to
 *   evaluate the logicalness (and even the results) of an action without
 *   having to execute it.
 *
 *   You're free to use this extension however you like, but please make any
 *   improvements or modifications available to the IF community.
 *
 *   (c) 2005 Steve Breslin (steve.breslin@gmail.com)
 *       2006 Krister Fundin (fundin@yahoo.com)
 */

#include <adv3.h>
#include "hypoTester.h"

/* ---------------------------------------------------------------------- */
/*
 *   Our ModuleID.
 */
ModuleID
    name = 'TADS 3 Hypothetical Action Tester'
    byLine = 'by Steve Breslin and Krister Fundin'
    htmlByLine =
        'by <a href="mailto:steve.breslin@gmail.com">Steve Breslin</a>
        and <a href="mailto:fundin@yahoo.com">Krister Fundin</a>'
    version = 4
;

/* ---------------------------------------------------------------------- */
/*
 *   Determine if a given action would pass verify() or not. Return true if
 *   the action is logical and allowed, or nil if it would have been stopped
 *   at this stage.
 */
hypoVerify(actor, action, [objs])
{
    /*
     *   get the verify results for this action and return true if they
     *   allow for the purpose of a hypothetical test
     */
    return getHypoVerifyResults(actor, action, objs...)
               .allowHypothetically;
}

/*
 *   Determine if a given action is achieved already. This will perform a
 *   hypothetical verification and look for an illogicalAlready result.
 */
isHypoActionAchieved(actor, action, [objs])
{
    /*
     *   get the verify results for this action and return true if they
     *   contain any illogicalAlready results
     */
    return getHypoVerifyResults(actor, action, objs...).isAlreadyAchieved;
}

/*
 *   Verify a given action and return the verify results. This function can
 *   be called directly if more detailed information is needed, such as
 *   whether the action would be allowed implicitly, or when comparing
 *   several actions to see which one would be the most logical.
 */
getHypoVerifyResults(actor, action, [objs])
{
    /* remember the current actor and action */
    local oldActor = gActor;
    local oldAction = gAction;

    try
    {
        /* set the new actor */
        gActor = actor;

        /* create a new action from our base class */
        gAction = action.createActionInstance();

        /* install the objects in the action */
        gAction.setResolvedObjects(objs...);
        gAction.setCurrentObjects(objs);

        /* verify the action */
        local results = gAction.verifyAction();

        /*
         *   The return value of verifyAction() can be nil, which means that
         *   there were no results and the action was logical and allowed,
         *   but to save our callers from having to worry about this special
         *   case, we make sure that we always return a VerifyResultList,
         *   even if it's a completely empty one.
         */
        if (results != nil)
            return results;
        else
            return new VerifyResultList();
    }
    finally
    {
        /* restore the old actor and action before leaving */
        gActor = oldActor;
        gAction = oldAction;
    }
}

/*
 *   Determine if a given action would pass check() or not. Return true if
 *   there are no objections, or nil if the action is cancelled by an exit
 *   signal.
 */
hypoCheck(actor, action, [objs])
{
    /* remember the current actor, action and transcript */
    local oldActor = gActor;
    local oldAction = gAction;
    local oldTranscript = gTranscript;

    try
    {
        /* set the new actor */
        gActor = actor;

        /* create a new action from our base class */
        gAction = action.createActionInstance();

        /* install the objects in the action */
        gAction.setResolvedObjects(objs...);
        gAction.setCurrentObjects(objs);

        /* block all output using a dummy transcript */
        gTranscript = dummyTranscript;

        /* check the action */
        gAction.checkAction();

        /* if there were no exit signals, return success */
        return true;
    }
    catch (ExitSignal es)
    {
        /* an exit signal means that the action failed check() */
        return nil;
    }
    finally
    {
        /* restore the old actor, action and transcript before leaving */
        gActor = oldActor;
        gAction = oldAction;
        gTranscript = oldTranscript;
    }
}

/*
 *   Determine if a given action would pass both verify() and check().
 */
hypoVerifyCheck(actor, action, [objs])
{
    return (hypoVerify(actor, action, objs...)
            && hypoCheck(actor, action, objs...));
}

/*
 *   Modify the VerifyResultList class to support our special hypoIllogical
 *   verify result. Also add a method to check for illogicalAlready verify
 *   results.
 */
modify VerifyResultList
    /* is the action allowed hypothetically? */
    allowHypothetically()
    {
        /*
         *   if it's not allowed normally, then it's not allowed
         *   hypothetically either
         */
        if (!allowAction)
            return nil;

        /* see if we have any hypoIllogical results */
        return (results_.indexWhich(
                    { x: x.ofKind(HypoIllogicalVerifyResult) }) == nil);
    }

    /* is the action illogical because it has already been achieved? */
    isAlreadyAchieved()
    {
        /* search for an illogicalAlready result */
        return (results_.indexWhich(
                    { x: x.ofKind(IllogicalAlreadyVerifyResult) }) != nil);
    }
;

/*
 *   A special type of verification result which allows an action to proceed
 *   unconditionally during normal execution, but not during a hypothetical
 *   evaluation.
 */
class HypoIllogicalVerifyResult: VerifyResult
    /* override the constructor, since we don't take any arguments */
    construct { }

    /*
     *   give us a higher rank than all the other verify results, since we
     *   don't want to interfer with the normal verification
     */
    resultRank = 150
;

/*
 *   A dummy transcript for use when calling check() routines. We could use
 *   an output filter to keep printed text away from the screen, but this
 *   wouldn't stop the reportFailure() macro from adding reports directly to
 *   the transcript, where they won't be printed until we have removed the
 *   filter again. To stop both reports and text printed using the say()
 *   function, we need our own transcript.
 */
dummyTranscript: CommandTranscript
    /* override the addReport() method to do nothing */
    addReport(report) { }

    /*
     *   Disregard all text printed through us. Normally, we would have
     *   wrapped the text in a report, which would then have been ignored by
     *   addReport(), but we may as well save some time by skipping this
     *   redundant step completely.
     */
    filterText(ostr, txt)
    {
        return nil;
    }

    /*
     *   An empty list will do as report vector. We won't ever add anything
     *   to it, but someone might call one of our methods that read from it.
     */
    reports_ = []
;

/* ---------------------------------------------------------------------- */
/*
 *   Execute one or more actions, returning true if they all succeeded or
 *   nil of any of them failed, as signaled by a failure report - then use
 *   undo to take everything back again.
 */
hypoTest([actions])
{
    /*
     *   The arguments for the action can be either a single action
     *   specification or several action specifications where each one is
     *   wrapped in a list. If we find that the first item of actions is not
     *   a list, then we assume that we have a single action specification
     *   and wrap it in a list.
     */
    if (dataType(actions[1]) != TypeList)
        actions = [actions];

    /* remember the current transcript */
    local oldTranscript = gTranscript;

    try
    {
        /* set an undo savepoint */
        savepoint();

        /* flag that we're performing an undo test */
        gUndoTesting = true;

        /*
         *   switch to a fresh transcript, so that we don't get confused by
         *   previous failure reports in the current transcript
         */
        gTranscript = new CommandTranscript();

        /* go through the actions */
        foreach (local cur in actions)
        {
            /* execute this one */
            _nestedAction(nil, cur[1], cur[2], cur.sublist(3)...);

            /* return nil if we have a failure report */
            if (gTranscript.isFailure)
                return nil;
        }

        /* all the actions were carried out successfully */
        return true;
    }
    finally
    {
        /* take us back to the savepoint we set earlier */
        undo();

        /*
         *   restore the old transcript before leaving (this variable is
         *   transient, so it won't be affected by the undo operation)
         */
        gTranscript = oldTranscript;
    }
}

/*
 *   Execute one or more actions, return the result of a callback function
 *   and use undo to take everything back again.
 */
hypoWhatIf(func, [actions])
{
    /* check for a single action specification and wrap it in a list */
    if (dataType(actions[1]) != TypeList)
        actions = [actions];

    /* remember the current transcript */
    local oldTranscript = gTranscript;

    try
    {
        /* set an undo savepoint */
        savepoint();

        /* flag that we're performing an undo test */
        gUndoTesting = true;

        /*
         *   Block all output using a dummy transcript. This isn't really
         *   necessary, since any reports added during the action would be
         *   removed again when we undo, but we can save a few undo records
         *   by not adding any reports at all.
         */
        gTranscript = dummyTranscript;

        /* execute the actions */
        foreach (local cur in actions)
        {
            _nestedAction(nil, cur[1], cur[2], cur.sublist(3)...);
        }

        /* invoke the callback and return the result */
        return func();
    }
    finally
    {
        /* take us back to the savepoint we set earlier */
        undo();

        /* restore the old transcript before leaving */
        gTranscript = oldTranscript;
    }
}

/*
 *   An object for storing global variable type properties.
 */
hypoGlobal: object
    /*
     *   Flag: are we currently in the middle of an undo test, I.E. a call
     *   to hypoWhatIf()? There are certain things that we can't do during
     *   such as test. We can't ask the player for input, for example, which
     *   in turn means that we can't finish the game. If we still have to
     *   test actions that could potentially finish the game, then it is
     *   important to know whether we're executing those actions for real or
     *   just testing them, so that we can do something else in the latter
     *   case, and this property gives us that information.
     */
    isUndoTesting = nil
;

