/* pbcache.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

#ifndef LINT
static char *RcsId = "@(#) $Id: pbcache.c,v 1.4 92/02/15 06:44:59 lee Exp $";
#endif

/* Block cache for lq-text */

/* Functions defined in this file:
 *
 * OpenDataBase()
 * ReadBlock(unsigned long Offset)
 * WriteBlock(unsigned long Block; char *Data)
 *
 * FlushCache()
 *
 * OpenFreeFile()
 * WriteFreeBlock()
 * char *ReadFreeBitBlock(long Where)
 * static INLINE int FindUnsetBitInByte(unsigned int Value)
 * unsigned long FindFreeBlock(t_WID WID)
 * void SetBlockStatus(unsigned long Offset; int Status)
 *
 * And for convenience, although they should really be elsewhere:
 * void Deletepblock(t_pblock *pblock)
 * void DeleteWordPlaces(unsigned long FirstBlock; t_WID WID)
 * (having them here means they can get at DataFile and OpenDataBase)
 *
 */

#include "globals.h" /* defines and declarations for database filenames */
#include "error.h"

#include <stdio.h> /* stderr, also for fileinfo.h */
#include <sys/types.h> /* for fileinfo.h, which uses time_t */
#include "fileinfo.h" /* for wordinfo.h */
#include "wordinfo.h" /* for t_WID */
#include <pblock.h>
#include "blkheader.h"

/** Unix system calls that need to be declared: **/
/* Some systems have a stdlib.h or stddef.h with some of these in,
 * but most don't (sigh)
 */
extern int open();
extern int read(), write();
extern long lseek();

/** C library functions that need to be declared: **/

/** lqtext library functions that need to be declared: **/

/** Functions within this file that need to be declared: **/
void FlushCache();
static void OpenDataBase();
static void OpenFreeFile();
static void WriteFreeBlock();
void SetBlockStatus();

unsigned long FindFreeBlock();
/** **/

#ifdef ASCIITRACE
extern int AsciiTrace;
#endif

static int DataFile = -1;
static unsigned long FreeBlockStart = 3L;
	/* i.e. not a multiple of 2, and hence not a valid value */

static int FreeFile = -1;
#define FREEBITLEN 512
#define BYTESPERFREEBIT (BLOCKSIZE*8)
static unsigned char FreeBitBlock[FREEBITLEN];
static char FreeBitBlockIsDirty = 0;
static int BlockCacheIsDirty = 0;
static int LowestFreeBlock = 0L;

char *ReadBlock();
void WriteBlock();

#define CACHELEN (BLOCKSIZE * 128)

static unsigned long BlockCacheTimer = 0;

typedef struct s_CacheEntry {
    unsigned long CacheStart;
    long CacheLen;
    int TimeLastUsed;
    char ReadAheadBuffer[CACHELEN];
    short IsDirty;
} t_CacheEntry;

#define CACHEDBLOCKS 4
static t_CacheEntry BlockCache[CACHEDBLOCKS] = {
    0,
};
static t_CacheEntry *CurrentBlock = BlockCache;

#ifdef ASCIITRACE
static void
printcache()
{
    register int i;
    fprintf(stderr, "**** cache at %ld is [", BlockCacheTimer);
    for (i = 0; i < CACHEDBLOCKS; i++) {
	fprintf(stderr, " %ld-%ld",
	    BlockCache[i].CacheStart,
	    BlockCache[i].CacheStart + BlockCache[i].CacheLen
	);
    }
    fprintf(stderr, "]\n");
}
#endif

static int
IsCached(Block)
    unsigned long Block;
{
    /* see if the requested block is cached, and, if so, make
     * CurrentBlock point to it.
     */

    register t_CacheEntry *cp;

    if (DataFile < 0) {
	OpenDataBase();
    }

    for (cp = BlockCache; cp - BlockCache < CACHEDBLOCKS; cp++) {
	if (cp->CacheStart <= Block && cp->CacheStart + cp->CacheLen >=
							Block + BLOCKSIZE) {
	    /* Note: use CacheLen, not CACHELEN, in case the cache entry
	     * isn't valid, in which case CacheLen is zero.
	     */
	    CurrentBlock = cp;
	    return 1;
	}
    }

#ifdef ASCIITRACE
    if (AsciiTrace > 10) {
	fprintf(stderr, "** IsCached Miss for %ld\n", Block);
	printcache();
    }
#endif

    return 0;
}

static void
FlushOneBlock(cp)
    t_CacheEntry *cp;
{
    int i;

    if (lseek(DataFile, cp->CacheStart, 0) < 0) {
	Error(E_SYS|E_FATAL,
	    "FlushCache: lseek(%d=%s, %ld, 0) failed",
	    DataFile, DataBase, cp->CacheStart
	);
    } 
    
    i = write(DataFile, cp->ReadAheadBuffer, cp->CacheLen);

    if (i != cp->CacheLen) {
	Error(E_FATAL|E_SYS,
	    "FlushCache: write(%d=%s, 0x%s, %d) failed, returned %d",
	    DataFile, DataBase, cp->ReadAheadBuffer, cp->CacheLen, i
	);
    }

    cp->IsDirty = 0;
    /* Don't change CacheLen, so that we will still find the block
     * if we want it
     */
}

void
FlushCache(MinToFree)
    int MinToFree;
{
    register t_CacheEntry *cp;
    t_CacheEntry *Oldest = &BlockCache[0];

#ifdef ASCIITRACE
    if (AsciiTrace > 4) {
	(void) fprintf(stderr, " (flush cache) ");
	(void) fflush(stderr);
    }
#endif

    if (DataFile <= 0) {
	if (BlockCacheIsDirty) {
	    Error(E_BUG,
		"dirty cache %ld len %ld, fd %d=%s",
		CurrentBlock->CacheStart,
		CurrentBlock->CacheLen,
		DataFile,
		DataBase
	    );
	}
	BlockCacheIsDirty = 0;
	return;
    }

    if (MinToFree) {
	/* check that the cache is in fact full */
	/* find the oldest block and free it */
	for (cp = BlockCache; cp - BlockCache < CACHEDBLOCKS; cp++) {
	    if (cp->TimeLastUsed && cp->TimeLastUsed < Oldest->TimeLastUsed) {
		Oldest = cp;
	    } else if (cp->CacheLen == 0) {
		/* Only on the first few times round... */
		Oldest = cp;
	    }
	}
	CurrentBlock = Oldest;
	if (CurrentBlock->CacheLen && CurrentBlock->IsDirty) {
	    FlushOneBlock(CurrentBlock = Oldest);
	    return;
	}
    }

    for (cp = BlockCache; cp - BlockCache < CACHEDBLOCKS; cp++) {

	if (cp->TimeLastUsed && cp->TimeLastUsed < Oldest->TimeLastUsed) {
	    Oldest = cp;
	}

	if (cp->CacheLen && cp->IsDirty) {
	    FlushOneBlock(cp);
	}
    }

    BlockCacheIsDirty = 0;
    CurrentBlock = Oldest;

    if (FreeBitBlockIsDirty) {
	WriteFreeBlock();
    }
}

void
WriteBlock(Block, Data)
    unsigned long Block;
    char *Data;
{
    if (DataFile < 0) {
	OpenDataBase();
    }

    if (!Data) {
	Error(E_BUG,
	    "WriteBlock %ld, 0x0: second argument (Data) is invalid",
	    Block
	);
    }

    if (!IsCached(Block)) {
	(void) ReadBlock(Block);
    }

    (void) memcpy(
	&(CurrentBlock->ReadAheadBuffer[Block - CurrentBlock->CacheStart]),
	Data,
	BLOCKSIZE
    );
    CurrentBlock->TimeLastUsed = ++BlockCacheTimer;
    CurrentBlock->IsDirty = 1;
    BlockCacheIsDirty = 1;

    return;
}

static void
OpenDataBase()
{
    extern void lqGetFileModes();
    int Flags, Modes;

    lqGetFileModes(&Flags, &Modes);

    if ((DataFile = open(DataBase, Flags, Modes)) < 0) {
	Error(E_FATAL|E_SYS, "Can't open database file \"%s\"", DataBase);
    }

    (void) ReadBlock(0L);
}

static void
WriteFreeBlock()
{
    int i;
    unsigned long OffsetInFile;

    if (FreeFile < 0 || !FreeBitBlockIsDirty) {
	return;
    }

    OffsetInFile = FreeBlockStart / (BLOCKSIZE * 8);

    if (lseek(FreeFile, OffsetInFile, 0) < 0) {
	Error(E_FATAL|E_SYS, "WriteFreeBlock: lseek(%d=\"%s\", %ld, 0) failed",
				FreeFile, FreeFileName, FreeBlockStart);
    }
    
    i = write(FreeFile, FreeBitBlock, FREEBITLEN);

    if (i != FREEBITLEN) {
	Error(E_FATAL|E_SYS,
	    "WriteFreeBlock: write(%d=\"%s\", 0x%x, %d) returned %d",
	    FreeFile, FreeFileName, FreeBitBlock, FREEBITLEN, i);
    }

    FreeBitBlockIsDirty = 0;
}

static void
ReadFreeBitBlock(Where)
    long Where;
{
    long AmountRead;
    unsigned long OffsetInFile;

    /* open the Free file if necessary... */

    if (FreeFile < 0) {
	OpenFreeFile();
    }

    OffsetInFile = Where / (BLOCKSIZE * 8);

    /* Round FREEBITLEN down to the start of the block */
    OffsetInFile /= FREEBITLEN;
    OffsetInFile *= FREEBITLEN;

    if (OffsetInFile * BLOCKSIZE * 8 == FreeBlockStart) {
	return;
    }

    if (FreeBitBlockIsDirty) {
	WriteFreeBlock();
    }

    if (lseek(FreeFile, OffsetInFile, 0) < 0) {
	Error(E_FATAL|E_SYS, "ReadFreeBlock: lseek(%d=\"%s\", %ld, 0) failed",
	    FreeFile, FreeFileName, Where);
    }

    AmountRead = read(FreeFile, FreeBitBlock, FREEBITLEN);

    if (AmountRead < 0) {
	Error(E_FATAL|E_SYS,
	    "ReadFreeBlock: read(%d=\"%s\", ...) returned %d, not %d",
			FreeFile, FreeFileName, AmountRead, FREEBITLEN
	);
    }

    FreeBlockStart = OffsetInFile * 8 * BLOCKSIZE;

    /* If we have gone past the end of the file, set the rest of the
     * block to zeros.
     */
    if (AmountRead < FREEBITLEN) {
#ifdef SYSV
	(void) memset(&FreeBitBlock[AmountRead], '\0', FREEBITLEN - AmountRead);
#else
	(void) bzero(&FreeBitBlock[AmountRead], FREEBITLEN - AmountRead);
#endif
    }
}

#define ReadNextFreeBitBlock() \
    ReadFreeBitBlock(FreeBlockStart + (8 * BLOCKSIZE * FREEBITLEN + 1))

#define SetBitInByte(Which) (01 << (Which))

static INLINE int
FindUnsetBitInByte(Value)
    unsigned int Value;
{
    register int WhichBit = 0;

    while (Value & 01) {
	Value >>= 1;
	WhichBit++;
    }

    return WhichBit;
}

static void
OpenFreeFile()
{
    extern void lqGetFileModes();
    int Flags, Modes;
    
    lqGetFileModes(&Flags, &Modes);

    if ((FreeFile = open(FreeFileName, Flags, Modes)) < 0) {
	Error(E_FATAL|E_SYS, "Can't open database free bitmap file \"%s\"",
			FreeFileName);
    }

    ReadFreeBitBlock(0L);

    if (!(FreeBitBlock[0] & 01)) {
	SetBlockStatus(0L, SET_BLOCK_AS_USED);
    }
}

/*ARGSUSED1*/
unsigned long
FindFreeBlock(WID)
    t_WID WID;
{
    register unsigned char *p;
    int GotOne = 0;
    unsigned long Here;

    if (FreeFile < 0) {
	OpenFreeFile();
	/* OpenFreeFile() calls ReadFreeBitBlock(0L) */
	LowestFreeBlock = 0L;
    } else {
	ReadFreeBitBlock(LowestFreeBlock);
    }

    do {
	for (p = FreeBitBlock; p - FreeBitBlock < FREEBITLEN; p++) {
	    if (*p != (unsigned char) 0xff) {
		GotOne++;
		break;
	    }
	}

	if (GotOne) break;

	ReadNextFreeBitBlock();
    } while (!GotOne);

    /* Now we've found a byte `containing' a free block... ( a zero bit)
     * so we have to identify the block.
     * There are 
     */

    Here = FreeBlockStart + (p - FreeBitBlock) * BYTESPERFREEBIT +
		(FindUnsetBitInByte((unsigned int)*p) * BLOCKSIZE);
    SetBlockStatus(Here, SET_BLOCK_AS_USED);

    /* Find the next free block */
    while (p - FreeBitBlock < FREEBITLEN) {
	if (*p != 0xff) {
	    break;
	}
	p++;
    }

    LowestFreeBlock = FreeBlockStart + (p - FreeBitBlock) * BYTESPERFREEBIT;
    /* This ignores the actual bit in the byte, but that's OK, we only want
     * to be able to start looking at the right byte next time round
     */

    return Here;
}

void
SetBlockStatus(Offset, Status)
    unsigned long Offset;
    int Status;
{
    register unsigned char *p;
    register unsigned int ui;
    unsigned char OldValue ;

    if (FreeFile < 0) {
	OpenFreeFile();
	LowestFreeBlock = 0L;
    }

    /* First make sure that the necessary bitmap block is loaded: */
    if (Offset >= (FreeBlockStart + (BYTESPERFREEBIT * FREEBITLEN)) ||
						(Offset < FreeBlockStart)) {
	ReadFreeBitBlock(Offset);
    }

    ui = (Offset - FreeBlockStart) / BLOCKSIZE; /* this is the bit address */
    p = &FreeBitBlock[ui >> 3]; /* i.e. ui/8, i.e. the right byte */
    OldValue = (*p);
    if (Status == SET_BLOCK_AS_FREE) {
	/* ui & 07 is ui % 8 is the bit within the byte: */
	*p &= (unsigned char)~(unsigned char) SetBitInByte(ui & 07);
	if (Offset < LowestFreeBlock || !LowestFreeBlock) {
	    LowestFreeBlock = Offset;
	}
    } else if (Status == SET_BLOCK_AS_USED) {
	*p |= SetBitInByte(ui & 07);
	if (Offset == LowestFreeBlock) {
	    LowestFreeBlock = 0;
	}
    } else {
	Error(E_BUG, "SetBlockStatus(%ld, %d)", Offset, Status);
    }
    if (OldValue != *p) {
	FreeBitBlockIsDirty = 1;
    }
}

#define BLOCKSINREADBUFCACHE 2
static char ReadBufferCache[BLOCKSINREADBUFCACHE][BLOCKSIZE];
static int BufferNumber = -1;
/* If sWriteNumber didn't overshoot buffers, BLOCKSINREADBUFCACHE could be
 * set to 1, and if we didn't need to go back and write to the previous
 * block to set the NextOffset, it could be zero and we could return a
 * pointer into the main cache without even the copy!
 */

/* Get a single disk block from the database */
char *
ReadBlock(Offset)
    unsigned long Offset;
{
    char *Buffer;

    if (DataFile < 0) {
	OpenDataBase();
    }

    if (++BufferNumber >= BLOCKSINREADBUFCACHE) BufferNumber = 0;
    Buffer = ReadBufferCache[BufferNumber];

#ifdef ASCIITRACE
    if (AsciiTrace > 15) {
	fprintf(stderr, "(R %ld) ", Offset);
	(void) fflush(stderr);
    }
#endif

    if (IsCached(Offset)) {
	(void) memcpy(
	    Buffer,
	    &(CurrentBlock->ReadAheadBuffer[Offset - CurrentBlock->CacheStart]),
	    BLOCKSIZE
	);
	CurrentBlock->TimeLastUsed = ++BlockCacheTimer;
	return Buffer;
    } 

    /* So it's not cached...
     * Which means we have to read a new cache.
     * So let's make some room:
     */

    if (BlockCacheIsDirty) {
	FlushCache(1);
	/* The argument to FlushBlock means it'll only write out a
	 * single dirty block, and set CurrentBlock top point to it.
	 */
    }

    /* cache only on CACHELEN (typically 8K) boundaries, as this helps the
     * file system, especially over NFS.
     */
    CurrentBlock->CacheStart = Offset / CACHELEN;
    CurrentBlock->CacheStart *= CACHELEN;

    if (lseek(DataFile, CurrentBlock->CacheStart, 0) < 0) {
	Error(E_SYS|E_WARN,
	    "ReadBlock %ld: lseek(%d=\"%s\", %ld 0) failed",
	    Offset, DataFile, DataBase, CurrentBlock->CacheStart
	);
	return (char *) 0;
    }
    /* Now we are in the right place */

    CurrentBlock->CacheLen =
		read(DataFile, CurrentBlock->ReadAheadBuffer, CACHELEN);

    if (CurrentBlock->CacheLen < 0) {
	Error(E_FATAL|E_SYS,
	    "ReadBlock: read(%d=\"%s\"...) returned %d, not %d",
	    DataFile, DataBase, CurrentBlock->CacheLen, CACHELEN
	);
    }

    if (CurrentBlock->CacheLen < CACHELEN) {
	CurrentBlock->CacheLen = CACHELEN;
	/* If CacheLen was less than CACHELN this can only mean that we
	 * have tried to read beyond the end of the file.  In this case,
	 * we might as well consider that we have the extra bytes cached
	 * as well.  The disadvantage is that the file will always grow to
	 * a multiple of CACHELEN, but the advantage is fewer reads when
	 * the file is growing.
	 */
    }

    (void) memcpy(
	Buffer,
	&(CurrentBlock->ReadAheadBuffer[Offset - CurrentBlock->CacheStart]),
	BLOCKSIZE
    );
    CurrentBlock->TimeLastUsed = ++BlockCacheTimer;
    return Buffer;
}

/*ARGSUSED1*/
void
DeleteWordPlaces(FirstBlock, WID)
    unsigned long FirstBlock;
    t_WID WID;
{
    char *p;
    t_BlockHeader *H;
    unsigned long Block = FirstBlock;

    if (!FirstBlock || !WID) {
	Error(E_BUG, "DeleteWordPlaces(%ld, %ld) invalid", FirstBlock, WID);
    }

    if (DataFile < 0) {
	OpenDataBase();
    }

    while (Block != 0L) {
	p = ReadBlock(Block);
	H = (t_BlockHeader *) p;
#ifdef WIDINBLOCK
	if (H->WID != WID) {
	    Error(E_BUG,
		"DeleteWordPlaces(Start=%ld, WID=%ld): block %ld had WID %ld",
		FirstBlock, WID, Block, H->WID
	    );
	}
	H->WID = 0;
# ifdef ASCIITRACE
	/* help debugging: */
	(void) sprintf(&H->Data[0], "Deleted -- was %ld\n", WID);
	WriteBlock(Block, p);
# endif /* ASCIITRACE */
#endif
	SetBlockStatus(Block, SET_BLOCK_AS_FREE);
	Block = H->NextOffset;
    }
    return;
}

void
Deletepblock(pblock)
    t_pblock *pblock;
{
    if (!pblock) {
	Error(E_BUG, "Attempt to delete null pblock (address 0x%x)", pblock);
    } else if (!pblock->ChainStart || !pblock->WID) {
	Error(E_BUG,
	    "Attempt to delete pblock 0x%x with WID %d, ChainStart %ld",
	    pblock,
	    pblock->WID,
	    pblock->ChainStart
	);
    }

    DeleteWordPlaces(pblock->ChainStart, pblock->WID);

    return;
}
