/* rpblock.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

#ifndef LINE
static char *RcsId = "@(#) $Id: rpblock.c,v 1.2 92/03/08 20:16:16 lee Exp $";
#endif

/* This file defines:
 *
 * t_pblock *Getpblock(t_WordInfo *WordInfo)
 *
 * t_WordPlace *GetWordPlaces(
 *    t_WID WID;
 *    char *Block;
 *    unsigned BlockLength;
 *    unsigned long NextOffset;
 *    unsigned long NumberExpected;
 * );
 *
 */

#include "globals.h" /* defines and declarations for database filenames */
#include "error.h"

#include <stdio.h> /* stderr, also for fileinfo.h */
#include <fcntl.h>
#include <malloc.h>
#include <sys/types.h>
#include "fileinfo.h" /* for wordinfo.h */
#include "wordinfo.h"
#include "pblock.h"
#include "numbers.h"
#include "emalloc.h"
#include "wordrules.h"

/** Unix system calls that need to be declared: **/

/** C library functions that need to be declared: **/
extern char *memcpy();

/** lqtext library functions that need to be declared: **/
extern unsigned char _GetByte();
extern unsigned long GetLong();
#ifdef ASCIITRACE
extern void fprintWordInfo(/*stream, W, Caller*/);
#endif

/** Functions within this file that need to be declared: **/

t_WordPlace *GetWordPlaces();
t_pblock *Getpblock();
/** **/

/* GetByte gets called literally millions of times, so it's a macro.
 * See the comment in wpblock.c.
 */
#define GetByte(WID, sp, Blockp, BlockLength, NextBlock) \
    ( (*(sp) - (unsigned char *) *(Blockp) >= *(BlockLength)) ? \
       _GetByte(WID, sp, Blockp, BlockLength, NextBlock) : *((*(sp))++) )

#ifdef ASCIITRACE
extern int AsciiTrace;
#endif

/* Look up a word in the database...
 * and return a list of all the WordPlaces where it's found
 * DESIGN BUG: should be called WordInfo2pblock().
 */
t_pblock *
Getpblock(WordInfo)
    t_WordInfo *WordInfo;
{
    t_pblock *pblock = 0;
    unsigned long HowManyToGet = 0L;
    t_WordPlace *WordPlaces;
    unsigned long CurrentPair = 0L;

    if (!WordInfo->NumberOfWordPlaces) {
#ifdef ASCIITRACE
	if (AsciiTrace > 2) {
	    fprintWordInfo(stderr, WordInfo, "Getpblock");
	}
#endif
	Error(E_BUG, "Getpblock: attempt to fetch \"%s\" with no matches!",
	    WordInfo->Word ? WordInfo->Word : "(null)"
	);
    }

    HowManyToGet = WordInfo->NumberOfWordPlaces;

    pblock = (t_pblock *) emalloc( sizeof(t_pblock) +
		    (unsigned) (HowManyToGet + 1) * sizeof(t_WordPlace));

    WordPlaces = pblock->WordPlaces;
    pblock->WID = WordInfo->WID;
    pblock->ChainStart = WordInfo->Offset;
    pblock->NumberOfWordPlaces = WordInfo->NumberOfWordPlaces;

    /* First, the pairs in the WordInfo might suffice: */
    if (WordInfo->WordPlacesInHere >= HowManyToGet) {
	for (; CurrentPair < WordInfo->WordPlacesInHere; CurrentPair++) {
	    WordPlaces[CurrentPair] = WordInfo->WordPlaces[CurrentPair];
	}
    }

    /* If they all fitted in the WordInfo block, well, that was a big win! */
    if (CurrentPair >= HowManyToGet) {
	pblock->ChainStart = 0L;
	return pblock;
    }

    /* So we need to read the entire list of WordPlaces from the database.
     * Although we may have already done the first few, I'm going to do them
     * all again because that ensures that the last few bytes in the
     * WordInfo data block can get used!
     */
    
    WordPlaces = GetWordPlaces(
		WordInfo->WID,
		WordInfo->WordPlaceStart,
		(unsigned) WIDBLOCKSIZE -
			    (WordInfo->WordPlaceStart - WordInfo->DataBlock),
		WordInfo->Offset,
		HowManyToGet);

  
  
    if (WordPlaces == (t_WordPlace *) 0) {
#ifdef ASCIITRACE
	if (AsciiTrace > 2) {
	    fprintWordInfo(stderr, WordInfo, "Getpblock");
	}
#endif
	Error(E_BUG, "no wordplaces for WID %ld, wanted %ld",
			WordInfo->WID, HowManyToGet);
    }

    /* copy the result... */
    (void) memcpy((char *) pblock->WordPlaces, (char *) WordPlaces,
				(int) (sizeof(t_WordPlace) * HowManyToGet));
    (void) efree((char *) WordPlaces);
    return pblock;
}

t_WordPlace *
GetWordPlaces(WID, Block, BlockLength, NextOffset, NumberExpected)
    t_WID WID;
    char *Block;
    unsigned BlockLength;
    unsigned long NextOffset;
    unsigned long NumberExpected;
{
    unsigned char *q = (unsigned char *) Block;
    unsigned long L;
    t_WordPlace *Places = (t_WordPlace *) 0;
    long CurrentPlace = 0;
    t_FID LastFID = (t_FID) 0;
    unsigned LastBlock = 0L;
    unsigned char LastFlags = 0;

#ifdef ASCIITRACE
    if (AsciiTrace > 10) {
	fprintf(stderr,
		"GetWordPlaces WID %ld Blk 0x%x len %d next %ld No. %ld\n",
		    WID, Block, BlockLength, NextOffset, NumberExpected);
			
    }
#endif

    if (Block == (char *) 0) {
	Error(E_BUG, "GetWordPlaces WID %lu, zero block", WID);
    }

    /*NOSTRICT*/
    Places = (t_WordPlace *) emalloc(sizeof(t_WordPlace) * NumberExpected);

    while (CurrentPlace < NumberExpected) {
	unsigned short NumberOfRepeats;
	unsigned char Uchar;
	t_FID FID;

	/** First get the FID.  The bottom bit of the number stored
	 ** actually determines whether there are multiple Places
	 ** stored here for the same FID.
	 **/
	L = GetLong(WID, &q, &Block, &BlockLength, &NextOffset);

	FID = (L >> 1) + LastFID; /* Get rid of flag bit */

	if (FID == 0) {
	    Error(E_BUG, "GetWordPlaces WID %ld, FID %ld is Zero!",
		WID, CurrentPlace
	    );
	}

	LastFID = FID;
	NumberOfRepeats = (L & 01L) ? 
		GetByte(WID, &q, &Block, &BlockLength, &NextOffset) : 1;

	/* Quick Sanity check */

	/* This is probably cheap enough that we can do it all the time */
	switch (NumberOfRepeats) {
	case 0L:
	    Error(E_BUG, "GetWordPlaces WID %ld: no entries! for FID %lu",
							WID, FID);
	case 1L:
	    if (L & 01L) {
		Error(E_BUG,
			"GetWordPlaces: WID %ld, FID %lu repeated 1 times!",
								WID, FID);
	    }
	}

	LastBlock = 0L;
	if (CurrentPlace + NumberOfRepeats > NumberExpected) {
	    Error(E_BUG,
		"GetWordPlaces: FID %lu WID %ld has %lu matches, not %lu",
		FID, WID, CurrentPlace + NumberOfRepeats + 1, NumberExpected);
	}
	for (; NumberOfRepeats != 0; --NumberOfRepeats) {
	    Places[CurrentPlace].FID = FID;
	    /* block number */
	    L = GetLong(WID, &q, &Block, &BlockLength, &NextOffset);
	    LastBlock += L;
	    Places[CurrentPlace].BlockInFile = LastBlock;
	    Uchar = GetByte(WID, &q, &Block, &BlockLength, &NextOffset);
	    Places[CurrentPlace].WordInBlock = (Uchar >> 1);

#ifdef ASCIITRACE
	    /* Sanity check: */
	    if (CurrentPlace > 0 && Places[CurrentPlace].FID ==
					Places[CurrentPlace - 1].FID) {
		if (Places[CurrentPlace - 1].BlockInFile ==
				    Places[CurrentPlace].BlockInFile) {
		    if (Places[CurrentPlace - 1].WordInBlock >=
				Places[CurrentPlace].WordInBlock) {
			Error(E_BUG,
	    "GetWordPlaces: match %d for word %ld FID %ld WIB decreases!",
				    CurrentPlace, WID, FID);
		    }
		} else if (Places[CurrentPlace - 1].BlockInFile >
				Places[CurrentPlace].BlockInFile) {
			Error(E_BUG,
	"GetWordPlaces: match %d for WID %ld FID %ld BIF decreases!",
			    CurrentPlace, WID, FID);
		}
	    }
	    /* end of sanity test */
#endif ASCIITRACE

	    if (Uchar & 01) { /* use if, not ?:, for profiler */
		LastFlags = Places[CurrentPlace].Flags = 
		    GetByte(WID, &q, &Block, &BlockLength, &NextOffset);
	    } else {
		Places[CurrentPlace].Flags = LastFlags;
	    }

	    /* If there are flags, there still might not be a separate
	     * entry for the number of preceding skipped bytes.
	     */
	    if (Places[CurrentPlace].Flags & WPF_HASSTUFFBEFORE) {
		Places[CurrentPlace].StuffBefore = 
		    GetByte(WID, &q, &Block, &BlockLength, &NextOffset);
	    } else {
		Places[CurrentPlace].StuffBefore = 1;
	    }
	    ++CurrentPlace;
	}
    }
    return Places;
}
