/* smalldb.c -- Copyright 1989 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* Simple interface to start and end dbm.
 * You may also need to supply dbm_store() and dbm_fetch(), but these
 * should certainly be macros.
 *
 * $Id: smalldb.c,v 1.9 92/02/15 06:46:03 lee Exp $
 *
 * $Log:	smalldb.c,v $
 * Revision 1.9  92/02/15  06:46:03  lee
 * removed unused variable `tmp'.
 * 
 * Revision 1.8  92/02/15  05:21:09  lee
 * Improved an error message.
 * 
 * Revision 1.7  92/02/15  04:27:54  lee
 * Improved the cache somewhat.
 * 
 * Revision 1.6  91/08/12  21:23:16  lee
 * Added GetFileModes() call to export FileModes and FileFlags.
 * 
 * Revision 1.5  91/03/03  00:15:22  lee
 * Improved an error message and fixed a permissions bug.
 * 
 * Revision 1.4  91/03/02  18:52:48  lee
 * Default access is now read only -- lqWriteAccess must be called otherwise.
 * 
 * Revision 1.3  90/10/06  00:12:20  lee
 * Prepared for first beta release.
 * 
 * Revision 1.2  90/09/20  17:53:26  lee
 * slight error reporting improvement.
 * 
 * Revision 1.1  90/08/09  19:16:56  lee
 * Initial revision
 * 
 * Revision 2.2  89/10/08  20:47:14  lee
 * Working version of nx-text engine.  Addfile and wordinfo work OK.
 * 
 * Revision 2.1  89/10/02  01:15:55  lee
 * New index format, with Block/WordInBlock/Flags/BytesSkipped info.
 * 
 * Revision 1.2  89/09/16  21:18:39  lee
 * First demonstratable version.
 * 
 * Revision 1.1  89/09/07  21:06:11  lee
 * Initial revision
 * 
 */

#include "globals.h"
#include "error.h"

#include <fcntl.h> /* for O_RDONLY, etc. */
#include "emalloc.h"
#include "smalldb.h"
#include "emalloc.h"

/* Actually we don't need this, but ANSI C requires it */
#ifdef __STDC__
# include <stdio.h>
#endif

extern int strcmp();
extern char *strcpy();

/* The physical database for the list of words, and for the list
 * of files, uses ndbm.
 * The advantage of this is that it takes only two file system accesses
 * to retrieve any data item (honest!).
 * It's also reasonably fast at insertion.
 * One disadvantage is that it doesn't cope if too many words have the
 * same (32-bit) hash function, although some of the publicly available
 * replacements such as the 4.4 BSD db package fix this.
 *
 * Since starting the database is expensive (two opens and a malloc),
 * I have a cache of DBM pointers and keep them open.  Versions of the
 * dbm routines that don't support more than one database will have to
 * have a cache-size of one!
 * I am not sure what the impact of this would be on performance; for
 * adding a new file it shouldn't be too bad, as the file list is examined
 * only once for each file, during reading, and the word database is looked
 * at (at least once for each distinct word) only on writing.
 * For retrieval, however, the word database will be looked at for each
 * word in the query, and the file database for (potentially) each match
 * of each word, so the requests will be more interspersed.
 * Under no circumstances is it acceptable to dispense with the cache, as
 * otherwise you will be doing (literally) thousands of calls to
 * open() and close() per second!
 *
 */

#undef startdb

#ifndef CACHE
# define CACHE 4 /* too many and we'll run out of file descriptors */
#endif

typedef struct s_DatabaseCache {
    char *Name;
    DBM *Value;
    struct s_DatabaseCache *Next;
    short NameLength;
} t_DatabaseCache;

static int MaxInCache = CACHE;
static int NumberInCache = 0;
static t_DatabaseCache *DatabaseCache;

/* FileFlags and Mode are passed to dbm_open */
static int FileFlags = O_RDONLY;
static int FileModes = 0;

void
lqWriteAccess()
{
    FileFlags = O_RDWR|O_CREAT;
    FileModes = 0664; /* owner and group write, others read only */
}

void
lqGetFileModes(Flagsp, Modesp)
    int *Flagsp;
    int *Modesp;
{
    if (!Flagsp || !Modesp) {
	Error(E_BUG, "lqGetFileModes(Flagsp=0x%x, Modesp=0x%x): %s zero",
	    Flagsp, Modesp,
	    (Flagsp) ? "Flagsp" : "Modesp"
	);
    }
    *Flagsp = FileFlags;
    *Modesp = FileModes;
}

DBM *
startdb(FilePrefix)
    char *FilePrefix;
{
    extern int errno;
    t_DatabaseCache *cp;
    int NameLength = strlen(FilePrefix);

    for (cp = DatabaseCache; cp; cp = cp->Next) {
	if (cp->NameLength == NameLength && cp->Value &&
				strcmp(cp->Name, FilePrefix) == 0) {
	    return cp->Value;
	}
    }

    /* assert: cp == 0 */

    /* not in the cache */

    /* if the cache is too big, close one entry, the last one */
    if (NumberInCache > MaxInCache) {
	t_DatabaseCache **cpp;

	for (cpp = &DatabaseCache; (*cpp); cpp = &(*cpp)->Next) {
	    if (!(*cpp)->Next) break;
	}

	if (*cpp && !(*cpp)->Next) {
	    /* Actually if this isn't true, MaxInCache is probably zero! */

	    if ((*cpp)->Value) {
		(void) dbm_close((*cpp)->Value);
		(*cpp)->Value = (DBM *) 0;
	    }
	    if ((*cpp)->Name) {
		(void) efree((*cpp)->Name);
		(*cpp)->Name = (char *) 0;
	    }
	    cp = (*cpp);
	    *cpp = (t_DatabaseCache *) 0; /* so it isn't pointed to any more */
	    --NumberInCache;
	} else {
	    Error(E_BUG,
		"startdb(%s) - cache is full up, none can be discarded",
		FilePrefix
	    );
	}
    }
    
    if (!cp) {
	cp = (t_DatabaseCache *) emalloc(sizeof(t_DatabaseCache));
    }

    errno = 0;

    if ((cp->Value = dbm_open(FilePrefix, FileFlags, FileModes)) == (DBM *) 0){
	Error(E_FATAL|E_SYS,
	    "couldn't open dbm database \"%s\"; is $LQTEXTDIR correct?",
	    FilePrefix
	);
    }

    cp->NameLength = NameLength;
    cp->Name = emalloc(NameLength + 1);
    (void) strcpy(cp->Name, FilePrefix);
    /* Put the new element at the start of the list, since if we just called
     * dbstart, we're certain to want this dbm database almost immediately,
     * and in any case before any other database.
     */
    cp->Next = DatabaseCache; /* cp->Next was previously invalid */
    DatabaseCache = cp;
    ++NumberInCache;

    return cp->Value;
}

#undef enddb

/*ARGSUSED*/
void
enddb(db)
    DBM *db;
{
    /* no-op */
    /* This could check the named db and move it to the bottom of the list,
     * I suppose.
     */
}

void
cleanupdb()
{
    register t_DatabaseCache *cp;
    t_DatabaseCache *Next = 0;

    cp = DatabaseCache;
    while (cp) {
	if (cp->Value) {
	    (void) dbm_close(cp->Value);
	}
	if (cp->Name) {
	    (void) efree(cp->Name);
	}

	Next = cp->Next;
	(void) efree((char *) cp);
	/* can no longer refer to cp->Next...*/
	cp = Next;
    }
    NumberInCache = 0;
}
