/* lqaddfile.c -- Copyright 1989, 1990 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* addfile -- add a file to the LQ-Text text retrieval index
 * Liam Quin, August 1989 and later...
 *
 * $Id: lqaddfile.c,v 1.20 92/03/22 22:14:53 lee Exp $ 
 */

static char *Version = "@(#) $Id: lqaddfile.c,v 1.20 92/03/22 22:14:53 lee Exp $";

#ifdef SYSV
extern int _filbuf(); /* used but not defined in stdio.h */
#endif
#include <stdio.h>
#include <malloc.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef BSD
# include <strings.h>
#else
# include <string.h>
#endif

#include "globals.h" /* defines and declarations for database filenames */
#include "error.h"

#include "fileinfo.h"
#include "wordinfo.h"
#include "wordrules.h"
#include "filter.h"
#include "emalloc.h"

#define enew(var, type) (var = (type *) emalloc(sizeof(type)))

#ifdef SYSV
#define TOLOWER(ch) ch = tolower(ch)
#else
#define TOLOWER(ch) if (isupper(ch)) ch = tolower(ch)
#endif

void DestroyFileInfo(), SaveFileInfo(), AddStream(), AddFrom();
extern lqWriteAccess(); /* Allow write access to the database */
/* Symbol Table Interface */
extern void AddWord(), WriteCurrentMaxWID();
extern void DumpCache(), cleanupdb();
extern char *WordRoot();
extern int TooCommon(), GetFilterType();
int RealGetChar(), AddFile();

/** System calls and library routines used in this file: **/
/** System calls: **/
extern void exit();
extern int stat();
/** Library Functions: **/
extern int atoi();
#ifndef tolower
 extern int tolower();
#endif
extern void perror();
/**/

char *progname = "@(#) $Id: lqaddfile.c,v 1.20 92/03/22 22:14:53 lee Exp $";

#ifdef USE_LINENUMBERS
static int UseLineNumbers = 0;
#endif

/* FROM pblock.c */
extern int AsciiTrace; /* provide increasingly verbose info if not zero */

static int LastChar = 0;
static int _chForLee = 0;

#define GetChar(F) \
    ( LastChar ? \
	(++BytesRead, (_chForLee = LastChar), (LastChar = 0), _chForLee) : \
	( (_chForLee = getc(FileInfo->Stream)) != '\'' || !InWord) ? \
		(++BytesRead, _chForLee) : RealGetChar(F) )
	 
int
main(argc, argv)
    int argc;
    char *argv[];
{
    extern char *strrchr();
    extern int getopt(), cknatstr();
    extern void SetDefaults();
    extern char *optarg;
    extern int optind;
    extern int MaxWordsInCache; /* see wordtable.c */

    int c;
    int ErrorFlag = 0;
    int DoNothing = 0;
    char *InputFile = (char *) 0;

#ifdef MALLOCTRACE
    malloc_debug(2);
#endif

    progname = argv[0]; /* retain the full path at first */

#ifdef M_MXFAST
    /* The new symbol table doesn't allocate large numbers of t_WordPlace,
     * so we want this number small...
     */
# if 0
    (void) mallopt(M_MXFAST, sizeof(t_WordPlace));
# else
    (void) mallopt(M_MXFAST, 8);
# endif
    /* may need to comment mallopt() out entirely for BSD -- use ifndef.
     * seems to work under SunOS, though.
     * When it works, it says "Allocate 100 or so chunks of this size at a
     * time, and whenver I ask for this much or less, give me one of the
     * chunks".  Clearly it had better not be too large, but it is a big
     * win with a structure allocated for every occurrence of every word!
     */
#endif

    SetDefaults(argc, argv);

    while ((c = getopt(argc, argv, "w:f:xVZz:")) != -1) {
	switch (c) {
	case 'w':
	    if (!cknatstr(optarg)) {
		fprintf(stderr,
			    "%s: -w must be given a number >= 0, not \"%s\"\n",
							    progname, optarg);
		fprintf(stderr, "\tuse %s -xv for further information\n");
		exit(1);
	    }
	    MaxWordsInCache = atoi(optarg);
	    break;
	case 'Z':
	case 'z':
	    break; /* work done in SetDefault() */
	case 'V':
	    fprintf(stderr, "%s: version: %s\n", progname, Version);
	    DoNothing = 1;
	    break;
	case 'f':
	    if (InputFile) {
		fprintf(stderr,
"%s: only one -f option allowed; use -xv for explanation\n", progname);

		exit(1);
	    }
	    InputFile = optarg;
	    break;
	case 'x':
	    ErrorFlag = (-1);
	    break;
	default:
	case '?':
	    ErrorFlag = 1;
	}
    }

    if ((progname = strrchr(progname, '/')) != (char *) NULL) {
	++progname; /* step over the last / */
    } else {
	progname = argv[0];
    }

    if (ErrorFlag > 0) {
	fprintf(stderr, "use %s -x or %s -xv for an explanation.\n",
							progname, progname);
	exit(1);
    } else if (ErrorFlag < 0) { /* -x was used */
	fprintf(stderr, "%s -- add files to an lq-text retrieval database\n",
								    progname);

	fputs("Options are:\n\
	-f file -- read the list of files to index from \"file\"\n\
	-c file	-- cfile contains a list of common words to be ignored\n\
	-d dir	-- use the lq-text database in the named directory\n\
	-t N	-- set the trace level to N [default: N = 0]\n\
	-V	-- print Version number and exit\n\
	-v	-- be verbose (equivalent to -t 1)\n\
	-w n	-- dump the word-cache every n words\n\
	-x	-- print this eXplanation and exit\n\
	--	-- all following arguments are file names\n\
\n\
", stderr);
	if (AsciiTrace == 1) {
	    /* used -v or -t1 */
	fprintf(stderr, "\n\
    Any remaining arguments are taken to be file names.  The current\n\
DOCPATH (%s) is searched for the files, and they are read and added\n\
to the index.  (If you use the -f option, you should not give filename\n\
arguments on the command line, although you can use \"-f -\" to read the\n\
list of files from standard input, one per line.\n\
Setting (with -w) the size of the cache may dramatically\n\
improve performance.  Systems with memory larger than the data can try -w0.\n\
See lqtext(1) for more information.\n", DocPath);
	}
    exit(0);

    }

#ifdef WIDINBLOCK
    {
	char *S = "********";
# ifdef ASCIITRACE
	/* remind people to recompile... */
	fprintf(stderr,
	    progname,
	    "%s\n%s Compiled with -DWIDINBLOCK for debugging\n%s\n",
	    s,
	    s,
	    s
	);
# else
	/* don't allow -DWIDINBLOCK without -DASCIITRACE */
	Error(E_BUG, "Compiled with -DWIDINBLOCK but not -DASCIITRACE!");
	syntax error; this prevents compilation here;
# endif /* ASCIITRACE */
    }
#endif

    if (DoNothing) {
	if (optind < argc) {
	    fprintf(stderr, "%s: warning: %d extra argument%s ignored...\n",
				progname, argc - optind,
				argc - optind == 1 ? "" : "%s" );
	    fprintf(stderr, "Use %s -x for an explanation\n", progname);
	}
	exit(0);
    }

    lqWriteAccess();

    if (InputFile) {
	if (optind < argc) {
	    fprintf(stderr, "%s: -f: too many arguments; use -xv\n", progname);
	    exit(1);
	}
	AddFrom(InputFile);
    } else for (; optind < argc; ++optind) {
	if (AddFile(argv[optind]) < 0 && AsciiTrace >= 1) {
	    fprintf(stderr, "%s: %s not added to index\n",
			progname, argv[optind]);
	}
    }

#ifndef MALLOCTRACE
    DumpCache(0); /* the 0 means don't bother calling free() */
#else
    DumpCache(1); /* Free everthing so whatever is left is a memory leak */
#endif

    cleanupdb(); /* empty the dbm cache */
    WriteCurrentMaxWID();

#ifdef MALLOCTRACE
    (void) fprintf(stderr, "%s: Malloctrace: checking...\n", progname);
    malloc_verify();
    (void) fprintf(stderr, "%s: Malloc Map\n", progname);
    mallocmap();
#endif

    exit(0);
    /*NOTREACHED*/
    return 1; /* disaster if we get here -- it's just for lint! */
}

void
AddFrom(Name)
    char *Name;
{
    extern char *GetLine();

    FILE *fp;
    char *Line;

    if (Name[0] == '-' && Name[1] == '\0') {
	fp = stdin;
    } else {
	fp = fopen(Name, "r");
    }

    if (fp == (FILE *) 0) {
	Error(E_FATAL|E_SYS, "-f: can't open \"%s\"", Name);
    }

    while ((Line = GetLine(fp, Name)) != (char *) 0) {
	if (AddFile(Line) < 0 && AsciiTrace >= 1) {
	    /* we already got one error message from AddFile() */
	    Error(E_WARN, "%s not added to database.", Line);
	}
    }

    if (fp != stdin) {
	(void) fclose(fp);
    }
}

extern int fclose(), pclose();

t_FileInfo *
MakeFileInfo(Name)
    char *Name;
{
#ifdef BSD
    extern time_t time();
#else
    extern long time();
#endif
    extern t_FID Name2FID();
    extern t_FileInfo *GetFileInfo();
    extern t_FID GetNextFID();
    FILE *MakeInput();
    struct stat StatBuf;

    t_FileInfo *FileInfo = 0;
    t_FID FID;

    if (!Name || !*Name) return (t_FileInfo *) 0; /* sanity */

    if (stat(Name, &StatBuf) < 0) {
#ifndef FindFile /* it is a macro these days... */
	extern char *FindFile();
#endif
	extern int errno;

	int e = errno;
	char *doc;

	if ((doc = FindFile(Name)) == (char *) 0) {
	    errno = e;
	    Error(E_SYS, "Can't index \"%s\"", Name);
	    return (t_FileInfo *) 0;
	}

	{
	    int len = strlen(doc);

	    if (len > 3 && doc[len - 2] == '.' && doc[len - 1] == 'Z') {
		/* TODO: allow lqaddfile and filters to unpack archives etc. */
		fprintf(stderr,
		    "%s: %s: files must be compressed after indexing\n",
				progname, doc
		);
		return (t_FileInfo *) 0;
	    }
	}

	if (stat(doc, &StatBuf) < 0) {
	    Error(E_SYS, "Can't index \"%s\"", Name);
	    return (t_FileInfo *) 0;
	}
	Name = doc;
    }

    if (StatBuf.st_size == 0L) {
	if (AsciiTrace) {
	    fprintf(stderr, "%s: %s empty -- not indexed\n", progname, Name);
	}
	return (t_FileInfo *) 0;
    }
    /* See if it's in the index already: */
    if ((FID = Name2FID(Name)) != (t_FID) 0) {

	if ((FileInfo = GetFileInfo(FID)) != (t_FileInfo *) 0) {
	    /* Check to see if the file hass changed since it was last
	     * indexed.  If it has, we should delete the old one from
	     * the database and give this one a new FID, but I have
	     * not done that yet -- that's /usr/local/lib/lqtextd or
	     * something, I suppose!
	     */
	    if (FileInfo->Date >= StatBuf.st_mtime &&
				FileInfo->FileSize == StatBuf.st_size) {
		if (AsciiTrace) {
		    fprintf(stderr, "%s: %s unchanged -- not indexed\n",
				progname, Name);
		}
		DestroyFileInfo(FileInfo);
		return (t_FileInfo *) 0;
	    }
	}
    } else {
	FID = GetNextFID((long) StatBuf.st_size);
    }

    if (FileInfo == (t_FileInfo *) 0) {
	/* Allocate Structure */
	enew(FileInfo, t_FileInfo);

	/* Although not always necessary, call emalloc here so that a
	 * FileInfo can always be deleted with DestroyFileInfo()
	 */
	FileInfo->Name = emalloc((unsigned)(strlen(Name) + 1));
	(void) strcpy(FileInfo->Name, Name);

	/* Other bits to set: */

	/* date */
	FileInfo->Date = StatBuf.st_mtime;
	FileInfo->FileSize = StatBuf.st_size;

	/* file type */
	if ((FileInfo->FilterType = GetFilterType(FileInfo, &StatBuf)) < 0) {
	    if (AsciiTrace) {
		fprintf(stderr, "%s: %s unknown file type -- not indexed\n",
							progname, Name);
	    }
	    (void) efree(FileInfo->Name);
	    (void) efree((char *) FileInfo);
	    return (t_FileInfo *) 0;
	}
    }

    FileInfo->FID = FID;
    FileInfo->Date = (long) time((long *) 0); /* it's a time_t on BSD */

    if ((FileInfo->Stream = MakeInput(FileInfo)) == (FILE *) 0) {
	fprintf(stderr, "%s: couldn't open filter for %s -- not indexed\n",
						    progname, FileInfo->Name);
	(void) efree(FileInfo->Name);
	(void) efree((char *) FileInfo);
	return (t_FileInfo *) 0;
    }

    return FileInfo;
}

void
DestroyFileInfo(FileInfo)
    t_FileInfo *FileInfo;
{
    if (FileInfo->Stream) {
	if (FileInfo->FilterType >= 0 && FileInfo->FilterType < MaxFilterType){
	    (* FilterTable[FileInfo->FilterType].close)(FileInfo->Stream);
	}
	FileInfo->Stream = (FILE *) 0;
    }
    if (FileInfo->Name) (void) efree(FileInfo->Name);
    (void) efree((char *) FileInfo);
}

int
AddFile(Name)
    char *Name;
{
    t_FileInfo *FileInfo;

    if (!Name || !*Name) return -1;
    if ((FileInfo = MakeFileInfo(Name)) == (t_FileInfo *) 0) return -1;

    AddStream(FileInfo);
    SaveFileInfo(FileInfo);
    DestroyFileInfo(FileInfo);

    return 0;
}

FILE *
MakeInput(FileInfo)
    t_FileInfo *FileInfo;
{
    FILE *fp;
    char *Buffer;
    unsigned BufLen;
    extern FILE *fopen(), *popen();

#define FSTRING FilterTable[FileInfo->FilterType].String

    if (FileInfo->FilterType > MaxFilterType) {
	Error(E_WARN, "filter type %d for %s too high (max %d)",
		FileInfo->FilterType, FileInfo->Name, MaxFilterType);
	return (FILE *) 0;
    }

    if (FilterTable[FileInfo->FilterType].Type != FileInfo->FilterType) {
	Error(E_FATAL|E_INTERNAL,
	    "Filter table entry %d has type %d, expected %d",
	    FileInfo->FilterType,
	    FilterTable[FileInfo->FilterType].Type,
	    FileInfo->FilterType
	);
    }

    if (FSTRING == (char *) 0) {
	return fopen(FileInfo->Name, "r");
    }

    BufLen = strlen(FileInfo->Name) * 2 + 4 + strlen(FSTRING);
	/* The +4 is to allow for an embedded " < " plus a \0;
	 * we append "< Name", but also expand %s to be the Name, hence
	 * the strlen * 2
	 */
    Buffer = emalloc(BufLen);

    (void) sprintf(Buffer, FSTRING, FileInfo->Name);
    (void) strcat(Buffer, " < ");
    (void) strcat(Buffer, FileInfo->Name);

    fp = popen(Buffer, "r");
    (void) efree(Buffer);
    return fp;
}

static long BytesRead = 0L;
static int InWord = 0;

/* Character input */

INLINE int
RealGetChar(FileInfo)
    t_FileInfo *FileInfo;
{
    /* ASSERT: InWord && _chForLee == '\'' */
    LastChar = getc(FileInfo->Stream);
    if (WithinWord(LastChar) && LastChar != '\'') {
	BytesRead++;
	return '\'';
    } else {
	/* delete the single quote, as it was at the end of
	 * a word, not in the middle
	 */
	BytesRead++;
	return ' ';
    }
    /*NOTREACHED*/
    /* exit(1); */
}

static char ThisWord[MaxWordLength + 1];
static char ThatWord[MaxWordLength + 1];

t_WordInfo *
ReadWord(FileInfo)
    t_FileInfo *FileInfo;
{
    /* use two static storage areas so we can be called twice in a row.
     * This is necessary to implement the WPF_LASTINBLOCK flag.
     */
    static t_WordInfo This, That;
    static int ThisOrThat = 0;
    register char *q;
    t_WordInfo *WordInfo;
    char *Buffer;
    int ch;
    static int WordInBlock = -1;
    static t_FID LastFid = 0L;
    static long LastPos = 0L;
    static int SawCommon = 0;
    static int SawLetters = 0;
    static int BlockInFile = 0L;
    static unsigned long LastBlock;
    unsigned long Start;

    if (ThisOrThat) {
	WordInfo = &This;
	WordInfo->Word = ThisWord;
    } else {
	WordInfo = &That;
	WordInfo->Word = ThatWord;
    }

    if (FileInfo->FID != LastFid) {
	/* Initialise the static WordInfo variables that are used to
	 * return functions, if not already done:
	 */
	if (LastFid == (t_FID) 0) {
	    WordInfo->WID = (t_WID) 0;
	    WordInfo->Next = (t_WordInfo *) 0;
	    WordInfo->WordPlaces = (t_WordPlace *) 0;
	    WordInfo->DataBlock = (char *) 0;
	    WordInfo->WordPlacesInHere = 0;
	}
	WordInBlock = (-1); /* none, yet! */
	LastPos = BlockInFile = LastBlock = 0L;
	BytesRead = 0L;
	SawCommon = SawLetters = 0;
	if (AsciiTrace) {
	    fprintf(stderr, "%s: Reading file \"%s\"",
			progname, FileInfo->Name);
	}
	This.Word = ThisWord;
	That.Word = ThatWord;
	LastFid = FileInfo->FID;
    }
    
    if (WordInfo->FID != LastFid) {
	WordInfo->WordPlace.FID = WordInfo->FID = LastFid;
	WordInfo->WID = (t_WID) 0;
	WordInfo->Next = (t_WordInfo *) 0;
	WordInfo->WordPlaces = (t_WordPlace *) 0;
	WordInfo->DataBlock = (char *) 0;
	WordInfo->WordPlacesInHere = 0;
	This.Word = ThisWord;
	That.Word = ThatWord;
    }

    /* The following loop is simply to avoid needless recursion. */
    for (;;) {

	q = Buffer = WordInfo->Word;

	/* Skip non-word characters */
	while ((ch = GetChar(FileInfo)) != EOF) {
	    if (StartsWord(ch)) {
		break;
#if 0
	    } else if (isdigit(ch)) {
		/* the character was something that is not the start of the
		 * word, but it might not have been a space -- for example,
		 * if we're not indexing numbers, consider 0xFFE6, which
		 * we don't want to index as xFFE6!
		 */
		do {
		    ch = GetChar(FileInfo);
		} while (isalnum(ch));

		if (ch == EOF) break;
#endif
	    }
	}

	/* ASSERT: we have read at least one character */

	if (ch == EOF) {
	    if (AsciiTrace) {
		fprintf(stderr, "\n");
	    }
	    return (t_WordInfo *) 0;
	}

	Start = BytesRead - 1;

#ifdef USE_LINENUMBERS
	if (UseLineNumbers) {
	    extern int lqLineInFile();

	    BlockInFile = lqLineInFile();
	} else {
	    BlockInFile = Start / FileBlockSize;
	}
#else /*!USE_LINENUMBERS */
	    BlockInFile = Start / FileBlockSize;
#endif /*USE_LINENUMBERS */

	if (BlockInFile != LastBlock) {
	    LastBlock = BlockInFile;
	    if (AsciiTrace > 1) {
		(void) fprintf(stderr, ".");
		(void) fflush(stderr);
	    }
	    WordInBlock = (-1);
	}

	if (isupper(ch)) {
	    WordInfo->WordPlace.Flags = WPF_UPPERCASE;
	    ch = tolower(ch);
	} else {
	    WordInfo->WordPlace.Flags = 0;
	}

	InWord = 1; /* For GetChar() */

	do {
	    if (q - Buffer < MaxWordLength) {
		*q++ = ch;
	    }
	    ch = GetChar(FileInfo);
	    TOLOWER(ch);
	} while (WithinWord(ch) || EndsWord(ch));

	*q = '\0';
	InWord = 0;

#ifdef __GNUC__
	/* this is to get round a gcc bug... */
	{
	    int i = q - Buffer;
	    WordInfo->Length = i;

	    if (i < MinWordLength) {
		register char *p;

		for (p = Buffer; p < q; p++) {
		    if (isalpha(*p)) {
			SawLetters = 1;
			break;
		    }
		}
		continue;
	    }
	}
#else
	if ((WordInfo->Length = q - Buffer) < MinWordLength) {
	    register char *p;

	    for (p = Buffer; p < q; p++) {
		if (isalpha(*p)) {
		    SawLetters = 1;
		    break;
		}
	    }
	    continue;
	}
#endif

	(void) WordRoot(WordInfo);

	WordInfo->Length = strlen(WordInfo->Word);

	if (TooCommon(WordInfo)) {
	    SawCommon++;
	    WordInBlock++;
#ifdef ASCIITRACE
	    if (AsciiTrace > 9) {
		fprintf(stderr, "%s: %s too common to index\n",
				progname, WordInfo->Word);
	    }
#endif
	    continue;
	} else if (SawCommon) {
	    SawCommon = 0;
	    WordInfo->WordPlace.Flags |= (WPF_LASTWASCOMMON|WPF_LASTHADLETTERS);
	}
	if (SawLetters) {
	    SawLetters = 0;
	    WordInfo->WordPlace.Flags |= WPF_LASTHADLETTERS;
	}

	/* StuffBefore is the # of chars between the end of the last word and
	 * the start of this one.
	 */
	if (Start > 1L) {
	    if (Start - (LastPos + 1) <= 0) { /* save a byte in the index */
		WordInfo->WordPlace.StuffBefore = 1;
	    } else if (Start - (LastPos + 1) >= 127 ) {
		WordInfo->WordPlace.StuffBefore = 127;
	    } else {
		WordInfo->WordPlace.StuffBefore = Start - (LastPos + 1);
	    }
	} else {
	    WordInfo->WordPlace.StuffBefore = 1; /* i.e., the default */
	}

	WordInfo->WordPlace.WordInBlock = (++WordInBlock);
	WordInfo->WordPlace.BlockInFile = BlockInFile;

	WordInfo->Word[WordInfo->Length] = '\0';

	LastPos = BytesRead - 1;

	ThisOrThat = !ThisOrThat;
	/* toggle between 0 and 1.  Boring life, really */

	if (!WordInfo->Word[0]) {
	    Error(E_WARN|E_INTERNAL, "%s: Null word ignored in ReadWord()",
		FileInfo->Name
	    );
	    continue;
	}

	break;
    }
    return WordInfo;
}

void
AddStream(FileInfo)
    t_FileInfo *FileInfo;
{
    /* I have to mark the last word in the block.
     * I do that by marking the previous word if it was in a differant block
     * than the current one.
     */
    t_WordInfo *WordInfo;
    t_WordInfo *LastWord = 0;

    BytesRead = 0;

    while ((WordInfo = ReadWord(FileInfo)) != (t_WordInfo *) 0) {
	if (LastWord) {
	    if (LastWord->WordPlace.BlockInFile !=
	    				WordInfo->WordPlace.BlockInFile) {
		LastWord->WordPlace.Flags |= WPF_LASTINBLOCK;
	    }
	    AddWord(LastWord);
	}
	LastWord = WordInfo;
    }
    if (LastWord) {
	/* it's the last in the file, so it is also the last in the block */
	LastWord->WordPlace.Flags |= WPF_LASTINBLOCK;
	AddWord(LastWord);
    }

    if (AsciiTrace) {
	fprintf(stderr, "%s: %s: indexed ", progname, FileInfo->Name);

	if (BytesRead > 1024) {
	    int bytes = BytesRead & 1023;
	    BytesRead >>= 10;
	    if (BytesRead > 1024) {
		int kbytes = BytesRead & 1023;
		BytesRead >>= 10;
		fprintf(stderr,
			"%ld.%-03.3d Megabytes, (wow!)\n",
			BytesRead, kbytes
		);

		/* Retrieving and displaying a file this size would be
		 * absurd!  Fortunately we don't ever look at the files
		 * when determining matches, but if the user asks to browse...
		 *
		 * This message is printed whenever we're being verbose,
		 * although the message for files > 16k is only printed in
		 * verbose mode because we aren't so sure about it...
		 */
		fprintf(stderr,
	"%s: Note: smaller files give better performance & index size\n",
		    progname);

	    } else {
		fprintf(stderr, "%d.%-02.2d Kilobytes\n", BytesRead, bytes);
		if (BytesRead > 16) {
		    /* Files larger than 16K have block numbers using two
		     * bytes, so they take up more index space.  To some
		     * extent this is offset by the saving in storing
		     * fewer FIDs, though.
		     */
		    if (AsciiTrace > 1) {
			fprintf(stderr,
	"%s: Note: smaller files may give better performance & index size\n",
			    progname);
		    }
		}
	    }
	} else {
	    fprintf(stderr, "%d bytes\n", BytesRead);
	}
    }
}

/* lqaddfile has been carried through several incarnations of lq-text,
 * and hence has more than one Inital Revision in the following history.
 *
 * $Log:	lqaddfile.c,v $
 * Revision 1.20  92/03/22  22:14:53  lee
 * Check for WIDINBLOCk and exit if ASCIITRACE not defined, else print msg.
 * 
 * Revision 1.19  92/02/15  05:44:40  lee
 * Improved report of bytes added, and added a couple of comments.
 * 
 * Revision 1.18  92/02/15  05:40:53  lee
 * now uses replacement for writewordchain, and has a numbner of speedups.
 * 
 * Revision 1.16  92/02/01  22:00:06  lee
 * Took GetLine() out and put it in lqutil.
 * Removed recursion in ReadWord(), and fixed a bug that made hex numbers
 * get indexed wrongly (e.g. 0xfa6b being indexed as xfa6b instead of not
 * at all).
 * 
 * Revision 1.15  92/01/29  17:41:20  lee
 * checks file is not compressed.
 * 
 * Revision 1.14  91/03/02  21:22:39  lee
 * Added write access call.
 * 
 * Revision 1.13  91/03/02  18:53:25  lee
 * Common words are now counted, so you can now edit the common word list
 * without invalidating the index.
 * 
 * Revision 1.12  90/10/06  00:50:54  lee
 * Prepared for first beta release.
 * 
 * Revision 1.11  90/10/05  23:46:11  lee
 * Allow compilation with -UASCIITRACE
 * 
 * Revision 1.10  90/10/04  17:54:46  lee
 * fixed a typo in the usage message.
 * 
 * Revision 1.9  90/09/28  23:20:22  lee
 * Put more of GetChar into a macro and parameterised TOLOWER.
 * 
 * Revision 1.8  90/09/28  22:19:04  lee
 * Did the previous fix _properly_!
 * 
 * Revision 1.7  90/09/28  22:12:35  lee
 * Made getchar a macro, and deleted the call to CallFree...
 * 
 * Revision 1.6  90/09/20  18:46:03  lee
 * Closed up a (very small) memory leak.
 * 
 * Revision 1.5  90/09/19  20:16:41  lee
 * Fixed problems associated with indexing an empty file.
 * 
 * Revision 1.4  90/08/29  21:45:18  lee
 * Alpha release
 * 
 * Revision 1.3  90/08/09  19:17:12  lee
 * *** empty log message ***
 * 
 * Revision 1.1  90/02/27  11:05:02  lee
 * Initial revision
 * 
 * Revision 2.2  89/10/08  20:45:13  lee
 * Working version of nx-text engine.  Addfile and wordinfo work OK.
 * 
 * Revision 2.1  89/10/02  01:14:12  lee
 * New index format, with Block/WordInBlock/Flags/BytesSkipped info.
 * 
 * Revision 1.3  89/09/17  23:02:42  lee
 * Various fixes; NumberInBlock now a short...
 * 
 * Revision 1.2  89/09/16  21:16:11  lee
 * First demonstratable version.
 * 
 * Revision 1.1  89/09/07  21:05:52  lee
 * Initial revision
 *
 */
