/* lqshow.c -- Copyright 1989, 1990 Liam R. Quin.  All Rights Reserved.
 * This code is NOT in the public domain.
 * See the file COPYRIGHT for full details.
 */

/* lqshow -- show a file according to keywords, highlighting matches
 * Liam R. Quin, September 1989 and later...
 *
 * $Id: lqshow.c,v 1.17 92/04/02 14:15:12 lee Exp $
 */

#include "globals.h" /* defines and declarations for database filenames */
#include "error.h" /* actually should use ShowInfo() really */
/* TODO: use SetErrorHandler() */

#ifdef SYSV
 /* for lint: */
 extern int w32addch();
#endif
#ifdef ultrix
# include <cursesX.h>
#else
# include <curses.h>
#endif
#include <malloc.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h> /* for fileinfo.h */
#include <sys/stat.h>

/* Check for old (or BSD) curses: */
#define HASUNDERLINE
#ifndef A_STANDOUT
# undef HASUNDERLINE
# define A_STANDOUT 10193 /* random */
# define A_UNDERLINE 312
# define attrset(a) ((a == 0) ? standend() : standout())
typedef char chtype; /* long on sysV */
static void
beep() {
    fprintf(stderr, "\007");
    fflush(stderr);
}
#else
# ifndef beep
 extern int beep();
# endif
#endif

#include "fileinfo.h"
#include "wordinfo.h"
#include "wordrules.h"
#include "pblock.h"
#include "emalloc.h"
#include "readfile.h"

/** Unix system calls that need declaring: **/
extern long lseek();
extern int open(), close();
extern int read();
extern void exit();
extern int stat();

/** Unix/C Library Functions that need declaring: **/
#ifndef tolower
 extern int tolower();
#endif
extern int strlen();
extern int strcmp();
extern unsigned sleep();
extern int atoi();
extern long atol();
extern void perror();

/** Curses library functions: **/
#ifdef SYSV
extern int box32();
#endif
#ifndef nonl
 extern int nonl();
#endif /**nonl*/
#ifndef noecho
extern int noecho();
#endif
#ifndef wmove
 extern int wmove();
#endif
#ifndef waddstr
 extern int waddstr();
#endif
#ifndef wrefresh
 extern int wrefresh();
#endif
#ifndef wprintw
 extern int printw();
#endif
extern int mvwprintw(), delwin(), wclear(), wclrtoeol(), endwin();

/** lqtext library functions that need declaring: **/
extern int MySystem();
extern int TooCommon();
extern void SetDefault();
extern void DefaultUsage();

/** Functions within this file that are used before being defined: **/
long ReadMatchFile();
void ShowFile();
void Output();

/** **/

/** some useful macros: **/
#define max(choir,boy) (((choir)>(boy))?(choir):(boy))
#define min(choir,boy) (((choir)<(boy))?(choir):(boy))

/** **/

int AsciiTrace = 0;

/* number of lines above and below each match to show by default. */
#define DFLTABOVE 6
#define DFLTBELOW 9

int LinesBelow = DFLTBELOW;
int LinesAbove = DFLTABOVE;

#define DISPLAY_TOP 3

extern int errno;

char *progname;
int ThisRow, ThisCol;
int SelectedNames = -1;
FILE *InfoStream = 0;

static char *Revision = "@(#) showfile.c 2.2";

int
main(argc, argv)
    int argc;
    char *argv[];
{
    extern int optind, getopt();
    extern char *optarg; /* for getopt */
    int ch; /* for getopt */
    int ErrFlag = 0; /* see how getopt makes programs cleaner? */
    int NumberOfFiles;
    char **Origv;
    int Origc;
    char *FileWithMatches = (char *) 0;
    char **MatchList;
    int MatchCount = 0;

    progname = argv[0];

    SetDefaults(argc, argv);

    /* All lq-text programs must call SetDefaults() before getopt, and
     * must then be prepared to ignore options z with arg and Z without.
     */
    while ((ch = getopt(argc, argv, "a:b:f:o:z:ZVvx")) != EOF) {
	switch (ch) {
	case 'z':
	    break; /* done by SetDefaults(); */
	case 'V':
	    fprintf(stderr, "%s version %s\n", progname, Revision);
	    break;
	case 'v':
	    AsciiTrace = 1;
	    break;
	case 'a': /* lines above */
	    LinesAbove = atoi(optarg); /* need cknum() */
	    break;
	case 'b':
	    LinesBelow = atoi(optarg);
	    break;
	case 'f':
	    FileWithMatches = optarg;
	    break;
	case 'o': /* -o fd --- write the selected files to fp */
	    if (SelectedNames >= 0) {
		fprintf(stderr,
		"%s: -o %d -o %s: you must not give more than one -o option.\n",
					progname, SelectedNames, optarg);
		ErrFlag = (-1);
	    } else {
		if (!isdigit(*optarg)) {
		    fprintf(stderr, "%s: -o must be followed by a number\n",
								progname);
		    exit(1);
		}
		SelectedNames = atoi(optarg);
		break;
	    }
	    break;
	case 'x':
	    ErrFlag = (-1);
	    break;
	case '?':
	default:
	    ErrFlag = 1;
	}
    }

    if (ErrFlag < 0) { /* -x or -xv was used */
	fprintf(stderr, "usage: %s [-xv] [options] [matches...]\n", progname);
	fprintf(stderr,
	"use %s -x, -xv or -xvv for more detailed explanations.\n", progname);

	if (AsciiTrace) {
	    DefaultUsage();
	    fprintf(stderr, "\n\
	-a above - set the number of lines shown above each matching\n\
		   match to \"above\" [default is %d]\n", DFLTABOVE);
	    fprintf(stderr, "\
	-b below - set the number of lines shown below each match\n\
		   match to \"above\" [default is %d]\n", DFLTBELOW);
	    fprintf(stderr, "\
	-f file -- \"file\" contains a list of matches, one per line\n");
	}
	if (AsciiTrace > 1) {
	    fputs("\
	Matches should be in the form of\n\
		BlockNumber  WordInBlock  FileName\n\
	where BlockBumber and WordInBlock are positive numbers.\n\
	(This is the format produced by the lqword -l command.)\n\
", stderr);
	}
	exit(0);
    } else if (ErrFlag > 0) {
	fprintf(stderr, "use %s -x for an explanation.\n", progname);
	exit(1);
    }

    /* open the file for the selected output */
    if (SelectedNames > 0) {
	if ((InfoStream = fdopen(SelectedNames, "w")) == (FILE *) 0) {
	    int e = errno;

	    fprintf(stderr, "%s: -o %d: can't open stream ",
	    					progname, SelectedNames);
	    errno = e;
	    perror("for writing");
	    exit(1);
	}
    }

    argv += optind;
    argc -= optind;

    /* check that we can get at the file containing the matches, if one
     * was supplied.
     */
    if (FileWithMatches) {
	struct stat StatBuf;
	char *msg = 0;

	if (stat(FileWithMatches, &StatBuf) < 0) {
	    int e = errno; /* on many systems, fprintf() changes errno! */
	    fprintf(stderr, "%s: can't open match-list file ", FileWithMatches);
	    errno = e;
	    perror(progname);
	    exit(1);
	} else if (AsciiTrace) {
	    switch (StatBuf.st_mode & S_IFMT) {
	    case S_IFDIR:
		fprintf(stderr,
		"%s: can't read matches from \"%s\" -- it's a directory!\n",
						progname, FileWithMatches);
		exit(1);
	    case S_IFREG:
		break;
#ifdef S_IFIFO
	    case S_IFIFO:
		msg = "named pipe or fifo";
		/* fall through */
#endif
	    case S_IFCHR:
		if (!msg) msg = "raw special device";
		/* fall through */
	    case S_IFBLK:
		if (!msg) msg = "block special device";
		/* fall through */
#ifdef S_IFNAM
	    case S_IFNAM:
		if (!msg) msg = "named special file"; /* wot dat? */
		/* fall through */
#endif
	    default:
		if (!msg) msg = "special file";

		fprintf(stderr,
		    "%s: warning: file \"%s\" containing matches is a %s\n",
		    progname, FileWithMatches, msg);
		
		/* but continue anyway... */

	    }
	}
	/* Now read the file, and make an array of matches... */
	if (ReadMatchFile(FileWithMatches, StatBuf.st_size, &MatchCount, &MatchList) < 0) {
	    fprintf(stderr, "%s: couldn't read matches from \"%s\"\n",
						progname, FileWithMatches);
	    exit(1);
	}
    }

    if (MatchCount) {
	argc = MatchCount;
	argv = MatchList;
    }

    if (argc < 3) {
	fprintf(stderr,
	"%s: matches must have at least 3 parts (not %d); use -xv for an explanation\n",
						    progname, argc);
	exit(1);
    } else if (argc % 3) {
	/* Note: I could detect lqword output here (i.e., without -l) */
	fprintf(stderr, "%s: can't understand match format;\n", progname);
	fprintf(stderr, "%s: use -xv for more explanation.\n", progname);
	exit(1);
    }

    Origv = argv;
    Origc = argc;

    ThisRow = DISPLAY_TOP - 1;
    NumberOfFiles = argc / 3;

    initscr();
    nonl();
    raw();
    noecho();

    while (argc > 0) {
	char Buffer[120];
	int Where;

	ThisRow = DISPLAY_TOP;
	ThisCol = (-1);
	ShowFile(argv[2], atol(*argv), (unsigned) atoi(argv[1]), argc - Origc);
	(void) sprintf(Buffer, "Match %d of %d", 
			    NumberOfFiles - (argc / 3) + 1, NumberOfFiles);
	Where = COLS - (strlen(Buffer) + 10);
	mvwaddstr(stdscr, LINES - 1, Where, Buffer);
	refresh();	/* Where (above) is in case mvwaddstr is a macro */

	if (argc > 0) {
	    attrset(A_STANDOUT);
	    mvwaddstr(stdscr, LINES - 1, 0, "[Press SPACE to continue]");
	    attrset(0);
	    wmove(stdscr, 0, 0);
	    (void) refresh();
	    switch (getch()) {
	    case '?':
	    case 'x':
	    case 'h':
	    case 'i':
#ifdef KEY_HELP
	    case KEY_HELP:
#endif
		{
		    WINDOW *HelpWin = newwin(12, 40, 5, (COLS - 40) / 2);

		    if (HelpWin == (WINDOW *) 0) {
			(void) beep();
		    } else {
#ifndef ACS_HLINE
			box(HelpWin, '#', '#');
#else
			box(HelpWin, 0, 0);
			/* Versions of curses with ASC_HLINE take 0 to
			 * mean that line-drawing should be done
			 * "properly".
			 */
#endif
			wmove(HelpWin, 1, 2);
			mvwprintw(HelpWin, 1,2, "x, ?    -- print this explanation");
			mvwprintw(HelpWin, 2,2, "space   -- go to next match");
			mvwprintw(HelpWin, 3,2, "return  -- go to next match");
			mvwprintw(HelpWin, 4,2, "0, ^, F -- go to First match");
			mvwprintw(HelpWin, 5,2, "$, L    -- go to the Last match");
			mvwprintw(HelpWin, 6,2, "n, +    -- go to the next file");
			mvwprintw(HelpWin, 7,2, "p, -    -- go to previous file");
			if (InfoStream) {
			    mvwprintw(HelpWin, 8,2,
					"s, g    -- save this filename");
			    mvwprintw(HelpWin, 9,2,
					"u, d    -- drop this filename");
			}
			mvwprintw(HelpWin, 10,2, "q, Q    -- quit browsing");
			wrefresh(HelpWin);
			(void) getch();
			delwin(HelpWin);
#ifndef CURSESX /* This is because 4.2 BSD a brain-dead curses... */
			clearok(stdscr, TRUE);
			wrefresh(stdscr);
#endif
		    }
		}
		break;
	    case 'q':
	    case 'Q':
		goto AllDone;
		    /* the goto is to surmount an AT&T compiler bug */
	    case '0': /* reset to beginning */
	    case '1':
	    case 'f':  case 'F':
	    case '^': case '6': /* (6 is often unshifted ^) */
		argc = Origc;
		argv = Origv;
		break;
	    case '$': /* to the end */
	    case 'l': case 'L': /* Last match */
		argv += (argc - 3);
		argc = 3;
		break;
	    case 'v': /* view the file -- use PAGER */
		{
		    char Buffer[4096];
		    char *doc;
		    int e = errno;

		    if ((doc = FindFile(argv[2])) == (char *) 0) {
			errno = e;
			perror(argv[2]);
			sleep(2);
			goto AllDone;
		    }

		    (void) sprintf(Buffer, "%s \"%s\"", PAGER, doc);
		    (void) MySystem(Buffer);
		    clearok(stdscr, TRUE);
		    wrefresh(stdscr);
		}
		break;
	    case 's': /* keep this filename for later use */
	    case 'k': case 'g': /* keep, get */
		if (InfoStream) {
		    fprintf(InfoStream, "%c %s\n", 's', argv[2]);
		} else {
		    (void) beep();
		}
		break;
	    case 'd': /* delete this file from the list */
	    case 'u':
		if (InfoStream) {
		    fprintf(InfoStream, "%c %s\n", 'd', argv[2]);
		} else {
		    (void) beep();
		}
		break;
	    case 'R': /* revert to initial state */
		break; /*NOTDONE*/
	    case '-':
	    case 'p':
		{
		    char *p = argv[2];
		    char **Argv = argv;
		    int Argc = argc;

		    while (Argc + 3 <= Origc && STREQ(Argv[2], p)) {
			Argv -= 3;
			Argc += 3;
		    }

		    if (Argc == argc) {
			(void) beep();
		    } else {
			argv = Argv;
			argc = Argc;
		    }
		}
		break;
	    case '+':
	    case 'n':
		{
		    char *p = argv[2];
		    char **Argv = argv;
		    int Argc = argc;

		    while (Argc > 3 && STREQ(Argv[2], p)) {
			Argv += 3;
			Argc -= 3;
		    }

		    if (Argc == argc) {
			(void) beep();
		    } else {
			argv = Argv;
			argc = Argc;
		    }
		}
		break;
	    case 'R' ^ 64: /* control-R */
	    case 'L' ^ 64: /* control-L */
		clearok(stdscr, TRUE);
		wrefresh(stdscr);
		break;
	    case '=':
		clearok(stdscr, TRUE);
		wrefresh(stdscr);
		{
		    FILE *Pager = popen(PAGER, "w");
		    char **p;
		    int i;

		    if (!Pager) {
			(void) beep();
			break;
		    }
		    for (p = Origv, i = Origc; i > 0; i -= 3, p += 3) {
			(void) fprintf(Pager, "%s\n", *p);
		    }
		    (void) pclose(Pager);
		}
		clearok(stdscr, TRUE);
		wrefresh(stdscr);
		break;
	    case ' ':
	    case '\r':
	    case '\n':
		argv += 3;
		argc -= 3;
		break;
	    default:
		(void) beep();
		break;
	    }
	}
    }

AllDone:
    wmove(stdscr, LINES - 1, 0);
    clrtoeol();
    /* Try to revent the screen from scrolling when we exit */
    wmove(stdscr, LINES - 2, 0);
    refresh();
    endwin();
    return 0;
}

long
ReadMatchFile(FileWithMatches, FileSize, MatchCount, MatchList)
    char *FileWithMatches;
    off_t FileSize;
    int *MatchCount;
    char ** *MatchList;
{
    char **Lines;
    char **Result;
    long n_matches;
    int i;
    char **Lppp;

    if (!FileWithMatches || !*FileWithMatches) {
	Error(E_FATAL, "match-list file (from -f) has empty name!");
    }

    n_matches = ReadFile(
	E_FATAL,
	FileWithMatches,
	"match list",
	&Lines, /* yes, a (char ***) */
	UF_IGNBLANKS|UF_IGNSPACES|UF_IGNHASH|UF_ESCAPEOK
    );

    if (n_matches < 1L) {
	Error(E_FATAL,
	    "Match file \"%s\" contains no matches",
	    FileWithMatches
	);
    }

    Result = (char **) malloc((unsigned) n_matches * 3 * sizeof(char *));

    if (Result == (char **) 0) {
	Error(E_FATAL|E_MEMORY, "%u bytes for match list \"%s\"",
	    (unsigned) n_matches * sizeof(char *) * 3,
	    FileWithMatches
	);
    }

    /* Now construc a new argv[] from the file we just read */

    Lppp = &Result[0];
    for (i = 0; i < n_matches; i++) {
	register char *p;

	p = Lines[i];
	/* ASSERT: There are no leading or trailing spaces on the line */
	if (!*p) {
	    continue; /* blank line */
	}

	/* block in file */
	*Lppp++ = p;
	/* find the end */

	while (*p && !isspace(*p)) {
	    p++;
	}
	*p = '\0'; /* terminate the string */

	/* move to the start of the next one: */
	do {
	    p++;
	} while (*p && isspace(*p));

	if (!*p) {
	    Error(E_FATAL,
		"\"%s\": format is: number number pathname, not \"number\"",
		FileWithMatches
	    );
	}

	/* word in block */
	*Lppp++ = p;

	/* find the end */

	while (*p && !isspace(*p)) {
	    p++;
	}
	*p = '\0'; /* terminate the string */

	/* move to the start of the next one: */
	do {
	    p++;
	} while (*p && isspace(*p));

	if (!*p) {
	    Error(E_FATAL,
		"\"%s\": contains a line (%s) with no filename",
		FileWithMatches,
		Lines[i]
	    );
	}

	/* file name, already null-terminated */
	*Lppp++ = p;
    }

    (*MatchList) = Result;
    return (*MatchCount = Lppp - Result);
}

void
ShowFile(FileName, BlockInFile, WordInBlock, UniqueID)
    char *FileName;
    unsigned long BlockInFile;
    unsigned int WordInBlock;
    int UniqueID;
{
    static char *Buffer = 0;
    int fd;
    static unsigned int BufLen;
    int AmountRead;
    register char *p;
    int LinesFound;
    int InTargetWord = 0;
    char *StartOfMyWord;
    int ThisWord = 0;
    unsigned long FirstLumpSize;
    char *Start;
    static int LastID = (-1);

    if (UniqueID == LastID) {
	return;
    } else {
	LastID = UniqueID;
    }
    wclear(stdscr);
    ThisRow = DISPLAY_TOP;

    if (Buffer == (char *) 0) {
	BufLen = COLS * (LinesAbove + LinesBelow + 1) + 1;
	if (BufLen < FileBlockSize * 3) BufLen = FileBlockSize * 3;
	Buffer = emalloc(BufLen);
    }

    errno = 0;

#ifdef COMPRESS_SUPPORT
    fd = UnpackAndOpen(FileName);
#else /* COMPRESS_SUPPORT */
    fd = open(FileName, O_RDONLY, 0);
#endif /* COMPRESS_SUPPORT */
    if (fd < 0) {
	int e = errno;
	char *doc;

	if ((doc = FindFile(FileName)) == (char *) 0) {
	    errno = e;
	    perror(FileName);
	    sleep(1);
	    return;
	}

#ifdef COMPRESS_SUPPORT
	fd = UnpackAndOpen(doc);
#else /* COMPRESS_SUPPORT */
	fd = open(doc, O_RDONLY, 0);
#endif /* COMPRESS_SUPPORT */
	if (fd < 0) {
	    perror(doc);
	    sleep(1);
	    return;
	}

	FileName = doc;
    }

    /* display a helpful message: */
    move(DISPLAY_TOP, 0);
    clrtoeol();
    move(DISPLAY_TOP - 1, 0);
    clrtoeol();
    mvwprintw(stdscr, DISPLAY_TOP - 1, 0,
		"Block %lu/Word %u in document: ", BlockInFile, WordInBlock);
    attrset(A_UNDERLINE);
    wprintw(stdscr, "%s", FileName);
    attrset(0);

    errno = 0;
    if (lseek(fd, BlockInFile? (long) ((BlockInFile - 1) * FileBlockSize) : 0L,
								    0) < 0) {
	perror("lseek");
	sleep(2);
	clearok(stdscr, TRUE);
	close(fd);
	return;
    }

    errno = 0;
    if ((AmountRead = read(fd, Buffer, BufLen)) < MinWordLength) {
	perror("read");
	sleep(5);
	close(fd);
	clearok(stdscr, TRUE);
	return;
    }

    /* clear the bottom bit of screen */
    {
	register int i;

	for (i = ThisRow; i < LINES; i++) {
	    move(i, 0);
	    wclrtoeol(stdscr);
	}
    }

    /** Find the required word */
    if (BlockInFile) {
	/* start 1 char before the end of the previous block */
	StartOfMyWord = &Buffer[FileBlockSize - 1];
	/* perhaps the last word of the previous block spans the block
	 * boundary?
	 */
	while (WithinWord(*StartOfMyWord)) StartOfMyWord++;
	if (StartOfMyWord < &Buffer[FileBlockSize]) {
	    StartOfMyWord = &Buffer[FileBlockSize];
	}
    } else {
	StartOfMyWord = Buffer;
    }

    for (ThisWord = 0; ThisWord <= WordInBlock + 1; ThisWord++) {
bored:
	/* skip to the start of a word */
	while (!StartsWord(*StartOfMyWord)) {
	    ++StartOfMyWord;
	}

	Start = StartOfMyWord;

	/* find the end of the word */
	while (WithinWord(*StartOfMyWord)) {
	    if (*StartOfMyWord == '\'' && !EndsWord(StartOfMyWord[1])) break;
	    StartOfMyWord++;
	}

	/* Assert: StartOfMyWord points 1 character beyond the end of the
	 * word pointed to by Start
	 */
	/* see if it's long enough */
	if (StartOfMyWord - Start < MinWordLength) {
	    goto bored;
	}

#if 0
	/* see if it is too common */
	{
	    extern char *WordRoot();

	    t_WordInfo W;
	    register char *p, *q;
	    char RootBuf[MaxWordLength + 1];

	    for (p = RootBuf, q = Start; *q; p++, q++) {
		if (q == StartOfMyWord) break;
		*p = isupper(*q) ? tolower(*q) : *q;
	    }
	    *p = '\0';

	    W.Word = RootBuf;
	    W.Length = strlen(W.Word);
	    W.WordPlace.Flags = 0;

	    (void) WordRoot(&W);

	    if (TooCommon(&W)) goto bored;

	}
#endif

	/** See if it's the right one */
	if (ThisWord == WordInBlock) {
	    StartOfMyWord = Start;
	    break;
	}
    }

    FirstLumpSize = StartOfMyWord - Buffer;

    /* Find N lines before it */
    LinesFound = 0;
    for (p = StartOfMyWord; p > Buffer; --p) {
	if (*p == '\n') {
	    if (++LinesFound > LinesAbove) break;
	}
    }

    /* display them */
    while (p < StartOfMyWord) {
	Output(*p); /* Output might be a macro later */
	p++;
    }

    /* find N lines after it */

    LinesFound = 0;
    while (p - Buffer < AmountRead) {
	switch (InTargetWord) {
	case 0:
	    if (StartsWord(*p)) {
		attrset(A_STANDOUT);
		InTargetWord = 1;
	    }
	    break;
	case 1:
	    if (!WithinWord(*p)) {
		InTargetWord = 2;
		attrset(0);
	    }
	}
	Output(*p);

	if (*p == '\n') {
	    if (++LinesFound > LinesBelow) break;
	}
	p++;
    }

    (void) refresh();

    (void) close(fd);
    return;
}

void
Output(ch)
    int ch;
{
    static int UnderlineFlag = 0;

    switch(ch) {
    default:
	if (++ThisCol > COLS) {
	    if (++ThisRow >= LINES - 1) {
		ThisRow = DISPLAY_TOP;
	    }
	    ThisCol = 0;
	}
	if (ThisCol <= 0) {
	    ThisCol = 0;
	    move(ThisRow, ThisCol);
	    clrtoeol();
	}
	wmove(stdscr, ThisRow, ThisCol);
	if (UnderlineFlag) {
	    int c = inch();

	    if ((ch == '_' && c && c != ' ') ||
		(c == '_' && ch != '_')) {
#ifdef HASUNDERLINE
		attrset(A_UNDERLINE);
#endif
		if (ch == '_') ch = c;
	    } else {
#ifdef HASUNDERLINE
		attrset(0); /* BUG what if we were standouting? */
#endif
		UnderlineFlag  = 0;
	    }
	}
	mvwaddch(stdscr, ThisRow, ThisCol, (chtype) ch);
	break;
    case '\b':
	if (ThisCol > 0) {
	    --ThisCol;
	    UnderlineFlag = 1;
	}
	break;
    case '\t':
	do {
	    if (++ThisCol > COLS) {
		if (++ThisRow >= LINES - 1) {
		    ThisRow = DISPLAY_TOP;
		}
		ThisCol = 0;
		break;
	    }
	    mvwaddch(stdscr, ThisRow, ThisCol, (chtype) ' ');
	} while (ThisCol & 07);
	break;
    case '\r':
	ThisCol = 0;
	break;
    case '\n':
	if (++ThisRow >= LINES - 1) {
	    ThisRow = DISPLAY_TOP;
	}
	ThisCol = (-1);
	move(ThisRow, 0);
	clrtoeol();
	break;
    case ' ':
	ThisCol++;
    }
}

/*
 * $Log:	lqshow.c,v $
 * Revision 1.17  92/04/02  14:15:12  lee
 * Allow spaces in filenames.
 * 
 * Revision 1.16  92/02/15  07:01:47  lee
 * hmmm... made declarations of beep() that don't coflict.
 * 
 * Revision 1.15  92/02/15  06:53:03  lee
 * made beep() void if defined here, and cast it everywhere.
 * Was checking the wrong variable for the = command, which I recall is
 * broken anyway becasue of tty modes.
 * 
 * Revision 1.14  92/02/15  05:13:17  lee
 * Fixed a bug in compress support.
 * 
 * Revision 1.13  92/01/29  17:58:07  lee
 * Added calls to uncompress compressed files...
 * 
 * Revision 1.12  92/01/29  17:50:55  lee
 * fixed a problem with tabs.
 * 
 * Revision 1.11  91/08/08  16:18:16  lee
 * d,s,g,u commands now only available if -o was used.
 * 
 * Revision 1.10  91/08/08  16:10:56  lee
 * Now copes with overstriking (for underlining and emboldening) a little
 * better, and is more tolerant about extra spaces in the match-file.
 * 
 * Revision 1.9  91/03/03  00:18:26  lee
 * No longer needs to check common words.
 * 
 * Revision 1.8  90/10/06  00:50:58  lee
 * Prepared for first beta release.
 * 
 * Revision 1.7  90/10/05  23:49:25  lee
 * Moved the Match %d of %d message left somewhat.
 * 
 * Revision 1.6  90/10/03  21:26:47  lee
 * Removed BSD/SYSV diffs and used CURSESX instead.
 * 
 * Revision 1.5  90/08/29  21:45:34  lee
 * Alpha release
 * 
 * Revision 1.4  90/08/09  19:17:17  lee
 * BSD lint and Saber
 * 
 * Revision 1.3  90/07/11  10:57:46  lee
 * Added limited ultrix support...  also some small optimisations and
 * changes to the help screen.
 * 
 * Revision 1.2  90/04/21  16:06:24  lee
 * Cleaned up the gode for gcc -W
 * 
 * Revision 1.1  90/02/14  18:32:39  lee
 * Initial revision
 * 
 * Revision 2.1  89/10/02  01:15:51  lee
 * New index format, with Block/WordInBlock/Flags/BytesSkipped info.
 * 
 * Revision 1.3  89/09/17  23:04:04  lee
 * Various fixes; NumberInBlock now a short...
 * 
 * Revision 1.2  89/09/16  21:18:35  lee
 * First demonstratable version.
 * 
 * Revision 1.1  89/09/16  20:02:58  lee
 * Initial revision
 * 
 */
