/* xlqtxt - front end to LQ-TEXT
 *
 * Liam Quin, 1991, 1992
 *
 * $Id: xlqtext.c,v 1.50 92/04/15 19:21:55 lee Exp Locker: lee $
 *
 */

#include <stdio.h>
#include <ctype.h>
#include <xview/xview.h>
#include <xview/frame.h>
#include <xview/panel.h>
#include <xview/svrimage.h>
#include <xview/icon.h>
#include <gfm.h>

#ifdef DEBUG
# define DEBUG_WHAT_WHERE(What) \
	fprintf(stderr, "%s at %d, %d\n", \
		What, \
		xv_get(XThings.MainPanel, PANEL_ITEM_X), \
		xv_get(XThings.MainPanel, PANEL_ITEM_Y));
#else
# define DEBUG_WHAT_WHERE(What)	/* nothing at all */
#endif

#define STREQ(boy,foot) ((*(boy) == *(foot)) && strcmp(boy,foot) == 0)
    /* STREQ is an optimisation suggested by Henry Spencer, utzoo!henry */
#define STRCMP(boy,foot) ((*(boy) == *(foot)) ? \
	strcmp(boy,foot) : (*(boy) - *(foot)) )

/* Structure for the list of matches that we have...
 */

typedef struct s_MatchList {
    struct s_MatchList *Next;
    unsigned long Block;
    unsigned int WordInBlock;
} t_MatchList;

typedef struct s_FileList {
    struct s_FileList *Next;
    char *FileName;
    unsigned long MatchCount;
    t_MatchList *MatchList;
} t_FileList;

t_FileList *FileList;

#define PhraseFieldCount 8

struct {
    Frame MainFrame;
    Panel MainPanel;
    Panel MainList;
    int MaxInList;
    Panel_item Phrases[PhraseFieldCount];

    Frame PropsFrame;
    Panel PropsPanel;

    Menu FileMenu;
    Menu EditMenu;
    Menu ViewMenu;
    Menu WordsMenu;
    Menu PrecisionMenu;
} XThings;

/* props stuff: */
int PropsStarted = 0;
static char *CaseMatch = "-mh";
static int ContextLinesAbove = 6;
static int ContextLinesBelow = 9;

#define DEF_TERM \
	"$OPENWINHOME/bin/cmdtool -fn lucidasanstypewriter-18 -name xvtext"

/* #define DEF_TERM "xterm -fn lucidasanstypewriter-18 -T xv-text" */

char *TerminalEmulator = DEF_TERM;
char *ActionProgram = "lqshow";

/* */

#define MaxPhraseLength 500 /* maximum length of a query */

void FillList();
void EmptyList();
void ShowList();
void UpdateFooter();
char *emalloc();
void ShowSelectedFiles();
void ShowEmptyMessage();
#define FilesAll 0
#define FilesSelected 1

/* call-backs */
void CB_Delete();
void CB_MakeList();
void CB_FileMenu();
void CB_EditMenu();
void CB_ViewMenu();
int CB_SetAction();
Panel_setting CB_SetDocPath();
Panel_setting CB_SetDataBaseDir();
Panel_setting CB_InnerPhrase();
Panel_setting CB_LastPhrase();
Panel_setting CB_SetTerminalEmulator();

char *progname;
Attr_attribute	INSTANCE; /* must be global */

static short icon_bits[] = {
# include "xlqtext.icon"
};

main(argc, argv)
    int argc;
    char *argv[];
{
    int i;

    argv[0] = "lq-text";

    for (i = 0; i < argc; i++) {
	if (STREQ(argv[i], "-help")) {
	    fprintf(stderr, "%s: OPEN LOOK GUI to lq-text\n", argv[0]);
	    fprintf(stderr, "Appliaction-specific options:\n");
	    fprintf(stderr,
		"\t-terminal \"command\" - specify terminal emulator to use\n");
	    fprintf(stderr,
		"\t\tdefault value: %s\n", TerminalEmulator
	    );
	}
	fflush(stderr);
    }

    xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL);
    INSTANCE = xv_unique_key();

    progname = argv[0];

    for (i = 0; i < argc; i++) {
	if (STREQ(argv[i], "-terminal")) {
	    if (i < argc) {
		TerminalEmulator = argv[i + 1];
		++i;
	    }
	}
    }

    XThings.MainFrame = (Frame) xv_create(NULL, FRAME,
	FRAME_LABEL,		argv[0],
	FRAME_SHOW_FOOTER,	TRUE,
	FRAME_LEFT_FOOTER,	"Type a Phrase to match and click on FIND",
	NULL
    );

    {
	Server_image icon_image;
	Icon icon;

	icon_image = (Server_image) xv_create(NULL, SERVER_IMAGE,
	    XV_WIDTH, 64,
	    XV_HEIGHT, 64,
	    SERVER_IMAGE_BITS, icon_bits,
	    NULL
	);

	icon = (Icon) xv_create(XThings.MainFrame, ICON,
	    ICON_IMAGE, icon_image,
	    ICON_TRANSPARENT, TRUE,
	    ICON_LABEL, "xlqtext",
	    NULL
	);

	xv_set(XThings.MainFrame,
	    FRAME_ICON,		icon,
	    NULL
	);

    }

    XThings.MainPanel = (Panel) xv_create(XThings.MainFrame, PANEL,
	PANEL_LAYOUT,		PANEL_HORIZONTAL,
	XV_HELP_DATA,		"lqtext:mainpanel",
	XV_X,			0,
	NULL
    );

    /** The menu bar **/

    XThings.FileMenu = (Menu) xv_create(NULL, MENU,
	MENU_TITLE_ITEM,	"File",
	MENU_NOTIFY_PROC,	CB_FileMenu,
	MENU_ITEM,
	    MENU_STRING,	"Save...",
	    XV_HELP_DATA,	"lqtext:file-save",
	    NULL,
	MENU_ITEM,
	    MENU_STRING,	"Load...",
	    XV_HELP_DATA,	"lqtext:file-load",
	    NULL,
	MENU_ITEM,
	    XV_HELP_DATA,	"lqtext:file-quit",
	    MENU_STRING,	"Quit",
	    NULL,
	NULL
    );

    (void) xv_create(XThings.MainPanel, PANEL_BUTTON,
	PANEL_LABEL_STRING,	"File",
	PANEL_ITEM_MENU,	XThings.FileMenu,
	XV_HELP_DATA,		"lqtext:filemenu",
	NULL
    );
    
    DEBUG_WHAT_WHERE("File Button");

    XThings.EditMenu = (Menu) xv_create(NULL, MENU,
	MENU_TITLE_ITEM,	"Edit",
	MENU_NOTIFY_PROC,	CB_EditMenu,
	MENU_ITEM,
	    MENU_STRING,	"Select All",
	    XV_HELP_DATA,	"lqtext:edit-selectall",
	    NULL,
	MENU_ITEM,
	    MENU_STRING,	"Delete",
	    XV_HELP_DATA,	"lqtext:edit-delete",
	    NULL,
	MENU_ITEM,
	    MENU_STRING,	"Properties...",
	    XV_HELP_DATA,	"lqtext:edit-properties",
	    NULL,
	NULL
    );

    (void) xv_create(XThings.MainPanel, PANEL_BUTTON,
	PANEL_LABEL_STRING,	"Edit",
	PANEL_ITEM_MENU,	XThings.EditMenu,
	XV_HELP_DATA,		"lqtext:editmenu",
	NULL
    );

    DEBUG_WHAT_WHERE("Edit Button");

    XThings.ViewMenu = (Menu) xv_create(NULL, MENU,
	MENU_TITLE_ITEM,	"View",
	MENU_NOTIFY_PROC,	CB_ViewMenu,
	MENU_ITEM,
	    XV_HELP_DATA,	"lqtext:view-selected",
	    MENU_STRING,	"Selected",
	    NULL,
	MENU_ITEM,
	    MENU_STRING,	"All",
	    XV_HELP_DATA,	"lqtext:view-all",
	    NULL,
	MENU_ITEM,
	    MENU_STRING,	"Vocabulary",
	    XV_HELP_DATA,	"lqtext:view-vocabulary",
	    NULL,
	MENU_DEFAULT, 1,
	NULL
    );

    (void) xv_create(XThings.MainPanel, PANEL_BUTTON,
	PANEL_LABEL_STRING,	"View",
	PANEL_ITEM_MENU,	XThings.ViewMenu,
	XV_HELP_DATA,		"lqtext:viewmenu",
	NULL
    );

    DEBUG_WHAT_WHERE("View Button");


    /** The buttons **/

    (void) xv_create(XThings.MainPanel, PANEL_BUTTON,
	PANEL_LABEL_STRING,	"Find",
	PANEL_NOTIFY_PROC,	CB_MakeList,
	XV_HELP_DATA,		"lqtext:findmenu",
	NULL
    );

    DEBUG_WHAT_WHERE("Find Button");


    /** The phrases: **/

    xv_set(XThings.MainPanel,
	PANEL_LAYOUT, PANEL_VERTICAL,
	NULL
    );

    for (i = 0; i < PhraseFieldCount - 1; i++) {
	XThings.Phrases[i] = (Panel_item) xv_create(
	    XThings.MainPanel, PANEL_TEXT,
	    PANEL_LABEL_STRING,	"Phrase:",
	    PANEL_VALUE_DISPLAY_LENGTH,	30,
	    PANEL_VALUE_STORED_LENGTH,	MaxPhraseLength,
	    PANEL_NOTIFY_STRING,		"\r\n\t\033",
	    PANEL_NOTIFY_PROC,		CB_InnerPhrase,
	    XV_HELP_DATA,		"lqtext:phraseline",
	    NULL
	);

	DEBUG_WHAT_WHERE("Phrase");
    }

    /* The last one is special: */
    XThings.Phrases[i] = (Panel_item) xv_create(XThings.MainPanel, PANEL_TEXT,
	PANEL_LABEL_STRING,	"Phrase:",
	PANEL_VALUE_DISPLAY_LENGTH,	30,
	PANEL_VALUE_STORED_LENGTH,	MaxPhraseLength,
	PANEL_NOTIFY_STRING,		"\r\n\t\033",
	PANEL_NOTIFY_PROC,		CB_LastPhrase,
	XV_HELP_DATA,		"lqtext:lastphraseline",
	NULL
    );
    DEBUG_WHAT_WHERE("Last Phrase");

    xv_set(XThings.MainPanel,
	PANEL_LAYOUT, PANEL_HORIZONTAL,
	NULL
    );

    /* The purpose of the DEBUG_WHAT_WHERE stuff is to let me position
     * this list exactly where I want it with XV_X and XV_Y...
     */
    XThings.MainList = (Panel) xv_create(XThings.MainPanel, PANEL_LIST,
	PANEL_LIST_DISPLAY_ROWS,	20,
	PANEL_CHOOSE_ONE,		FALSE,
	XV_X,				340,
	XV_Y,				4,
	XV_HELP_DATA,			"lqtext:matchlist",
	NULL
    );

    /* This is really to make the list large enough, but the extra help
     * doesn't go amiss...
     */

    ShowEmptyMessage();
    DEBUG_WHAT_WHERE("Match List");

    xv_set(XThings.MainFrame,
	PANEL_LAYOUT,		PANEL_HORIZONTAL,
	NULL
    );


    window_fit(XThings.MainPanel);
    window_fit(XThings.MainFrame);
    xv_main_loop(XThings.MainFrame);
    exit(0);
}

void
ShowEmptyMessage()
{
    xv_set(XThings.MainList,
	PANEL_LIST_STRING, 0,
"    Type one or more phrases and press FIND;                       ",
	NULL
    );

    xv_set(XThings.MainList,
	PANEL_LIST_STRING, 1,
"    The list of matching files will display here. ",
	NULL
    );
}

void
FreeFileList()
{
    t_FileList *fp;
    t_MatchList *mp;

    for (fp = FileList; fp; ) {
	t_FileList *NextF = fp->Next;

	for (mp = fp->MatchList; mp; ) {
	    t_MatchList *NextM = mp->Next;

	    (void) free((char *) mp);
	    mp = NextM;
	}

	(void) free(fp->FileName);
	(void) free((char *) fp);
	fp = NextF;
    }

    FileList = (t_FileList *) NULL;
}

/* InsertMatch() returns nonzero if the name was inserted. */
int
InsertMatch(FileName, Block, WordInBlock)
    char *FileName;
    unsigned long Block;
    int WordInBlock;
{
    register t_FileList **fp;
    register int i = -1;
    t_FileList *New;
    register t_MatchList **mp;
    t_MatchList *NewMatch;
    int InsertedName = 0;

    /* TODO remember last position (although this routine is very fast) */

    /* We insert the match, keeping the list in sorted order.
     * This is important if there were several phrases, since we want the
     * overall list to end up sorted.
     */
    for (fp = &FileList; *fp; fp = &((*fp)->Next) ) {
	if ((i = STRCMP((*fp)->FileName, FileName)) >= 0) {
	    break;
	}
    }

    if (i) {
	/* The matched file is not there already, so we need to insert
	 * it into the list.
	 */
	New = (t_FileList *) emalloc(sizeof(t_FileList));
	New->Next = (*fp);
	New->MatchCount = 1L;
	New->FileName = strcpy(emalloc(strlen(FileName) + 1), FileName);
	*fp = New;
	(*fp)->MatchList = (t_MatchList *) emalloc(sizeof(t_MatchList));
	mp = &(*fp)->MatchList;
	(*mp)->Block = Block;
	(*mp)->WordInBlock = WordInBlock;
	(*mp)->Next = (t_MatchList *) NULL;
	return 1; /* inserted a name */
    } else {
	/* The file is already there, so we simpy need to insert an entry.
	 */
	New = (*fp);
    }

    /* Now allocate the match */
    for (mp = &(New->MatchList); *mp; mp = &(*mp)->Next) {
	if ((*mp)->Block == Block) {
	    if ((*mp)->WordInBlock == WordInBlock) {
		return 0;
	    } else if ((*mp)->WordInBlock > WordInBlock) {
		break;
	    }
	}
    }

    /* insert before */
    NewMatch = (t_MatchList *) emalloc(sizeof(t_MatchList));
    NewMatch->Block = Block;
    NewMatch->WordInBlock = WordInBlock;
    NewMatch->Next = (*mp);
    New->MatchCount++;
    *mp = NewMatch;
    return 0;
}

static int
ParseLine(Line, Blockp, Wordp, Namep)
    char *Line;
    unsigned long *Blockp;
    int *Wordp;
    char **Namep;
{
    register char *p = Line;

    if (!p) return -1;

    while (isspace(*p)) p++; /* skip space */

    if (!*p || !isdigit(*p)) return -1;

    *Blockp = 0;

    while (isdigit(*p)) {
	*Blockp *= 10;
	*Blockp += (*p) - '0';
	p++;
    }

    if (!isspace(*p)) return -1;

    while (isspace(*p)) p++; /* skip space */

    if (!*p || !isdigit(*p)) return -1;

    *Wordp = 0;

    while (isdigit(*p)) {
	*Wordp *= 10;
	*Wordp += (*p) - '0';
	p++;
    }

    if (!isspace(*p)) return -1;

    p++; /* skip over the space */

    {
	register char *q = p;

	while (*q) q++;
	if (q[-1] == '\n') {
	    q--;
	    *q = '\0';
	}
	*Namep = emalloc(q - p + 1);
	strcpy(*Namep, p);
    }

    return 0;
}

void
FillList(Word)
    char *Word;
{
    FILE *Input;
    char Buf[1000];
    register char *p;

    sprintf(Buf, "lqphrase %s \"%s\" | sort +2 +0n -2", CaseMatch, Word);

    if ((Input = popen(Buf, "r")) == (FILE *) NULL) {
	xv_set(XThings.MainList,
	    PANEL_LIST_STRING, 0, "Open Error: ",
	    NULL
	);
	xv_set(XThings.MainList,
	    PANEL_LIST_STRING, 1, Buf,
	    NULL
	);
	return;
    }

    GetListFromFile("lqphrase pipeline", Input);
    (void) pclose(Input);
}

GetListFromFile(FileName, Input)
    char *FileName;
    FILE *Input;
{
    char Buf[2000];
    int ListTotal = 0;
    int i;

    for (;;) {
	unsigned long Block;
	int Word;
	char *FileName;

	if (fgets(Buf, sizeof(Buf), Input) == (char *) NULL) {
	    break;
	}

	if (Buf[0] == '#') {
	    char Buf2[1024];

	    if (sscanf(Buf, "#%d %[^\n]\n", &i, Buf2) == 2) {
		char *p = Buf2;

		while (*p == ' ') p++;

		xv_set(XThings.Phrases[i], PANEL_VALUE, p, NULL);
	    }
	    continue;
	}
	/* parse the line */
	if (ParseLine(Buf, &Block, &Word, &FileName) < 0) {
	    fprintf(stderr, "%s: Warning: Error parsing \"%s\"\n", progname, Buf);
	    continue;
	}

	if (FileName && *FileName) {
	    (void) InsertMatch(FileName, Block, Word);
	    if (++ListTotal % 50 == 0) {
		(void) sprintf(Buf, "%d matches...", ListTotal);
		xv_set(XThings.MainFrame,
		    FRAME_LEFT_FOOTER,	Buf,
		    NULL
		);
	    }
	}
    }
}

void
ShowList()
{

    register int i;
    t_FileList *Fp;

    for (i = XThings.MaxInList - 1; i >= 0; --i) {
	xv_set(XThings.MainList, PANEL_LIST_DELETE, i, NULL);
	if (--XThings.MaxInList == 0) {
	    break;
	}
    }

    i = 0;

    for (Fp = FileList; Fp; Fp = Fp->Next) {
	unsigned int nbytes = strlen(Fp->FileName);
	char *tmp = emalloc( ( (i < 60) ? 60 : i ) + 10);

	(void) sprintf(tmp, "%6d: %-60s", Fp->MatchCount, Fp->FileName);

	xv_set(XThings.MainList,
	    PANEL_LIST_STRING, i, tmp,
	    NULL
	);

	if (i % 50 == 0) {
	    static char Buf[50];

	    (void) sprintf(Buf, "%d files matched...", i);
	    xv_set(XThings.MainFrame,
		FRAME_LEFT_FOOTER,	Buf,
		NULL
	    );
	}

	i++;

	(void) free(tmp);
    }

    XThings.MaxInList = (int) xv_get(XThings.MainList, PANEL_LIST_NROWS);

    if (XThings.MaxInList == 1) {
	xv_set(XThings.MainList, PANEL_LIST_SELECT, 0, TRUE, NULL);
    }

    UpdateFooter();
}

static char *TmpFile = 0;
static FILE *TmpF = (FILE *) 0;

void
StartShow()
{
    extern char *mktemp();
    char Buf[2000];
    static char TempFile[30]; /* ML for MatchList... */
    (void) strcpy(TempFile, "/tmp/xlqMLXXXXXX"); /* (avoid read-only memory) */
    TmpFile = mktemp(TempFile);

    if (TmpF) {
	(void) fclose(TmpF);
    }

    if ((TmpF = fopen(TmpFile, "w")) == (FILE *) NULL) {
	(void) sprintf(Buf, "Cannot create tmp file \"%s\"", TmpFile);
	xv_set(XThings.MainFrame,
	    FRAME_LEFT_FOOTER,	Buf,
	    NULL
	);
	return;
    }

}

void
EndShow()
{
    char Buf[2000];

    (void) fclose(TmpF);
    if (STREQ(ActionProgram, "lqkwik2")) {
	(void) sprintf(Buf, "exec %s -f %s &\n", ActionProgram, TmpFile);
    } else if (STREQ(ActionProgram, "ae")) {
	(void) sprintf(Buf, "exec ae2 %s &\n", TmpFile);
    } else {
	(void) sprintf(Buf,
	    "exec /bin/sh -c \"%s -e %s -f %s -a %d -b %d; rm -f %s\" &",
	    TerminalEmulator,
	    ActionProgram,
	    TmpFile,
	    ContextLinesAbove,
	    ContextLinesBelow,
	    TmpFile
	);
    }

    system(Buf);
}

void
ShowFile(String)
    char *String;
{
    char Buf[2000];
    t_FileList *Fp;
    t_MatchList *Mp;


    /* find the string in the list: */
    for (Fp = FileList; Fp; Fp = Fp->Next) {
	if (STRCMP(Fp->FileName, String) >= 0) {
	    break;
	}
    }

    if (!Fp || !Fp->FileName || !STREQ(Fp->FileName, String)) {
	(void) sprintf(Buf, "Cannot find \"%s\" in the list of strings!",
								String);
	xv_set(XThings.MainFrame,
	    FRAME_LEFT_FOOTER,	Buf,
	    NULL
	);
	return;
    }

    if (!(Mp = Fp->MatchList)) {
	(void) sprintf(Buf, "File \"%s\" has no matches!", Fp->FileName);
	xv_set(XThings.MainFrame,
	    FRAME_LEFT_FOOTER,	Buf,
	    NULL
	);
	return;
    }

    while (Mp) {
	(void) fprintf(TmpF, "%ul %d %s\n",
	    Mp->Block, Mp->WordInBlock, Fp->FileName);
	Mp = Mp->Next;
    }
}

void
EmptyList()
{
    register int i;

    XThings.MaxInList = (int) xv_get(XThings.MainList, PANEL_LIST_NROWS);

    for (i = XThings.MaxInList - 1; i >= 0; --i) {
	xv_set(XThings.MainList, PANEL_LIST_DELETE, i, NULL);
	if (--XThings.MaxInList == 0) {
	    break;
	}
    }

    FreeFileList();
}

void
UpdateFooter()
{
    char Buf[40];
    t_FileList *Fp;
    int total = 0;
    t_MatchList *Mp;
    t_FileList *fp;

    XThings.MaxInList = (int) xv_get(XThings.MainList, PANEL_LIST_NROWS);

    for (fp = FileList; fp; fp = fp->Next) {
	if (!(Mp = fp->MatchList)) {
	    continue;
	}

	while (Mp) {
	    ++total;
	    Mp = Mp->Next;
	}
    }


    switch (XThings.MaxInList) {
    case 0:
	(void) sprintf(Buf, "No document matches.");
	break;
    case 1:
	(void) sprintf(Buf, "%d match%s in one document.",
	    total,
	    (total == 1) ? "" : "es"
	);
	break;
    default:
	(void) sprintf(Buf, "%d matches in %d documents.",
	    total, XThings.MaxInList);
	break;
    }

    xv_set(XThings.MainFrame,
	FRAME_LEFT_FOOTER,	Buf,
	NULL
    );
}

void
ShowSelectedFiles(Which)
    int Which;
{
    register int i;
    int FoundCount = 0;

    xv_set(XThings.MainFrame, FRAME_BUSY, TRUE, NULL);

    for (i = 0; i < XThings.MaxInList; i++) {
	char *p;
	char *q;

	if (Which == FilesSelected) {
	    if (xv_get(XThings.MainList, PANEL_LIST_SELECTED, i) == FALSE) {
		continue;
	    }
	}

	p = (char *) xv_get(XThings.MainList, PANEL_LIST_STRING, i);

	/* step over any leading whitespace */
	q = p; while (isspace(*q)) q++; p = q;

	/* step over optional "  matchcount: " */
	while (isdigit(*q)) q++;
	if (q != p && *q == ':' && q[1] == ' ') {
	    q++;
	    while (isspace(*q)) q++;
	    p = q;
	}

	/* strip trailing spaces */
	q = p;
	while (*q) q++;
	do {
	    --q;
	} while (isspace(*q));
	q++; /* gone one too far */
	*q = '\0';

	if (!FoundCount) {
	    StartShow();
	}
	/* look at the result */
	ShowFile(p);

	++FoundCount;
    }

    if (FoundCount) {
	EndShow();
    } else {
	if (Which == FilesSelected) {
	    xv_set(XThings.MainFrame,
		FRAME_LEFT_FOOTER,	"No selected filenames to browse",
		NULL
	    );
	} else {
	    xv_set(XThings.MainFrame,
		FRAME_LEFT_FOOTER,	"No filenames to browse",
		NULL
	    );
	}
    }
    xv_set(XThings.MainFrame, FRAME_BUSY, FALSE, NULL);
}

char *
emalloc(n)
    unsigned int n;
{
    char *p;

    if (n == 0) n = 4;

    if ((p = malloc(n)) == (char *) NULL) {
	(void) fprintf(stderr, "%s: attempt to get %ud bytes failed\n",
							progname, n);
	exit(1);
    }
    return p;
}


void
StartProps()
{
    void CB_SetMatchLevel();
    Panel_setting CB_SetContextAbove();
    Panel_setting CB_SetContextBelow();
    Xv_opaque ApplyButton, CancelButton;

    XThings.PropsFrame = (Panel) xv_create(XThings.MainFrame, FRAME_CMD,
	FRAME_LABEL,		"Xlqtext Properties",
	NULL
    );

    XThings.PropsPanel = (Panel) xv_get(XThings.PropsFrame, FRAME_CMD_PANEL);

    (void) xv_set(XThings.PropsPanel,
	XV_HELP_DATA,		"lqtext:propspanel",
	PANEL_LAYOUT,		PANEL_VERTICAL,
	NULL
    );

    (void) xv_create(XThings.PropsPanel, PANEL_CHOICE,
	XV_X,			88,
	PANEL_LABEL_STRING,	"Match Level: ",
	PANEL_CHOICE_STRINGS,
	    "Exact",
	    "Heuristic",
	    "Rough",
	     NULL,
	PANEL_NOTIFY_PROC,	CB_SetMatchLevel,
	PANEL_VALUE,		1,
	NULL
    );

    (void) xv_create(XThings.PropsPanel, PANEL_CHOICE,
	XV_X,			131, /* 100 too small */
	PANEL_LABEL_STRING,	"Action: ",
	PANEL_CHOICE_STRINGS,
	    "Browse Files",
	    "Show Index",
	    "View SGML",
	     NULL,
	PANEL_NOTIFY_PROC,	CB_SetAction,
	PANEL_VALUE,		0,
	NULL
    );

    (void) xv_create(XThings.PropsPanel, PANEL_NUMERIC_TEXT,
	XV_X, 5,
	PANEL_MIN_VALUE,	0,
	PANEL_MAX_VALUE,	100,
	PANEL_LABEL_STRING,	"Lines of context above: ",
	PANEL_NOTIFY_PROC,	CB_SetContextAbove,
	PANEL_VALUE,		ContextLinesAbove,
	NULL
    );

    (void) xv_create(XThings.PropsPanel, PANEL_NUMERIC_TEXT,
	XV_X, 6,
	PANEL_MIN_VALUE,	0,
	PANEL_MAX_VALUE,	100,
	PANEL_LABEL_STRING,	"Lines of context below: ",
	PANEL_NOTIFY_PROC,	CB_SetContextBelow,
	PANEL_VALUE,		ContextLinesBelow,
	NULL
    );

    (void) xv_create(XThings.PropsPanel, PANEL_TEXT,
	XV_X, 39,
	PANEL_LABEL_STRING,	"Terminal Emulator: ",
	PANEL_NOTIFY_PROC,	CB_SetTerminalEmulator,
	PANEL_VALUE, TerminalEmulator,
	NULL
    );

    {
	extern char *getenv();
	char *p = getenv("DOCPATH");

	if (!p) {
	    p = "(default)";
	}

	(void) xv_create(XThings.PropsPanel, PANEL_TEXT,
	    PANEL_LABEL_STRING,	"Document search path: ",
	    XV_X, 31 - 27, /* see Database directory */
	    PANEL_NOTIFY_PROC,	CB_SetDocPath,
	    PANEL_VALUE, p,
	    NULL
	);
    }

    {
	extern char *getenv();
	char *p = getenv("LQTEXTDIR");

	if (!p) {
	    p = "$HOME/LQTEXTDIR";
	}

	(void) xv_create(XThings.PropsPanel, PANEL_TEXT,
XV_X, 16,

	    XV_X, 31,
	    PANEL_LABEL_STRING,	"Database directory: ",
	    PANEL_NOTIFY_PROC,	CB_SetDataBaseDir,
	    PANEL_VALUE, p,
	    NULL
	);
    }

    /**
     * To add:
     * Help file location (or HELPPATH?)
     * docpath
     * database directory
     * context view width
     **/

    {
	int w, x;
	int cw, cx;
	int y;

	ApplyButton = xv_create(XThings.PropsPanel, PANEL_BUTTON,
	    PANEL_LABEL_STRING,	"Apply",
	    NULL
	);


	y = (int) xv_get(ApplyButton, XV_Y);

	CancelButton = xv_create(XThings.PropsPanel, PANEL_BUTTON,
	    XV_Y, y,
	    PANEL_LABEL_STRING,	"Cancel",
	    NULL
	);

	xv_set(XThings.PropsPanel, PANEL_DEFAULT_ITEM, ApplyButton, NULL);


	window_fit(XThings.PropsPanel);
	window_fit(XThings.PropsFrame);


	w = (int) xv_get(ApplyButton, XV_WIDTH);
	cw = (int) xv_get(CancelButton, XV_WIDTH);
	x = (int) xv_get(XThings.PropsPanel, XV_WIDTH);
	x /= 2;
	x -= (w + cw + 5) / 2;
	xv_set(ApplyButton, XV_X, x, NULL);
	xv_set(CancelButton, XV_X, x + w + 5, XV_Y, y, NULL);

	window_fit(XThings.PropsPanel);
	window_fit(XThings.PropsFrame);
    }

    PropsStarted = 1;
}

void
DisplayProps()
{
    if (!PropsStarted) {
	StartProps();
    }

    (void) xv_set(XThings.PropsFrame,
	XV_SHOW,	TRUE,
	NULL
    );
}

char *
ShellExpand(Str)
    char *Str;
{
    FILE *f;
    static char Buf[1024];
    register char *p, *q;

    /* sprintf(Buf, "echo -n \"%s\"", Str), but with " and \ escaped */

    p = Buf;
    q = "/bin/echo -n \"";
    while (*q) {
	*p++ = *q++;
    }
    q = Str;
    while (*q) {
	if (*q == '\\' || *q == '\"') {
	    *p++ = '\\';
	}
	*p++ = *q++;
    }
    *p++ = '\"';
    *p = '\0';

    /* get the shell to give us the result: */

    f = popen(Buf, "r");
    if (f) {
	fgets(Buf, sizeof Buf, f);
	(void) fflush(f);
	pclose(f);
	if (Buf[0]) {
	    return Buf;
	}
    }
    return Str;
}

/** callbacks **/

void
CB_Delete()
{
    register int i;

    xv_set(XThings.MainList,
	XV_SHOW, FALSE,
	NULL
    );

    for (i = XThings.MaxInList - 1; i >= 0; --i) {
	if (xv_get(XThings.MainList, PANEL_LIST_SELECTED, i) == TRUE) {
	    xv_set(XThings.MainList, PANEL_LIST_DELETE, i, NULL);
	    /* TODO: delete from linked list too! */
	    if (--XThings.MaxInList == 0) {
		break;
	    }
	}
    }

    xv_set(XThings.MainList,
	XV_SHOW, TRUE,
	NULL
    );
    UpdateFooter();
}


void
CB_MakeList()
{
    int i;

    xv_set(XThings.MainFrame, FRAME_BUSY, TRUE, NULL);

    xv_set(XThings.MainList,
	XV_SHOW, FALSE,
	NULL
    );

    EmptyList();

    for (i = 0; i < PhraseFieldCount; i++) {
	char *Phrase = (char *) xv_get(XThings.Phrases[i], PANEL_VALUE);

	if (Phrase && *Phrase) {
	    FillList(Phrase);
	}
    }

    ShowList();

    xv_set(XThings.MainFrame, FRAME_BUSY, FALSE, NULL);

    xv_set(XThings.MainList,
	XV_SHOW, TRUE,
	NULL
    );
}

static void
ReadListFromFile(FileName)
    char *FileName;
{
    FILE *f;
    char Buf[2000];

    xv_set(XThings.MainFrame, FRAME_BUSY, TRUE, NULL);

    if ((f = fopen(FileName, "r")) == (FILE *) 0) {
	xv_set(XThings.MainFrame,
	    FRAME_BUSY,		FALSE,
	    FRAME_LEFT_FOOTER,	"Couldn't open file",
	    NULL
	);
	return;
    }

    xv_set(XThings.MainList,
	XV_SHOW, FALSE,
	NULL
    );

    EmptyList();
    GetListFromFile(FileName, f);
    (void) fclose(f);

    ShowList();

    xv_set(XThings.MainFrame, FRAME_BUSY, FALSE, NULL);

    xv_set(XThings.MainList,
	XV_SHOW, TRUE,
	NULL
    );
}

SaveListToFile(FileName)
    char *FileName;
{
    FILE *f;
    char Buf[2000];
    t_FileList *Fp;
    t_MatchList *Mp;
    int i;

    xv_set(XThings.MainFrame, FRAME_BUSY, TRUE, NULL);

    if ((f = fopen(FileName, "w")) == (FILE *) 0) {
	xv_set(XThings.MainFrame,
	    FRAME_BUSY,		FALSE,
	    FRAME_LEFT_FOOTER,	"Couldn't open file for writing",
	    NULL
	);
	return;
    }

    xv_set(XThings.MainList,
	XV_SHOW, FALSE,
	NULL
    );


    for (i = 0; i < PhraseFieldCount; i++) {
	char *Phrase = (char *) xv_get(XThings.Phrases[i], PANEL_VALUE);

	if (Phrase && *Phrase) {
	    fprintf(f, "# %d %s\n", i, Phrase);
	} else {
	    fprintf(f, "# %d   \n", i);
	}
    }

    for (Fp = FileList; Fp; Fp = Fp->Next) {
	if (!(Mp = Fp->MatchList)) {
	    continue;
	}

	while (Mp) {
	    (void) fprintf(f, "%d %d %s\n",
		Mp->Block, Mp->WordInBlock, Fp->FileName);
	    Mp = Mp->Next;
	}
    }

    (void) fclose(f);

    xv_set(XThings.MainFrame, FRAME_BUSY, FALSE, NULL);

    xv_set(XThings.MainList,
	XV_SHOW, TRUE,
	NULL
    );
}

static Xv_opaque FileDialogue = NULL;

int
filter_callback(ip, Path) 
    Xv_opaque ip;
    char *Path;
{
    return TRUE;
}

int
get_file(ip, dir, file)
    Xv_opaque ip;
    char *dir;
    char *file;
{
    char *path;

    if (!dir || !*dir) dir = ".";

    path = malloc(strlen(dir) + strlen(file) + 2);
    (void) sprintf(path, "%s/%s", dir, file);
    ReadListFromFile(path);
    return GFM_OK;
}

save_list(ip, dir, file)
    Xv_opaque ip;
    char *dir;
    char *file;
{
    char *path;

    if (!dir || !*dir) dir = ".";

    path = malloc(strlen(dir) + strlen(file) + 2);
    (void) sprintf(path, "%s/%s", dir, file);
    SaveListToFile(path);
    return GFM_OK;
}

void
CB_FileMenu(WhichMenu, WhichItem)
    Menu WhichMenu;
    Menu_item WhichItem;
{
    char *String = (char *) xv_get(WhichItem, MENU_STRING);
    extern void gfm_activate();

    if (!String || !*String) return; /* paranoia... */

    if (STREQ(String, "Quit")) {
	xv_destroy_safe(XThings.MainFrame);
	exit(1);
    } else if (STREQ(String, "Load...")) {
	extern char *getcwd();
	char *Here;
	char buf[2000];

	FileDialogue =
	    gfm_initialize(FileDialogue, XThings.MainFrame, "Load List");

	Here = getcwd(buf, sizeof buf);
#if 0
void
gfm_activate(ip, directory, filter_pat, filter_callback, callback, glyph, mode)
        gfm_popup_objects       *ip;
        char                    *directory;
        char                    *filter_pat;
        int                     (*filter_callback)();
        int                     (*callback)();
        Xv_opaque               glyph;
        GFM_MODE                mode;
#endif
	gfm_activate(
	    FileDialogue,
	    Here,
	    "^.*$",
	    filter_callback,
	    get_file,
	    NULL,
	    GFM_LOAD
	);
    } else if (STREQ(String, "Save...")) {
	extern char *getcwd();
	char *Here;
	char buf[2000];

	FileDialogue =
	    gfm_initialize(FileDialogue, XThings.MainFrame, "Save List");

	Here = getcwd(buf, sizeof buf);
	gfm_activate(FileDialogue,
	    Here,
	    "^.*$",
	    filter_callback,
	    save_list,
	    NULL,
	    GFM_SAVE
	);
    } else {
	fprintf(stderr, "%s: unknown menu choice: %s\n",
	    progname, String);
    }
}

void
CB_EditMenu(WhichMenu, WhichItem)
    Menu WhichMenu;
    Menu_item WhichItem;
{
    char *String = (char *) xv_get(WhichItem, MENU_STRING);

    if (!String || !*String) return; /* paranoia... */

    if (STREQ(String, "Select All")) {
	int i = (int) xv_get(XThings.MainList, PANEL_LIST_NROWS) - 1;

	xv_set(XThings.MainList, XV_SHOW, FALSE, NULL);
	while (i >= 0) {
	    xv_set(XThings.MainList, PANEL_LIST_SELECT, i, TRUE, NULL);
	    i--;
	}
	xv_set(XThings.MainList, XV_SHOW, TRUE, NULL);

    } else if (STREQ(String, "Delete")) {
	CB_Delete();
    } else if (STREQ(String, "Properties...")) {
	DisplayProps();
    } else {
	fprintf(stderr, "%s: unknown menu choice: %s\n",
	    progname, String);
    }
}

static Frame F_WordList = 0;
static Xv_opaque F_List = 0;

MakeWordList()
{
    Panel P;
    static void CB_ChangePattern();

    F_WordList = (Frame) xv_create(XThings.MainFrame, FRAME,
	FRAME_LABEL,		"lq-text wordlist",
	FRAME_SHOW_FOOTER,	TRUE,
	NULL
    );
    /*TODO: give it an icon */

    P = (Panel) xv_create(F_WordList, PANEL,
	PANEL_LAYOUT,		PANEL_VERTICAL,
	XV_HELP_DATA,		"lqtext:vocabulary-panel",
	XV_X,			0,
	NULL
    );

    /* make a text field for the pattern */
    xv_create(P, PANEL_TEXT,
	PANEL_LABEL_STRING,	"Pattern:",
	PANEL_VALUE_DISPLAY_LENGTH,	30,
	PANEL_VALUE_STORED_LENGTH,	1000,
	PANEL_NOTIFY_STRING,		"\r\n\t\033",
	PANEL_NOTIFY_PROC,		CB_ChangePattern,
	PANEL_VALUE,			".*",
	XV_HELP_DATA,		"lqtext:pattern",
	NULL
    );

    F_List = (Xv_opaque) xv_create(P, PANEL_LIST,
	PANEL_LIST_DISPLAY_ROWS,	40,
	PANEL_CHOOSE_ONE,		FALSE,
	XV_X,				10,
	XV_HELP_DATA,			"lqtext:wordlist",
	NULL
    );
}

void
ShowWordList()
{
    if (!F_WordList) {
	MakeWordList();
    }

    xv_set(F_WordList,
	XV_SHOW, TRUE,
	WIN_FRONT, TRUE,
	NULL
    );
}

void
CB_ViewMenu(WhichMenu, WhichItem)
    Menu WhichMenu;
    Menu_item WhichItem;
{
    char *String = (char *) xv_get(WhichItem, MENU_STRING);

    if (!String || !*String) return; /* paranoia... */

    if (STREQ(String, "All")) {
	ShowSelectedFiles(FilesAll);
    } else if (STREQ(String, "Selected")) {
	ShowSelectedFiles(FilesSelected);
    } else if (STREQ(String, "Vocabulary")) {
	ShowWordList();
    } else {
	fprintf(stderr, "%s: unknown menu choice: %s\n",
	    progname, String);
    }
}

Panel_setting
CB_InnerPhrase(WhichItem, WhichEvent)
    Panel_item WhichItem;
    Event *WhichEvent;
{
    switch (event_action(WhichEvent)) {
    case '\n': case '\r': case '\t':
	/* move to the next field */
	return PANEL_NEXT; /* green book p. 167 */
    case '\033':
	CB_MakeList();
	break;
    default:
	return panel_text_notify(WhichItem, WhichEvent);
    }
    return 0;
}

Panel_setting
CB_LastPhrase(WhichItem, WhichEvent)
    Panel_item WhichItem;
    Event *WhichEvent;
{
    CB_MakeList();
    return 0;
}

void
CB_SetMatchLevel(item, value, event)
    Panel_item item;
    int value;
    Event *event;
{
    switch (value) {
    case 0:
	CaseMatch = "-mp";
	break;
    case 1:
	CaseMatch = "-mh";
	break;
    case 2:
	CaseMatch = "-ma";
	break;
    default:
	fprintf(stderr, "%s: invalid choice %d in CB_SetMatchLevel\n",
		progname, item);
    }
}

Panel_setting
CB_SetContextAbove(item, event)
    Panel_item item;
    Event *event;
{
    ContextLinesAbove = xv_get(item, PANEL_VALUE);
    return PANEL_INSERT;
}

Panel_setting
CB_SetContextBelow(item, event)
    Panel_item item;
    Event *event;
{
    ContextLinesBelow = xv_get(item, PANEL_VALUE);
    return PANEL_INSERT;
}

Panel_setting
CB_SetDocPath(item, event)
    Panel_item item;
    Event *event;
{
    char *DocPath = (char *) xv_get(item, PANEL_VALUE);
    static char *Value = 0;

    if (Value) free(Value);

    if (!DocPath || !*DocPath || STREQ(DocPath, "(default)")) {
	DocPath = "`getdefaults docpath`";
    }
    DocPath = ShellExpand(DocPath);
    Value = malloc(sizeof("DOCPATH=") + strlen(DocPath));
    (void) sprintf(Value, "DOCPATH=%s", DocPath);
    (void) putenv(Value);
    xv_set(item, PANEL_VALUE, DocPath, NULL);

    return PANEL_INSERT;
}

Panel_setting
CB_SetDataBaseDir(item, event)
    Panel_item item;
    Event *event;
{
    char *DataBaseDir = (char *) xv_get(item, PANEL_VALUE);
    static char *Value = 0;
    char Buf[2000];

    if (Value) free(Value);

    if (!DataBaseDir || !*DataBaseDir) {
	DataBaseDir = "$HOME/LQTEXTDIR";
    }

    DataBaseDir = ShellExpand(DataBaseDir);
    Value = malloc(sizeof("LQTEXTDIR=") + strlen(DataBaseDir));
    (void) sprintf(Value, "LQTEXTDIR=%s", DataBaseDir);
    (void) putenv(Value);
    xv_set(item, PANEL_VALUE, DataBaseDir, NULL);

    return PANEL_INSERT;
}

Panel_setting
CB_SetTerminalEmulator(item, event)
    Panel_item item;
    Event *event;
{
    char *te = (char *) xv_get(item, PANEL_VALUE);
    static char *Value = 0;

    if (Value) free(Value);

    if (!te || !*te) {
	te = DEF_TERM;
    }

    Value = ShellExpand(te);
    Value = malloc(strlen(te) + 1);
    (void) strcpy(Value, te);
    TerminalEmulator = Value;
    xv_set(item, PANEL_VALUE, te, NULL);

    return PANEL_INSERT;
}

int
CB_SetAction(item, value, event)
    Panel_item item;
    int value;
    Event *event;
{
    switch (value) {
    case 0:
	ActionProgram = "lqshow";
	break;
    case 1:
	ActionProgram = "lqkwik2";
	break;
    case 2:
	ActionProgram = "ae";
	break;
    }
    return value;
}

static void
CB_ChangePattern()
{
}
