#
# This file is part of ElasticSearchX-Model
#
# This software is Copyright (c) 2011 by Moritz Onken.
#
# This is free software, licensed under:
#
#   The (three-clause) BSD License
#
package ElasticSearchX::Model::Document;
{
  $ElasticSearchX::Model::Document::VERSION = '0.0.3';
}

use strict;
use warnings;

use Moose 1.15 ();
use Moose::Exporter;
use ElasticSearchX::Model::Document::Trait::Class;
use ElasticSearchX::Model::Document::Trait::Attribute;
use ElasticSearchX::Model::Document::Types qw();

my ( undef, undef, $init_meta ) = Moose::Exporter->build_import_methods(
    install         => [qw(import unimport)],
    with_meta       => [qw(has)],
    class_metaroles => {
        class     => ['ElasticSearchX::Model::Document::Trait::Class']
    }, );

sub has {
    my $meta = shift;
    my $name = shift;

    Moose->throw_error('Usage: has \'name\' => ( key => value, ... )')
      if @_ % 2 == 1;
    my %options = ( definition_context => Moose::Util::_caller_info(), @_ );
    $options{traits} ||= [];
    push(@{$options{traits}}, 'ElasticSearchX::Model::Document::Trait::Attribute')
        if($options{property} || !exists $options{property});
    delete $options{property};
    
    my $attrs = ( ref($name) eq 'ARRAY' ) ? $name : [ ($name) ];
    $meta->add_attribute( $_, %options ) for @$attrs;
}

sub init_meta {
    my $class = shift;
    my %p     = @_;
    Moose::Util::ensure_all_roles( $p{for_class},
                                   qw(ElasticSearchX::Model::Document::Role) );
    $class->$init_meta(%p);
}

1;



=pod

=head1 NAME

ElasticSearchX::Model::Document

=head1 VERSION

version 0.0.3

=head1 SYNOPSIS

 package MyClass;
 use Moose;
 use ElasticSearchX::Model::Document;
 use ElasticSearchX::Model::Document::Types qw(Location);
 use MooseX::Types -declare => ['Resources'];
 use MooseX::Types::Structured qw(Dict Optional);
 use MooseX::Types::Moose qw(Str ArrayRef);

 subtype Resources,
  as Dict [ license => Optional [ ArrayRef [Str] ],
            homepage => Optional [Str],
            bugtracker => Optional [ Dict [ web => Str, mailto => Str ] ] ];

 has default  => ( is => 'ro' );
 has date     => ( is => 'ro', isa => 'DateTime' );
 has location => ( is => 'ro', isa => Location );
 has res      => ( is => 'ro', isa => Resources );
 has abstract => ( is => 'ro', analyzer => 'lowercase' );

=head1 DESCRIPTION

This class extends Moose to include meta information for ElasticSearch.
By default, each attribute is treated as property of an ElasticSearch
type (i.e. the L<ElasticSearchX::Model::Document::Trait::Attribute>
trait is applied). The type name is derived from the class name. See
L<ElasticSearchX::Model::Document::Trait::Class>.

See L<ElasticSearchX::Model::Document::Trait::Attribute/ATTRIBUTES> for a full 
list of attribute options available.

=head1 ATTRIBUTES

=head2 index

 $document->index($model->index('index_v2'));
 $document->put;

The L<ElasticSearchX::Model::Index> object this document belongs to.
This attribute is writeable, which allows you to reindex a document
to a different index. Make sure that the type is part of the new
index. Otherwise indexing is likely to fail.

=head2 _id

This is a read-only attribute that holds the value of the id of
the document. If you want to set the id by yourself, create an
attribute in your document class which has the C<id> attribute
set to C<1>. Otherwise the id will be generated by ElasticSearch.

 has id => ( is => 'ro', id => 1 );

=head1 METHODS

=head2 put

This puts a document to the ElasticSearch server. It calls
L<ElasticSearchX::Model::Document::Trait::Class/get_data> to retrieve the
data from an L<ElasticSearchX::Model::Document> object.

=head2 delete

Delete the document from the index.

=head1 AUTHOR

Moritz Onken

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2011 by Moritz Onken.

This is free software, licensed under:

  The (three-clause) BSD License

=cut


__END__

