# This script is expected to run with the non-local version of the admin
# library linked in.  So when the database is in fact local, we need to
# run kadmin.local to perform the operations.  Once the KDC and admin
# server are running, of course, we can use the non-local version with
# no problems; we just wind up having to authenticate to it despite having
# direct write access to the database.
#
# With tcl's dynamic library support, it'd be nice to load the admin
# library only after we've decided which version we need.  But we aren't
# using that support at this point in time.  Oh well.

#set kdb5_util_prog "../kadmin/dbutil/kdb5_util"
#set kadmin_local_prog "../kadmin/cli/kadmin.local"
set kdb5_util_prog "$krb5_path(prefix)/sbin/kdb5_util"
set kadmin_local_prog "$krb5_path(prefix)/sbin/kadmin.local"

# assumes config files have already been written

# run a command, feeding it passwords, and check for expected output
proc create_db.run { cmd passwords tmpname expected } {
    catch {
	set f [open "|$cmd >&$tmpname" w]
	puts $f $passwords
	close $f
    } errs
    if [string length $errs] {
	error "error running \"$cmd\": $errs"
    }

    # Now look at the output from running the program.
    set output ""
    catch {
	set f [open $tmpname r]
	set output [read -nonewline $f]
	close $f
    } errs
    if [string length $errs] {
	error "can't check log of Kerberos database commands: $errs"
    }
    set output [split $output "\n"]

    # Check for certain OS-specific warning messages we can safely discard.
    if [regexp "^ld.so:" [lindex $output 0]] {
	set output [lrange $output 1 end]
    }

    # Check for the expected output strings.
    foreach x $expected {
	set out [lindex $output 0]
	if [regexp "^$x\$" $out] {
	    set output [lrange $output 1 end]
	} else {
	    regsub "^$x" $out "" out
	    error "unexpected output from \"$cmd\":\n$out"
	}
    }
    if [llength $output] {
	error "unexpected output from \"$cmd\":\n[join $output \"\n\"]"
    }
}
proc get_temp_file { } {
    global env tmpdir
    # Find a place for a temporary file.
    if ![info exists tmpdir] {
	if [info exists env(TMPDIR)] {
	    set tmpdir $env(TMPDIR)
	    if {![file isdirectory $tmpdir] || ![file writable $tmpdir]} {
		error "Environment variable TMPDIR says $tmpdir but it's not writable."
	    }
	} else {
	    foreach d { /tmp /var/tmp /usr/tmp } {
		if [file isdirectory $d]&&[file writable $d] {
		    set tmpdir $d
		    break
		}
	    }
	    if ![info exists tmpdir] {
		error "Unable to find a writable temporary directory."
	    }
	}
    }
    return "$tmpdir/kn[pid]inst.tmp"
}
proc create_db { password } {
    global kdb5_util_prog
    global realm

    set tmpname [get_temp_file]

    # Run "kdb5_util create"
    create_db.run "$kdb5_util_prog create -s" \
	    "$password\n$password\n" $tmpname [list		\
	"Initializing database '.*' for realm '$realm',"	\
	"master key name 'K/M@$realm'"				\
	"You will be prompted for the database Master Password." \
	"It is important that you NOT FORGET this password."	\
	"Enter KDC database master key:"			\
	"Re-enter KDC database master key to verify:"		\
    ]
    file delete $tmpname
}
proc create_admin_inst { apassword } {
    global realm kadmin_local_prog admin_principal
    set tmpname [get_temp_file]
    # Okay.  Now, create the admin/admin entry.
    create_db.run "$kadmin_local_prog -q \"ank $admin_principal\"" \
	    "$apassword\n$apassword\n" $tmpname [list		\
	"Enter password for principal \"$admin_principal@$realm\": "	\
	"Re-enter password for principal \"$admin_principal@$realm\": " \
	"Principal \"$admin_principal@.*\" created."			\
    ]
    file delete $tmpname
}
proc load_db { dumpfile } {
    global kdb5_util_prog

    exec $kdb5_util_prog load $dumpfile
}

proc setup_database { realm dbdir } {
    set dbfile $dbdir/principal.db
    set stash_file $dbdir/.k5.$realm
    set kadmin_keytab $dbdir/kadm5.keytab

    if [file exists $dbfile] then {
	puts \
"\nAn old KDC database appears to be present.  If you want to continue using
that database, see the documentation for how to proceed; this script is
intended primarily for new installations, and does not yet handle existing
databases.\n"
	if [ask-yesno "Delete the database?" n] {
	    eval file delete [glob $dbdir/principal.*] \
		    $stash_file $kadmin_keytab
	} else {
	    puts "Okay.  Check the documentation and start this configuration"
	    puts "program again later."
	    exit 0
	}
    }
    # Okay.  Now, there's no database.
    puts "\nPreparing to create the KDC database..."
    set password [askpw "Enter the database master password:" \
	    "For verification, enter the master password again:"]
    puts "Creating the KDC database..."
    create_db $password
}
