/*
 * admin/edit/cpw.c
 *
 * Copyright 1995 by the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * Export of this software from the United States of America may
 *   require a specific license from the United States Government.
 *   It is the responsibility of any person or organization contemplating
 *   export to obtain such a license before exporting.
 * 
 * WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
 * distribute this software and its documentation for any purpose and
 * without fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of M.I.T. not be used in advertising or publicity pertaining
 * to distribution of the software without specific, written prior
 * permission.  M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 * 
 * Change passwords for a KDC db entry.
 */

#include "k5-int.h"
#include "com_err.h"
#include "adm.h"
#include "adm_proto.h"
#include <stdio.h>
#include <time.h>

#include "kdb5_edit.h"
#include "kascvt.h"

extern char	*Err_no_master_msg;
extern char	*Err_no_database;
extern char	*current_dbname;


/*
 * I can't figure out any way for this not to be global, given how ss
 * works.
 */
extern int exit_status;
extern krb5_context edit_context;
extern krb5_keyblock master_keyblock;
extern krb5_principal master_princ;
extern krb5_db_entry master_entry;
extern krb5_encrypt_block master_encblock;
extern int	valid_master_key;
extern char *krb5_default_pwd_prompt1, *krb5_default_pwd_prompt2;
extern krb5_boolean dbactive;
extern FILE *scriptfile;

/* static routines stolen from lib/kdb/kdb_cpw.c, since the interfaces
   there aren't good enough for raw keys. */

/* unmodified */
static int
get_key_data_kvno(context, count, data)
    krb5_context	  context;
    int			  count;
    krb5_key_data	* data;
{
    int i, kvno;
    /* Find last key version number */
    for (kvno = i = 0; i < count; i++) {
	if (kvno < data[i].key_data_kvno) {
	    kvno = data[i].key_data_kvno;
	}
    }
    return(kvno);
}

/* unmodified */
static void
cleanup_key_data(context, count, data)
    krb5_context	  context;
    int			  count;
    krb5_key_data	* data;
{
    int i, j;

    for (i = 0; i < count; i++) {
	for (j = 0; j < data[i].key_data_ver; j++) {
	    if (data[i].key_data_length[j]) {
	    	free(data[i].key_data_contents[j]);
	    }
	}
    }
}
/*
 * Add key_data for a krb5_db_entry 
 * If passwd is NULL the assumes that the caller wants a random password.
 */
/* but if salttype is afs (which it must be) use the passwd as raw key */
static krb5_error_code
add_key_afs_pwd(context, master_eblock, ks_tuple, ks_tuple_count, keyval, 
	    db_entry, kvno)
    krb5_context	  context;
    krb5_encrypt_block  * master_eblock;
    krb5_key_salt_tuple	* ks_tuple;
    int			  ks_tuple_count;
    char 		* keyval;
    krb5_db_entry	* db_entry;
    int			  kvno;
{
    krb5_error_code	  retval;
    krb5_encrypt_block    key_eblock;
    krb5_keysalt	  key_salt;
    krb5_keyblock	  key;
    krb5_data	  	  pwd;
    krb5_boolean	  found;
    int			  i, j;

    for (i = 0; i < ks_tuple_count; i++) {
	krb5_enctype new_enctype, old_enctype;

	switch (new_enctype = ks_tuple[i].ks_enctype) {
	case ENCTYPE_DES_CBC_MD4:
	case ENCTYPE_DES_CBC_MD5:
	case ENCTYPE_DES_CBC_RAW:
	    new_enctype = ENCTYPE_DES_CBC_CRC;
	default:
	    break;
	}
	/*
	 * We could use krb5_keysalt_iterate to replace this loop, or use
	 * krb5_keysalt_is_present for the loop below, but we want to avoid
	 * circular library dependencies.
	 */
	for (found = j = 0; j < i; j++) {
	    if (ks_tuple[j].ks_salttype == ks_tuple[i].ks_salttype) {
		switch (old_enctype = ks_tuple[j].ks_enctype) {
		case ENCTYPE_DES_CBC_MD4:
		case ENCTYPE_DES_CBC_MD5:
		case ENCTYPE_DES_CBC_RAW:
	    	    old_enctype = ENCTYPE_DES_CBC_CRC;
		default:
	    	    break;
		}
	        if (old_enctype == new_enctype) {
		    found = 1;
		    break;
		}
	    }
	}
	if (found)
	    continue;
	krb5_use_enctype(context, &key_eblock, ks_tuple[i].ks_enctype);
	if (retval = krb5_dbe_create_key_data(context, db_entry)) 
	    return(retval);

	/* Convert password string to key using appropriate salt */
	switch (key_salt.type = ks_tuple[i].ks_salttype) {
    	case KRB5_KDB_SALTTYPE_AFS3: {
            krb5_data * saltdata;
            if (retval = krb5_copy_data(context, krb5_princ_realm(context,
					db_entry->princ), &saltdata))
	 	return(retval);

	    key_salt.data = *saltdata;
	    /* key_salt.data.length = -1; can't do this here! */
	    krb5_xfree(saltdata);
	}
		break;
	default:
	    abort();
	    return(KRB5_KDB_BAD_SALTTYPE);
	}

	key.contents = keyval;
	key.magic = KV5M_KEYBLOCK;
	key.length = 8;
	key.enctype = key_eblock.crypto_entry->proto_enctype;

	if (retval = krb5_dbekd_encrypt_key_data(context, master_eblock, &key,
		     (const krb5_keysalt *)&key_salt,
		     kvno + 1, &db_entry->key_data[db_entry->n_key_data-1])) {
	    return(retval);
	}
    }
    return(retval);
}

/*
 * Change password for a krb5_db_entry 
 * Assumes the max kvno
 *
 * As a side effect all old keys are nuked.
 */
krb5_error_code
afs_dbe_cpw(context, master_eblock, ks_tuple, ks_tuple_count, keyval, db_entry)
    krb5_context	  context;
    krb5_encrypt_block  * master_eblock;
    krb5_key_salt_tuple	* ks_tuple;
    int			  ks_tuple_count;
    char 		* keyval;
    krb5_db_entry	* db_entry;
{
    int 		  key_data_count;
    krb5_key_data 	* key_data;
    krb5_error_code	  retval;
    int			  kvno;

    /* First save the old keydata */
    kvno = get_key_data_kvno(context, db_entry->n_key_data, db_entry->key_data);
    key_data_count = db_entry->n_key_data;
    key_data = db_entry->key_data;
    db_entry->key_data = NULL;
    db_entry->n_key_data = 0;

    if (retval = add_key_afs_pwd(context, master_eblock, ks_tuple, ks_tuple_count,
				 keyval, db_entry, kvno)) {
	cleanup_key_data(context, db_entry->n_key_data, db_entry->key_data);
	db_entry->n_key_data = key_data_count;
	db_entry->key_data = key_data;
    } else {
	cleanup_key_data(context, key_data_count, key_data);
    }
    return(retval);
}

static krb5_key_salt_tuple ks_tuple_afs[] = 
{{ ENCTYPE_DES_CBC_CRC, KRB5_KDB_SALTTYPE_AFS3 }};

static krb5_key_salt_tuple *afs_ks_tuple = ks_tuple_afs;
static int afs_ks_tuple_count = 1;

void 
enter_afs_pwd_key(cmdname, princ, ks_tuple, ks_tuple_count, entry, keyval)
    char 		* cmdname;
    char 		* princ;
    krb5_key_salt_tuple	* ks_tuple;
    int			  ks_tuple_count;
    krb5_db_entry 	* entry;
    char *keyval;
{
    krb5_error_code 	  retval;
    int			  one = 1;
  
    if (ks_tuple_count == 0) {
	ks_tuple_count = afs_ks_tuple_count;
	ks_tuple = afs_ks_tuple;
    }
    if ((retval = afs_dbe_cpw(edit_context, &master_encblock, ks_tuple,
			      ks_tuple_count, keyval, entry))) {
	com_err(cmdname, retval, "while storing entry for '%s'\n", princ);
        memset(keyval, 0, 8); /* erase it */
	krb5_dbe_free_contents(edit_context, entry);
	goto errout;
    }
    memset(keyval, 0, 8); /* erase it */

    /* Write the entry back out and we're done */
    if ((retval = krb5_db_put_principal(edit_context, entry, &one))) {
	com_err(cmdname, retval, "while storing entry for '%s'\n", princ);
    }

    if (one != 1) {
        com_err(cmdname, 0, "entry not stored in database (unknown failure)");
        exit_status++;
    }

errout:;
    krb5_db_free_principal(edit_context, entry, one);
    if (retval)
        exit_status++;
    return;
}


static int
afs_pre_key(progname, newprinc, entry)
    char		* progname;
    krb5_principal 	* newprinc;
    krb5_db_entry 	* entry;
{
    krb5_boolean 	  more;
    krb5_error_code 	  retval;
    int 		  nprincs = 1;

    if (!dbactive) {
	com_err(progname, 0, Err_no_database);
    } else if (!valid_master_key) {
	com_err(progname, 0, Err_no_master_msg);
    } else if ((retval = krb5_db_get_principal(edit_context, *newprinc, entry, 
					       &nprincs, &more))) {
        com_err(progname,retval,"while trying to get principal's db entry");
    } else if ((nprincs > 1) || (more)) {
	krb5_db_free_principal(edit_context, entry, nprincs);
    	krb5_free_principal(edit_context, *newprinc);
    } else if (nprincs) 
	return(1);
    else 
	return(0);
    return(-1);
}

/* was add_pwd_key */
int kdb_write_one_entry(progname, r, v4realm)
     char *progname;
     struct kas_entry *r;
     char *v4realm;
{
    krb5_error_code 	  retval;
    krb5_principal 	  newprinc;
    krb5_db_entry	  entry;
    krb5_error_code v5code;
    char *v4name, *v4instance;
    int v4len;

    v4instance = strchr(r->name, '.'); /* skip to the dot */
    if(v4instance) { 
      v4len = v4instance - r->name;	/* measure the name */
      v4name = malloc(v4len+1);
      if(!v4name) { return -1; }
      strncpy(v4name, r->name, v4len);
      v4name[v4len] = 0;
      v4instance++;			/* just skip the dot */
    } else {
      /* no instance */
      v4name = r->name;
      v4instance = "";
    }

    v5code = krb5_425_conv_principal(edit_context, 
				     v4name, v4instance, v4realm, 
				     &newprinc);
    if(v5code) {
      com_err(progname, v5code, "parsing v4 name %s@%s", r->name, v4realm);
      return v5code;
    }
    switch (afs_pre_key(progname, &newprinc, &entry)) {
    case 0:
	if ((retval = create_db_entry(newprinc, &entry))) {
	    com_err(progname, retval, "While creating new db entry.");
	    exit_status++;
	    return;
	}
        enter_afs_pwd_key(progname, r->name, NULL, 0, &entry, r->keyval);
    	krb5_free_principal(edit_context, newprinc);
	return;
    case 1:
	com_err(progname, 0, "Principal '%s' already exists.", r->name);
        krb5_db_free_principal(edit_context, &entry, 1);
        krb5_free_principal(edit_context, newprinc);
    default:
    	exit_status++;
	break;
    }
}

#include "consume.h"

/*
User data for AuthServer.Admin (NOTGS+NOSEAL)
  key (1):k\266\367mFQ\266\357, last cpw: Thu Jul 27 16:22:09 1995
  password will never expire.
  An unlimited number of unsuccessful authentications is permitted.
  entry never expires.  Max ticket lifetime 10.00 hours.
  last mod on Thu Jul 27 16:22:09 1995 by <none>

*/

static int add_to_list(l,s)
     struct list **l;
     char *s;
{
  /* drop s onto the list some how. Not ordered */
  struct list *n;
  n = malloc(sizeof(struct list));
  if (!n) { /* abort? */ return ENOMEM; }
  n->el = s;
  n->next = (*l);
  *l = n;
  return 0;
}

static void print_one_entry(k)
     struct kas_entry *k;
{
  struct list* l;
  printf("Name: %s kvno: %d life(hours) %f\n",
	 k->name, k->kvno, k->lifehours);
  printf("Key: %03o %03o %03o %03o  %03o %03o %03o %03o\n",
	 k->keyval[0],
	 k->keyval[1],
	 k->keyval[2],
	 k->keyval[3],
	 k->keyval[4],
	 k->keyval[5],
	 k->keyval[6],
	 k->keyval[7]);
  if (k->flags) printf("Flags: (");
  for(l = k->flags; l; l=l->next) {
    printf("{%s}%c", l->el, l->next?'+':')');
  }
  if (k->flags) printf("\n");
  printf("change: {%s}\npw expire: {%s}\nslimit: {%s}\nexp date: {%s}\n",
	 k->change_date, k->password_expire, 
	 k->success_limit, k->expires_date);

  printf("last mod: {%s}\n\n", k->last_mod);
}

static void print_one_line(k)
     struct kas_entry *k;
{
  struct list* l;
  printf("%s kvno: %d life(hours) %f\n",
	 k->name, k->kvno, k->lifehours);
}

struct kas_entry *chew_one_entry(c)
     struct consume_data* c;
{
  char *noflags;
  struct kas_entry *r = malloc(sizeof(struct kas_entry));
  int i;

  /* error on !r */
  if (!r) { /* abort? */ return 0; }
  r->flags = 0;

  skip_ws(c);
  if (found_eof(c)) {
    free(r);
    return 0;
  }
  skip_string(c,"User data for ");
  r->name = consume_until(c," \n");	/* stops at match */
  if (looking_at(c," ")) {
    skip_string(c," (");
    do {
      noflags = consume_until(c,"+)");
      add_to_list(&(r->flags), noflags);
      if (looking_at(c,"+")) skip_string(c,"+");
    } while(!looking_at(c,")"));
    skip_string(c,")\n");
  }
  skip_ws(c);
  skip_string(c,"key (");
  r->kvno = consume_int_decimal(c,")");
  skip_string(c,"):");
  for (i = 0; i<8; i++) {
    if (looking_at(c,"\\")) {
      skip_string(c,"\\");
      r->keyval[i] = consume_int_3octal(c); /* grabs 3 digits exactly */
    } else {
      r->keyval[i] = consume_byte(c);	/* grab byte, return as int, skip */
    }
  }
  skip_string(c,", last cpw: ");
  r->change_date = consume_date(c,"\n");
  skip_ws(c);
  r->password_expire = consume_until(c,"\n");
  skip_ws(c);
  r->success_limit = consume_until(c,"\n");
  skip_ws(c);
  r->expires_date = consume_until(c,".");
  skip_string(c,".  Max ticket lifetime ");
  r->lifehours = consume_float(c," ");
  skip_string(c," hours.\n  ");
  r->last_mod = consume_until(c,"\n");
  return r;
}

extern char *cur_realm;

int add_afs_keys(argc, argv)
     int argc;
     char *argv[];
{
  struct kas_entry *r;
  /* setup context */
  struct consume_data *c;
  char *v4realm = 0;
  int verbose = 0;
  extern char *optarg;	
  extern int optind;
  int optchar;
  char *afsname = 0;
  FILE *afsfile = stdin;

  optind = 1;
  while ((optchar = getopt(argc, argv, "a:4:v")) != EOF) {
    switch(optchar) {
    case 'a':			/* afs input file name */
      afsname = optarg;
      afsfile = fopen(afsname, "r");
      if (!afsfile) perror("kascvt: opening afs input file");
      break;
    case 'v':
      verbose++;
      break;
    case '4':			/* v4 realm name */
      v4realm = optarg;
      printf("original v4-style realm name %s\n", v4realm);
      break;
    default:
      fprintf(stderr,"kascvt: usage: kascvt [-a database] [-v] [-4 v4name]\n");
      exit(1);
      /*NOTREACHED*/
    }
  }
  if (!v4realm) {
    v4realm = cur_realm;
    printf("defaulting to %s for original v4-style realm name\n", v4realm);
  }
  if (!afsfile) {
    fprintf(stderr, "kascvt: init failed\n");
    exit(2);
  }
  c = consume(afsfile);
  while (r = chew_one_entry(c)) {
    if(verbose>=2) print_one_entry(r);
    if(verbose>=1) print_one_line(r);
    if(r->kvno == 0) {
      r->kvno = 1;
      if(verbose>=1) printf("kascvt: kvno changed from 0 to 1\n");
    }
    kdb_write_one_entry(argv[0], r, v4realm); /* v4realm from cmd line */
  }
  return 0;
}
