#if defined(_WIN32) && defined(__CYGWIN32__)
#include <windows.h>
#include <exceptions.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include <string.h>
#include <time.h>
#include "win32service.h"

#include "osconf.h"

/* Local #defines. */
#ifdef KERBNET
#define KRB5_CONFIG_ENV "KRB5_CONFIG"
#define KRB5_CONFIG_ENV_VALUE "etc\\krb5.conf"
#define KRB5_KDC_PROFILE_ENV "KRB5_KDC_PROFILE"
#define KRB5_KDC_PROFILE_ENV_VALUE "etc\\kdc.conf"
#endif /* KERBNET */

#define PWS_DLL_WATCHKEY_REG_NAME \
"SOFTWARE\\Cygnus Solutions\\PasswordSync\\WatchKey"

#define PWS_REG_NAME "SOFTWARE\\Cygnus Solutions\\PasswordSync"
#define PWS_ARGS_REG_NAME "PasswordSync args"
#define NOFORK_ARG "-n"

extern int request_exit(int);
extern int original_main(int argc, char **argv);

/*
 * Stop mutex - this will go away when select can be interrupted
 * by a signal in cygwin32.
 */

static HANDLE can_stop = INVALID_HANDLE_VALUE;

/*
 * Calls to lock/unlock mutex.
 */

DWORD lock_exit(DWORD timeout)
{
  DWORD ret;
  if(can_stop == INVALID_HANDLE_VALUE)
    return WAIT_OBJECT_0;
  ret = WaitForSingleObject(can_stop, timeout);
  if(ret == WAIT_ABANDONED || ret == WAIT_FAILED) {
    log_message(LOG_ERR, "Failed to wait for stop mutex. Error was",
                    str_oserr(GetLastError()));
  }
  return ret;
}

void unlock_exit()
{
  if(can_stop != INVALID_HANDLE_VALUE)
    ReleaseMutex(can_stop);
}

/*
 * Function to change a value on a known key
 * that the pwsync dll is watching. This lets the pwsync
 * dll know that we are starting/stopping and it should
 * close its named pipe handle.
 */

int update_watch_value() 
{
  static HKEY watch_key = INVALID_HANDLE_VALUE;
  DWORD val;
  DWORD err;

  if(watch_key == INVALID_HANDLE_VALUE) {
    /* Open the watchkey */
    if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE,
			   PWS_DLL_WATCHKEY_REG_NAME, 0, KEY_WRITE,
			   &watch_key)) != ERROR_SUCCESS) {
      log_message(LOG_ERR, "Failed to open registry key %s. Error was %s",
		PWS_DLL_WATCHKEY_REG_NAME, str_oserr(err));
      return FALSE;
    }
    srand((unsigned)time(0));
  }
  /* Set a dummy value on the watchkey */
  val = rand();
  if((err = RegSetValueEx( watch_key, "watchval", 0, REG_DWORD,
		   (CONST BYTE *)&val, sizeof(val))) != ERROR_SUCCESS) {
    log_message(LOG_ERR, 
	      "Failed to write the watch value in registry key %s. Error was %s",
	      PWS_DLL_WATCHKEY_REG_NAME, str_oserr(err));
    return FALSE;
  }
}

#ifdef KERBNET
/*
 * Setup the environment variables KRB5_CONFIG and
 * KRB5_KDC_PROFILE from the registry. 
 */

int setup_environment_variables()
{
  char *kerbnet_home;
  HKEY hkey;
  DWORD type;
  DWORD datasize;
  DWORD home_length;
  DWORD err;

  if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, KERBNET_BASE_REG_NAME, 0,
		KEY_READ, &hkey)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to open Registry key %s. Error was %s",
                KERBNET_BASE_REG_NAME, str_oserr(err));
    return -1;
  }
  
  /* Try and get the KERBNET_HOME value size. */
  datasize = 0;
  if((err = RegQueryValueEx(hkey, KERBNET_HOME_REG_NAME, 0, &type, 0, &datasize)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to find Registry value %s. Error was %s", 
                KERBNET_HOME_REG_NAME, str_oserr(err));
    RegCloseKey(hkey);
    return -1;
  }

  /* Add one char for a possibly missing '\' character */
  datasize += sizeof(char);
  /* Add in the size we will use to append to it. */
  datasize += 20;
 
  /* Allocate the memory for the value */
  if((kerbnet_home = (char *)LocalAlloc(LMEM_FIXED, datasize)) == 0) {
    log_message(LOG_ERR, "Failed to allocate space for value %s. Error was %s", 
                KERBNET_HOME_REG_NAME, str_oserr(GetLastError()));
    RegCloseKey(hkey);
    return -1;
  }

  /* Now get the value */
  if((err = RegQueryValueEx(hkey, KERBNET_HOME_REG_NAME, 0, &type, (LPBYTE)kerbnet_home, 
							&datasize)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to retrieve Registry value %s. Error was %s.", 
                KERBNET_HOME_REG_NAME, str_oserr(err));
    RegCloseKey(hkey);
    LocalFree((HLOCAL)kerbnet_home);
    return -1;
  }

  /* Ensure it ends in a '\' */
   if(kerbnet_home[strlen(kerbnet_home)-1] != '\\')
    strcat(kerbnet_home, "\\");

  home_length = strlen(kerbnet_home);

  /*
   * Set KERBNET_HOME as an environment variable, then set KRB5_CONFIG and
   * KRB5_KDC_PROFILE.
   */
  setenv(KERBNET_HOME_REG_NAME, kerbnet_home, 1);
  /* Add on the extra for KRB5_CONFIG */
  strcat( kerbnet_home, KRB5_CONFIG_ENV_VALUE);
  setenv(KRB5_CONFIG_ENV, kerbnet_home,1);
  /* Truncate back to KERBNET_HOME */
  kerbnet_home[home_length] = '\0';
  /* Add on the extra for KRB5_KDC_PROFILE */
  strcat(kerbnet_home, KRB5_KDC_PROFILE_ENV_VALUE);
  setenv(KRB5_KDC_PROFILE_ENV, kerbnet_home,1);

  LocalFree((HLOCAL)kerbnet_home);
  RegCloseKey(hkey);
  return 0;   
}
#endif /* KERBNET */

/*
 * Add any needed args to the argc, argv array.
 * Fish them from the registry from a REG_MULTI_SZ datatype.
 * By default there is always one extra arg, NOFORK_ARG.
 */
 
int add_extra_args( struct argc_argv *args)
{
  int extra_args;
  char *pws_args = 0;
  char *p;
  char **new_argv;
  int i;
  DWORD datasize;
  DWORD type;
  HKEY hkey;
  DWORD err;
  
  if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, PWS_REG_NAME, 0,
		KEY_READ, &hkey)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "Failed to open Registry key %s. Error was %s",
		PWS_REG_NAME, str_oserr(err));
    return -1;
  }
  
  /* Try and get the PasswordSync args value size. */
  datasize = 0;
  if((err = RegQueryValueEx(hkey, PWS_ARGS_REG_NAME, 0, &type, 0, &datasize)) == ERROR_SUCCESS) {
    /* Allocate the memory for the value */
    if((pws_args = (char *)LocalAlloc(LMEM_FIXED, datasize)) == 0) {
      log_message(LOG_ERR, "Failed to allocate space for value %s. Error was %s",
                  PWS_ARGS_REG_NAME, str_oserr(GetLastError()));
      RegCloseKey(hkey);
      return -1;
    }

    /* Now get the value */
    if((err = RegQueryValueEx(hkey, PWS_ARGS_REG_NAME, 0, &type, (LPBYTE)pws_args, 
  							&datasize)) != ERROR_SUCCESS) {
      log_message(LOG_ERR, "Failed to retrieve Registry value %s. Error was %s",
                  PWS_ARGS_REG_NAME, str_oserr(err));
      RegCloseKey(hkey);
      LocalFree((HLOCAL)pws_args);
      return -1;
    }
  }

  RegCloseKey(hkey);

  /* Now go through the REG_MULTI_SZ pws_args and add them into a re-allocated
     argv array. extra_args starts at 1 due to the NOFORK_ARG arg. */
  extra_args = 1;
  if(pws_args != 0) {
    for( p = pws_args; *p; p += strlen(p) + 1, ++extra_args)
      ;
  }

  if((new_argv = (char **)LocalAlloc(LMEM_FIXED,
                          (args->argc + extra_args + 1)*sizeof(char *)))==0) {
      log_message(LOG_ERR,
                  "Failed to find allocate space for new PasswordSync arg array. Error was %s",
                  str_oserr(GetLastError()));
      LocalFree((HLOCAL)pws_args);
      return -1;
  }

  /* Setup nofork as the first extra arg. */
  new_argv[0] = args->argv[0];
  new_argv[1] = NOFORK_ARG;

  /* Add in the remaining args from the command line. */
  for( i = 1; i < args->argc; ++i)
    new_argv[i+1] = args->argv[i];

  /* Increment by one for the extra NOFORK_ARG. */
  ++i;
  
  /* Now add in the args from the registry */
  if(pws_args != 0) {
    for( p = pws_args; *p; p += strlen(p) + 1) {
      if(strcmp(p, NOFORK_ARG) == 0)
        continue; /* Don't need two NOFORK_ARG args. */
      new_argv[i++] = p;
    }
  }

  /* Null terminate argv array */
  new_argv[i] == 0;
  args->argv = new_argv;
  args->argc = i;
  return 0;
}

/*
 * Code to install a dll into the %SYSTEMROOT%\SYSTEM32 directory.
 * Sets file security correctly (if NTFS filesystem). Returns 0
 * if successful, non-zero (and deletes the file) if fails.
 */

int install_dll_into_system( char *dll_path, char *systempath, int path_size)
{
  char *p;
  char rootpath[4];
  DWORD dummy;
  DWORD flags;
  SECURITY_DESCRIPTOR sd;

  /* Check that dll_path exists. */
  if(GetFileAttributes(dll_path) == 0xFFFFFFFF) {
    log_message( LOG_ERR, "install_dll_into_system: Cannot access file %s. Error was %s",
                 dll_path, str_oserr(GetLastError()));
    return -1;
  }

  /* Ensure the dll_path ends in .dll */
  p = strrchr(dll_path, '.');
  if( p == 0 || (strcasecmp( p, ".dll")) != 0) {
    log_message( LOG_ERR, "install_dll_into_system: filename %s must end in .dll.",
                 dll_path);
    return -1;
  }
  /* Get the systemroot environment variable. */
  if(GetEnvironmentVariable("SYSTEMROOT", systempath, path_size) == 0) {
    log_message( LOG_ERR, 
           "install_dll_into_system: Unable to get environment variable SYSTEMROOT.");
    return -1;
  }

  if(strlen(systempath) + strlen("\\SYSTEM32\\") + 1 > path_size) {
    log_message(LOG_ERR, "install_dll_into_system: System path too long !");
    return -1;
  }

  strcat(systempath, "\\SYSTEM32\\");
  p = strrchr( dll_path, '\\');
  if(p == 0)
    p = dll_path;
  else
    p++;

  if(strlen(systempath) + strlen(p) + 1 > path_size) {
    log_message(LOG_ERR, "install_dll_into_system: Combined dll and system path too long !");
    return -1;
  }
  strcat(systempath, p);

  /* Check if this filesystem is NTFS */
  if(systempath[1] != ':') {
    log_message(LOG_ERR, 
           "install_dll_into_system: unable to determine drive letter for SYSTEMROOT.");
    DeleteFile(systempath);
    return -1;
  }
  strncpy( rootpath, systempath, 2);
  rootpath[2] = '\\';
  rootpath[3] = '\0';
  if(GetVolumeInformation(rootpath, 0, 0, &dummy, &dummy, &flags, 0, 0 ) == FALSE) {
    log_message(LOG_ERR, 
           "install_dll_into_system: GetVolumeInformation for path %s failed. Error was %s", 
           rootpath, str_oserr(GetLastError()));
    DeleteFile(systempath);
    return -1;
  }

  if((flags & FS_PERSISTENT_ACLS) == 0) {
    printf("install_dll_into_system: WARNING: You are installing security critical components \
into a file system (%s) with no security ! Please reconsider this installation.\n", rootpath);
    return -1;
  }

  /* Get the required Security Descriptor. */
  if(create_sd_from_list( &sd, 2, "Administrators", GENERIC_ALL,
                                  "SYSTEM", GENERIC_ALL) == FALSE) {
    DeleteFile(systempath);
    return -1;
  }

  if(copy_file_with_sd( dll_path, systempath, TRUE, &sd) == 0) {
    return -1;
  }

  return 0;
}

/*
 * Code to install the sync dll into
 * %SYSTEMROOT%\SYSTEM32 and update the
 * HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Lsa\Notification Packages
 * key.
 */

int add_sync_dll( char *dll_path)
{
  char *p;
  DWORD err;
  DWORD size;
  DWORD type;
  char *value;
  HKEY hkey;
  char systempath[MAX_PATH];

  if(install_dll_into_system( dll_path, systempath, MAX_PATH) != 0)
    return -1;

  /* Remove the directory component from dll_path */
  p = strrchr( dll_path, '\\');
  if(p == 0)
    p = dll_path;
  else
    *p++ = 0;
  dll_path = p;
  /* 
   * Truncate the .dll from the dll_path (we know this will 
   * succeed as install_dll_into_system checks that dll_path
   * ends in .dll).
   */
  p = strrchr(dll_path, '.');
  *p = 0;

  /* Get the lsa registry key and get the Notification Packages value. */
  if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, 
               "SYSTEM\\CurrentControlSet\\Control\\Lsa", 
               0, KEY_READ|KEY_WRITE, &hkey)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "add_sync_dll: RegOpenKeyEx on key SYSTEM\\CurrentControlSet\\Control\\Lsa failed. Error was %s",
                str_oserr(err));
    DeleteFile( systempath );
    return -1;
  }
 
  size = 0;
  if((err = RegQueryValueEx( hkey, "Notification Packages", 0, &type,
                             0, &size)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "add_sync_dll: RegQueryValueEx for size on value Notification Packages failed. Error was %s",
                 str_oserr(err));
    DeleteFile( systempath );
    RegCloseKey(hkey);
    return -1;
  }

  size += (strlen( dll_path ) + 1);
  if((value = malloc( size )) == 0) {
    log_message( LOG_ERR, "add_sync_dll: malloc fail.");
    DeleteFile( systempath );
    RegCloseKey(hkey);
    return -1;
  }

  if((err = RegQueryValueEx( hkey, "Notification Packages", 0, &type,
                             value, &size)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "add_sync_dll: RegQueryValueEx for size on value Notification Packages failed. Error was %s",
                 str_oserr(err));
    DeleteFile( systempath );
    free(value);
    RegCloseKey(hkey);
    return -1;
  }

  /* Check if this value already exists */
  for( p = value; *p; p += (strlen(p) + 1)) {
    if(strcasecmp( p, dll_path)==0) {
      log_message( LOG_ERR, "add_sync_dll: value %s already exists in value Notification Packages", dll_path);
      DeleteFile( systempath );
      free(value);
      RegCloseKey(hkey);
      return -1;
    }
  }

  /* Append the new value */
  strcpy(p, dll_path);
  p += (strlen(p) + 1);
  *p = 0;

  if((err = RegSetValueEx( hkey, "Notification Packages", 0, REG_MULTI_SZ,
                           value, p - value)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "add_sync_dll: RegSetValueEx for value Notification Packages failed. Error was %s",
      str_oserr(err));
    DeleteFile( systempath );
    free(value);
    RegCloseKey(hkey);
    return -1;
  }

  free(value);
  RegCloseKey(hkey);

  printf("Password Synchonization DLL successfully installed. You must reboot\
 your machine for this to take effect.\n");

  return 0;
}

/*
 * Code to remove sync dll and update the
 * HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Lsa\Notification Packages
 * key.
 */

int remove_sync_dll( char *dll_path)
{
  char *p;
  char *p1;
  DWORD err;
  DWORD size;
  DWORD type;
  char *value;
  char *newvalue;
  HKEY hkey;
  BOOL found;

  /* Remove the directory component from dll_path */
  p = strrchr( dll_path, '\\');
  if(p == 0)
    p = dll_path;
  else
    *p++ = 0;
  dll_path = p;
  /* 
   * Truncate the .dll from the dll_path (we know this will 
   * succeed as install_dll_into_system checks that dll_path
   * ends in .dll).
   */
  p = strrchr(dll_path, '.');
  if((p != 0) && (strcasecmp( p, ".dll") == 0))
    *p = 0;

  /* Get the lsa registry key and get the Notification Packages value. */
  if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, 
               "SYSTEM\\CurrentControlSet\\Control\\Lsa", 
               0, KEY_READ|KEY_WRITE, &hkey)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "remove_sync_dll: RegOpenKeyEx on key SYSTEM\\CurrentControlSet\\Control\\Lsa failed. Error was %s",
                str_oserr(err));
    return -1;
  }
 
  size = 0;
  if((err = RegQueryValueEx( hkey, "Notification Packages", 0, &type,
                             0, &size)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "remove_sync_dll: RegQueryValueEx for size on value Notification Packages failed. Error was %s",
                 str_oserr(err));
    RegCloseKey(hkey);
    return -1;
  }

  if((value = malloc( size )) == 0) {
    log_message( LOG_ERR, "remove_sync_dll: malloc fail.");
    RegCloseKey(hkey);
    return -1;
  }

  if((err = RegQueryValueEx( hkey, "Notification Packages", 0, &type,
                             value, &size)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "remove_sync_dll: RegQueryValueEx for size on value Notification Packages failed. Error was %s",
                 str_oserr(err));
    free(value);
    RegCloseKey(hkey);
    return -1;
  }

  
  if((newvalue = malloc( size )) == 0) {
    log_message( LOG_ERR, "remove_sync_dll: malloc fail.");
    RegCloseKey(hkey);
    return -1;
  }

  /* Check to see that this value already exists - removing it in the process. */
  found = FALSE;
  for( p = value, p1 = newvalue; *p; p += (strlen(p) + 1), p1 += (strlen(p1)+1)) {
    if(strcasecmp( p, dll_path)==0) {
      found = TRUE;
      continue;
    }
    strcpy(p1, p);
  }

  *p1 = 0;

  if(found == FALSE) {
    log_message( LOG_ERR, "add_sync_dll: DLL %s doesn't exist in the value Notification Packages", dll_path);
    free(value);
    free(newvalue);
    RegCloseKey(hkey);
    return -1;
  }

  if((err = RegSetValueEx( hkey, "Notification Packages", 0, REG_MULTI_SZ,
                           newvalue, p1 - newvalue)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "remove_sync_dll: RegSetValueEx for value Notification Packages failed. Error was %s",
      str_oserr(err));
    free(value);
    free(newvalue);
    RegCloseKey(hkey);
    return -1;
  }

  free(value);
  free(newvalue);
  RegCloseKey(hkey);

  printf("Password Synchonisation DLL successfully removed. You must reboot\
your machine for this to take effect.\n");

  return 0;
}

/*
 * Code to install a generic sync dll into the Registry key
 * HKEY_LOCAL_MACHINE\SOFTWARE\Cygnus Support\PasswordSync\Dlls
 * as a REG_MULTI_SZ value whose value is the given args.
 */

int add_lib_dll( int argc, char **argv)
{
  char *p;
  char systempath[MAX_PATH];
  char *dll_path = argv[0];
  HKEY hkey;
  char *value;
  int i;
  int size;
  DWORD err;

  if(install_dll_into_system( dll_path, systempath, MAX_PATH) != 0)
    return -1;

  /* Remove the directory component from dll_path */
  p = strrchr( dll_path, '\\');
  if(p == 0)
    p = dll_path;
  else
    *p++ = 0;
  dll_path = p;
  /* 
   * Truncate the .dll from the dll_path (we know this will 
   * succeed as install_dll_into_system checks that dll_path
   * ends in .dll).
   */
  p = strrchr(dll_path, '.');
  *p = 0;

  /* Get a handle to HKEY_LOCAL_MACHINE\SOFTWARE\Cygnus Solutions\PasswordSync\Dlls */
  if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, PWS_DLL_REG_NAME,
                          0, KEY_READ|KEY_WRITE, &hkey)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "add_lib_dll: RegOpenKeyEx on key %s Failed. Error was %s",
                PWS_DLL_REG_NAME, str_oserr(err));
    DeleteFile( systempath );
    return -1;
  }

  /* Now create the value from the rest of the arguments. */
  argc--; /* We used the first arg as the dll path */
  argv = &argv[1];
  size = 0;
  for( i = 0; i < argc; i++)
    size += (strlen(argv[i])+1);
  size++; /* Space for last '\0' */
  if(( value = malloc( size )) == 0) {
    log_message( LOG_ERR, "add_lib_dll: malloc failed.");
    DeleteFile( systempath );
    RegCloseKey(hkey);
    return -1;
  }
  p = value;
  for( i = 0; i < argc; i++) {
    strcpy( p, argv[i]);
    p += (strlen(argv[i])+1);
  }
  *p++ = 0;
  /* Now set this as the value */
  if((err = RegSetValueEx( hkey, dll_path, 0, REG_MULTI_SZ, value, p - value)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "add_lib_dll: RegSetValueEx on value %s Failed. Error was %s",
                dll_path, str_oserr(err));
    free(value);
    DeleteFile( systempath );
    RegCloseKey(hkey);
    return -1;
  }
  free(value);
  RegCloseKey(hkey);

  printf("Password Synchonisation plugin dll %s successfully installed.\n", dll_path);

  return 0;
}

/*
 * Code to remove a generic sync dll from the Registry key
 * HKEY_LOCAL_MACHINE\SOFTWARE\Cygnus Support\PasswordSync\Dlls
 */

int remove_lib_dll( char *dll_name)
{
  char *p;
  DWORD err;
  HKEY hkey;

  /* Strip any leading path. */
  p = strrchr(dll_name, '\\');
  if(p != 0) {
    *p++ = 0;
  } else
    p = dll_name;
  dll_name = p;

  /* Strip any trailing .dll */
  p = strrchr(dll_name, '.');
  if((p != 0) && (strcasecmp( p, ".dll") == 0))
    *p = 0;

  /* Get a handle to HKEY_LOCAL_MACHINE\SOFTWARE\Cygnus Solutions\PasswordSync\Dlls */
  if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, PWS_DLL_REG_NAME,
                          0, KEY_READ|KEY_WRITE, &hkey)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "remove_lib_dll: RegOpenKeyEx on key %s Failed. Error was %s",
                PWS_DLL_REG_NAME, str_oserr(err));
    return -1;
  }

  if((err = RegDeleteValue( hkey, dll_name)) != ERROR_SUCCESS) {
    log_message( LOG_ERR, "remove_lib_dll: RegDeleteValue on value %s Failed. Error was %s",
                dll_name, str_oserr(err));
    RegCloseKey(hkey);
    return -1;
  }

  printf("Password Synchonisation plugin dll %s successfully removed.\n", dll_name);
  RegCloseKey(hkey);
  return 0;
}

/*
 * Service install code called by boilerplate service code.
 */
 
int do_service_install(int *argc_ret, char **argv, int *cont)
{
  int err;
  int i, saved_i;
  int argc = *argc_ret;
  SECURITY_DESCRIPTOR sd;
  char username[512];
  DWORD usize = sizeof(username);

  *cont = TRUE;

  /*
   * Check we are running as Administrator - exit otherwise.
   */

  if(GetUserName(username, &usize) == FALSE) {
    *cont = FALSE;
    log_message(LOG_ERR, "do_service_install: GetUserName failed. Error was %s.",
                str_oserr(GetLastError()));
    return -1;
  }

  if(strcasecmp(username, "Administrator") != 0) {
    log_message(LOG_ERR, "You must be running as user Administrator to use this program. You are currently user %s.", username);
    *cont = FALSE;
    free(username);
    return -1;
  }

  free(username);

  /* Get the required Security Descriptor. */
  if(create_sd_from_list( &sd, 2, "Administrators", GENERIC_ALL,
                                  "SYSTEM", GENERIC_ALL) == FALSE) {
    *cont = FALSE;
    return -1;
  }

  /* Ensure that the registry heirarchy exists */
  if(err = create_registry_tree( HKEY_LOCAL_MACHINE, PWS_DLL_REG_NAME, &sd)) {
    *cont = FALSE;
    return err;
  }

  /*
   * If we are invoked as /install /args then we are being invoked
   * to change the args in the registry path
   * SOFTWARE\Cygnus Solutions\PasswordSync.
   */
  if((argc >= 3) && (*argv[2] == '/' || *argv[2] == '-') && (strcasecmp( &argv[2][1], "args") == 0)) {
    err = setup_service_args( HKEY_LOCAL_MACHINE, PWS_REG_NAME,
                              PWS_ARGS_REG_NAME, argc - 3, &argv[3]);
    *cont = FALSE;
    return err;
  }

  /*
   * If we are invoked as /install /lib then we are being invoked
   * to add an entry in the registry path
   * SOFTWARE\Cygnus Solutions\PasswordSync\Dlls.
   */
  if((argc >= 3) && (*argv[2] == '/' || *argv[2] == '-') && (strcasecmp( &argv[2][1], "lib") == 0)) {
    *cont = FALSE;
    if(argc == 3) {
      log_message(LOG_ERR, "Usage: /install /lib <path to .dll> <dll args>");
      return -1;
    }
    err = add_lib_dll( argc - 3, &argv[3]);
    return err;
  }

  /*
   * If we are invoked as /install /syncdll then we are being invoked
   * to add an entry in the registry path
   * SOFTWARE\Cygnus Solutions\PasswordSync\Dlls.
   */
  if((argc >= 3) && (*argv[2] == '/' || *argv[2] == '-') && (strcasecmp( &argv[2][1], "syncdll") == 0)) {
    *cont = FALSE;
    if(argc != 4) {
      log_message(LOG_ERR, "Usage: /install /syncdll <path to sync.dll>");
      return -1;
    }
    err = add_sync_dll( argv[3]);
    return err;
  }

  /*
   * We are really being installed. Add the args
   * given on the command line.
   */
  for( i = 2; i < argc; i++)
    if(((argv[i][0] != '/') && (argv[i][0] != '-')) ||
       ((strncasecmp( &argv[i][1], "username", 9) != 0) &&
        (strncasecmp( &argv[i][1], "password", 9) != 0)))
      break;

  if(i == argc) /* No args */
    return 0;

  saved_i = i;

  if((argv[i] != 0) && (*argv[i] == '/' || *argv[i] == '-') && (strcasecmp( &argv[i][1], "args") == 0))
    i++;

  err = setup_service_args( HKEY_LOCAL_MACHINE, PWS_REG_NAME,
                            PWS_ARGS_REG_NAME, argc - i, &argv[i]);
  /* Truncate the args here */
  argv[saved_i] = 0;
  *argc_ret = saved_i;

  return err;
}

/*
 * Service delete code called by boilerplate service code.
 */

int do_service_delete(int *argc, char **argv, int *cont)
{
  DWORD err;
  char username[512];
  SECURITY_DESCRIPTOR sd;
  DWORD usize = sizeof(username);

  *cont = TRUE;

  /*
   * Check we are running as Administrator - exit otherwise.
   */

  if(GetUserName(username, &usize) == FALSE) {
    *cont = FALSE;
    log_message(LOG_ERR, "do_service_install: GetUserName failed. Error was %s.",
                str_oserr(GetLastError()));
    return -1;
  }

  if(strcasecmp(username, "Administrator") != 0) {
    log_message(LOG_ERR, "You must be running as user Administrator to use this program. You are currently user %s.", username);
    *cont = FALSE;
    free(username);
    return -1;
  }

  free(username);

  /* Get the required Security Descriptor. */
  if(create_sd_from_list( &sd, 2, "Administrators", GENERIC_ALL,
                                  "SYSTEM", GENERIC_ALL) == FALSE) {
    *cont = FALSE;
    return -1;
  }

  /* Ensure that the registry heirarchy exists */
  if(err = create_registry_tree( HKEY_LOCAL_MACHINE, PWS_DLL_REG_NAME, &sd)) {
    *cont = FALSE;
    return err;
  }

  /*
   * If we are invoked as /delete /lib then we are being invoked
   * to remove an entry in the registry path
   * SOFTWARE\Cygnus Solutions\PasswordSync\Dlls.
   */
  if((*argc >= 3) && (*argv[2] == '/' || *argv[2] == '-') && (strcasecmp( &argv[2][1], "lib") == 0)) {
    *cont = FALSE;
    if(*argc != 4) {
      log_message(LOG_ERR, "Usage: /delete /lib <dllname>");
      return -1;
    }
    err = remove_lib_dll( argv[3]);
    return err;
  }

  /*
   * If we are invoked as /delete /syncdll then we are being invoked
   * to remove an entry in the registry path
   * SOFTWARE\Cygnus Solutions\PasswordSync\Dlls.
   */
  if((*argc >= 3) && (*argv[2] == '/' || *argv[2] == '-') && (strcasecmp( &argv[2][1], "syncdll") == 0)) {
    *cont = FALSE;
    if(*argc != 4) {
      log_message(LOG_ERR, "Usage: /delete /syncdll <sync.dll>");
      return -1;
    }
    err = remove_sync_dll( argv[3]);
    return err;
  }

  if((*argc == 3) && (*argv[2] == '/' || *argv[2] == '-') && (strcasecmp( &argv[2][1], "registry") == 0)) {
    *cont = FALSE;
    if(err = delete_registry_tree( HKEY_LOCAL_MACHINE, PWS_DLL_REG_NAME))
      return err;
  }

  return 0;
}

/*
 * Exception handler.
 */

void except_handler(EXCEPTION_RECORD *a, void *b, CONTEXT *c, void *d)
{
  log_message(LOG_ERR, "Exception trapped. Service terminating. Code = %x, at ip = %x",
	    a->ExceptionCode, c->Eip);
  exit(0);
}

/*
 * This is x86 specific. We need to make this generic for
 * ALPHA also (as currently that's the only other processor
 * NT runs on, with PowerPC being cancelled).
 */
 
extern exception_list *_except_list asm ("%fs:0");

/*
 * This is called in a separate thread.
 */

int do_service_main(struct argc_argv *args)
{
  int ret;
  
  /*
   * Catch any exceptions.
   */
  exception_list el;
  el.handler = except_handler;
  el.prev = _except_list;
  _except_list = &el;

  /* Open the syslog. */
  openlog( global_service_name, LOG_PID, LOG_DAEMON);
  
  /*
   * Create the can_stop mutex.
   */
  if((can_stop = CreateMutex( 0, FALSE, 0)) == 0) {
    log_message(LOG_ERR, "Failed to create stop mutex. Error was %s",
                str_oserr(GetLastError()));
    return -1;
  }

  if(global_debug_flag == TRUE) {
    /*
     * Running in foreground - just call
     * original_main() after removing -debug arg.
     */
    int i;
    for( i = 0; i < args->argc; ++i) {
      if(strcasecmp(&args->argv[i][1], "debug")==0) {
        --(args->argc);
      	for(; i < args->argc; ++i) {
      	  args->argv[i] = args->argv[i+1];
      	}
        args->argv[args->argc] = 0;
      	break;
      }
    }
    ret = original_main(args->argc, args->argv);
    closelog();
    return ret;
  }

#ifdef KERBNET
  /*
   * Running as a service.
   * Set up the neccessary environment variables from
   * the registry. The need to be KRB5_CONFIG and KRB5_KDC_PROFILE.
   */
  if(setup_environment_variables() != 0)
    return -1;
#endif /* KERBNET */

  /*
   * Add in the neccessary
   * args then call the original_main().
   */
  if(add_extra_args(args) != 0)
    return -1;

  /* Tell the service manager we have started. */
  TellServiceManager(SERVICE_RUNNING, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
  
  /* Inform the event log we have been started. */
  syslog( LOG_INFO, "Service %s has been started", global_service_name);

  /*
   * Update the watch value to tell the pwsync dll
   * that we have started.
   */
  if(update_watch_value() != 0)
    return -1;

  ret = original_main(args->argc, args->argv);
  closelog();

  update_watch_value();

  return ret;
}

/*
 * Call to ask service to stop. Currently, as cygwin32 does not
 * interrupt a select when a signal fires, we have to use the
 * can_stop mutex and TerminateThread(). When this is fixed we
 * will just use kill().
 */
 
int request_service_stop(SERVICE_STATUS *ssh)
{
  openlog( global_service_name, LOG_PID, LOG_DAEMON);
  lock_exit(5000);
  request_exit(SIGINT);
  TerminateThread(global_service_thread, 0);
  unlock_exit();
  /* Log that we exited. */
  syslog( LOG_INFO, "Service %s has been stopped", global_service_name);
  update_watch_value();
  closelog();
  return 0;
}

/*
 * Request the service pause. Get the mutex
 * and then suspend the thread.
 */
 
int request_service_pause(SERVICE_STATUS *ssh)
{
  openlog( global_service_name, LOG_PID, LOG_DAEMON);
  if(lock_exit(4500) != WAIT_OBJECT_0) {
    /* Failed to get mutex */
    TellServiceManager(SERVICE_RUNNING, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
    closelog();
    return 0;
  }
  SuspendThread(global_service_thread);
  TellServiceManager(SERVICE_PAUSED, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
  closelog();
  return 0;
}

/*
 * Request the service continue. Release the thread
 * then release the mutex.
 */

int request_service_continue(SERVICE_STATUS *ssh)
{
  openlog( global_service_name, LOG_PID, LOG_DAEMON);
  unlock_exit();
  ResumeThread(global_service_thread);
  TellServiceManager(SERVICE_RUNNING, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
  closelog();
  return 0;
}

int do_service_usage(int argc, char **argv)
{
  printf("usage: pwsync.exe\n");
  printf("\t/install          -  installs service & creates registry entries.\n");
  printf("\t/delete           -  deletes service & removes registry entries.\n");
  printf("\t/install /syncdll sync.dll   - installs password synchonization dll\n");
  printf("\t\tinto %SYSTEMROOT%\\SYSTEM32\n");
  printf("\t/delete /syncdll sync.dll    - removes password synchonization\n");
  printf("\t\tdll from registry.\n");
  printf("\t/install /lib synclib.dll [arg1] [arg2] ... [argn] - installs plugin\n");
  printf("\t\tpassword synchonization dll into %SYSTEMROOT%\\SYSTEM32\n");
  printf("\t\tand installs the args into the registry. Args may be optional.\n");
  printf("\t/delete /lib synclib.dll  - removes plugin password synchonization\n");
  printf("\t\tdll and it's args from the registry.\n");
  return 0;
}

#endif /* _WIN32 && __CYGWIN32__ */
