/*
 * Copyright, OpenVision Technologies, Inc., 1996, All Rights Reserved
 *
 * WARNING:  Retrieving the OpenVision Kerberos Administration system
 * source code, as described below, indicates your acceptance of the
 * following terms.  If you do not agree to the following terms, do not
 * retrieve the OpenVision Kerberos administration system.
 *
 * You may freely use and distribute the Source Code and Object Code
 * compiled from it, but this Source Code is provided to you "AS IS"
 * EXCLUSIVE OF ANY WARRANTY, INCLUDING, WITHOUT LIMITATION, ANY
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR
 * ANY OTHER WARRANTY, WHETHER EXPRESS OR IMPLIED.  IN NO EVENT WILL
 * OPENVISION HAVE ANY LIABILITY FOR ANY LOST PROFITS, LOSS OF DATA OR
 * COSTS OF PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, OR FOR ANY
 * SPECIAL, INDIRECT, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THIS
 * AGREEMENT, INCLUDING, WITHOUT LIMITATION, THOSE RESULTING FROM THE
 * USE OF THE SOURCE CODE, OR THE FAILURE OF THE SOURCE CODE TO PERFORM,
 * OR FOR ANY OTHER REASON.
 *
 * OpenVision retains all rights, title, and interest in the donated
 * Source Code.  With respect to OpenVision's copyrights in the donated
 * Source Code, OpenVision also retains rights to derivative works of
 * the Source Code whether created by OpenVision or a third party.
 *
 * OpenVision Technologies, Inc. has donated this Kerberos
 * Administration system to MIT for inclusion in the standard Kerberos 5
 * distribution. This donation underscores our commitment to continuing
 * Kerberos technology development and our gratitude for the valuable
 * work which has been performed by MIT and the Kerberos community.
 */

#include "k5-int.h"		/* we need krb5_context::clockskew */
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <krb.h>
#include <krb4-proto.h>
#include "krb524.h"

/* rather than copying the cmu code, these values are derived from
   a calculation based on the table and comments found there.
   the expression (in elisp) is:
   (defun cmu-to-secs2 (j)
      (if (< j 128) (* j 5 60)
         (round (* 38400 (expt 1.06914489 (- j 128))))))
   and is low by one for 16 values but is exact for the others.
 */

static long cmu_seconds[] = 
{
  38400,  41055,  43894,  46929,  50174,  53643,  57352,  61318,
  65558,  70091,  74937,  80119,  85658,  91581,  97914,  104684,
  111922,  119661,  127935,  136781,  146239,  156350,  167161,  178720,
  191077,  204289,  218415,  233517,  249663,  266926,  285383,  305116,
  326213,  348769,  372885,  398668,  426233,  455705,  487215,  520903,
  556921,  595430,  636600,  680618,  727679,  777995,  831789,  889303,
  950794,  1016536,  1086825,  1161973,  1242317,  1328217,  1420057,  1518246,
  1623225,  1735463,  1855462,  1983757,  2120924,  2267575,  2424366, 2591999,
  0
};

/* Convert v5 time to v4 cmu ticks */
int calculate_cmu_seconds(int deltatime)
{
    int lifetime = deltatime / 300;
    /* if (lifetime > 255) lifetime = 255; */
    if (lifetime > 127) {
        /* use the CMU algorithm instead: */
        long *clist = cmu_seconds;
        while(*clist && *clist < deltatime) clist++;
        lifetime = 128 + (clist - cmu_seconds);
    }
    return(lifetime);
}

/*
 * Convert a v5 ticket for server to a v4 ticket, using service key
 * skey for both.
 */
int krb524_convert_tkt_skey(context, v5tkt, v4tkt, v5_skey, v4_skey)
     krb5_context context;
     krb5_ticket *v5tkt;
     KTEXT_ST *v4tkt;
     krb5_keyblock *v5_skey, *v4_skey;
{
     char pname[ANAME_SZ], pinst[INST_SZ], prealm[REALM_SZ];
     char sname[ANAME_SZ], sinst[INST_SZ];
     krb5_enc_tkt_part *v5etkt;
     int ret, lifetime, deltatime;
     krb5_timestamp server_time;

     v5tkt->enc_part2 = NULL;
     if ((ret = krb5_decrypt_tkt_part(context, v5_skey, v5tkt))) {
	  krb5_free_ticket(context, v5tkt);
	  return ret;
     }
     v5etkt = v5tkt->enc_part2;

     if ((ret = krb524_convert_princs(context, v5etkt->client, v5tkt->server,
				     pname, pinst, prealm, sname,
				     sinst))) {
	  krb5_free_enc_tkt_part(context, v5etkt);
	  v5tkt->enc_part2 = NULL;
	  return ret;
     }
     
     if (v5etkt->session->enctype != ENCTYPE_DES_CBC_CRC ||
	 v5etkt->session->length != sizeof(C_Block)) {
	  if (krb524_debug)
	       fprintf(stderr, "v5 session keyblock type %d length %d != C_Block size %d\n",
		       v5etkt->session->enctype,
		       v5etkt->session->length,
		       sizeof(C_Block));
	  krb5_free_enc_tkt_part(context, v5etkt);
	  v5tkt->enc_part2 = NULL;
	  return KRB524_BADKEY;
     }
     
     /* V4 has no concept of authtime or renew_till, so ignore them */
     /* V4 lifetime is 1 byte, in 5 minute increments */
     if (v5etkt->times.starttime == 0)
	  v5etkt->times.starttime = v5etkt->times.authtime;
     /* rather than apply fit an extended v5 lifetime into a v4 range,
	give out a v4 ticket with as much of the v5 lifetime is available
	"now" instead. */
     if ((ret = krb5_timeofday(context, &server_time))) {
         if (krb524_debug)
	      fprintf(stderr, "krb5_timeofday failed!\n");
	 krb5_free_enc_tkt_part(context, v5etkt);
	 v5tkt->enc_part2 = NULL;
	 return ret;       
     }
     if (   (server_time+context->clockskew >= v5etkt->times.starttime)
	 && (server_time-context->clockskew <= v5etkt->times.endtime)) {
	  deltatime = v5etkt->times.endtime - v5etkt->times.starttime;
	  lifetime = deltatime / 300;
	  /* if (lifetime > 255) lifetime = 255; */
	  if (lifetime > 127) {
	      /* use the CMU algorithm instead: */
	      long *clist = cmu_seconds;
	      while(*clist && *clist < deltatime) clist++;
	      lifetime = 128 + (clist - cmu_seconds);
	  }
     } else {
          if (krb524_debug)
	       fprintf(stderr, "v5 ticket time out of bounds\n");
	  krb5_free_enc_tkt_part(context, v5etkt);
	  v5tkt->enc_part2 = NULL;
	  if (server_time+context->clockskew < v5etkt->times.starttime)
	       return KRB5KRB_AP_ERR_TKT_NYV;
	  else if (server_time-context->clockskew > v5etkt->times.endtime)
	       return KRB5KRB_AP_ERR_TKT_EXPIRED;
	  else /* shouldn't happen, but just in case... */
	    return KRB5KRB_AP_ERR_TKT_NYV;
     }

     /* XXX perhaps we should use the addr of the client host if */
     /* v5creds contains more than one addr.  Q: Does V4 support */
     /* non-INET addresses? */
     if (!v5etkt->caddrs || !v5etkt->caddrs[0] ||
	 v5etkt->caddrs[0]->addrtype != ADDRTYPE_INET) {
	  if (krb524_debug)
	       fprintf(stderr, "Invalid v5creds address information.\n");
	  krb5_free_enc_tkt_part(context, v5etkt);
	  v5tkt->enc_part2 = NULL;
	  return KRB524_BADADDR;
     }
     
     if (krb524_debug)
	printf("startime = %ld, authtime = %ld, lifetime = %ld\n",
	       (long) v5etkt->times.starttime,
	       (long) v5etkt->times.authtime,
	       (long) lifetime);

     /* XXX are there V5 flags we should map to V4 equivalents? */
     ret = krb_create_ticket(v4tkt,
			     0, /* flags */			     
			     pname,
			     pinst,
			     prealm,
			     *((unsigned long *)v5etkt->caddrs[0]->contents),
			     (char *) v5etkt->session->contents,
			     lifetime,
			     /* issue_data */
			     /* v5etkt->times.starttime, */
			     server_time,
			     sname,
			     sinst,
			     v4_skey->contents);

     krb5_free_enc_tkt_part(context, v5etkt);
     v5tkt->enc_part2 = NULL;
     if (ret == KSUCCESS)
	  return 0;
     else
	  return KRB524_V4ERR;
}
