/*
 * Copyright, OpenVision Technologies, Inc., 1996, All Rights Reserved
 *
 * WARNING:  Retrieving the OpenVision Kerberos Administration system
 * source code, as described below, indicates your acceptance of the
 * following terms.  If you do not agree to the following terms, do not
 * retrieve the OpenVision Kerberos administration system.
 *
 * You may freely use and distribute the Source Code and Object Code
 * compiled from it, but this Source Code is provided to you "AS IS"
 * EXCLUSIVE OF ANY WARRANTY, INCLUDING, WITHOUT LIMITATION, ANY
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR
 * ANY OTHER WARRANTY, WHETHER EXPRESS OR IMPLIED.  IN NO EVENT WILL
 * OPENVISION HAVE ANY LIABILITY FOR ANY LOST PROFITS, LOSS OF DATA OR
 * COSTS OF PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, OR FOR ANY
 * SPECIAL, INDIRECT, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THIS
 * AGREEMENT, INCLUDING, WITHOUT LIMITATION, THOSE RESULTING FROM THE
 * USE OF THE SOURCE CODE, OR THE FAILURE OF THE SOURCE CODE TO PERFORM,
 * OR FOR ANY OTHER REASON.
 *
 * OpenVision retains all rights, title, and interest in the donated
 * Source Code.  With respect to OpenVision's copyrights in the donated
 * Source Code, OpenVision also retains rights to derivative works of
 * the Source Code whether created by OpenVision or a third party.
 *
 * OpenVision Technologies, Inc. has donated this Kerberos
 * Administration system to MIT for inclusion in the standard Kerberos 5
 * distribution. This donation underscores our commitment to continuing
 * Kerberos technology development and our gratitude for the valuable
 * work which has been performed by MIT and the Kerberos community.
 */

#include "gssapiP_generic.h"
#include <string.h>

/*
 * $Id: disp_major_status.c,v 1.6 1996/07/22 20:33:01 marc Exp $
 */

/* This code has knowledge of the min and max errors of each type
   within the gssapi major status */

#define GSS_ERROR_STR(value, array, select, min, max, num) \
   (((select(value) < (min)) || (select(value) > (max))) ? NULL : \
    (array)[num(value)])

/**/

static const char * const calling_error_string[] = {
   NULL,
   "A required input parameter could not be read",
   "A required input parameter could not be written",
   "A parameter was malformed",
};
 
static const char * const calling_error = "calling error";

#define GSS_CALLING_ERROR_STR(x) \
   GSS_ERROR_STR((x), calling_error_string, GSS_CALLING_ERROR, \
		 GSS_S_CALL_INACCESSIBLE_READ, GSS_S_CALL_BAD_STRUCTURE, \
		 GSS_CALLING_ERROR_FIELD)

/**/

static const char * const routine_error_string[] = {
   NULL,
   "An unsupported mechanism was requested",
   "An invalid name was supplied",
   "A supplied name was of an unsupported type",
   "Incorrect channel bindings were supplied",
   "An invalid status code was supplied",
   "A token had an invalid signature",
   "No credentials were supplied",
   "No context has been established",
   "A token was invalid",
   "A credential was invalid",
   "The referenced credentials have expired",
   "The context has expired",
   "Miscellaneous failure",
   "The quality-of-protection requested could not be provided",
   "The operation is forbidden by the local security policy",
   "The operation or option is not available",
};   

static const char * const routine_error = "routine error";

#define GSS_ROUTINE_ERROR_STR(x) \
   GSS_ERROR_STR((x), routine_error_string, GSS_ROUTINE_ERROR, \
		 GSS_S_BAD_MECH, GSS_S_FAILURE, \
		 GSS_ROUTINE_ERROR_FIELD)

/**/

/* this becomes overly gross after about 4 strings */

static const char * const sinfo_string[] = {
   "The routine must be called again to complete its function",
   "The token was a duplicate of an earlier token",
   "The token's validity period has expired",
   "A later token has already been processed",
};

static const char * const sinfo_code = "supplementary info code";

#define LSBGET(x) ((((x)^((x)-1))+1)>>1)
#define LSBMASK(n) ((1<<(n))^((1<<(n))-1))

#define GSS_SINFO_STR(x) \
   ((((1<<(x)) < GSS_S_CONTINUE_NEEDED) || ((1<<(x)) > GSS_S_UNSEQ_TOKEN)) ? \
    /**/NULL:sinfo_string[(x)])

/**/

static const char * const no_error = "No error";
static const char * const unknown_error = "Unknown %s (field = %d)";

/**/

int display_unknown(kind, value, buffer)
     const char *kind;
     OM_uint32 value;
     gss_buffer_t buffer;
{
   int len;
   char *str;

   if ((str =
	(char *) xmalloc(len = strlen(unknown_error)+strlen(kind)+7)) == NULL)
      return(0);

   sprintf(str, unknown_error, kind, value);

   buffer->length = len;
   buffer->value = str;

   return(1);
}

/* code should be set to the calling error field */

static OM_uint32 display_calling(minor_status, code, status_string)
     OM_uint32 *minor_status;
     OM_uint32 code;
     gss_buffer_t status_string;
{
   const char *str;

   if (str = GSS_CALLING_ERROR_STR(code)) {
      if (! g_make_string_buffer(str, status_string)) {
	 *minor_status = ENOMEM;
	 return(GSS_S_FAILURE);
      }
   } else {
      if (! display_unknown(calling_error, GSS_CALLING_ERROR_FIELD(code),
			    status_string)) {
	 *minor_status = ENOMEM;
	 return(GSS_S_FAILURE);
      }
   }
   *minor_status = 0;
   return(GSS_S_COMPLETE);
}

/* code should be set to the routine error field */

static OM_uint32 display_routine(minor_status, code, status_string)
     OM_uint32 *minor_status;
     OM_uint32 code;
     gss_buffer_t status_string;
{
   const char *str;

   if (str = GSS_ROUTINE_ERROR_STR(code)) {
      if (! g_make_string_buffer(str, status_string)) {
	 *minor_status = ENOMEM;
	 return(GSS_S_FAILURE);
      }
   } else {
      if (! display_unknown(routine_error, GSS_ROUTINE_ERROR_FIELD(code),
			    status_string)) {
	 *minor_status = ENOMEM;
	 return(GSS_S_FAILURE);
      }
   }
   *minor_status = 0;
   return(GSS_S_COMPLETE);
}

/* code should be set to the bit offset (log_2) of a supplementary info bit */

static OM_uint32 display_bit(minor_status, code, status_string)
     OM_uint32 *minor_status;
     OM_uint32 code;
     gss_buffer_t status_string;
{
   const char *str;

   if (str = GSS_SINFO_STR(code)) {
      if (! g_make_string_buffer(str, status_string)) {
	 *minor_status = ENOMEM;
	 return(GSS_S_FAILURE);
      }
   } else {
      if (! display_unknown(sinfo_code, 1<<code, status_string)) {
	 *minor_status = ENOMEM;
	 return(GSS_S_FAILURE);
      }
   }
   *minor_status = 0;
   return(GSS_S_COMPLETE);
}

/**/

/* return error messages, for routine errors, call error, and status,
   in that order.
     message_context == 0 : print the routine error
     message_context == 1 : print the calling error
     message_context > 2  : print supplementary info bit (message_context-2)
     */

OM_uint32 g_display_major_status(minor_status, status_value, 
				 message_context, status_string)
     OM_uint32 *minor_status;
     OM_uint32 status_value;
     OM_uint32 *message_context;
     gss_buffer_t status_string;
{
   OM_uint32 ret, tmp;
   int bit;

   /*** deal with no error at all specially */

   if (status_value == 0) {
      if (! g_make_string_buffer(no_error, status_string)) {
	 *minor_status = ENOMEM;
	 return(GSS_S_FAILURE);
      }
      *message_context = 0;
      *minor_status = 0;
      return(GSS_S_COMPLETE);
   }

   /*** do routine error */

   if (*message_context == 0) {
      if (tmp = GSS_ROUTINE_ERROR(status_value)) {
	 status_value -= tmp;
	 if (ret = display_routine(minor_status, tmp, status_string))
	    return(ret);
	 *minor_status = 0;
	 if (status_value) {
	    (*message_context)++;
	    return(GSS_S_CONTINUE_NEEDED);
	 } else {
	    *message_context = 0;
	    return(GSS_S_COMPLETE);
	 }
      } else {
	 (*message_context)++;
      }
   } else {
      status_value -= GSS_ROUTINE_ERROR(status_value);
   }

   /*** do calling error */

   if (*message_context == 1) {
      if (tmp = GSS_CALLING_ERROR(status_value)) {
	 status_value -= tmp;
	 if (ret = display_calling(minor_status, tmp, status_string))
	    return(ret);
	 *minor_status = 0;
	 if (status_value) {
	    (*message_context)++;
	    return(GSS_S_CONTINUE_NEEDED);
	 } else {
	    *message_context = 0;
	    return(GSS_S_COMPLETE);
	 }
      } else {
	 (*message_context)++;
      }
   } else {
      status_value -= GSS_CALLING_ERROR(status_value);
   }

   /*** do sinfo bits (*message_context == 2 + number of bits done) */

   tmp = GSS_SUPPLEMENTARY_INFO_FIELD(status_value);
   /* mask off the bits which have been done */
   if (*message_context > 2) {
      tmp &= ~LSBMASK(*message_context-3);
      status_value &= ~LSBMASK(*message_context-3);
   }

   if (!tmp) {
      /* bogon input - there should be something left */
      *minor_status = (OM_uint32) G_BAD_MSG_CTX;
      return(GSS_S_FAILURE);
   }

   /* compute the bit offset */
   /*SUPPRESS 570*/
   for (bit=0; (((OM_uint32) 1)<<bit) != LSBGET(tmp); bit++) ;

   /* print it */
   if (ret = display_bit(minor_status, bit, status_string))
      return(ret);

   /* compute the new status_value/message_context */
   status_value -= ((OM_uint32) 1)<<bit;

   if (status_value) {
      *message_context = bit+3;
      return(GSS_S_CONTINUE_NEEDED);
   } else {
      *message_context = 0;
      return(GSS_S_COMPLETE);
   }
}
