/*
 * Code to install the kerbgina.dll into the 
 * SYSTEMROOT\SYSTEM32 directory with the correct
 * permissions on an NT system. Then updates the
 * registry and creates :
 * HKEY_LOCAL_MACHINE\Software\Microsoft\Windows NT\
 * CurrentVersion\Winlogon -> GinaDLL value.
 * and also the
 * HKEY_LOCAL_MACHINE\Software\Cygnus Support\Kerbnet\1\Gina
 * key.
 */

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define GINA_REG_NAME "SOFTWARE\\Cygnus Solutions\\Kerbnet\\1\\Gina"
#define SYSTEM_GINA_REG_NAME "SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Winlogon"
#define SYSTEM_GINA_REG_VALUE "GinaDLL"
#define DLL_NAME "kerbgina.dll"

/* Registry value names */
#define REQUIRE_KERBEROS_LOGON_REG_VALUE "RequireKerberosLogon"
#define EXCLUDED_USERS_REG_VALUE "ExcludedUsers"
#define REALMMAP_REG_VALUE "REALMS"

/*
 * Convert system error to char. Returns 
 * memory allocated with LocalAlloc.
 */

char *error_to_string(DWORD error)
{
  char *msgbuf;
  
  if(FormatMessage(
       FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
       NULL,
       error,
       MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* Default language */
       (char *)&msgbuf,
       0,
       NULL
       ) == 0)
    return 0;
  return msgbuf;
}

/*
 * Return a pointer to a string describing an os error.
 * error_to_string returns a pointer to LocalAlloc'ed
 * memory. Cache it and release when the next one is
 * requested.
 */

char *str_oserr(DWORD err)
{
  static char *lastmsg = 0;

  if(lastmsg)
    LocalFree((HLOCAL)lastmsg);

  lastmsg = error_to_string(err);
  return lastmsg;
}

/*
 * Utility function to get allocate a SID from a name.
 * Looks on local machine. SID is allocated with LocalAlloc
 * and must be freed by the caller.
 * Returns TRUE on success, FALSE on fail.
 */

BOOL get_sid(const char *name, SID **ppsid)
{
  SID_NAME_USE sid_use;
  DWORD sid_size = 0;
  DWORD dom_size = 0;
  char *domain;

  *ppsid = 0;
  if(LookupAccountName(0, name, 0, &sid_size, 0, &dom_size, &sid_use) == 0) {
    if(GetLastError() != ERROR_INSUFFICIENT_BUFFER) {
      fprintf( stderr, "get_sid: LookupAccountName for size on name %s failed. Error was %s",
            name, str_oserr(GetLastError()));
      return FALSE;
    }
  }

  *ppsid = (SID *)LocalAlloc( LMEM_FIXED, sid_size);
  domain = (char *)LocalAlloc( LMEM_FIXED, dom_size);
  if( *ppsid == 0 || domain == 0) {
    fprintf( stderr, "get_sid: LocalAlloc failed. Error was %s",
                 str_oserr(GetLastError()));
    if(*ppsid)
      LocalFree((HLOCAL)*ppsid);
    if(domain)
      LocalFree((HLOCAL)domain);
    *ppsid = 0;
    return FALSE;
  }

  if(LookupAccountName(0, name, *ppsid, &sid_size, domain, &dom_size, &sid_use) == 0) {
    fprintf( stderr, 
         "get_sid: LookupAccountName failed for name %s. Error was %s",
         name, str_oserr(GetLastError()));
    LocalFree((HLOCAL)*ppsid);
    LocalFree((HLOCAL)domain);
    *ppsid = 0;
    return FALSE;
  }

  LocalFree((HLOCAL)domain);
  return TRUE;
}

/*
 * Utility function to setup a security descriptor
 * from a varargs list of char *name followed by a DWORD access
 * mask. The access control list is allocated with LocalAlloc
 * and must be freed by the caller.
 * returns TRUE on success, FALSE on fail.
 */

BOOL create_sd_from_list( SECURITY_DESCRIPTOR *sdout, int num, ...)
{
  va_list ap;
  SID **sids = 0;
  char *name;
  DWORD amask;
  DWORD acl_size;
  PACL pacl = 0;
  int i;

  if((sids = (SID **)calloc(1,sizeof(SID *)*num)) == 0) {
    fprintf(stderr, "create_sd_from_list: calloc fail.");
    return FALSE;
  }

  acl_size = num * (sizeof(ACL) +
             sizeof(ACCESS_ALLOWED_ACE) +
             sizeof(DWORD));

  /* Collect all the SID's */
  va_start( ap, num);
  for( i = 0; i < num; i++) {
    name = va_arg( ap, char *);
    amask = va_arg(ap, DWORD);
    if(get_sid( name, &sids[i]) == FALSE)
      goto cleanup;
    acl_size += GetLengthSid(sids[i]);
  }
  va_end(ap);
  if((pacl = (PACL)LocalAlloc( LMEM_FIXED, acl_size)) == 0) {
    fprintf( stderr, "create_sd_from_list: LocalAlloc fail. Error was %s",
            str_oserr(GetLastError()));
    goto cleanup;
  }

  if(InitializeSecurityDescriptor( sdout, SECURITY_DESCRIPTOR_REVISION) == FALSE) {
    fprintf( stderr, "create_sd_from_list: InitializeSecurityDescriptor fail. Error was %s",
                 str_oserr(GetLastError()));
    goto cleanup;
  }
  if(InitializeAcl( pacl, acl_size, ACL_REVISION) == FALSE) {
    fprintf( stderr, "create_sd_from_list: InitializeAcl fail. Error was %s",
                 str_oserr(GetLastError()));
    goto cleanup;
  }
  va_start(ap, num);
  for( i = 0; i < num; i++) {
    ACE_HEADER *ace_p;
    name = va_arg( ap, char *);
    amask = va_arg( ap, DWORD);
    if(AddAccessAllowedAce( pacl, ACL_REVISION, amask, sids[i]) == FALSE) {
      fprintf( stderr, "create_sd_from_list: AddAccessAllowedAce fail. Error was %s",
                 str_oserr(GetLastError()));
      goto cleanup;
    }
    /* Make sure the ACE is inheritable */
    if(GetAce( pacl, 0, (LPVOID *)&ace_p) == FALSE) {
      fprintf( stderr, "create_sd_from_list: GetAce fail. Error was %s",
                 str_oserr(GetLastError()));
      goto cleanup;
    }
    ace_p->AceFlags |= ( CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE);
  }

  /* Add the ACL into the sd. */
  if(SetSecurityDescriptorDacl( sdout, TRUE, pacl, FALSE) == FALSE) {
    fprintf( stderr, "create_sd_from_list: SetSecurityDescriptorDacl fail. Error was %s",
               str_oserr(GetLastError()));
    goto cleanup;
  }
  for( i = 0; i < num; i++)
    if(sids[i] != 0)
      LocalFree((HLOCAL)sids[i]);
  free(sids);

  return TRUE;

cleanup:

  if(sids != 0) {
    for( i = 0; i < num; i++)
      if(sids[i] != 0)
        LocalFree((HLOCAL)sids[i]);
    free(sids);
  }
  if(pacl != 0)
    LocalFree((HLOCAL)pacl);
  return FALSE;
}

/*
 * Utility function to cause a registry tree to be created.
 * If the supplied tree doesn't exist it will walk down the
 * tree creating the branches.
 */

int create_registry_tree( HKEY key_start, const char *reg_path, SECURITY_DESCRIPTOR *sd)
{
  /* First see if the key already exists */
  HKEY key;
  DWORD err;
  DWORD disp;
  SECURITY_ATTRIBUTES sa, *psa = 0;
  char *cur_path = 0;
  char *p;

  if((err = RegOpenKeyEx( key_start, reg_path, 0, KEY_READ, &key)) == ERROR_SUCCESS) {
    /* Key exists */
    RegCloseKey(key);
    return 0;
  }

  if((cur_path = malloc(strlen(reg_path) + 1)) == 0) {
    fprintf( stderr, "create_registry_tree: malloc failed.");
    return -1;
  }

  if(sd != 0) {
    sa.nLength = sizeof(SECURITY_ATTRIBUTES);
    sa.lpSecurityDescriptor = sd;
    sa.bInheritHandle = FALSE;
    psa = &sa;
  }

  p = strchr(reg_path, '\\');
  while(1) {
    if( p == 0 ) {
      strcpy(cur_path, reg_path);
    } else {
      strncpy( cur_path, reg_path, p - reg_path);
      cur_path[p-reg_path] = '\0';
    }

    if((err = RegCreateKeyEx( key_start, cur_path, 0, 0, REG_OPTION_NON_VOLATILE,
                               KEY_READ, psa, &key, &disp)) != ERROR_SUCCESS) {
      free(cur_path);
      fprintf(stderr, 
            "create_registry_tree: RegCreateKeyEx failed for path %s. Error was %s.", 
            cur_path, str_oserr(err));
    }
    RegCloseKey(key);
    if(p == 0)
      break;
    p++;
    p = strchr(p, '\\');
  }
  return 0;
}

/*
 * Set the security on the kerbgina.dll file in SYSTEM32. Update the
 * registry value. Assumes kerbgina.dll is already in SYSTEM32.
 */

int gina_install(const char *name)
{
	SECURITY_DESCRIPTOR sd;
	SECURITY_DESCRIPTOR filesd;
	char systempath[_MAX_PATH];
	char rootpath[4];
	char gina_dll_value[512];
	DWORD dummy;
	DWORD flags;
	DWORD type;
	DWORD size;
	HKEY key;
	DWORD err;

	/* Get the systemroot environment variable. */
	if(GetEnvironmentVariable("SYSTEMROOT", systempath, sizeof(systempath)) == 0) {
		fprintf( stderr, 
			"%s: Unable to get environment variable SYSTEMROOT.", name);
		return -1;
	}

	if(strlen(systempath) + strlen("\\SYSTEM32\\") + strlen(DLL_NAME) + 1 > sizeof(systempath)) {
		fprintf(stderr, "%s: System path too long !", name);
		return -1;
	}

	strcat(systempath, "\\SYSTEM32\\");
	strcat(systempath, DLL_NAME);

	/* Check that the gina dll exists. */
	if(GetFileAttributes(systempath) == 0xFFFFFFFF) {
		fprintf(stderr, "%s: File %s cannot be accessed. Error was %s\n",
			name, systempath, str_oserr(GetLastError()));
		return -1;
	}

	/* Check if this filesystem is NTFS */
	if(systempath[1] != ':') {
		fprintf(stderr, 
			   "%s: unable to determine drive letter for SYSTEMROOT.", name);
		return -1;
	}
	strncpy( rootpath, systempath, 2);
	rootpath[2] = '\\';
	rootpath[3] = '\0';
	if(GetVolumeInformation(rootpath, 0, 0, &dummy, &dummy, &flags, 0, 0 ) == FALSE) {
		fprintf(stderr, 
			   "%s: GetVolumeInformation for path %s failed. Error was %s", 
			   name, rootpath, str_oserr(GetLastError()));
		return -1;
	}

	if((flags & FS_PERSISTENT_ACLS) != 0) {
		/* Get the required Security Descriptor. */
		if(create_sd_from_list( &filesd, 2, "Administrators", GENERIC_ALL,
									  "SYSTEM", GENERIC_ALL) == FALSE) {
			return -1;
		}
		if(SetFileSecurity( systempath, DACL_SECURITY_INFORMATION,
							&filesd) == 0) {
			fprintf(stderr, "%s: Failed to set file security on file %s. Error was %s\n",
				name, systempath);
			return -1;
		}
	}

	/* Create the required Security Descriptor. */
	if(create_sd_from_list( &sd, 3, "Administrators", GENERIC_ALL,
								  "SYSTEM", GENERIC_ALL,
								  "Users", GENERIC_READ) == FALSE) {
		return -1;
	}

	/* Ensure that the registry heirarchy exists */
	if(err = create_registry_tree( HKEY_LOCAL_MACHINE, GINA_REG_NAME, &sd)) {
		return err;
	}

	/* Finally install the registry entry in 
	   SYSTEM_GINA_REG_NAME\SYSTEM_GINA_REG_VALUE */
	if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, SYSTEM_GINA_REG_NAME, 0,
							KEY_READ|KEY_WRITE, &key)) != ERROR_SUCCESS) {

		fprintf(stderr, "%s: Failed to open registry key %s. Error was %s\n",
			name, SYSTEM_GINA_REG_NAME, str_oserr(err));
		return -1;
	}
	/* Get the current value of GinaDLL - ok if it doesn't exist. */
	size = sizeof(gina_dll_value);
	err = RegQueryValueEx( key, SYSTEM_GINA_REG_VALUE, 0, &type,
								gina_dll_value, &size);
	if(err == ERROR_SUCCESS) {
		/* If it already contains the string kerbgina.dll then
		   we're done. */
		if(strstr( gina_dll_value, DLL_NAME) != 0) {
			printf("%s: %s is already installed in the registry.\n", name, DLL_NAME);
			RegCloseKey(key);
			return 0;
		} 
		fprintf(stderr, "%s: A GinaDLL value already exists in the registry key %s and is not \
%s. Please remove this value and run the program again.\n", 
				name, SYSTEM_GINA_REG_NAME, DLL_NAME);
		RegCloseKey(key);
		return -1;
	}

	/* Add the new value. */
	if((err = RegSetValueEx( key, SYSTEM_GINA_REG_VALUE, 0, REG_SZ,
		                     systempath, strlen(systempath)+1)) != ERROR_SUCCESS) {
		fprintf(stderr, "%s: Failed to set registry value %s in key %s. Error was %s\n",
			name, SYSTEM_GINA_REG_VALUE, SYSTEM_GINA_REG_NAME, str_oserr(err));
		RegCloseKey(key);
		return -1;
	}
	RegCloseKey(key);

	printf("%s: New GINA installed. You need to reboot your machine before this will take effect\n",
		    name);
	return 0;
}

/*
 * Remove the GinaDLL entry in the registry.
 */

int gina_delete( const char *name)
{
	DWORD err;
	DWORD size;
	DWORD type;
	HKEY key;
 	char gina_dll_value[512];

	/* Finally install the registry entry in 
	   SYSTEM_GINA_REG_NAME\SYSTEM_GINA_REG_VALUE */
	if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, SYSTEM_GINA_REG_NAME, 0,
							KEY_READ|KEY_WRITE, &key)) != ERROR_SUCCESS) {

		fprintf(stderr, "%s: Failed to open registry key %s. Error was %s\n",
			name, SYSTEM_GINA_REG_NAME, str_oserr(err));
		return -1;
	}
	/* Get the current value of GinaDLL. */
	size = sizeof(gina_dll_value);
	if((err = RegQueryValueEx( key, SYSTEM_GINA_REG_VALUE, 0, &type,
								gina_dll_value, &size)) != ERROR_SUCCESS) {
		fprintf(stderr, "%s: The value %s in registry key %s does not exist. \
No GINA was installed\n",
			name, SYSTEM_GINA_REG_VALUE, SYSTEM_GINA_REG_NAME);
		RegCloseKey(key);
		return -1;
	}

	/* If it already contains the string kerbgina.dll then
	   we can delete it. */
	if(strstr( gina_dll_value, DLL_NAME) != 0) {
		RegDeleteValue( key, SYSTEM_GINA_REG_VALUE);
	} else {

		fprintf(stderr, "%s: A GinaDLL value already exists in the registry \
key %s and is not %s.\n", 
				name, SYSTEM_GINA_REG_NAME, DLL_NAME);
		RegCloseKey(key);
		return -1;
	}

	printf("%s: %s GINA is now deleted. You need to reboot your machine \
before this will take effect\n", DLL_NAME, name);
	return 0;
}

/*
 * Set the 'REALMS' map in the registry.
 * map must be of the format 'DOMAIN=realm,DOMAIN=realm......'
 */

int gina_realmmap( const char *name, char *arg)
{
	DWORD err;
	HKEY key;
	DWORD len;
	char *map;
	char *p, *p1;

        if((map = calloc(strlen(arg) + 2, 1)) == 0 ) {
                fprintf(stderr, "%s: gina_realmmap : calloc failed\n",
                        name);
                return -1;
        }
        strcpy(map, arg);
        len = strlen(map) + 2; /* Include extra '\0' */

	/* Process the map, replacing ',' with '\0' */
	for( p = map; *p; p += strlen(p)+1) {
		if((p1 = strchr(p, ',')) != 0)
			*p1 = 0;
		if(strchr(p, '=') == 0) {
			fprintf(stderr, "%s: Malformed realmmap. Map should be of the form \
'DOMAIN=realm,MACHINE=realm,....\n",
					name);
			return -1;
		}
	}
	if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, GINA_REG_NAME, 0,
							KEY_READ|KEY_WRITE, &key)) != ERROR_SUCCESS) {
		fprintf(stderr, "%s: Failed to open registry key %s. Error was %s\n",
			name, GINA_REG_NAME, str_oserr(err));
		return -1;
	}
	if((err = RegSetValueEx( key, REALMMAP_REG_VALUE, 0, REG_MULTI_SZ,
		                     (LPBYTE)map, len)) != ERROR_SUCCESS) {
		fprintf(stderr, "%s: Failed to set registry value %s in key %s. Error was %s\n",
			name, REALMMAP_REG_VALUE, GINA_REG_NAME, str_oserr(err));
		RegCloseKey(key);
		return -1;
	}
	RegCloseKey(key);
	printf("%s: REALMS value set\n", name);
	return 0;
}

/*
 * Set the 'ExcludedUser' entry in the registry.
 * map must be of the format 'user,user1,......'
 */

int gina_excluded_users( const char *name, char *arg)
{
	DWORD err;
	HKEY key;
	DWORD len;
        char *users;
	char *p, *p1;

	if((users = calloc(strlen(arg) + 2, 1)) == 0 ) {
		fprintf(stderr, "%s: gina_excluded_users : calloc failed\n",
                        name);
                return -1;
        }
        strcpy(users, arg);
        len = strlen(users) + 2; /* Include extra '\0' */

	/* Process the map, replacing ',' with '\0' */
	for( p = users; *p; p += strlen(p)+1) {
		if((p1 = strchr(p, ',')) != 0)
			*p1 = 0;
	}
	if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, GINA_REG_NAME, 0,
							KEY_READ|KEY_WRITE, &key)) != ERROR_SUCCESS) {
		fprintf(stderr, "%s: Failed to open registry key %s. Error was %s\n",
			name, GINA_REG_NAME, str_oserr(err));
		return -1;
	}
	if((err = RegSetValueEx( key, EXCLUDED_USERS_REG_VALUE, 0, REG_MULTI_SZ,
		                     (LPBYTE)users, len)) != ERROR_SUCCESS) {
		fprintf(stderr, "%s: Failed to set registry value %s in key %s. Error was %s\n",
			name, EXCLUDED_USERS_REG_VALUE, GINA_REG_NAME, str_oserr(err));
		RegCloseKey(key);
		return -1;
	}
	RegCloseKey(key);
	printf("%s: ExcludedUsers value set\n", name);
	return 0;
}


/*
 * Add or remove the 'RequireKerberosLogon' registry entry.
 */
int gina_require(const char *name, BOOL require)
{
	DWORD require_val = (DWORD)require;
	HKEY key;
	DWORD err;

	if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, GINA_REG_NAME, 0,
							KEY_READ|KEY_WRITE, &key)) != ERROR_SUCCESS) {
		fprintf(stderr, "%s: Failed to open registry key %s. Error was %s\n",
			name, GINA_REG_NAME, str_oserr(err));
		return -1;
	}
	if((err = RegSetValueEx( key, REQUIRE_KERBEROS_LOGON_REG_VALUE, 0, REG_DWORD,
		                     (LPBYTE)&require_val, sizeof(DWORD))) != ERROR_SUCCESS) {
		fprintf(stderr, "%s: Failed to set registry value %s in key %s. Error was %s\n",
			name, REQUIRE_KERBEROS_LOGON_REG_VALUE, GINA_REG_NAME, str_oserr(err));
		RegCloseKey(key);
		return -1;
	}
	RegCloseKey(key);
	printf("%s: RequireKerberosLogon set to %s\n", name, require ? "True" : "False");
	return 0;
}

/*
 * Print program usage.
 */

void usage(const char *name)
{
	fprintf(stderr, "Usage: %s [/install] [/delete] [/require] \n\
[/norequire] [/realmmap DOMAIN=realm,DOMAIN1=realm1] [/excluded user1,user2]\n", name);
}

int main(int argc, char **argv)
{
  OSVERSIONINFO vers;
  char username[512];
  DWORD usize = sizeof(username);

  /* Ensure we are running on Windows NT */
  vers.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  GetVersionEx( &vers );

  if(vers.dwPlatformId != VER_PLATFORM_WIN32_NT) {
	  fprintf(stderr, "%s: This program can only be run on Windows NT\n",
		      argv[0]);
	  return -1;
  }

  /*
   * Check we are running as Administrator - exit otherwise.
   */

  if(GetUserName(username, &usize) == FALSE) {
    fprintf(stderr, "%s: GetUserName failed. Error was %s.",
                argv[0], str_oserr(GetLastError()));
    return -1;
  }

  if(stricmp(username, "Administrator") != 0) {
	  fprintf(stderr, 
		  "%s: You must be running as user Administrator to use this program. You are currently user %s.", 
		    argv[0], username);
    free(username);
    return -1;
  }

  free(username);

  if((argc == 2) && (argv[1][0] == '-' || argv[1][0] == '/')) {
	  if(stricmp( &argv[1][1], "install") == 0)
		  return gina_install( argv[0] );
	  else if (stricmp( &argv[1][1], "delete") == 0)
		  return gina_delete( argv[0] );
	  else if (stricmp( &argv[1][1], "require") == 0)
		  return gina_require( argv[0], TRUE);
	  else if (stricmp( &argv[1][1], "norequire") == 0)
		  return gina_require( argv[0], FALSE);
  } else if ((argc == 3) && (argv[1][0] == '-' || argv[1][0] == '/')) {
	  if(stricmp( &argv[1][1], "realmmap") == 0)
		  return gina_realmmap( argv[0], argv[2]);
	  else if(stricmp( &argv[1][1], "excluded") == 0)
		  return gina_excluded_users(argv[0], argv[2]);
  }
  usage(argv[0]);
  return -1;
}

