#include <windows.h>
#include <process.h>
#include <stdio.h>
#include <stdlib.h>

struct UNI_STRING {
  USHORT len;
  USHORT maxlen;
  WCHAR *buff;
};

#define PWS_DLL_KEY_REG_NAME \
"SOFTWARE\\Cygnus Solutions\\PasswordSync"

#define PWS_DLL_TIMEOUT_REG_VALUE_NAME \
"SOFTWARE\\Cygnus Solutions\\PasswordSync\\Timeout"

#define PWS_DLL_TOKEN_KEY_REG_NAME \
"SOFTWARE\\Cygnus Solutions\\PasswordSync\\DLLToken"

#define PWS_DLL_WATCHKEY_REG_NAME \
"SOFTWARE\\Cygnus Solutions\\PasswordSync\\WatchKey"

#define PWS_DLL_TOKEN_REG_VALUE_NAME "Token"

#define PASSWD_SYNC_PIPE_NAME "\\\\.\\PIPE\\PASSWORDSYNC"

#define DEFAULT_TIMEOUT 3*1000

static HANDLE global_nph = INVALID_HANDLE_VALUE;
static DWORD global_timeout;
static char global_token[8];
static char global_msgbuf[2048];
static char global_err_buf[1024];
static CRITICAL_SECTION global_nph_cs;
static HKEY global_watch_key;

/*
 * Convert system error to char. Returns 
 * memory allocated with LocalAlloc.
 */

char *error_to_string(DWORD error)
{
  char *msgbuf;
  
  if(FormatMessage(
       FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
       NULL,
       error,
       MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* Default language */
       (char *)&msgbuf,
       0,
       NULL
       ) == 0)
    return 0;
  return msgbuf;
}

/*
 * Return a pointer to a string describing an os error.
 * error_to_string returns a pointer to LocalAlloc'ed
 * memory. Cache it and release when the next one is
 * requested.
 */

char *str_oserr(DWORD err)
{
  static char *lastmsg = 0;

  if(lastmsg)
    LocalFree((HLOCAL)lastmsg);

  lastmsg = error_to_string(err);
  return lastmsg;
}

static void log_err(const char *message)
{
  HANDLE hEventSrc = RegisterEventSource(NULL, "Password Sync DLL");
  if (hEventSrc == 0)	/* Not much we can do here... */
    return;
  ReportEvent(hEventSrc, EVENTLOG_ERROR_TYPE, 0, 0,
	      NULL, 1, 0, &message, NULL);
  DeregisterEventSource (hEventSrc);
}

BOOLEAN attempt_connect()
{
  DWORD pipe_mode;
  OVERLAPPED ov;
  char buf[8];
  DWORD amount_read;
  
  /*
   * Open the named pipe. Defaults to byte mode.
   */
  if((global_nph = CreateFile( PASSWD_SYNC_PIPE_NAME, GENERIC_READ|GENERIC_WRITE,
			       0, 0, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, 0)) == INVALID_HANDLE_VALUE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1, 
	      "Failed to open pipe %s. Error was %s",
	      PASSWD_SYNC_PIPE_NAME, str_oserr(GetLastError()));
    log_err(global_err_buf);
    return TRUE;
  }

  /*
   * Change the pipe mode to message.
   */
  pipe_mode = PIPE_READMODE_MESSAGE | PIPE_WAIT; 
  if(SetNamedPipeHandleState( global_nph, &pipe_mode, 0, 0) == FALSE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1, 
	      "Failed to set named pipe to message mode. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }

  memset(&ov, '\0', sizeof(ov));
  /* 
   * Start the read. Special case - ov.hEvent being null allows
   * the pipe handle to be waited on.
   */
  if(ReadFile(global_nph, buf, sizeof(global_token), &amount_read,&ov) == FALSE) {
    if(GetLastError() == ERROR_IO_PENDING) {
      
      /* 
       * Wait global_timeout milliseconds for a message to come in. If it is 
       * the token then we are talking to a SYSTEM level process. Else, drop 
       *the pipe and return.
       */
      if( WaitForSingleObject(global_nph, global_timeout) != WAIT_OBJECT_0) {
	log_err( "Timeout waiting for token to be returned.");
	goto cleanup;
      }
      if(GetOverlappedResult( global_nph, &ov, &amount_read, FALSE)==FALSE) {
	_snprintf(global_err_buf, sizeof(global_err_buf)-1,
		  "GetOverlappedResult failed on reading token. Error was %s",
		  str_oserr(GetLastError()));
	log_err(global_err_buf);
	goto cleanup;
      }
    } else {
      _snprintf(global_err_buf, sizeof(global_err_buf)-1,
		"ReadFile on pipe for token failed. Error was %s",
		str_oserr(GetLastError()));
      log_err(global_err_buf);
      goto cleanup;
    }
  }
  if(amount_read != sizeof(global_token)) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "Incorrect token size read. Was %d, should be %d", amount_read, 
	      sizeof(global_token));
    log_err(global_err_buf);
    goto cleanup;
  }
  if(memcmp( buf, global_token, sizeof(global_token)) != 0) {
    log_err("Incorrect token received.");
    goto cleanup;
  }
  /* Everything ok - we're connected. */
  return FALSE;

cleanup:

  CloseHandle(global_nph);
  global_nph = INVALID_HANDLE_VALUE;
  return TRUE;
  
}

/*
 * Function to create a security descriptor containing all
 * access, inheritable to SYSTEM only.
 */

BOOL setup_security(SECURITY_DESCRIPTOR *psd)
{
  PSID systemSid = 0;
  SID_IDENTIFIER_AUTHORITY ntauth = SECURITY_NT_AUTHORITY;
  DWORD size;
  PACL pacl = 0;
  ACE_HEADER *ace_p;
  
  if(AllocateAndInitializeSid( &ntauth, 1,
			       SECURITY_LOCAL_SYSTEM_RID,
			       0, 0, 0, 0, 0, 0, 0,
			       &systemSid) == FALSE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "AllocateAndInitializeSid failed. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }
  if(InitializeSecurityDescriptor( psd, SECURITY_DESCRIPTOR_REVISION) == FALSE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "InitializeSecurityDescriptor failed. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }
  if(SetSecurityDescriptorOwner( psd, systemSid, 0) == FALSE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "SetSecurityDescriptorOwner failed. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }
  
  /* Calculate the DACL size needed. */
  size = sizeof(ACL) +
    sizeof(ACCESS_ALLOWED_ACE) +
    sizeof(DWORD) +
    GetLengthSid(systemSid);
  if((pacl = (PACL)LocalAlloc( LMEM_FIXED, size)) == 0) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "Failed to allocate memory for the ACL. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }
  
  if(InitializeAcl( pacl, size, ACL_REVISION) == FALSE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "InitializeAce failed. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }
  if(AddAccessAllowedAce( pacl, ACL_REVISION, GENERIC_ALL, systemSid)	== FALSE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "AddAccessAllowedAce failed. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }
  /* Make sure the ACE is inheritable */
  if(GetAce( pacl, 0, (LPVOID *)&ace_p) == FALSE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "GetAce failed. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }
  
  ace_p->AceFlags |= ( CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE);
  if(SetSecurityDescriptorDacl( psd, TRUE, pacl, FALSE) == FALSE) {
    _snprintf(global_err_buf, sizeof(global_err_buf)-1,
	      "SetSecurityDescriptorDacl failed. Error was %s",
	      str_oserr(GetLastError()));
    log_err(global_err_buf);
    goto cleanup;
  }
  
  return TRUE;

cleanup:

  if(pacl != 0)
    LocalFree((HLOCAL)pacl);
  if(systemSid != 0)
    FreeSid(systemSid);
  return FALSE;
}

/*
 * Watch thread. Any change on the watch key means the password sync
 * service has either started or stopped. Close down the pipe handle.
 */

static void __cdecl watch_thread( void *arg)
{
  DWORD err;
  char local_msg[1024];

  while(1) {
    err = RegNotifyChangeKeyValue(global_watch_key, FALSE, REG_NOTIFY_CHANGE_LAST_SET,
			    FALSE, FALSE);
    if(err != ERROR_SUCCESS) {
      _snprintf(local_msg, sizeof(local_msg)-1,
		"RegNotifyChangeKeyValue failed. Error was %s",
		str_oserr(err));
      log_err(local_msg);
      return;
    }

    /* Get the critical section and close the pipe */
    EnterCriticalSection(&global_nph_cs);
    if(global_nph != INVALID_HANDLE_VALUE)
      CloseHandle(global_nph);
    global_nph = INVALID_HANDLE_VALUE;
    LeaveCriticalSection(&global_nph_cs);
  }
}

/*
 * Use this call to add a random 8 byte token into
 * the registry. The receiving service uses this to
 * prove to us it has system access.
 */

BOOLEAN __stdcall InitializeChangeNotify ()
{
  int num;
  DWORD err;
  DWORD val;
  DWORD disp, type, size;
  HKEY hkey;
  SECURITY_ATTRIBUTES sa;
  SECURITY_DESCRIPTOR sd;

  __try {
    /* Set the timeout to default */
    global_timeout = DEFAULT_TIMEOUT;
    global_nph = INVALID_HANDLE_VALUE;
    
    InitializeCriticalSection(&global_nph_cs);
    
    if(setup_security(&sd) == FALSE)
      return FALSE;
    
    sa.nLength = sizeof(SECURITY_ATTRIBUTES);
    sa.lpSecurityDescriptor = &sd;
    sa.bInheritHandle = FALSE;

    /* Seed the random number generator */
    srand( (unsigned)GetCurrentTime());
    num = rand();
    memcpy( &global_token[0], &num, 4);
    num = rand();
    memcpy( &global_token[4], &num, 4);

    /* Open the watchkey */
    if((err = RegCreateKeyEx(HKEY_LOCAL_MACHINE,
			     PWS_DLL_WATCHKEY_REG_NAME, 0, 0,
			     REG_OPTION_VOLATILE,
			     KEY_ALL_ACCESS, &sa, &global_watch_key, &disp)) != ERROR_SUCCESS) {
      _snprintf(global_err_buf, sizeof(global_err_buf)-1, 
		"Failed to create registry key %s. Error was %s",
		PWS_DLL_WATCHKEY_REG_NAME, str_oserr(err));
      log_err(global_err_buf);
      return FALSE;
    }

    /* Set a dummy value on the watchkey */
    if((err = RegSetValueEx( global_watch_key, "watchval", 0, REG_DWORD,
	     (CONST BYTE *)&val, sizeof(val))) != ERROR_SUCCESS) {
      _snprintf(global_err_buf, sizeof(global_err_buf)-1, 
		"Failed to write the watch value in registry key %s. Error was %s",
		PWS_DLL_WATCHKEY_REG_NAME, str_oserr(err));
      log_err(global_err_buf);
      return FALSE;
    }

    /* Delete the old token. Ignore errors. */
    RegDeleteKey(HKEY_LOCAL_MACHINE, PWS_DLL_TOKEN_KEY_REG_NAME);

    /* Add the token. */
    if((err = RegCreateKeyEx(HKEY_LOCAL_MACHINE, 
			     PWS_DLL_TOKEN_KEY_REG_NAME, 0, 0,
			     REG_OPTION_VOLATILE, 
			     KEY_ALL_ACCESS, &sa, &hkey, &disp)) != ERROR_SUCCESS) {
      _snprintf(global_err_buf, sizeof(global_err_buf)-1, 
		"Failed to create registry key %s. Error was %s",
		PWS_DLL_TOKEN_KEY_REG_NAME, str_oserr(err));
      RegCloseKey(global_watch_key);
      log_err(global_err_buf);
      return FALSE;
    }
    if((err = RegSetValueEx(hkey, PWS_DLL_TOKEN_REG_VALUE_NAME,	0,
			    REG_BINARY, (BYTE *)global_token, 
			    sizeof(global_token))) != ERROR_SUCCESS) {
      _snprintf(global_err_buf, sizeof(global_err_buf)-1, 
		"Failed to set registry value %s. Error was %s",
		PWS_DLL_TOKEN_REG_VALUE_NAME, str_oserr(err));
      log_err(global_err_buf);
      RegCloseKey(hkey);
      RegCloseKey(global_watch_key);
      return FALSE;
    }
    RegCloseKey(hkey);
    if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, 
			   PWS_DLL_KEY_REG_NAME, 0,
			   KEY_READ, &hkey)) == ERROR_SUCCESS) {
      size = sizeof(global_timeout);
      err = RegQueryValueEx(hkey, PWS_DLL_TIMEOUT_REG_VALUE_NAME,
			    0, &type, (LPBYTE)&global_timeout,
			    &size);
      if((err == ERROR_SUCCESS) && (type != REG_DWORD))
	global_timeout == DEFAULT_TIMEOUT;
    }
    RegCloseKey(hkey);
    
    /* Now create the watchthread that will close the named pipe handle on change */
    if(_beginthread( watch_thread, 0, 0) == -1) {
      _snprintf(global_err_buf, sizeof(global_err_buf)-1, 
		"Failed to create watch thread.");
      log_err(global_err_buf);
      RegCloseKey(global_watch_key);
      return FALSE;
    }

  } __except(	EXCEPTION_EXECUTE_HANDLER ) {
    log_err("Exception in InitializeChangeNotify");
  }
  return TRUE;
} 

LONG __stdcall PasswordChangeNotify (
				     struct UNI_STRING *user,
				     ULONG rid,
				     struct UNI_STRING *passwd
)
{
  DWORD written;
  char *buf;
  DWORD len;
  
  __try {
    /* Get a valid handle */
    EnterCriticalSection(&global_nph_cs);
    if(global_nph == INVALID_HANDLE_VALUE) {
      if(attempt_connect() == TRUE) {
	LeaveCriticalSection(&global_nph_cs);
	return FALSE;
      }
    }
    buf = (char *)global_msgbuf;
    len = sizeof(global_msgbuf);
    
    /* Set the uid */
    memcpy(buf, &rid, sizeof(rid));
    buf += sizeof(rid);
    len -= sizeof(rid);
    
    if(user->len + sizeof(wchar_t) > len) {
      log_err("Message buffer too small for username.");
      LeaveCriticalSection(&global_nph_cs);
      return FALSE;
    }
    memcpy(buf, user->buff, user->len);
    memset(&buf[user->len], '\0', sizeof(wchar_t));
    buf += (user->len + sizeof(wchar_t));
    len -= (user->len + sizeof(wchar_t));
    
    if(passwd->len + sizeof(wchar_t) > len) {
      log_err("Message buffer too small for password.");
      LeaveCriticalSection(&global_nph_cs);
      return FALSE;
    }
    memcpy(buf, passwd->buff, passwd->len);
    memset(&buf[passwd->len], '\0', sizeof(wchar_t));
    buf += (passwd->len + sizeof(wchar_t));
    len -= (passwd->len + sizeof(wchar_t));
    
    /* Now write down the pipe */
    if(WriteFile(global_nph, global_msgbuf, buf - global_msgbuf, &written, 0) == FALSE) {
      _snprintf(global_err_buf, sizeof(global_err_buf)-1,
		"WriteFile failed. Error was %s", str_oserr(GetLastError()));
      log_err(global_err_buf);
      CloseHandle(global_nph);
      global_nph = INVALID_HANDLE_VALUE;
      LeaveCriticalSection(&global_nph_cs);
      return FALSE;
    }
    if(written != (DWORD)(buf - global_msgbuf)) {
      log_err("WriteFile : short write.");
      CloseHandle(global_nph);
      global_nph = INVALID_HANDLE_VALUE;
      LeaveCriticalSection(&global_nph_cs);
      return FALSE;
    }
    LeaveCriticalSection(&global_nph_cs);
  } __except(	EXCEPTION_EXECUTE_HANDLER ) {
    log_err("Exception in PasswordChangeNotify");
  }
  return TRUE;
}

