/***********************************************************
	Copyright 1992 by Carnegie Mellon University
  
                      All Rights Reserved
  
Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.  
  
CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/
/*
 * parse.c - MIB compiler done in the author's spare time.
 *
 * This is compiler for MIB (Management Information Base) specifications
 * based on ISO/DIS ASN.1 grammar. It is a recursive descent parser
 * with bultin lexer and MIB oriented semantics. It is NOT an ASN.1
 * grammar parser.
 *
 * The expectation is that MIB semantics will be enhanced over the 
 * following years and some parse code will be augmented. With this
 * expectation the compiler was designed with the following in mind:
 *	1) eventually compiled indexed binary MIBs will be used possibly
 *	   generated by use of this compiler.
 *	2) understandable error messages.
 *	3) minimal load time
 *	4) minimal memory usage
 *
 * Compared to the previous version of parse.c, it supports a variety of
 * functions to enable a large number of independently created MIB
 * files to be loaded without problems. It provides to some extent
 * the following features:
 *	- module scoping
 *	- very limited macro support (hacked)
 *	- limited type checking
 *	- simple but effective Unix "C" like compiler error msgs.
 *	- MIB semantics
 * It does however use alot more memory and takes longer.
 *
 * Undoubtedly there are lots of speed improvements that can be done
 * but I am out of time.
 *
 *			Rudy Nedved, November 1992.
 *
 * HISTORY
 * 06-Nov-92  Rudy Nedved (ern) at Carnegie-Mellon University
 *	Initial version.
 *	Undoubtedly lots of bugs.
 *	Some known. Some not known.
 *
 */
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <varargs.h>
#include <errno.h>
#include "asn1.h"
#include "parse.h"
extern int *malloc();
extern int errno;
extern char *strerror();

/****************************************************************
 *		FEATURE DEFINES
 *****************************************************************
 */

/* DODEBUG is non-zero for including debug output. Off eliminates
 * along of static strings and additional debugging code.
 */
#define	DODEBUG 0

/* MEMACCT is non-zero for counting memory allocations and deallocations.
 * This is useful for checking memory leaks. Off eliminates print outs
 * and speeds up program.
 */
#define	MEMACCT	0

/* MEMCHECK is non-zero so each chunk of memory is checked and garbaged
 * before freeing. Each chunk of memory is marked with a code and a
 * length so free() calls can check if the call is valid and can garbage
 * the memory to insure freed memory is not used for valid information.
 */
#define MEMCHECK 0

/* HASHSTRACCT is non-zero of hash string stats should be maintained and
 * printed. The primary goal is to compress string usage, minimize byte
 * comparisons done and overall comparisons.
 */
#define HASHSTRACCT 0

/* MAXTOKENSIZ is the length of the maximum token we can parse in the
 * lexer. Normally it is the maximum sized character string with
 * white space compression we can handle. If it is exceeded then
 * error messages will occur and the token will be truncated.
 */
#define	MAXTOKENSIZ (4*1024)

/* MAXLINESIZ is the length of the maximum source input line we can
 * maintain in the parser. The parser will not fail but error message
 * line displays will be truncated.
 */
#define	MAXLINESIZ 512
/*****************************************************************
 *		VARIOUS MODULE GLOBALS
 *****************************************************************
 */

#if	DODEBUG
int LexerDebug = 2;		/* non-zero for debug lexer */
    /* 1 print fetched tokens */
    /* 2 print lines fetched */
    /* 3 print peek tokens */
    /* 4 print eat tokens */
    /* 5 print every fetched token */
int ParseDebug = 1;		/* non-zero for debug parser */
#define	PPRINTF(XX)	if (ParseDebug) printf XX
#define	VPPRINTF(XX)	if (ParseDebug>1) printf XX
#define	VVPPRINTF(XX)	if (ParseDebug>2) printf XX
    /* 1 print descent routines */
    /* 2 print major parse information */
    /* 3 print minor parse information */
    /* 4 print very minor parse */
int SymblDebug = 1;		/* non-zero for debug symbol table */
#define SPRINTF(XX)	if (SymblDebug) printf XX
#define VSPRINTF(XX)	if (SymblDebug>1) printf XX
#define VVSPRINTF(XX)	if (SymblDebug>2) printf XX
    /* 1 show major symbolic operations */
    /* 2 show minor symbolic operations */
    /* 3 show very minor symbolic operations */
    /* 4 show types, values and macros added */
    /* 5 show intermediate types, values and macros */
int MacroDebug = 1;		/* non-zero for debug macro */
#define MPRINTF(XX)	if (MacroDebug) printf XX
#define VMPRINTF(XX)	if (MacroDebug>1) printf XX
#define VVMPRINTF(XX)	if (MacroDebug>2) printf XX
    /* 1 print major macro operations */
    /* 2 print minor macro operations */
    /* 3 print very minor macro operations */
    /* 4 print macro parts */
int TreeDebug = 2;		/* non-zero for debug tree code */
#define TPRINTF(XX)	if (TreeDebug>2) printf XX
    /* 1 print tree at very end */
    /* 2 print tree when module augmentation done */
    /* 3 print tree messages */
    /* 4 print subtrees after addition and fill in */
    /* 5 print subtree after addiition */

#else
#define	PPRINTF(XX)
#define	VPPRINTF(XX)
#define	VVPPRINTF(XX)
#define SPRINTF(XX)
#define VSPRINTF(XX)
#define VVSPRINTF(XX)
#define MPRINTF(XX)
#define VMPRINTF(XX)
#define VVMPRINTF(XX)
#define TPRINTF(XX)

#endif DODEBUG

#if	MEMACCT
static long MemCntMalloc = 0;	/* outstanding memory (global) */
static long RelCntMalloc = 0;	/* freed memory (global) */
static long MemCntTokLoad = 0;	/* outstanding tokens */
static long RelCntTokLoad = 0;	/* freed tokens */
static long MemCntDNODE = 0;	/* outstanding delay nodes */
static long RelCntDNODE = 0;	/* freed delay nodes */
static long MemCntMNODE = 0;	/* outstanding module nodes */
static long RelCntMNODE = 0;	/* freed module nodes */
static long MemCntTNAME = 0;	/* outstanding type name nodes */
static long RelCntTNAME = 0;	/* freed type name nodes */
static long MemCntVNAME = 0;	/* outstanding value name nodes */
static long RelCntVNAME = 0;	/* freed value name nodes */
static long MemCntSNODE = 0;	/* outstanding symbol nodes */
static long RelCntSNODE = 0;	/* freed symbol nodes */
static long MemCntMPARSE = 0;	/* outstanding macro nodes */
static long RelCntMPARSE = 0;	/* freed macro nodes */
static long MemCntTREE = 0;	/* outstanding tree nodes */
static long RelCntTREE = 0;	/* freed tree nodes */
static long MemCntSTR = 0;	/* outstanding strings */
static long RelCntSTR = 0;	/* freed strings nodes */
static long MemCntENUMS = 0;	/* outstanding enums */
static long RelCntENUMS = 0;	/* freed enums */
static long MemCntINODE = 0;	/* outstanding import/export nodes */
static long RelCntINODE = 0;	/* freed import/export nodes */
#endif	MEMACCT

/*****************************************************************
 *		RANDOM SUPPORT ROUTINES
 *****************************************************************
 */

/*
 * general print error routine
 *
 * PrintError(module,flag,submsg,msg,nextline)
 *
 * module - name of module calling this routine
 * flag - 2=fatal,1=error,0=warning
 * submsg - sub information message
 * msg - message
 * nextline - a next line to print
 */
static PrintError(module,flag,submsg,msg,nextline)
char *module;
int flag;
char *submsg,*msg,*nextline;
{
    register char *p;
    static char errbuf[BUFSIZ];

    p = errbuf;
    *p = '\0';

    /* append module if we have one */
    if (module != NULL) {
	strcat(p,module);
	strcat(p,": ");
	p += (int)strlen(p);
    }

    /* append error code */
    if (flag == 1)
	strcat(p,"Error: ");
    else if (flag == 2)
	strcat(p,"Fatal Error: ");
    else
	strcat(p,"Warning: ");
    p += (int)strlen(p);

    /* append sub message if we have one */
    if (submsg != NULL) {
	strcat(p,submsg);
	strcat(p," ");
	p += (int)strlen(p);
    }

    /* append message */
    strcat(p,msg);

    /* flush buffers out first */
    fflush(stdout);
    fflush(stderr);

    /* dump it out */
    fputs(errbuf,stdout);
    fputs("\n",stdout);
    if (nextline != NULL) {
	fputs("       ",stdout);
	fputs(nextline,stdout);
	fputs("\n",stdout);
    }
    fflush(stdout);

    /* see if we should die */
    if (flag == 2) {
	fprintf(stderr,"Fatal error; can not continue\n");
	exit(-2);
    }
#if	DODEBUG
    exit(-3);
#endif
}

/*
 * An error unrelated to parse actions has occured (out of memory)
 *
 * PrintErr(module,flag,fmt,args...)
 *
 * module - name of module calling this routine
 * flag - 2=fatal,1=error,0=warning
 * fmt,args - printf error message
 */
/*VARARGS*/
static PrintErr(va_alist)
va_dcl
{
    va_list ap;
    int flag;
    char *module,*fmt;
    char msgbuf[BUFSIZ];

    /* collect arguments */
    va_start(ap);
    module = va_arg(ap,char *);
    flag = va_arg(ap,int);
    fmt = va_arg(ap,char *);
    vsprintf(msgbuf, fmt, ap);
    va_end(ap);

    PrintError(module,flag,NULL,msgbuf,NULL);
}

/*
 * allocate general memory
 */
static char *Malloc(len,noclear)
register int len;
register int noclear;
{
    register char *ptr;

    /* deal with IBM RT C library malloc bug */
#if ibmrt
    if (len < 16) len = 16;
#endif ibmrt

#if	MEMCHECK
    len += 2*sizeof(long);
    ptr = (char *) malloc(len);
#else
    ptr = (char *) malloc(len);
#endif
    if (ptr == NULL)
	PrintErr("Malloc",2,"Can not allocate memory - %s",strerror(errno));
#if	MEMACCT
    MemCntMalloc++;
#endif	/* MEMACCT */
    if (noclear == 0)
	bzero(ptr,len);

#if	MEMCHECK
    *(long *)ptr = len;
    ptr += sizeof(long);
#define	MEMCHECKCODE	0x504d4e73	/* sNMP backwards */
    *(long *)ptr = MEMCHECKCODE;
    ptr += sizeof(long);
#endif
    return ptr;
}

/*
 * free general memory
 */
#if	MEMCHECK
static Free(ptr)
register char *ptr;
{
    register int len,cod,i;
    register char *p;
    ptr -= sizeof(long);
    cod = *(long *)ptr;
    ptr -= sizeof(long);
    len = *(long *)ptr;
    if (cod != MEMCHECKCODE || len <= 2*sizeof(long)) {
	fflush(stdout);
	fprintf(stderr,"memory problem cod=%x (vs %x) len=%d\n",
		cod,MEMCHECKCODE,len);
	exit(-1);
    }
    for(p=ptr,i=0;i<len;i++)
	*p++ = 0xc3;
#if	MEMACCT
    MemCntMalloc--;
    RelCntMalloc++;
#endif
    free((char *)ptr);
}
#else
#if	MEMACCT
static Free(x)
char *x;
{
    MemCntMalloc--;
    RelCntMalloc++;
    free(x);
}
#else
#define	Free(ptr) free(ptr)
#endif
#endif


/****************************************************************
 *		COMPRESSED STRING SUPPORT
 *****************************************************************
 */

/*
 * string allocation for static (unmodified) strings. We use a hash
 * table to find the strings quickly. Alas this mechanism causes a bit
 * more overhead for finally allocated strings but what the heck.
 */
static struct HashStrEntry {
    struct HashStrEntry *Next;	/* next bucket entry */
#if	MEMCHECK
    long Code;			/* mem check code */
#endif	MEMCHECK
    unsigned long Hash;		/* hash value */
    short Len;			/* length of the string */
    short Cnt;			/* count of users of string */
    /* the rest is the string */
    char Val[4];
};
typedef struct HashStrEntry HASHSTR;

/* we have 62 characters \n, space and printable */
#define	HASHSTRTABSIZ	701
static HASHSTR **HashStrTable;

#if	HASHSTRACCT
static long hashstrnew = 0;	/* how many strings created */
static long hashstrhits = 0;	/* how many times we found a str */
static long hashstrmiss = 0;	/* how many times we did not find a str */
static long hashstrhcmp = 0;	/* how many hash compares were done */
static long hashstrlcmp = 0;	/* how many length compares were done */
static long hashstrscmp = 0;	/* how many string compares were done */
static long hashstrbcmp = 0;	/* how many bytes compares were done */
static long hashstrbcnt = 0;	/* how many bytes in strings */

static HashStats()
{
    register int i;
    register HASHSTR *ptr;
    int min,max,tot,below,above,avg;

    min = max = -1;
    above = below = tot = 0;
    avg = hashstrnew/HASHSTRTABSIZ;
    for(i=0;i<HASHSTRTABSIZ;i++){
	register int cnt;
	cnt = 0;
	for(ptr = HashStrTable[i];ptr != NULL;ptr = ptr->Next)
	    cnt++;
	tot += cnt;
	if (max < 0 || cnt > max) max = cnt;
	if (min < 0 || cnt < min) min = cnt;
	if (cnt < avg) below++; 
	else if (cnt > avg) above++;
    }
    avg = tot/i;
    printf("HashStr new %d hits %d miss %d\n",
	   hashstrnew,hashstrhits,hashstrmiss);
    printf("HashStr total %d avg %d min %d max %d below %d above %d\n",
	   tot,avg,min,max,below,above);
    printf("HashStr hash %d len %d str %d bytes %d (%d stored)\n",
	   hashstrhcmp,hashstrlcmp,hashstrscmp,hashstrbcmp,hashstrbcnt);
    printf("HashStr h+l %d, h+l+b %d\n",
	   hashstrhcmp+hashstrlcmp,hashstrhcmp+hashstrlcmp+hashstrbcmp);
	   
}

static int hashstrcmp(s1,s2)
register char *s1,*s2;
{
    register int i,ch;

    for(;;) {
	i = (ch = *s1++) - *s2++;
	hashstrbcmp++;
	if (ch == 0 || i != 0) break;
    }
    return i;
}
#endif

/*
 * generate a hash value that attempts to 1) identify the string uniquely
 * especially when compared to similiar strings, 2) spread same length
 * strings over different buckets and 3) generate good bucket values
 * as compared to the prime number table size
 *
 * return a unsigned 32-bit number for comparisons as by side effect set
 * the length.
 */
static unsigned long HashStr(str,lenp)
char *str;
int *lenp;
{
    register unsigned long hash;
    register int tmp;
    register char *p;

    hash = 0;
    for (p = str;*p;) {
	tmp = *p++;
	hash <<= 1;
	hash += (tmp-7);
    }

    *lenp = tmp = p - str;
    hash *= tmp;
    return hash;
}

/*
 * we maintain the hash buckets in a length increasing order to
 * maximize hits since smaller words tend to be used more frequently.
 */
static char *MakeStr(str)
register char *str;
{
    int len, i;
    register HASHSTR *ptr,*after;
    register unsigned long hash;

    /* generate a hash and length */
    hash = HashStr(str,&len);
    i = hash % HASHSTRTABSIZ;

    /* see if we have it already */
    after = NULL;
    for(ptr = HashStrTable[i];ptr != NULL;ptr = ptr->Next) {
#if	HASHSTRACCT
	hashstrlcmp++;
#endif
	if (ptr->Len != len) {
	    /* if key is smaller then entry then done */
	    if (ptr->Len > len)
		break;
	    after = ptr;
	    continue;
	}
	else
	    after = ptr;
#if	HASHSTRACCT
	hashstrhcmp++;
#endif
	if (ptr->Hash != hash) continue;
#if	HASHSTRACCT
	hashstrscmp++;
#endif
#if	HASHSTRACCT
	if(hashstrcmp(ptr->Val,str)) continue;
	hashstrhits++;
#else
	if(strcmp(ptr->Val,str)) continue;
#endif
	ptr->Cnt++;
	return ptr->Val;
    }

#if	HASHSTRACCT
    hashstrmiss++;
    hashstrnew++;
    hashstrbcnt += len + 1;
#endif
#if	MEMACCT
    MemCntSTR++;
#endif	MEMACCT
    ptr = (HASHSTR *) Malloc(sizeof(HASHSTR)+len+1-4,1);
    ptr->Hash = hash;
    ptr->Cnt = 1;
    ptr->Len = len;
#if	MEMCHECK
    ptr->Code = MEMCHECKCODE;
#endif	MEMCHECK
    ptr->Next = NULL;
    strcpy(ptr->Val,str);

    if (after == NULL) {
	/* insert at head */
	ptr->Next = HashStrTable[i];
	HashStrTable[i] = ptr;
    }
    else {
	/* insert behind */
	ptr->Next = after->Next;
	after->Next = ptr;
    }

    return ptr->Val;
}

/*
 * increment the use count associated with a string
 */
static char *CopyStr(str)
register char *str;
{
    register HASHSTR *ptr;

    /* generate the pointer to the head */
    ptr = (HASHSTR *) (str - (sizeof(HASHSTR)-4));

#if	MEMCHECK
    /* check and see if code is vald */
    if (ptr->Code != MEMCHECKCODE || ptr->Len < 0) {
	fflush(stdout);
	fprintf(stderr,"copystr memory problem cod=%x (vs %x) len=%d '%s'\n",
		ptr->Code,MEMCHECKCODE,ptr->Len,str);
	exit(-1);
    }
#endif	MEMCHECK

    /* increment the pointer */
    ptr->Cnt++;

    return str;
}

/*
 * decrement the use count associated with a string
 */
static UnMakeStr(str)
register char *str;
{
    register HASHSTR *ptr;

    /* generate the pointer to the head */
    ptr = (HASHSTR *) (str - (sizeof(HASHSTR)-4));

#if	MEMCHECK
    /* check and see if code is vald */
    if (ptr->Code != MEMCHECKCODE || ptr->Len < 0) {
	fflush(stdout);
	fprintf(stderr,"unmkstr memory problem cod=%x (vs %x) len=%d '%s'\n",
		ptr->Code,MEMCHECKCODE,ptr->Len,str);
	exit(-1);
    }
#endif	MEMCHECK

    /* decrement the pointer */
    if (--ptr->Cnt < 0)
	fprintf(stderr,"String '%s' freed too many times\n",str);

}

/*****************************************************************
 *
 *		LEXICAL ANALYSIS RECORDS
 *
 *****************************************************************/

/* 
 * token types
 */
#define LEXUNKNOWN	-7	/* unknown token */
#define LEXUIDENT	-6	/* identifier, 1st character uppercase */
#define LEXLIDENT	-5	/* identifier, 1st character lowercase */
#define LEXNUMBER	-4	/* base 10 digit string */
#define LEXBSTRING	-3	/* base 2 digit string */
#define	LEXHSTRING	-2	/* base 16 digit/alpha string */
#define LEXCSTRING	-1	/* character string */
/* the above tokens use allocated memory, ones below do not */
#define LEXEOF		0	/* end of file */
/* keywords */
#define LEXBOOLEAN	1
#define LEXINTEGER	2
#define LEXBIT		3
#define LEXSTRING	4
#define LEXOCTET	5
#define LEXNULL		6
#define LEXSEQUENCE	7
#define LEXOF		8
#define	LEXSET		9
#define LEXIMPLICIT	10
#define LEXCHOICE	11
#define LEXANY		12
#define LEXEXTERNAL	13
#define LEXOBJECT	14
#define LEXIDENTIFIER	15
#define LEXOPTIONAL	16
#define LEXDEFAULT	17
#define LEXCOMPONENTS	18
#define LEXUNIVERSAL	19
#define LEXAPPLICATION	20
#define LEXPRIVATE	21
#define LEXTRUE		22
#define LEXFALSE	23
#define LEXBEGIN	24
#define LEXEND		25
#define LEXDEFINITIONS	26
/* keywords in MIBs and probably in ASN.1 */
#define LEXIMPORTS	27
#define LEXEXPORTS	28
#define LEXFROM		29
#define LEXSIZE		30

/* punctuation */
#define LEXASSIGN	50
#define LEXLEFTCURLY	51
#define LEXRIGHTCURLY	52
#define LEXLESS		53
#define LEXCOMMA	54
#define LEXPERIOD	55
#define LEXLEFTPAREN	56
#define LEXRIGHTPAREN	57
#define LEXLEFTBRACK	58
#define LEXRIGHTBRACK	59
#define LEXHYPEN	60
/* punctuation in MIBs and probably in ASN.1 */
#define LEXSEMI		61
#define LEXRANGE	62
/* macro keywords */
#define LEXMACRO	80
#define LEXTYPE		81
#define LEXNOTATION	82
#define LEXVALUE	83
/* macro punctation */
#define LEXALTERNATE	84
#define LEXGREATER	85

static FILE *CurFile=NULL;		/* the current file */
static char CurFileName[512];		/* the current file name */
static int CurLine=0;			/* the current line we are one */
static int LastPrint=0;			/* the last line we complained about */

static char PeekC[4];			/* char peak ahead buffer */
static int PeekC_Cnt = 0;		/* char peak ahead buffer count */

static char *TokenBufPtr = NULL;	/* token buffer */
static int TokenBufSiz = 0;		/* token buffer size */

static char *LBufPtr = NULL;		/* line buffer */
static int LBufSiz = 0;			/* line buffer size */
static char *LBufIn = NULL;		/* line buffer in pointer */
static char *LBufOut = NULL;		/* line buffer out pointer */
static int LBufCnt = 0;			/* line buffer active count */
static int LBufCode = 0;		/* line buffer term code */

static char *CurToken = NULL;		/* current token value */
static int CurTokenVal = LEXEOF;	/* current token lex value */
static char *NxtToken = NULL;		/* peek token value */
static int NxtTokenVal = LEXEOF;	/* peek token lex value */

static int MaxRWordLen = 0;		/* maximum reserved word length */

/* reserved word list */
static struct rsvdword {
    char *name;		   	/* reserved word value */
    int len;
    int val;
    int hash;
};

/* put most frequent keywords in front */
static struct rsvdword rsvdwords[] = {
/* very frequently making objects */
    {"OBJECT", sizeof("OBJECT")-1, LEXOBJECT, 0},
    {"IDENTIFIER", sizeof("IDENTIFIER")-1, LEXIDENTIFIER, 0},
/* usually the objects are integers */
    {"INTEGER", sizeof("INTEGER")-1, LEXINTEGER, 0},
/* or they are strings */
    {"SIZE", sizeof("SIZE")-1, LEXSIZE, 0},
    {"OCTET", sizeof("OCTET")-1, LEXOCTET, 0},
    {"STRING", sizeof("STRING")-1, LEXSTRING, 0},
    {"SEQUENCE", sizeof("SEQUENCE")-1, LEXSEQUENCE, 0},
    {"OF", sizeof("OF")-1, LEXOF, 0},
/* modles and macros */
    {"BEGIN", sizeof("BEGIN")-1, LEXBEGIN, 0},
    {"END", sizeof("END")-1, LEXEND, 0},
/* used when a module is defined */
    {"DEFINITIONS", sizeof("DEFINITIONS")-1, LEXDEFINITIONS, 0},
    {"IMPORTS", sizeof("IMPORTS")-1, LEXIMPORTS, 0},
    {"EXPORTS", sizeof("EXPORTS")-1, LEXEXPORTS, 0},
    {"FROM", sizeof("FROM")-1, LEXFROM, 0},
/* macro definitions and weird types */
    {"MACRO", sizeof("MACRO")-1, LEXMACRO, 0},
    {"TYPE", sizeof("TYPE")-1, LEXTYPE, 0},
    {"NOTATION", sizeof("NOTATION")-1, LEXNOTATION, 0},
    {"VALUE", sizeof("VALUE")-1, LEXVALUE, 0},
    {"APPLICATION", sizeof("APPLICATION")-1, LEXAPPLICATION, 0},
/* never should be used in MIB */
    {"BOOLEAN", sizeof("BOOLEAN")-1, LEXBOOLEAN, 0},
    {"BIT", sizeof("BIT")-1, LEXBIT, 0},
    {"NULL", sizeof("NULL")-1, LEXNULL, 0},
    {"SET", sizeof("SET")-1, LEXSET, 0},
    {"IMPLICIT", sizeof("IMPLICIT")-1, LEXIMPLICIT, 0},
    {"CHOICE", sizeof("CHOICE")-1, LEXCHOICE, 0},
    {"ANY", sizeof("ANY")-1, LEXANY, 0},
    {"EXTERNAL", sizeof("EXTERNAL")-1, LEXEXTERNAL, 0},
    {"OPTIONAL", sizeof("OPTIONAL")-1, LEXOPTIONAL, 0},
    {"DEFAULT", sizeof("DEFAULT")-1, LEXDEFAULT, 0},
    {"COMPONENTS", sizeof("COMPONENTS")-1, LEXCOMPONENTS, 0},
    {"UNIVERSAL", sizeof("UNIVERSAL")-1, LEXUNIVERSAL, 0},
    {"PRIVATE", sizeof("PRIVATE")-1, LEXPRIVATE, 0},
    {"TRUE", sizeof("TRUE")-1, LEXTRUE, 0},
    {"FALSE", sizeof("FALSE")-1, LEXFALSE, 0}
};
#define NUMRSVDWORDS	(sizeof(rsvdwords)/sizeof(struct rsvdword))

/*****************************************************************
 *
 *		LEXICAL ANALYSIS ROUTINES
 *
 *****************************************************************/

/*
 * allocate token memory
 */
#if	MEMACCT
static char *TokLoad(str)
char *str;
{
    MemCntTokLoad++;
    return MakeStr(str);
}
static TokLoadFree(x)
char *x;
{
    MemCntTokLoad--;
    RelCntTokLoad++;
    UnMakeStr(x);
}
#else
#define TokLoad(x) MakeStr(x)
#define TokLoadFree(x) UnMakeStr(x)
#endif

/*
 * open the file up we use
 */
static int OpenTok(filenam)
char *filenam;
{
    FILE *tmpfile;

    /* set up token buffer */
    if (TokenBufPtr == NULL) {
	TokenBufSiz = MAXTOKENSIZ;
	TokenBufPtr = (char *) Malloc(TokenBufSiz,1);
    }
    /* set up line buffer */
    if (LBufPtr == NULL) {
	LBufSiz = MAXLINESIZ;
	LBufPtr = (char *) Malloc(LBufSiz,1);
	LBufOut = LBufIn = LBufPtr;
	LBufCode = 0;
    }
    /* open the file */
    tmpfile = fopen(filenam,"r");
    if (tmpfile == NULL)
	return -1;

    /* set up */
    CurFile = tmpfile;
    *CurFileName = '\0';
    CurLine = 1;
    LastPrint = 0;
    strncat(CurFileName,filenam,sizeof(CurFileName));

    /* return success */
    return 0;
}

/*
 * close the file we use
 */
static CloseTok()
{
    /* release token buffer */
    if (TokenBufPtr != NULL) {
	TokenBufSiz = 0;
	Free(TokenBufPtr);
	TokenBufPtr = NULL;
    }
    /* release line buffer */
    if (LBufPtr != NULL) {
	LBufSiz = 0;
	Free(LBufPtr);
	LBufOut = LBufPtr = NULL;
    }

    /* no file */
    if (CurFile != NULL)
	fclose(CurFile);
    CurFile = NULL;
    *CurFileName = '\0';
}

/*
 * parse error message routine provides line information.
 * ParseErr(module,flag,fmt,args...)
 *
 * module - name of module calling this routine
 * flag - 2=fatal,1=error,0=warning
 * fmt,args - printf error message
 */
/*VARARGS*/
static ParseErr(va_alist)
va_dcl
{
    va_list ap;
    int flag;
    register char *p;
    char *module,*fmt;
    char msgbuf[BUFSIZ], submsgbuf[BUFSIZ/2];

    /* collect arguments */
    va_start(ap);
    module = va_arg(ap,char *);
    flag = va_arg(ap,int);
    fmt = va_arg(ap,char *);
    vsprintf(msgbuf, fmt, ap);
    va_end(ap);

    /* generate sub message */
    sprintf(submsgbuf,"%s, line %d: ",CurFileName,CurLine);

    /* generate next line message if different */
    p = NULL;
    if (CurLine != LastPrint) {
	if (LBufPtr != NULL && *LBufPtr != '\0') {
	    p = LBufPtr;
	    while (*p && isspace(*p)) p++;
	}
	LastPrint = CurLine;
    }

    PrintError(module,flag,submsgbuf,msgbuf,p);
}

/*
 * keep a peek buffer
 */
#define UnGetC(x) (x != EOF ? PeekC[PeekC_Cnt++] = x : 0)

/*
 * get a character from the input stream maintaing a line
 * buffer for error messages...slower but gets the job done.
 */
static int _GetC()
{
    register int ch;
    register char *p,*ep;

    /* check peek ahead buffer */
    if (PeekC_Cnt > 0)
	return PeekC[--PeekC_Cnt];

    /* load more characters up */
    for(;;) {
	/* see if we have a terminating character */
	if ((ch = LBufCode) == '\n') {
	    CurLine++;
	    LBufCode = 0;
	    return ch;
	}
	else if (ch == EOF)
	    return ch;

	/* load up a line */
	ep = p = LBufIn = LBufPtr;
	ep += LBufSiz-1;
	while (p < ep) {
	    ch = getc(CurFile);
	    if (ch == '\n' || ch == EOF) {
		LBufCode = ch;
		break;
	    }
	    *p++ = ch;
	}
	*p = '\0';
#if	DODEBUG
	if (LexerDebug>1) printf("LINE%d: %s\n",CurLine,LBufPtr);
#endif	DODEBUG

	/* see if we have any data */
	LBufOut = p;
	LBufCnt = p - LBufPtr;
	if (--LBufCnt >= 0)
	    return (*LBufIn++);

    }
}
#define GetC() (PeekC_Cnt <= 0 && --LBufCnt >= 0 ? *LBufIn++ : _GetC())

/*
 * Parses a token from the input stream and determines the type of
 * token.
 */
static int Load_Token(TokPtr)
char **TokPtr;
{
    static char *mod = "Load_Token";
    register int ch;
    register char *cp, *ep;
    char *token;
    int overflow;

    /* handle white space and comments */
    for (;;) {
	/* process new characters until no white space */
	do {
	    ch = GetC();
	} while (isspace(ch));
	/* non-white space found, check for comment */
	if (ch != '-') 
	    break;	/* not a comment */
	/* process the possible comment */
	ch = GetC();
	if (ch != '-') {
	    /* not a double hypen, backup and exit */
	    UnGetC(ch);
	    ch = '-';
	    break;
	}
	/* it is a comment, eat until EOL, EOF or double hypen */
	for(;;) {
	    ch = GetC();
	    /* see if we hit EOL or EOF */
	    if (ch == EOF || ch == '\n')
		break;	/* yep terminate */
	    /* possible end comment */
	    if (ch == '-') {
		ch = GetC();
		if (ch == '-')
		    break;
		/* else embedded hypen */
	    }
	}
	/* loop again processing new characters */
    }

    /* check for EOF */
    if (ch == EOF) {
	*TokPtr = " >>EOF<< ";
	return LEXEOF;
    }

    /* set up for token load */
    token = cp = TokenBufPtr;
    ep = token + TokenBufSiz-1;
    overflow = 0;

    /* parse the token */
    if (isalpha(ch)) {
	register long hash;

	/* looks like an identifier or type reference */
	/* NOTE: we allow underscore even through ASN.1 does not */
	hash = 0;
	for(;;) {
	    if (cp < ep) {
		hash = hash<<1 + ch;
		*cp++ = ch;
	    }
	    else if (!overflow) {
		ParseErr(mod,0,"Buffer Overflow for identifier");
		overflow++;
	    }
	    ch = GetC();
	    if (isalnum(ch))
		continue;
	    /* we have a possible break condition */
	    if (ch == '-' || ch == '_') {
		int oldch;
		oldch = ch;
		/* hypen or underscore, must seperate */
		ch = GetC();
		if (isalnum(ch)) {
		    /* record hypen */
		    if (cp < ep) {
			hash = hash<<1 + oldch;
			*cp++ = oldch;
		    }
		    else if (!overflow) {
			ParseErr(mod,0,"Buffer Overflow for identifier");
			overflow++;
		    }
		    /* continue and record charatcer */
		    continue;
		}
		/* ooops */
		UnGetC(ch);
		ch = oldch;
	    }
	    /* not a valid character - break */
	    UnGetC(ch);
	    *cp = '\0';
	    break;
	}
	/* determine type of identifier */
	if (isupper(*token)) {
	    register int len;
	    register struct rsvdword *rp;
	    /* if small token then check reserved word table */
	    if ((len = cp-token) <= MaxRWordLen) {
		for(rp = rsvdwords; rp < &rsvdwords[NUMRSVDWORDS]; rp++) {
		    if (hash != rp->hash) continue;
		    if (rp->len != len) continue;
		    if (strcmp(rp->name,token) != 0) continue;
		    *TokPtr = rp->name;
		    return rp->val;
		}
	    }
	    /* must be type identifier */
	    *TokPtr = TokLoad(token);
	    return LEXUIDENT;
	}
	else {
	    *TokPtr = TokLoad(token);
	    return LEXLIDENT;
	}
    }
    else if (isdigit(ch)) {
	/* must be a number */
	/* tolerate leading zeroes even though ASN.1 does not */
	do {
	    if (cp < ep)
		*cp++ = ch;
	    else if (!overflow) {
		ParseErr(mod,0,"Buffer Overflow for number");
		overflow++;
	    }
	    ch = GetC();
	} while (isdigit(ch));
	UnGetC(ch);
	*cp = '\0';
	*TokPtr = TokLoad(token);
	return LEXNUMBER;
    }
    else if (ch == '"') {
	/* must be a quoted string (cstring) */
	register int EatSpace, LineCnt;
	EatSpace = 2;
	LineCnt = 0;
	/* load it up with compression */
	for(;;) {
	    /* get a character */
	    ch = GetC();
	    /* see if we are done */
	    if (ch == EOF)
		break;
	    if (ch == '"') {
		/* see if this terminates (double quote does not follow) */
		ch = GetC();
		if (ch != '"') {
		    UnGetC(ch);
		    break;	/* yep */
		}
		/* nope, add the second quote since it is escaped */
	    }
	    /* add it using compression */
	    if (cp < ep) switch(EatSpace) {
		case 0:
		    /* inside of a word */
		    if (isspace(ch)) {
			/* break out of word */
			EatSpace = 1;
			if (ch == '\n') LineCnt++;
			continue;
		    }
		    else
			*cp++ = ch;	/* word continues */
		    break;
		case 1:
		    /* trailing space on a word */
		    if (isspace(ch)) {
			if (ch == '\n') LineCnt++;
			continue;
		    }
		    /* hit another word, add seperator */
		    if (LineCnt) {
			if (LineCnt > 1)
			    *cp++ = '\n';
			*cp++ = '\n';
		    }
		    else
			*cp++ = ' ';
		    *cp++ = ch;
		    LineCnt = EatSpace = 0;
		    break;
		case 2:
		    /* ignore leading white space */
		    if (isspace(ch))
			continue;
		    /* found first character */
		    *cp++ = ch;
		    LineCnt = EatSpace = 0;
		    break;
	    }
	    else if (!overflow) {
		ParseErr(mod,0,"Buffer Overflow for quoted string");
		overflow++;
	    }
	};
	/* all done, terminate cstring */
	*cp = '\0';
	*TokPtr = TokLoad(token);
	return LEXCSTRING;
    }
    else if (ch == '\'') {
	/* must be a hex or bit string */
	for(;;) {
	    ch = GetC();
	    if (!isxdigit(ch))
		break;
	    if (cp < ep)
		*cp++ = ch;
	    else if (!overflow) {
		ParseErr(mod,0,"Buffer Overflow for bit or hex string");
		overflow++;
	    }
	}
	*cp = '\0';
	if (ch == '\'') {
	    ch = GetC();
	    if (ch == 'H' || ch == 'h') {
		*TokPtr = TokLoad(token);
		return LEXHSTRING;
	    }
	    if (ch == 'B' || ch == 'b') {
		*TokPtr = TokLoad(token);
		return LEXBSTRING;
	    }
	}
	ParseErr(mod,1,"Malformed bit or hex string '%s'",token);
	UnGetC(ch);
	*TokPtr = TokLoad(token);
	return LEXUNKNOWN;
    }
    else {
	/* must be punctuation */
	*cp++ = ch;
	*cp = '\0';
	switch (ch) {
	    case '{':
		*TokPtr = "{";
		return LEXLEFTCURLY;
	    case '}':
		*TokPtr = "}";
		return LEXRIGHTCURLY;
	    case ',':
		*TokPtr = ",";
		return LEXCOMMA;
	    case '<':
		*TokPtr = "<";
		return LEXLESS;
	    case '>':
		*TokPtr = ">";
		return LEXGREATER;
	    case '.':
		ch = GetC();
		if (ch == '.') {
		    *cp++ = ch;
		    *cp = '\0';
		    *TokPtr = "..";
		    return LEXRANGE;
		}
		else
		    UnGetC(ch);
		*TokPtr = ".";
		return LEXPERIOD;
	    case '(':
		*TokPtr = "(";
		return LEXLEFTPAREN;
	    case ')':
		*TokPtr = ")";
		return LEXRIGHTPAREN;
	    case '[':
		*TokPtr = "[";
		return LEXLEFTBRACK;
	    case ']':
		*TokPtr = "]";
		return LEXRIGHTBRACK;
	    case '-':
		*TokPtr = "-";
		return LEXHYPEN;
	    case '|':
		*TokPtr = "|";
		return LEXALTERNATE;
	    case ':':
		ch = GetC();
		if (ch == ':') {
		    ch = GetC();
		    if (ch == '=') {
			*cp++ = ':';
			*cp++ = '=';
			*cp = '\0';
			*TokPtr = "::=";
			return LEXASSIGN;
		    }
		    else
			UnGetC(ch);
		}
		else
		    UnGetC(ch);
		break;
	    case ';':
		*TokPtr = ";";
		return LEXSEMI;
	}
	ParseErr(mod,1,"Unknown token '%s'",token);
	*TokPtr = TokLoad(token);
	return LEXUNKNOWN;
    }
}

/* init hash values in reserved words */
static InitRWords()
{
    register struct rsvdword *rp;
    register int hash;

    for(rp = rsvdwords; rp < &rsvdwords[NUMRSVDWORDS]; rp++) {
	register char *p;
	hash = 0;
	p = rp->name;
	while (*p)
	    hash = hash<<1 + *p++;
	rp->hash = hash;
	if (rp->len > MaxRWordLen) MaxRWordLen = rp->len;
    }
}

/* the decode token routine for humans */
static char *StrTok(val)
int val;
{
    char *p;
    p = "??";
    switch(val) {
	case LEXUNKNOWN:	return("Unknown token");
	case LEXUIDENT:		return("Type identifier");
	case LEXLIDENT:		return("Value identifier");
	case LEXNUMBER:		return("Number");
	case LEXBSTRING:	return("Bit string");
	case LEXHSTRING:	return("Hex string");
	case LEXCSTRING:	return("Character string");
	case LEXEOF:		return("End of File");
	/* keywords */
	case LEXBOOLEAN:	return("BOOLEAN");
	case LEXINTEGER:	return("INTEGER");
	case LEXBIT:		return("BIT");
	case LEXSTRING:		return("STRING");
	case LEXOCTET:		return("OCTET");
	case LEXNULL:		return("NULL");
	case LEXSEQUENCE:	return("SEQUENCE");
	case LEXOF:		return("OF");
	case LEXSET:		return("SET");
	case LEXIMPLICIT:	return("IMPLICIT");
	case LEXCHOICE:		return("CHOICE");
	case LEXANY:		return("ANY");
	case LEXEXTERNAL:	return("EXTERNAL");
	case LEXOBJECT:		return("OBJECT");
	case LEXIDENTIFIER:	return("IDENTIFIER");
	case LEXOPTIONAL:	return("OPTIONAL");
	case LEXDEFAULT:	return("DEFAULT");
	case LEXCOMPONENTS:	return("COMPONENTS");
	case LEXUNIVERSAL:	return("UNIVERSAL");
	case LEXAPPLICATION:	return("APPLICATION");
	case LEXPRIVATE:	return("PRIVATE");
	case LEXTRUE:		return("TRUE");
	case LEXFALSE:		return("FALSE");
	case LEXBEGIN:		return("BEGIN");
	case LEXEND:		return("END");
	case LEXDEFINITIONS:	return("DEFINITIONS");
	/* keywords in MIBs and probably in ASN.1 */
	case LEXIMPORTS:	return("IMPORTS");
	case LEXEXPORTS:	return("EXPORTS");
	case LEXFROM:		return("FROM");
	case LEXSIZE:		return("SIZE");
	/* punctuation */
	case LEXASSIGN:		return("\"::=\"");
	case LEXLEFTCURLY:	return("\"{\"");
	case LEXRIGHTCURLY:	return("\"}\"");
	case LEXLESS:		return("\"<\"");
	case LEXCOMMA:		return("\",\"");
	case LEXPERIOD:		return("\".\"");
	case LEXLEFTPAREN:	return("\"(\"");
	case LEXRIGHTPAREN:	return("\")\"");
	case LEXLEFTBRACK:	return("\"[\"");
	case LEXRIGHTBRACK:	return("\"]\"");
	case LEXHYPEN:		return("\"-\"");
	/* punctuation in MIBs and probably in ASN.1 */
	case LEXSEMI:		return("\";\"");
	case LEXRANGE:		return("\"..\"");
	/* macro keywords */
	case LEXMACRO:		return("MACRO");
	case LEXTYPE:		return("TYPE");
	case LEXNOTATION:	return("NOTATION");
	case LEXVALUE:		return("VALUE");
	/* macro punctation */
	case LEXALTERNATE:	return("\"|\"");
	case LEXGREATER:	return("\">\"");
    }
    return p;
}

#if	DODEBUG
/*
 * display a token
 */
static ShowTok(prfx,lex,str)
char *prfx;
int lex;
char *str;
{
    register char *p;
    p = StrTok(lex);
    printf("%s '%.20s' [%s]\n",prfx,str,p);
}
#endif	DODEBUG

/*
 * get a token, return the type associated with the token
 */
static int _GetTok()
{
    static char *mod = "GetTok";

    /* see if we have a valid token and if so return */
    if (CurToken != NULL) {
#if	DODEBUG
	if (LexerDebug>4) ShowTok(mod,CurTokenVal,CurToken);
#endif	DODEBUG
	return CurTokenVal;
    }
    /* see if we have a valid peek token, shit and return */
    if (NxtToken != NULL) {
	CurToken = NxtToken;
	NxtToken = NULL;
	CurTokenVal = NxtTokenVal;
#if	DODEBUG
	NxtTokenVal = LEXEOF;
	if (LexerDebug>4) ShowTok(mod,CurTokenVal,CurToken);
#endif	DODEBUG
	return CurTokenVal;
    }
    /* no current or peek, load one */
    CurTokenVal = Load_Token(&CurToken);
#if	DODEBUG
    if (LexerDebug) ShowTok(mod,CurTokenVal,CurToken);
#endif	DODEBUG
    return CurTokenVal;
}
#define GetTok() (CurToken ? CurTokenVal : _GetTok())

/*
 * one token lookahead, return the type associated with the next token
 */
static int _PeekTok()
{
    static char *mod = "PeekTok";

    /* see if we have a get token first */
    if (CurToken == NULL)
	GetTok();
    /* see if we have a valid peek token */
    if (NxtToken != NULL) {
#if	DODEBUG
	if (LexerDebug>2) ShowTok(mod,NxtTokenVal,NxtToken);
#endif	DODEBUG
	return NxtTokenVal;
    }
    /* no peek, load one */
    NxtTokenVal = Load_Token(&NxtToken);
#if	DODEBUG
    if (LexerDebug) ShowTok(mod,NxtTokenVal,NxtToken);
#endif	DODEBUG
    return (NxtTokenVal);
}
#define PeekTok() (NxtToken && CurToken ? NxtTokenVal : _PeekTok())

/*
 * eat the current token, allowing caller to keep the allocated memory
 */
static EatTok()
{
    static char *mod = "EatTok";

    /* get rid of current token */
    if (CurToken != NULL) {
#if	DODEBUG
	if (LexerDebug>3) ShowTok(mod,CurTokenVal,CurToken);
#endif	DODEBUG
	if (CurTokenVal < 0)
	    TokLoadFree(CurToken);
	CurTokenVal = LEXEOF;
	CurToken = NULL;
    }
    /* preload the next token */
    GetTok();
}

/*****************************************************************
 *		SYMBOL TABLE RECORD DEFINITIONS
 *****************************************************************
 */

/* The Macro Record is designed to efficiently handle macro definitions
 * by speeding up parsing at the expense of creating the initial structure
 */
static struct MacroParse {
    struct MacroParse *Next;	/* allocation chain */
    struct MacroParse *Prev;	/* allocation chain */
    int Code;			/* type of macro component */
    char *Symbol;		/* symbolic name if any */
    int Hit;			/* non-zero if this part has been parsed */
    int Recurse;		/* non-zero if a recursive component */
    union {
/* a sequential list of macro components to parse */
#define MACRO_LIST	1
	struct {
	    int Cnt;			/* count of macro parts in list */
	    struct MacroParse **List;	/* list of macro parts */
	} List;
/* a list of macro components to be parse with at least one being done */
#define MACRO_ALT	2
	struct {
	    int Cnt;			/* count of macro parts in list */
	    struct MacroParse **List;	/* list of macro parts */
	} Alt;
/* this is a lexical token */
#define MACRO_LEX	3
	char *Lexeme;
/* this is a type definition */
#define MACRO_TYPE	4
	struct {
	    int IsType;			/* non-zero if TYPE */
	    char *TName;		/* type name */
	    struct SymbolNode *Type;	/* type */
	} Type;
/* this is a value definition */
#define MACRO_VALUE	5
	struct {
	    int IsValue;		/* non-zero if VALUE */
	    char *Name;			/* name of the value */
	    char *TName;		/* type name */
	    struct SymbolNode *Type;	/* type */
	} Value;
/* this is a temporary symbol definition which is eliminated */
#define MACRO_SYMBOL	6
	struct MacroParse *SymTop;
/* this is a empty defintion which can be executed once */
#define MACRO_EMPTY	7
    } Part;
};
typedef struct MacroParse MPARSE;

/*
 * symbol record is designed to represent both values and types at the
 * expense of memory.
 */
static struct SymbolNode {
/* allocation and deallocation circular list pointers and marker */
    struct SymbolNode *AllocNext;
    struct SymbolNode *AllocPrev;
    int AllocMark;
/* association */
    struct SymbolNode *Assoc;		/* associated symbol information */
/* symbol's name and internal code */
    char *Name;				/* type or value name */
    int Code;				/* symbol code */
    union {				/* symbolic description */
/* types */
#define SYM_INTEGER	1	/* INTEGER */
	struct {
	    long Min;				/* min */
	    long Max;				/* max */
	    int HasRange;			/* has min/max */
	    struct SymbolNode *NamedList;	/* list of named values */
	    struct enum_list *EnumList;		/* non-null enum list */
	} TypInt;
#define SYM_OSTRING	2	/* OCTET STRING */
	struct {
	    long MinSiz;			/* min size */
	    long MaxSiz;			/* min size */
	    int HasSize;			/* has min/max */
	} TypOStr;
#define SYM_NULL	3	/* NULL */
#define SYM_SEQ		4	/* SEQUENCE */
	struct {
	    char *Sname;
	    struct SymbolNode *TypPtr;
	    struct SymbolNode *SNext;
	} Seq;
#define SYM_SEQOF	5	/* SEQUENCE OF */
	struct SymbolNode *TypSeqOf;
#define SYM_CHOICE	6	/* CHOICE */
	struct {
	    char *Cname;
	    struct SymbolNode *TypPtr;
	    struct SymbolNode *CNext;
	} Choice;
#define SYM_OBJID	7	/* OBJECT IDENTIFIER */
#define SYM_MACRO	8	/* MACRO */
	struct {
	    MPARSE *TypPart;
	    MPARSE *ValPart;
	} Macro;
/* values */
#define SYM_VALINTEGER	9	/* value INTEGER */
	struct {
	    long val;
	    char *id;
	    struct SymbolNode *vnext;
	} ValInt;
#define SYM_VALOSTRING	10	/* value STRING */
	char *ValStr;
#define	SYM_VALNULL	11	/* value NULL */
#define SYM_VALOBJID	12	/* value OBJ ID */
	struct {
	    char *idstr;
	    long idnum;
	    struct SymbolNode *idnext;
	} ValObjId;
#define SYM_VALNAME	13	/* named value */
	struct {
	    char *namedstr;
	    struct SymbolNode *namedtyp;
	} ValName;
#define SYM_DELAY	14	/* delayed type binding */
	struct {
	    int LineNum;
	} Delay;
#define SYM_VALDELAY	15	/* delayed value binding */
	struct {
	    struct SymbolNode *Type;
	    int LineNum;
	} ValDelay;
#define SYM_VALMACRO	16	/* macro string, type or value */
	struct {
	    char *partlex;
	    struct SymbolNode *parttype;
	    struct SymbolNode *partvalue;
	    struct SymbolNode *partvaluet;
	} ValMacro;
    } Desc;
};
typedef struct SymbolNode SNODE;

/*
 * type name record (associates a name to type)
 */
static struct TypeName {
    char *Name;				/* type or value name */
    struct TypeName *Next;		/* next entry in table */
    struct SymbolNode *TypPtr;		/* type pointer */
};
typedef struct TypeName TNAME;

/*
 * value name record (associates a name to value with a type)
 */
static struct ValueName {
    char *Name;				/* type or value name */
    struct ValueName *Next;		/* next entry in table */
    struct SymbolNode *TypPtr;		/* type pointer */
    struct SymbolNode *ValPtr;		/* value pointer */
    struct tree *TreePtr;		/* tree node pointer for this name */
};
typedef struct ValueName VNAME;

/*
 * import/export record
 */
static struct IENode {
    struct IENode *Next;
    char *Name;				/* type or value name */
};
typedef struct IENode INODE;

/*
 * module context
 */
static struct ModContext {
    struct ModContext *Next;
    char *Name;
    TNAME *Types;		/* Types and Macros */
    VNAME *Values;		/* Values */
    INODE *Exports;		/* Export symbols */
};
typedef struct ModContext MNODE;

/*
 * delay record
 */
static struct DelayNode {
    struct DelayNode *Next;
    char *Name;			/* optional name */
    SNODE *TypPtr;		/* pointer to type record */
    SNODE *ValPtr;		/* pointer to value record */
};
typedef struct DelayNode DNODE;

/*****************************************************************
 *		SYMBOL/MACRO MEMORY ROUTINES
 *****************************************************************
 */

static MPARSE *MacroHead = NULL;	/* head of macro parse records */
static SNODE *SymbolHead = NULL;	/* head of symbol records */
static MNODE *TopContext = NULL;	/* head of modules records */
static MNODE *CurContext = NULL;	/* current module record */
static DNODE *TDelayList = NULL;	/* head of delayed type symbols */
static DNODE *VDelayList = NULL;	/* head of delayed value symbols */

static int InMacro = 0;			/* non-zero for in a macro */
static TNAME *MacroTypes = NULL;	/* our local macro types */

/*
 * make/free a delay record
 */
#if	MEMACCT
static DNODE *MakeDNODE()
{
    MemCntDNODE++;
    return (DNODE *) Malloc(sizeof(DNODE),0);
}
static FreeDNODE(x)
{
    MemCntDNODE--;
    RelCntDNODE++;
    Free(x);
}
#else
#define MakeDNODE() (DNODE *) Malloc(sizeof(DNODE),0);
#define FreeDNODE(x) Free(x)
#endif

/*
 * Make a macro parse record
 */
static MPARSE *MakeMPARSE()
{
    register MPARSE *ptr;
#if	MEMACCT
    MemCntMPARSE++;
#endif
    ptr = (MPARSE *) Malloc(sizeof(MPARSE),0);
    if (MacroHead == NULL)
	ptr->Next = ptr->Prev = MacroHead = ptr;
    else {
	/* insert tail */
	ptr->Next = MacroHead;
	ptr->Prev = MacroHead->Prev;
	(MacroHead->Prev)->Next = ptr;
	MacroHead->Prev = ptr;
    }
    return ptr;
}

/*
 * free a macro parse record
 */
static FreeMPARSE(ptr)
register MPARSE *ptr;
{
    static char mod[] = "FreeMPARSE";
    if (MacroHead == NULL)
	PrintErr(mod,2,"macro parse list is empty");
    if (MacroHead == MacroHead->Next) {
	if (MacroHead != ptr)
	    PrintErr(mod,2,"macro parse not part of list");
	MacroHead = NULL;
    }
    else {
	if (MacroHead == ptr)
	    MacroHead = ptr->Next;
	(ptr->Prev)->Next = ptr->Next;
	(ptr->Next)->Prev = ptr->Prev;
    }
#if	MEMACCT
    MemCntMPARSE--;
    RelCntMPARSE++;
#endif
    Free(ptr);
}

/*
 * make a symbol record
 */
static SNODE *MakeSNODE()
{
    register SNODE *ptr;
#if	MEMACCT
    MemCntSNODE++;
#endif
    ptr = (SNODE *) Malloc(sizeof(SNODE),0);
    if (SymbolHead == NULL)
	ptr->AllocNext = ptr->AllocPrev = SymbolHead = ptr;
    else {
	/* insert tail */
	ptr->AllocNext = SymbolHead;
	ptr->AllocPrev = SymbolHead->AllocPrev;
	(SymbolHead->AllocPrev)->AllocNext = ptr;
	SymbolHead->AllocPrev = ptr;
    }
    return ptr;
}

/*
 * free a symbol record
 */
static FreeSNODE(ptr)
register SNODE *ptr;
{
    static char mod[] = "FreeSNODE";
    if (SymbolHead == NULL)
	PrintErr(mod,2,"symbol node list is empty");
    if (SymbolHead == SymbolHead->AllocNext) {
	if (SymbolHead != ptr)
	    PrintErr(mod,2,"symbol node not part of list");
	SymbolHead = NULL;
    }
    else {
	if (SymbolHead == ptr)
	    SymbolHead = ptr->AllocNext;
	(ptr->AllocPrev)->AllocNext = ptr->AllocNext;
	(ptr->AllocNext)->AllocPrev = ptr->AllocPrev;
    }
#if	MEMACCT
    MemCntSNODE--;
    RelCntSNODE++;
#endif
    Free(ptr);
}

/*
 * make/free a type name record
 */
#if	MEMACCT
static TNAME *MakeTNAME()
{
    MemCntTNAME++;
    return (TNAME *) Malloc(sizeof(TNAME),0);
}
static FreeTNAME(x)
{
    MemCntTNAME--;
    RelCntTNAME++;
    Free(x);
}
#else
#define MakeTNAME() (TNAME *) Malloc(sizeof(TNAME),0);
#define FreeTNAME(x) Free(x)
#endif

/*
 * make/free a value name record
 */
#if	MEMACCT
static VNAME *MakeVNAME()
{
    MemCntVNAME++;
    return (VNAME *) Malloc(sizeof(VNAME),0);
}
static FreeVNAME(x)
{
    MemCntVNAME--;
    RelCntVNAME++;
    Free(x);
}
#else
#define MakeVNAME() (VNAME *) Malloc(sizeof(VNAME),0);
#define FreeVNAME(x) Free(x)
#endif

/*
 * make/free an import/export record
 */
#if	MEMACCT
static INODE *MakeINODE()
{
    MemCntINODE++;
    return (INODE *) Malloc(sizeof(INODE),0);
}
static FreeINODE(x)
{
    MemCntINODE--;
    RelCntINODE++;
    Free(x);
}
#else
#define MakeINODE() (INODE *) Malloc(sizeof(INODE),0);
#define FreeINODE(x) Free(x)
#endif

/*****************************************************************
 *		SYMBOL/MACRO SUPPORT ROUTINES
 *****************************************************************
 */

#if	DODEBUG
/*
 * dump a symbol record
 */
static SymDump(prfx,Ptr,lvl)
char *prfx;
SNODE *Ptr;
int lvl;
{
    register int i;
    for(i=0;i<lvl;i++)
	fputs("  ",stdout);
    printf("%s",prfx);
    if (Ptr != NULL) {
	if (Ptr->Name != NULL)
	    printf(" Name:'%s'",Ptr->Name);
	if (Ptr->Assoc != NULL)
	    printf(" (A->%lx)",Ptr->Assoc);
    }
    printf(" %lx->",Ptr);
    if (Ptr == NULL) {
	printf("\n");
	return;
    }
    switch(Ptr->Code) {
	case SYM_INTEGER:
	    printf(" INTEGER");
	    if (Ptr->Desc.TypInt.HasRange)
		printf(" (RANGE %u => %u)",
		       Ptr->Desc.TypInt.Min,
		       Ptr->Desc.TypInt.Max);
	    if (Ptr->Desc.TypInt.NamedList != NULL) {
		printf(" ->%lx\n",Ptr->Desc.TypInt.NamedList);
		SymDump(prfx,Ptr->Desc.TypInt.NamedList,lvl+1);
	    }
	    else
		printf("\n");
	    break;
	case SYM_OSTRING:
	    printf(" OSTRING");
	    if (Ptr->Desc.TypOStr.HasSize)
		printf(" (SIZE %d => %d)",
		       Ptr->Desc.TypOStr.MinSiz,Ptr->Desc.TypOStr.MaxSiz);
	    printf("\n");
	    break;
	case SYM_NULL:
	    printf(" NULL\n");
	    break;
	case SYM_SEQ:
	    printf(" SEQ");
	    if (Ptr->Desc.Seq.Sname)
		printf(" '%s'",Ptr->Desc.Seq.Sname);
	    if (Ptr->Desc.Seq.SNext != NULL) {
		printf(" ->%lx\n",Ptr->Desc.Seq.SNext);
		SymDump(prfx,Ptr->Desc.Seq.SNext,lvl+1);
	    }
	    else
		printf("\n");
	    break;
	case SYM_SEQOF:
	    printf(" SEQOF ->%lx\n",Ptr->Desc.TypSeqOf);
	    if (Ptr->Desc.TypSeqOf != NULL)  {
		SymDump(prfx,Ptr->Desc.TypSeqOf,lvl+1);
	    }
	    else
		printf("\n");
	    break;
	case SYM_CHOICE:
	    printf(" CHOICE");
	    if (Ptr->Desc.Choice.Cname)
		printf(" '%s'",Ptr->Desc.Choice.Cname);
	    if (Ptr->Desc.Choice.CNext != NULL) {
		printf(" ->%lx\n",Ptr->Desc.Choice.CNext);
		SymDump(prfx,Ptr->Desc.Choice.CNext,lvl+1);
	    }
	    else
		printf("\n");
	    break;
	case SYM_OBJID:
	    printf(" OBJID\n");
	    break;
	case SYM_MACRO:
	    printf(" MACRO Type->%lx Value->%lx\n",
		   Ptr->Desc.Macro.TypPart,Ptr->Desc.Macro.ValPart);
	    break;
	case SYM_VALINTEGER:
	    printf(" VALINT");
	    printf(" %ld",Ptr->Desc.ValInt.val);
	    if (Ptr->Desc.ValInt.id) printf(" '%s'",Ptr->Desc.ValInt.id);
	    if (Ptr->Desc.ValInt.vnext != NULL) {
		printf(" ->%lx\n",Ptr->Desc.ValInt.vnext);
		SymDump(prfx,Ptr->Desc.ValInt.vnext,lvl+1);
	    }
	    else
		printf("\n");
	    break;
	case SYM_VALOSTRING:
	    printf(" VALSTR");
	    if (Ptr->Desc.ValStr) printf(" '%.20s'",Ptr->Desc.ValStr);
	    printf("\n");
	    break;
	case SYM_VALOBJID:
	    printf(" VALOBJID");
	    if (Ptr->Desc.ValObjId.idstr)
		printf(" '%s'",Ptr->Desc.ValObjId.idstr);
	    printf(" (%ld)",Ptr->Desc.ValObjId.idnum);
	    if (Ptr->Desc.ValObjId.idnext) {
		printf(" ->%lx\n",Ptr->Desc.ValObjId.idnext);
		SymDump(prfx,Ptr->Desc.ValObjId.idnext,lvl+1);
	    }
	    else
		printf("\n");
	    break;
	case SYM_VALNAME:
	    printf(" VALNAME");
	    if (Ptr->Desc.ValName.namedstr)
		printf(" '%s'",Ptr->Desc.ValName.namedstr);
	    if (Ptr->Desc.ValName.namedstr != NULL) {
		printf(" ->%lx\n",Ptr->Desc.ValName.namedtyp);
		SymDump(prfx,Ptr->Desc.ValName.namedtyp,lvl+1);
	    }
	    else
		printf("\n");
	    break;
	case SYM_DELAY:
	    printf(" DELAY");
	    printf(" LineNum %d",Ptr->Desc.Delay.LineNum);
	    printf("\n");
	    break;
	case SYM_VALDELAY:
	    printf(" VALDELAY");
	    printf(" LineNum %d",Ptr->Desc.ValDelay.LineNum);
	    if (Ptr->Desc.ValDelay.Type) {
		printf(" Type->%lx\n",Ptr->Desc.ValDelay.Type);
		SymDump(prfx,Ptr->Desc.ValDelay.Type,lvl+1);
	    }
	    printf("\n");
	    break;
	case SYM_VALMACRO:
	    printf(" VALMACRO");
	    if (Ptr->Desc.ValMacro.partlex)
		printf(" LEXEME '%s'",Ptr->Desc.ValMacro.partlex);
	    if (Ptr->Desc.ValMacro.parttype != NULL) {
		printf(" Type->%lx\n",Ptr->Desc.ValMacro.parttype);
		SymDump(prfx,Ptr->Desc.ValMacro.parttype,lvl+1);
	    }
	    if (Ptr->Desc.ValMacro.partvalue != NULL) {
		printf(" Value->%lx\n",Ptr->Desc.ValMacro.partvalue);
		SymDump(prfx,Ptr->Desc.ValMacro.partvalue,lvl+1);
	    }
	    if (!Ptr->Desc.ValMacro.partvalue && !Ptr->Desc.ValMacro.parttype)
		printf("\n");
	    break;
	default:
	    printf(" Unknown %d\n",Ptr->Code);
	    return;
    }
    if (Ptr->Assoc)
	SymDump(prfx,Ptr->Assoc,lvl);
}

/*
 * dump a macro tree
 */
static MacroDump(T,prfx,lvl)
MPARSE *T;
char *prfx;
int lvl;
{
    register int i;

    if (lvl >= 0) for(i=0;i<lvl;i++) fputs("  ",stdout);
    printf("%s %lx: ",prfx,T);

    if (T == NULL) {
	printf("NULL\n");
	return;
    }
    printf("#%d ",T->Code);
    if (T->Symbol != NULL) printf("'%s' ",T->Symbol);
    if (T->Recurse) printf(">>RECURSE<< ",T->Symbol);
    if (T->Hit) printf("**HIT** ",T->Symbol);
    switch (T->Code) {
	case MACRO_LIST:
	    printf("LIST %d:",T->Part.List.Cnt);
	    for(i=0;i<T->Part.List.Cnt;i++)
		printf(" %lx",T->Part.List.List[i]);
	    printf("\n");

	    if (lvl >= 0)
		for(i=0;i<T->Part.List.Cnt;i++)
		    MacroDump(T->Part.List.List[i],prfx,lvl+1);
	    break;
	case MACRO_ALT:
	    printf("ALT %d:",T->Part.Alt.Cnt);
	    for(i=0;i<T->Part.Alt.Cnt;i++)
		printf(" %lx",T->Part.Alt.List[i]);
	    printf("\n");

	    if (lvl >= 0)
		for(i=0;i<T->Part.Alt.Cnt;i++)
		    MacroDump(T->Part.Alt.List[i],prfx,lvl+1);
	    break;
	case MACRO_LEX:
	    printf("LEXEME '%s'\n",T->Part.Lexeme);
	    break;
	case MACRO_TYPE:
	    printf("TYPE %stype='%s' (%lx)\n",
		   T->Part.Type.IsType ? "(TYPE) " : "",
		   T->Part.Type.TName ? T->Part.Type.TName : "?",
		   T->Part.Type.Type);
	    if (T->Part.Type.Type)
		SymDump(prfx,T->Part.Type.Type,lvl+1);
	    break;
	case MACRO_VALUE:
	    printf("VALUE %sid='%s' type='%s' (%lx)\n",
		   T->Part.Value.IsValue ? "(VALUE) " : "",
		   T->Part.Value.Name ? T->Part.Value.Name : "?",
		   T->Part.Value.TName ? T->Part.Value.TName : "?",
		   T->Part.Value.Type);
	    if (T->Part.Value.Type)
		SymDump(prfx,T->Part.Value.Type,lvl+1);
	    break;
	case MACRO_SYMBOL:
	    printf("SYMBOL (%lx)\n",T->Part.SymTop);
	    break;
	case MACRO_EMPTY:
	    printf("EMPTY\n");
	    break;
	default:
	    printf("Unknown %d\n",T->Code);
    }
}
#endif	DODEBUG

/*
 * process a macro node and its children and mark any snodes
 */
static MarkMacro(T)
MPARSE *T;
{
    register int i;

    if (T == NULL)
	return;

    switch(T->Code) {
	case MACRO_LIST:
	    for(i=0;i<T->Part.List.Cnt;i++)
		MarkMacro(T->Part.List.List[i]);
	    break;
	case MACRO_ALT:
	    for(i=0;i<T->Part.Alt.Cnt;i++)
		MarkMacro(T->Part.Alt.List[i]);
	    break;
	case MACRO_VALUE:
	    if (T->Part.Value.Type)
		MarkSNODE(T->Part.Value.Type);
	    break;
	case MACRO_TYPE:
	    if (T->Part.Type.Type)
		MarkSNODE(T->Part.Type.Type);
	    break;
    }
}

/*
 * mark a node and its children as being in use by export
 */
static MarkSNODE(ptr)
register SNODE *ptr;
{
    if (ptr == NULL || ptr->AllocMark)
	return;
    ptr->AllocMark = 1;

    switch(ptr->Code) {
	case SYM_INTEGER:
	    MarkSNODE(ptr->Desc.TypInt.NamedList);
	    break;
	case SYM_SEQ:
	    MarkSNODE(ptr->Desc.Seq.TypPtr);
	    MarkSNODE(ptr->Desc.Seq.SNext);
	    break;
	case SYM_SEQOF:
	    MarkSNODE(ptr->Desc.TypSeqOf);
	    break;
	case SYM_CHOICE:
	    MarkSNODE(ptr->Desc.Choice.TypPtr);
	    MarkSNODE(ptr->Desc.Choice.CNext);
	    break;
	case SYM_MACRO:
	    MarkMacro(ptr->Desc.Macro.TypPart);
	    MarkMacro(ptr->Desc.Macro.ValPart);
	    break;
	case SYM_VALINTEGER:
	    MarkSNODE(ptr->Desc.ValInt.vnext);
	    break;
	case SYM_VALOBJID:
	    MarkSNODE(ptr->Desc.ValObjId.idnext);
	    break;
	case SYM_VALNAME:
	    MarkSNODE(ptr->Desc.ValName.namedtyp);
	    break;
	case SYM_VALMACRO:
	    MarkSNODE(ptr->Desc.ValMacro.parttype);
	    MarkSNODE(ptr->Desc.ValMacro.partvalue);
	    MarkSNODE(ptr->Desc.ValMacro.partvaluet);
	    break;
    }
    /* we don't mark the associated information */
}

/*
 * release and snode
 */
static RelSNODE(ptr)
register SNODE *ptr;
{
    if (ptr->Name)
	UnMakeStr(ptr->Name);
    switch(ptr->Code) {
	case SYM_SEQ:
	    if (ptr->Desc.Seq.Sname)
		UnMakeStr(ptr->Desc.Seq.Sname);
	    break;
	case SYM_CHOICE:
	    if (ptr->Desc.Choice.Cname)
		UnMakeStr(ptr->Desc.Choice.Cname);
	    break;
	case SYM_VALINTEGER:
	    if (ptr->Desc.ValInt.id)
		UnMakeStr(ptr->Desc.ValInt.id);
	    break;
	case SYM_VALOSTRING:
	    if (ptr->Desc.ValStr)
		UnMakeStr(ptr->Desc.ValStr);
	    break;
	case SYM_VALOBJID:
	    if (ptr->Desc.ValObjId.idstr)
		UnMakeStr(ptr->Desc.ValObjId.idstr);
	    break;
	case SYM_VALNAME:
	    if (ptr->Desc.ValName.namedstr)
		UnMakeStr(ptr->Desc.ValName.namedstr);
	    break;
	case SYM_VALMACRO:
	    if (ptr->Desc.ValMacro.partlex)
		UnMakeStr(ptr->Desc.ValMacro.partlex);
	    break;
    }
    FreeSNODE(ptr);
}

/*
 * scan and release symbol record nodes
 * if keep is non-zero then only unmarked ones are kept otherwise
 * all are released.
 */
static ScanRelSNODE(keep)
int keep;
{
    if (!keep) {
	/* release entire list */
	while (SymbolHead != NULL)
	    RelSNODE(SymbolHead);
    }
    else {
	register SNODE *ptr;
	/* release unmarked nodes */
	while (SymbolHead != NULL && SymbolHead->AllocMark == 0)
	    FreeSNODE(SymbolHead);
	if ((ptr = SymbolHead) != NULL) {
	    register SNODE *nptr;
	    do {
		nptr = ptr->AllocNext;
		if (ptr->AllocMark == 0)
		    RelSNODE(ptr);
	    } while ((ptr = nptr) != SymbolHead);
	}
    }
}

/*
 * find the module context
 */
static MNODE *FindModContext(nam)
char *nam;
{
    register MNODE *mptr;
    register int i;

    if (nam != NULL) {
	for(mptr=TopContext;mptr != NULL;mptr = mptr->Next) {
	    if (nam == mptr->Name) {
		VSPRINTF(("FindModContext found module '%s'\n",mptr->Name));
		return mptr;
	    }
	}
	SPRINTF(("FindModContext did NOT find module '%s'\n",nam));
	return NULL;
    }
    VVSPRINTF(("FindModContext using current\n"));
    return CurContext;
}


/*
 * Lookup the specified type name and return the associated type name
 *     pointer.
 * A module can be specified with NULL being the current module.
 */
static TNAME *FindType(nam,mptr)
char *nam;
register MNODE *mptr;
{
    static char *mod = "FindType";
    register TNAME *ptr;

    /* determine module */
    if (mptr == NULL)
	mptr = CurContext;
    ptr = mptr->Types;

    /* lookup type name in type list */
    for(ptr=mptr->Types;ptr != NULL;ptr = ptr->Next) {
	if (ptr->Name == nam) {
	    VSPRINTF(("FindType found type '%s'\n",ptr->Name));
	    return ptr;
	}
    }

    SPRINTF(("FindType did NOT find type '%s'\n",nam));
    return NULL;
}

/*
 * add a type to the current scoping level
 */
static TNAME *AddType(nam,tptr)
char *nam;
SNODE *tptr;
{
    TNAME *tabptr;

    VSPRINTF(("AddType adding type '%s'\n",nam));

    /* if type name is not defined then copy into type */
    if (tptr->Name == NULL)
	tptr->Name = CopyStr(nam);

    /* make a new entry */
    tabptr = MakeTNAME();
    tabptr->Name = CopyStr(nam);
    tabptr->TypPtr = tptr;

    /* add it to the list */
    tabptr->Next = CurContext->Types;
    CurContext->Types = tabptr;

#if	DODEBUG
    SPRINTF(("AddType added type '%s'\n",nam));
    if (SymblDebug>3 || (tptr->Code == SYM_MACRO && MacroDebug))
    	SymDump("AddType TYPE",tptr,0);
#endif	DODEBUG
    return tabptr;
}

/*
 * Lookup the specified value name and return the associated value name
 *     pointer.
 * A module can be specified with NULL being the current module.
 */
static VNAME *FindValue(nam,mptr)
char *nam;
register MNODE *mptr;
{
    static char *mod = "FindValue";
    register VNAME *ptr;

    /* determine module */
    if (mptr == NULL)
	mptr = CurContext;

    /* lookup value name in value list */
    for(ptr=mptr->Values;ptr != NULL;ptr = ptr->Next) {
	if (ptr->Name == nam) {
	    VSPRINTF(("FindValue found value '%s'\n",ptr->Name));
	    return ptr;
	}
    }

    SPRINTF(("FindValue did NOT find value '%s'\n",nam));
    return NULL;
}

/*
 * add an value to the current module value table
 */
 VNAME *AddValue(nam,tptr,vptr)
char *nam;
SNODE *tptr,*vptr;
{
    VNAME *tabptr;

    VSPRINTF(("AddValue adding value '%s'\n",nam));

    /* if value name is not defined then copy into type */
    if (vptr->Name == NULL)
	vptr->Name = CopyStr(nam);

    /* make a new entry */
    tabptr = MakeVNAME();
    tabptr->Name = CopyStr(nam);
    tabptr->TypPtr = tptr;
    tabptr->ValPtr = vptr;

    /* add it to the list */
    tabptr->Next = CurContext->Values;
    CurContext->Values = tabptr;

#if	DODEBUG
    SPRINTF(("AddValue added value '%s'\n",nam));
    if (SymblDebug>3) {
	SymDump("AddValue TYPE",tptr,0);
	SymDump("AddValue VALUE",vptr,0);
    }
#endif	DODEBUG
    return tabptr;
}

/*
 * Module Error.
 * ModErr(module,flag,fmt,args...)
 *
 * module - name of module calling this routine
 * flag - 2=fatal,1=error,0=warning
 * fmt,args - printf error message
 */
/*VARARGS*/
static ModErr(va_alist)
va_dcl
{
    va_list ap;
    int flag;
    char *module,*fmt;
    char msgbuf[BUFSIZ], submsgbuf[BUFSIZ/2];

    /* collect arguments */
    va_start(ap);
    module = va_arg(ap,char *);
    flag = va_arg(ap,int);
    fmt = va_arg(ap,char *);
    vsprintf(msgbuf, fmt, ap);
    va_end(ap);

    /* generate sub message */
    sprintf(submsgbuf,"%s: ",CurFileName);
    if (CurContext != NULL)
	sprintf(submsgbuf+(int)strlen(submsgbuf),
		"Module %s:",CurContext->Name);

    PrintError(module,flag,submsgbuf,msgbuf,NULL);
}

/*
 * define and set current module context
 */
static PushModule(nam)
char *nam;
{
    MNODE *ptr;

#if	MEMACCT
    MemCntMNODE++;
#endif
    ptr = (MNODE *) Malloc(sizeof(MNODE),0);
    ptr->Name = nam;
    ptr->Next = TopContext;
    CurContext = TopContext = ptr;
}

/*
 * eliminate all symbols from module
 */
DoExports(mptr)
register MNODE *mptr;
{
    static char mod[]="DoExports";

    /* first mark all exported types and values */
    while (mptr->Exports != NULL) {
	register INODE *ptr;
	/* unlink export node from list */
	ptr = mptr->Exports;
	mptr->Exports = ptr->Next;
	/* determine type or value */
	if (isupper(*ptr->Name)) {
	    register TNAME *tptr;
	    /* type */
	    tptr = FindType(ptr->Name,NULL);
	    if (tptr == NULL)
		ModErr(mod,1,"EXPORT of type '%s' not found",ptr->Name);
	    else
		MarkSNODE(tptr->TypPtr);
	}
	else {
	    /* value */
	    register VNAME *vptr;
	    vptr = FindValue(ptr->Name,NULL);
	    if (vptr == NULL)
		ModErr(mod,1,"EXPORT of value '%s' not found",ptr->Name);
	    else {
		MarkSNODE(vptr->TypPtr);
		MarkSNODE(vptr->ValPtr);
	    }
	}
	/* get rid of export record */
	UnMakeStr(ptr->Name);
	FreeINODE(ptr);
    }
}


/*
 * pop a module context.
 * handle tree augmentation.
 */
static PopModule()
{
    static char *mod = "PopModule";
    register TNAME *tptr;
    register VNAME *vptr;

    /* complain about unresolved delay types and values */
    ResolveDelay(1);

    /* generate node information */
    GenNodes(CurContext);

    /* handle exports */
    DoExports(CurContext);

    /* release symbol records not exported */
    ScanRelSNODE(1);

    /* no current context */
    CurContext = NULL;
}

/*****************************************************************
 *		TREE BUILDER
 *****************************************************************
 */

/* the top of the tree */
static struct tree *treetop = NULL;

/* cache of compressed strings "OBJECT-TYPE" , "SYNTAX", etc. */
static char *OBJECTTYPEstr = NULL;
static char *SYNTAXstr = NULL;
static char *NETADDRstr = NULL;
static char *IPADDRstr = NULL;
static char *COUNTERstr = NULL;
static char *GAUGEstr = NULL;
static char *TIMETICKSstr = NULL;
static char *OPAQUEstr = NULL;
static char *DESCRIPTIONstr = NULL;

/*
 * make/free enumerate lists
 */
#if	MEMACCT
static struct enum_list *MakeENUM()
{
    MemCntENUMS++;
    return (struct enum_list *) Malloc(sizeof(struct enum_list),0);
}
static FreeENUM(x)
{
    MemCntENUMS--;
    RelCntENUMS++;
    Free(x);
}
#else
#define MakeENUM() (struct enum_list *) Malloc(sizeof(struct enum_list),0);
#define FreeENUM(x) Free(x)
#endif

/*
 * make/free tree nodes
 */
#if	MEMACCT
static struct tree *MakeTREE()
{
    MemCntTREE++;
    return (struct tree *) Malloc(sizeof(struct tree),0);
}
static FreeTREE(x)
{
    MemCntTREE--;
    RelCntTREE++;
    Free(x);
}
#else
#define MakeTREE() (struct tree *) Malloc(sizeof(struct tree),0)
#define FreeTREE(x) Free(x)
#endif

#if	DODEBUG
/*
 * tree dumping routine
 */
static DumpTree(ptr,lvl)
struct tree *ptr;
int lvl;
{
    register int i;
    register struct tree *cp,*pp;

    /* go thru peer list and for each node print it out */
    for(pp = ptr; pp; pp = pp->next_peer) {

	/* print out the node */
	for(i=0;i<lvl;i++) fputs("| ",stdout);
	printf(".%s(%d) ",pp->label ? pp->label : "?",pp->subid);
	switch(pp->type) {
	    case TYPE_OTHER:
		printf("Other"); break;
	    case TYPE_OBJID:
		printf("ObjID"); break;
	    case TYPE_OCTETSTR:
		printf("OctetStr"); break;
	    case TYPE_INTEGER:
		printf("Integer"); break;
	    case TYPE_NETADDR:
		printf("NetAddr"); break;
	    case TYPE_IPADDR:
		printf("IpAddr"); break;
	    case TYPE_COUNTER:
		printf("Counter"); break;
	    case TYPE_GAUGE:
		printf("Gauge"); break;
	    case TYPE_TIMETICKS:
		printf("TimeTicks"); break;
	    case TYPE_OPAQUE:
		printf("Opaque"); break;
	    case TYPE_NULL:
		printf("Null"); break;
	    case TYPE_COUNTER64:
		printf("Counter64"); break;
	    case TYPE_BITSTRING:
		printf("Bit Str"); break;
	    case TYPE_NSAPADDRESS:
		printf("Nsapddress"); break;
	    default:
		printf("Unknown"); break;
	}
	printf("%s%s\n",
	       pp->enums ? " HAS ENUMS" : "",
	       pp->description ? " HAS DESC" : "");

	/* print out the children if any */
	if (pp->child_list != NULL)
	    DumpTree(pp->child_list,lvl+1);
    }
}
#endif	DODEBUG

/*
 * connect a child into the specified parent sorted by numeric number
 */
static Connect(parent,child)
register struct tree *parent, *child;
{
    register struct tree *chk;

    TPRINTF(("Connecting %s(%d) to %s(%d)\n",
	    child->label ? child->label : "??",
	    child->subid,
	    parent->label ? parent->label : "??",
	    parent->subid));

    /* init parent information */
    child->parent = parent;

    /* if parent has no child then simply set and return */
    if (parent->child_list == NULL) {
	parent->child_list = child;
	return;
    }

    /* get pointer to first child */
    chk = parent->child_list;

    /* check head pointer first */
    if (child->subid < chk->subid) {
	child->next_peer = chk;
	parent->child_list = child;
	return;
    }

    /* check the rest of the list */
    for(;chk->next_peer;chk = chk->next_peer) {
	/* see if new child is before next child */
	if (child->subid < chk->next_peer->subid) {
	    /* it is, add before */
	    child->next_peer = chk->next_peer;
	    chk->next_peer = child;
	    return;
	}
    }

    /* add to end */
    chk->next_peer = child;
}

/*
 * process an object id list for a name record
 * we use recursion and avoid loops of 
 *	foo OBJECT IDENTIFIER := { bar 1 }
 *      bar OBJECT IDENTIFIER := { foo 1 }
 * by creating the name record for "foo" before recursing.
 *
 * There is no doubt this code can be simplified and cleaned up...sigh.
 */
static ObjIDProcess(NPtr)
VNAME *NPtr;
{
    static char *mod = "ObjIDProcess";
    register struct tree *chk;
    struct tree *parent, *child, *tmp;
    register SNODE *Parent,*Child;
    VNAME *nptr;
    int idnum;
    char *idnam;


    /* make sure name record is valid */
    if (NPtr->Name == NULL || NPtr->ValPtr == NULL || NPtr->TypPtr == NULL) {
	ModErr(mod,2,"Called with bad name pointer");
	return;
    }
    if (NPtr->ValPtr->Code != SYM_VALOBJID) {
	ModErr(mod,1,"Wrong type for value '%s' %d",
	       NPtr->Name,NPtr->ValPtr->Code);
	return;
    }

    /* do not re-process nodes already resolved */
    if (NPtr->TreePtr != NULL) {
	ModErr(mod,1,"Node '%s' already processed",NPtr->Name);
	return;
    }

    TPRINTF(("ObjIDProcess working on %s\n",NPtr->Name));

    /* set up node and set name to have a pointer to prevent loops */
    child = MakeTREE();
    child->label = CopyStr(NPtr->Name);
    /* child->subid not set yet */
    NPtr->TreePtr = child;

    /* no parent node info */
    parent = NULL;

    /* process parent/child pairs */
    Parent = NPtr->ValPtr;
    while (Parent != NULL) {
	/* extract parent information */
	idnam = Parent->Desc.ValObjId.idstr; 
	idnum = Parent->Desc.ValObjId.idnum;
	Child = Parent->Desc.ValObjId.idnext;
	TPRINTF(("Parent %s(%d)\n",idnam ? idnam : "?", idnum));

	/* if we are processing a parent with a child then we 
         * require the parent tree node to exist.
	 */
	if (Child != NULL) {
	    /* if new parent does not have a name then look for the id num
	     * in the children of current parent. If no parent then use
	     * top level.
	     */
	    if (idnam == NULL) {
		/* scan for id num */
		tmp = parent ? parent : treetop;
		for(tmp = tmp->child_list;tmp;tmp = tmp->next_peer)
		    if (idnum == tmp->subid)
			break;
		if (tmp == NULL) {
		    /* we do not have a name and we did not find the id then
		     * we have to add it. This is *sort* of normal but
		     * complain any way.
		     */
		    ModErr(mod,1,"No parent found for %d in '%s'",idnum,
			   NPtr->Name);
		    /* make the new node */
		    tmp = MakeTREE();
		    tmp->subid = idnum;
		    /* link it into parent */
		    Connect(parent ? parent : treetop, tmp);
		}
		/* work on child */
		parent = tmp;
		Parent = Child;
		TPRINTF(("new %s(%d)\n",idnam ? idnam : "?", idnum));
		continue;
	    }

	    /* we have a node name, look it up */
	    nptr = FindValue(idnam,NULL);

	    /* if we did not find the name in the symbol table and we
	     * have a name and value and a parent then implicitly
 	     * we have defined a new symbol. Add it and shorten 
 	     * current obj id list.
	     *
	     * This mechanism allows us to have a context, minimize
	     * recursion and minimize exporting symbol information.
	     */
	    if (nptr == NULL && parent != NULL) {
		register SNODE *vptr;
		/* make tree node entry corresponding to this child */
		tmp = MakeTREE();
		tmp->label = CopyStr(idnam);
		tmp->subid = idnum;
		if (idnum < 0)
		    ModErr(mod,1,"No value found for '%s' in '%s'",idnam,
			   NPtr->Name);
		/* connect it to parent */
		Connect(parent,tmp);

		/* internet ::= { iso org dod 1}
		 * becomes
		 * org ::= { iso org }
		 * internet ::= { org dod 1 }
		 */
		/* clone child node and terminate */
		vptr = MakeSNODE();
		/* if child has a name then copy it */
		if (Child->Name != NULL)
		    vptr->Name = CopyStr(Child->Name);
		vptr->Code = SYM_VALOBJID;
		if (Child->Desc.ValObjId.idstr)
		    vptr->Desc.ValObjId.idstr = 
					CopyStr(Child->Desc.ValObjId.idstr);
		vptr->Desc.ValObjId.idnum = Child->Desc.ValObjId.idnum;
		vptr->Desc.ValObjId.idnext = NULL;

		/* connect parent to new child */
		Parent->Desc.ValObjId.idnext = vptr;

		/* add new symbol to point to parent */
		nptr = AddValue(idnam,NPtr->TypPtr,Parent);
		/* update symbol table entry for */
		nptr->TreePtr = tmp;

		/* fix current symbol to point to new obj id list */
		NPtr->ValPtr = Child;
		/* shift parent down */
		parent = tmp;
		Parent = Child;
		TPRINTF(("new %s(%d)\n",idnam ? idnam : "?", idnum));
		continue;
	    }

	    /* if we still do not have a symbol name then it should be
	     * first identifier in list and the top level should be searched
	     * It is unclear whether we should add this information to the
	     * symbol table.
	     */
	    if (nptr == NULL) {
		/* scan for name */
		tmp = treetop;
		for(tmp = tmp->child_list;tmp;tmp = tmp->next_peer)
		    if (idnam == tmp->label) {
			/* update num in node */
			if (idnum >= 0)
			    tmp->subid = idnum;
			else
			    idnum = tmp->subid;
			break;
		}
		/* if we did not find the name then check id if we have one */
		if (tmp == NULL && idnum >= 0) {
		    tmp = parent ? parent : treetop;
		    for(tmp = tmp->child_list;tmp;tmp = tmp->next_peer)
			if (idnum == tmp->subid) {
			    /* update name in node */
			    tmp->label = CopyStr(idnam);
			    break;
			}
		}
		/* if we still do not have a parent then use top level */
		if (tmp == NULL) {
		    ModErr(mod,1,"No parent found for '%s' in '%s'",idnam,
			   NPtr->Name);
		    parent = treetop;
		    /* make the new node */
		    tmp = MakeTREE();
		    tmp->label = CopyStr(idnam);
		    tmp->subid = idnum;
		    /* link it into parent */
		    Connect(treetop, tmp);
		}
		parent = tmp;
		Parent = Child;
		TPRINTF(("new %s(%d)\n",idnam ? idnam : "?", idnum));
		continue;
	    }

	    /* if we get here then we found the name in the module
	     * and should build based on that name. If that name has
	     * not had a node bound to it then recurse...
	     */
	    if (nptr->TreePtr == NULL)
		ObjIDProcess(nptr);

	    parent = nptr->TreePtr;
	    Parent = Child;
	    continue;
	}

	/* at this point, we have the last entity in object id list
	 * and this is the node we want to add connect it to the parent.
	 */
	TPRINTF(("Doing end child %s(%d) for %s\n",idnam ? idnam : "?",
		 idnum,NPtr->Name));

	/* fill in subbid for child */
	child->subid = idnum;

	/* check and make sure we have a valid parent */
	if (parent == NULL) {
	    /* we do not have a parent... should not happen */
	    ModErr(mod,1,"No parent for node %s",NPtr->Name);
	    /* set parent to top level */
	    parent = treetop;
	}

	/* connect the child to the parent */
	Connect(parent,child);

	/* all done */
	break;
    }
}

/*
 * fill in information from OBJECT-TYPE macro
 */
static FillIn(NodPtr,TypPtr)
register struct tree *NodPtr;
register SNODE *TypPtr;
{
    register SNODE *tp,*pp;

    /* loop thru associations */
    for(tp = TypPtr;tp;tp = tp ? tp->Assoc : NULL) {
	if (tp == NULL || tp->Code != SYM_VALMACRO)
	    continue;
	TPRINTF(("FillIn %s\n",
		tp->Desc.ValMacro.partlex ? tp->Desc.ValMacro.partlex : ""));
	if (tp->Desc.ValMacro.partlex == SYNTAXstr) {
	    tp = tp->Assoc;
	    if (tp == NULL || tp->Code != SYM_VALMACRO)
		continue;
	    pp = tp->Desc.ValMacro.parttype;
	    if (pp == NULL)
		continue;
	    switch (pp->Code) {
		case SYM_INTEGER:
		    if (pp->Name == COUNTERstr)
			NodPtr->type = TYPE_COUNTER;
		    else if (pp->Name == GAUGEstr)
			NodPtr->type = TYPE_GAUGE;
		    else if (pp->Name == TIMETICKSstr)
			NodPtr->type = TYPE_TIMETICKS;
		    else {
			register SNODE *p;
			NodPtr->type = TYPE_INTEGER;
			/* see if we have a cached enum list */
			if (pp->Desc.TypInt.EnumList != NULL)
			    NodPtr->enums = pp->Desc.TypInt.EnumList;
			else if ((p = pp->Desc.TypInt.NamedList) != NULL) {
			    register struct enum_list *el,*last;
			    /* we have named list */
			    last = NULL;
			    while (p != NULL && p->Code == SYM_VALINTEGER) {
				el = MakeENUM();
				el->value = p->Desc.ValInt.val;
				el->label = CopyStr(p->Desc.ValInt.id);
				if (last == NULL)
				    NodPtr->enums = el;
				else
				    last->next = el;
				last = el;
				p = p->Desc.ValInt.vnext;
			    }
			    /* cache the enum list */
			    pp->Desc.TypInt.EnumList = NodPtr->enums;
			}
		    }
		    break;
		case SYM_OSTRING:
		    if (pp->Name == IPADDRstr)
			NodPtr->type = TYPE_IPADDR;
		    else if (pp->Name == NETADDRstr)
			NodPtr->type = TYPE_NETADDR;
		    else if (pp->Name == OPAQUEstr)
			NodPtr->type = TYPE_OPAQUE;
		    else
			NodPtr->type = TYPE_OCTETSTR;
		    break;
		case SYM_OBJID:
		    NodPtr->type = TYPE_OBJID;
		    break;
		default:
		    break;
	    }
	}
	else if (tp->Desc.ValMacro.partlex == DESCRIPTIONstr) {
	    tp = tp->Assoc;
	    if (tp == NULL || tp->Code != SYM_VALMACRO)
		continue;
	    pp = tp->Desc.ValMacro.partvalue;
	    if (pp == NULL)
		continue;
	    if (pp->Code != SYM_VALOSTRING)
		continue;
	    NodPtr->description = CopyStr(pp->Desc.ValStr);
	}
	/* else we do not load this information */
    }
}

/*
 * generate node information from object identifier values
 */
static GenNodes(ptr)
MNODE *ptr;
{
    register VNAME *nptr;

    /* loop thru each value entry that is of type OBJID and
     * derive the node information recursively.
     */
    for (nptr=ptr->Values;nptr != NULL;nptr = nptr->Next) {
	/* ignore nodes with out values or type object identifier */
	if (nptr->ValPtr == NULL || nptr->ValPtr->Code != SYM_VALOBJID)
	    continue;
	/* if we do not have a tree node then process it */
	if (nptr->TreePtr == NULL)
	    ObjIDProcess(nptr);
#if	DODEBUG
	if (TreeDebug>4) DumpTree(treetop,0);
#endif	DODEBUG
	/* fill in missing node information if of type OBJECT-TYPE */
	if (nptr->TreePtr != NULL) {
	    if (nptr->TypPtr != NULL && nptr->TypPtr->Name == OBJECTTYPEstr) {
fflush(stdout);
		FillIn(nptr->TreePtr,nptr->TypPtr);
	    }
	}
#if	DODEBUG
	if (TreeDebug>3) DumpTree(treetop,0);
#endif	DODEBUG
    }
#if	DODEBUG
    if (TreeDebug>1) DumpTree(treetop,0);
#endif	DODEBUG
}


/*
/*****************************************************************
 *			PARSER SUPPORT ROUTINES
 *****************************************************************
 */

/*
 * general parse complaint routine
 */
static PError(mod,want,got)
char *mod;
{
    ParseErr(mod,1,"Expected %s, got %s",StrTok(want),StrTok(got));
}
/*
 * general check and complaint routine
 */
static int PChk(mod,want)
register char *mod;
register int want;
{
    register int lex;
    lex = GetTok();
    if (lex != want) {
	PError(mod,want,lex);
	EatTok();
	return 1;
    }
#if	DODEBUG
    if (ParseDebug>3) ShowTok("PCHK",lex,CurToken);
#endif	DODEBUG
    EatTok();
    return 0;
}

/*
 * myatoi which ignores overflow by limiting things to max values
 */
static long myatoi(str)
register char *str;
{
    register long val, oldval;
    register int i;

    /* build it up */
    val = 0;
    while (*str) {
	oldval = val;
	val *= 10;
	val += (*str++ - '0');
	/* check for overflow and max out */
	if (val < oldval) {
	    val = 0x7fffffff;
	    break;
	}
    }

    return val;
}

/*
 * general check and parse number routine
 */
static int PNum(mod,valptr)
char *mod;
long *valptr;
{
    int lex;
    lex = GetTok();
    if (lex != LEXNUMBER) {
	PError(mod,LEXNUMBER,lex);
	EatTok();
	return 1;
    }
    *valptr = myatoi(CurToken);
#if	DODEBUG
    if (ParseDebug>3) ShowTok("PNUM",lex,CurToken);
#endif	DODEBUG
    EatTok();
    return 0;
}

/*
 * Add a delay record
 */
static AddDelay(nam,typptr,valptr)
char *nam;
SNODE *typptr;
SNODE *valptr;
{
    register DNODE *ptr;

    VVSPRINTF(("AddDelay '%s'\n",nam ? nam : "?"));

    ptr = MakeDNODE();
    if (nam != NULL)
	ptr->Name = CopyStr(nam);
    ptr->TypPtr = typptr;
    ptr->ValPtr = valptr;

    if (ptr->ValPtr != NULL) {
	VSPRINTF(("AddDelay adding value '%s'\n",nam ? nam : "?"));
	/* add to value list */
	ptr->Next = VDelayList;
	VDelayList = ptr;
    }
    else {
	VSPRINTF(("AddDelay adding type '%s'\n",nam ? nam : "?"));
	/* add to type list */
	ptr->Next = TDelayList;
	TDelayList = ptr;
    }
}

/*
 * Resolve delay records
 */
static ResolveDelay(flg)
int flg;
{
    static char *mod = "ResolveDelay";
    register DNODE *NewList,*ptr;


    VVSPRINTF(("ResolveDelay doing Types\n"));

    /* handle types first */
    NewList = NULL;
    while (TDelayList != NULL) {
	ptr = TDelayList;
	TDelayList = ptr->Next;
	/* type */
	VVSPRINTF(("ResolveDelay value '%s'\n",ptr->Name ? ptr->Name : "?"));
	if (ptr->TypPtr->Code != SYM_DELAY) {
	    /* resolved */
	    if (ptr->Name)
		UnMakeStr(ptr->Name);
	    FreeDNODE(ptr);
	}
	else if (flg) {
	    ModErr(mod,1,"Type '%s' never defined on line %d",
		   ptr->Name ? ptr->Name : "?",
		   ptr->TypPtr->Desc.Delay.LineNum);
	    if (ptr->Name)
		UnMakeStr(ptr->Name);
	    FreeDNODE(ptr);
	}
	else {
	    ptr->Next = NewList;
	    NewList = ptr;
	}
    }
    TDelayList = NewList;

    VVSPRINTF(("ResolveDelay doing Values\n"));

    /* handle values last */
    NewList = NULL;
    while (VDelayList != NULL) {
	ptr = VDelayList;
	VDelayList = ptr->Next;
	/* type */
	VVSPRINTF(("ResolveDelay value '%s'\n",ptr->Name ? ptr->Name : "?"));
	if (ptr->ValPtr->Code != SYM_VALDELAY) {
	    /* resolved */
	    if (ptr->Name)
		UnMakeStr(ptr->Name);
	    FreeDNODE(ptr);
	}
	else if (flg) {
	    ModErr(mod,1,"Value '%s' never defined on line %d",
		   ptr->Name ? ptr->Name : "?",
		   ptr->ValPtr->Desc.ValDelay.LineNum);
	    if (ptr->Name)
		UnMakeStr(ptr->Name);
	    FreeDNODE(ptr);
	}
	else {
	    ptr->Next = NewList;
	    NewList = ptr;
	}
    }
    VDelayList = NewList;

    VVSPRINTF(("ResolveDelay DONE\n"));
}

/*
 * Make sure a type exists for a specified type identifier
 */
static SNODE *TypeBind(nam,noforce)
char *nam;
int noforce;
{
    TNAME *NamPtr;
    SNODE *NewOne;

    VSPRINTF(("TypeBind looking for type '%s'\n",nam));

    NamPtr = FindType(nam,NULL);
    if (NamPtr != NULL)
	return NamPtr->TypPtr;

    if (noforce)
	return NULL;

    VSPRINTF(("TypeBind binding type '%s'\n",nam));

    NewOne = MakeSNODE();
    NewOne->Code = SYM_DELAY;
    NewOne->Desc.Delay.LineNum = CurLine;
    AddDelay(nam,NewOne,NULL);

    NamPtr = AddType(nam,NewOne);

    return NamPtr->TypPtr;
}

/*
 * Make sure a value exists for a specified value identifier
 */
static SNODE *ValueBind(nam,TypPtr,noforce)
char *nam;
SNODE *TypPtr;
int noforce;
{
    VNAME *NamPtr;
    SNODE *NewOne;

    VSPRINTF(("ValueBind looking for value '%s'\n",nam));

    NamPtr = FindValue(nam,NULL);
    if (NamPtr != NULL)
	return NamPtr->ValPtr;

    if (noforce)
	return NULL;

    VSPRINTF(("ValueBind binding value '%s'\n",nam));

    NewOne = MakeSNODE();
    NewOne->Code = SYM_VALDELAY;
    NewOne->Desc.ValDelay.LineNum = CurLine;
    NewOne->Desc.ValDelay.Type = TypPtr;

    AddDelay(nam,TypPtr,NewOne);

    NamPtr = AddValue(nam,TypPtr,NewOne);

    return NamPtr->ValPtr;
}

/*
 * see if a lexeme looks like a type reference
 */
static int IsTypeRef(lex)
int lex;
{
    switch(lex) {
	case LEXINTEGER:
	case LEXOCTET:
	case LEXNULL:
	case LEXSEQUENCE:
	case LEXCHOICE:
	case LEXLEFTBRACK:
	case LEXOBJECT:
	case LEXUIDENT:
	    return 1;
	default:
	    return 0;
    }
}

/*
 * see if a lexeme looks like a value reference
 */
static int IsValueRef(lex)
int lex;
{
    switch(lex) {
	case LEXNUMBER:
	case LEXHYPEN:
	case LEXLIDENT:
	case LEXHSTRING:
	case LEXBSTRING:
	case LEXCSTRING:
	case LEXLEFTCURLY:
	case LEXNULL:
	    return 1;
	default:
	    return 0;
    }
}

/*
 * similiar to P_Value but used to determine if a value matches
 * a type selection...normally called on a CHOICE.
 */
static SNODE *FindValueType(Ptr,lex,str,IdTyp)
SNODE *Ptr;
int lex;
char *str;
SNODE *IdTyp;
{
    SNODE *TPtr, *Tmp;
    static char *mod = "FindValueType";

    PPRINTF(("FindValueType for token '%.20s' (%lx)\n",str,Ptr));

    /* process the type */
    switch(Ptr->Code) {
	case SYM_INTEGER:
	    PPRINTF(("FindValueType INTEGER\n"));
	    /* see if it is in the named typed */
	    if (lex == LEXLIDENT) {
		for(Tmp = Ptr->Desc.TypInt.NamedList;Tmp != NULL;
		    Tmp = Tmp->Desc.ValInt.vnext) {
		    if (Tmp->Desc.ValInt.id != NULL)
			if (strcmp(Tmp->Desc.ValInt.id,str) == 0)
			    return Ptr;
		}
	    }
	    /* do normal checks */
	    if (IdTyp != NULL && IdTyp->Code == SYM_VALINTEGER)
		return Ptr;
	    if (lex == LEXHYPEN)
		return Ptr;
	    if (lex == LEXNUMBER)
		return Ptr;
	    if (lex == LEXHSTRING)
		return Ptr;
	    break;
	case SYM_OSTRING:
	    PPRINTF(("FindValueType OSTRING\n"));
	    if (IdTyp != NULL && IdTyp->Code == SYM_VALOSTRING)
		return Ptr;
	    if (lex == LEXCSTRING)
		return Ptr;
	    if (lex == LEXHSTRING)
		return Ptr;
	    break;
	case SYM_NULL:
	    PPRINTF(("FindValueType NULL\n"));
	    if (IdTyp != NULL && IdTyp->Code == SYM_VALNULL)
		return Ptr;
	    if (lex == LEXNULL)
		return Ptr;
	    break;
	case SYM_OBJID:
	    PPRINTF(("FindValueType OBJID\n"));
	    if (IdTyp != NULL && IdTyp->Code == SYM_VALOBJID)
		return Ptr;
	    if (lex == LEXLEFTCURLY)
		return Ptr;
	    break;
	case SYM_CHOICE:
	    PPRINTF(("FindValueType CHOICE\n"));
	    /* go thru the type we were given and try to find a match */
	    for (TPtr=Ptr;TPtr != NULL;TPtr = TPtr->Desc.Choice.CNext) {
		PPRINTF(("FindValueType of '%s'\n",
			 TPtr->Desc.Choice.Cname 
				? TPtr->Desc.Choice.Cname : "??"));
		if ((Tmp = TPtr->Desc.Choice.TypPtr) == NULL) {
		    PPRINTF(("FindValueType no type\n"));
		    continue;
		}
		/* if we have a found type then check it immediately */
		Tmp = FindValueType(Tmp,lex,str,IdTyp);
		if (Tmp != NULL) return Tmp;
	    }
	    break;
	default:
	    PPRINTF(("FindValueType unknown %d\n",Ptr->Code));
	    break;
    }

    VPPRINTF(("FindValueType no choice match\n"));
    return NULL;
}

/*
 * find macro expansion that seems to fit the best
 */
static MPARSE *TryMacro(Part,LexP)
MPARSE *Part;
int *LexP;
{
    static char mod[] = "TryMacro";
    char *lexstr;
    int i, lex, LexN;
    SNODE *SPtr;

#if	DODEBUG
    VVMPRINTF(("TryMacro %d\n",*LexP));
    if (MacroDebug>3)MacroDump(Part,"TryMacro",-1);
#endif	DODEBUG

    switch(Part->Code) {
	case MACRO_LEX:
	    VVMPRINTF(("TryMacro LEX\n"));
	    /* get a token */
	    if (*LexP == 0) {
		lex = GetTok();
		lexstr = CurToken;
	    }
	    else if (*LexP == 1) {
		lex = GetTok();
		lex = PeekTok();
		lexstr = NxtToken;
	    }
	    else {
		/* assume success we can not check any farther */
		VVMPRINTF(("TryMacro LEX ASSUME\n"));
		break;
	    }
	    /* check token */
	    VMPRINTF(("TryMacro LEX %s vs %s\n",Part->Part.Lexeme,lexstr));
	    if ((lex < 0 && Part->Part.Lexeme != lexstr) ||
		(lex >= 0 && strcmp(Part->Part.Lexeme,lexstr) != 0)) {
		VMPRINTF(("TryMacro LEX FAIL\n"));
		return NULL;
	    }
	    /* "eat" token */
	    (*LexP)++;
	    VMPRINTF(("TryMacro LEX OK\n"));
	    break;

	case MACRO_TYPE:
	    VVMPRINTF(("TryMacro TYPE\n"));
	    /* get a token */
	    if (*LexP == 0) {
		lex = GetTok();
		lexstr = CurToken;
	    }
	    else if (*LexP == 1) {
		lex = GetTok();
		lex = PeekTok();
		lexstr = NxtToken;
	    }
	    else {
		/* assume success we can not check any farther */
		VVMPRINTF(("TryMacro TYPE ASSUME\n"));
		break;
	    }
	    /* must be a type of some sort */
	    if (!IsTypeRef(lex)) {
		VVMPRINTF(("TryMacro TYPE FAIL (not a type reference)\n"));
		return NULL;
	    }
	    VMPRINTF(("TryMacro TYPE OK\n"));
	    /* "eat" token */
	    (*LexP)++;
	    break;

	case MACRO_VALUE:
	    VVMPRINTF(("TryMacro VALUE\n"));
	    /* get a token */
	    if (*LexP == 0) {
		lex = GetTok();
		lexstr = CurToken;
	    }
	    else if (*LexP == 1) {
		lex = GetTok();
		lex = PeekTok();
		lexstr = NxtToken;
	    }
	    else {
		/* assume success we can not check any farther */
		VVMPRINTF(("TryMacro VALUE ASSUME\n"));
		break;
	    }
	    if (!IsValueRef(lex)) {
		VVMPRINTF(("TryMacro VALUE FAIL (not a value reference)\n"));
		return NULL;
	    }
	    VMPRINTF(("TryMacro VALUE OK\n"));
	    /* "eat" token */
	    (*LexP)++;
	    break;

	case MACRO_EMPTY:
	    /* always matches */
	    VMPRINTF(("TryMacro EMPTY OK\n"));
	    break;

	case MACRO_SYMBOL:
	    VVMPRINTF(("TryMacro SYMBOL\n"));
	    if (Part->Part.SymTop != NULL && Part->Part.SymTop->Hit > 0) {
		VMPRINTF(("TryMacro SYMBOL OK [HIT]\n"));
		break;
	    }
	    VVMPRINTF(("TryMacro SYMBOL FAIL\n"));
	    return NULL;

	case MACRO_LIST:
	    VVMPRINTF(("TryMacro LIST\n"));
	    /* elements must match in order */
	    LexN = *LexP;
	    for(i=0;i<Part->Part.List.Cnt;i++) {
		if (TryMacro(Part->Part.List.List[i],&LexN) == NULL) {
		    VVMPRINTF(("TryMacro LIST FAIL\n",i));
		    return NULL;
		}
	    }
	    VMPRINTF(("TryMacro LIST OK\n",i));
	    /* "eat" token(s) */
	    *LexP = LexN;
	    break;

	case MACRO_ALT:
	    VVMPRINTF(("TryMacro ALT\n"));
	    /* pick first selection that matches */
	    for(i=0;i<Part->Part.Alt.Cnt;i++) {
		LexN = *LexP;
		if (TryMacro(Part->Part.Alt.List[i],&LexN) != NULL) {
		    break;
		}
	    }
	    if (i >= Part->Part.Alt.Cnt) {
		VVMPRINTF(("TryMacro ALT FAIL\n",i));
		return NULL;
	    }

	    VMPRINTF(("TryMacro ALT OK\n"));
	    /* "eat" token(s) */
	    *LexP = LexN;
	    break;

	default:
	    /* never should happen */
	    MPRINTF(("TryMacro DEFAULT FAIL\n"));
	    return NULL;
    }

    /* if we get here then everything is okay */
    return Part;
}

/* forward references */
static SNODE *P_Type();
static SNODE *P_Value();

/*
 * perform a macro parse
 * returns non-zero on failure
 */
static int DoMacro(MP,Tail)
MPARSE *MP;
SNODE *Tail;
{
    static char mod[] = "DoMacro";
    int i, LexN, tmp, lex;
    SNODE *NewOne, *TmpOne, *TypPtr, *ValPtr;
    TNAME *TNam;
    VNAME *VNam;
    MPARSE *Tmp;

#if	DODEBUG
    VMPRINTF(("DoMacro\n"));
    if(MacroDebug>3)MacroDump(MP,"DoMacro_OF",-1);
#endif	DODEBUG

    switch(MP->Code) {
	case MACRO_LEX:
	    /* must match token */
	    VMPRINTF(("DoMacro LEX '%s'\n",MP->Part.Lexeme));
	    lex = GetTok();
	    if ((lex < 0 && CurToken != MP->Part.Lexeme) ||
		(lex >= 0 && strcmp(CurToken,MP->Part.Lexeme) != 0)) {
		ParseErr(mod,1,"Expected macro string '%s', found '%s'",
			 MP->Part.Lexeme,CurToken);
		MPRINTF(("DoMacro LEX '%s' FAIL\n",MP->Part.Lexeme));
		return 1;
	    }
	    EatTok();
	    MPRINTF(("DoMacro LEX '%s' OK\n",MP->Part.Lexeme));
	    /* add to associates */
	    NewOne = MakeSNODE();
	    NewOne->Code = SYM_VALMACRO;
	    /* clone the lexeme in case it is a lexical token */
	    NewOne->Desc.ValMacro.partlex = MakeStr(MP->Part.Lexeme);
	    Tail->Assoc = NewOne;
	    return 0;

	case MACRO_TYPE:
	    /* process ANY type */
	    VMPRINTF(("DoMacro TYPE\n"));
	    TmpOne = P_Type();
	    if (TmpOne == NULL) {
		MPRINTF(("DoMacro TYPE FAIL\n"));
		return 1;
	    }
	    MPRINTF(("DoMacro TYPE OK\n"));

	    /* add to associates */
	    NewOne = MakeSNODE();
	    NewOne->Code = SYM_VALMACRO;
	    NewOne->Desc.ValMacro.parttype = TmpOne;
	    Tail->Assoc = NewOne;

	    /* add name to current scoping list */
	    TNam = MakeTNAME();
	    if (MP->Part.Type.TName)
		TNam->Name = MP->Part.Type.TName;
	    else
		TNam->Name = MP->Part.Type.Type->Name;
	    TNam->TypPtr = TmpOne;
	    TNam->Next = MacroTypes;
	    MacroTypes = TNam;
	    MPRINTF(("DoMacro TYPE Add local type %s\n",TNam->Name));
	    return 0;

	case MACRO_VALUE:
	    /* process symbol description */
	    VMPRINTF(("DoMacro VALUE\n"));
	    /* see if we have a local macro type */
	    TypPtr = NULL;
	    if (MP->Part.Value.TName) {
		for(TNam = MacroTypes;TNam != NULL;TNam = TNam->Next)
		    if (TNam->Name == MP->Part.Value.TName)
			break;
		if (TNam != NULL)
		    TypPtr = TNam->TypPtr;
	    }
	    /* if we did not derive a type then use the specified type */
	    if (TypPtr == NULL)
	 	TypPtr = MP->Part.Value.Type;
	    /* if not type at this point, then we are in trouble */
	    if (TypPtr == NULL) {
		ParseErr(mod,1,"No macro type for value");
		MPRINTF(("DoMacro VALUE FAIL (parse)\n"));
		return 1;
	    }
	    /* parse the type value */
	    TmpOne = P_Value(TypPtr);
	    if (TmpOne == NULL) {
		ParseErr(mod,1,"No macro value");
		MPRINTF(("DoMacro VALUE FAIL (parse)\n"));
		return 1;
	    }
	    MPRINTF(("DoMacro VALUE OK\n"));
	    /* add to associates */
	    NewOne = MakeSNODE();
	    NewOne->Code = SYM_VALMACRO;
	    NewOne->Desc.ValMacro.partvalue = TmpOne;
	    NewOne->Desc.ValMacro.partvaluet = TypPtr;
	    Tail->Assoc = NewOne;
	    return 0;

	case MACRO_EMPTY:
	    /* matches anything, never generates anything */
	    MPRINTF(("DoMacro EMPTY\n"));
	    return 0;

	case MACRO_SYMBOL:
	    VMPRINTF(("DoMacro SYMBOL\n"));
	    if (MP->Part.SymTop != NULL && MP->Part.SymTop->Hit > 0) {
		VMPRINTF(("DoMacro SYMBOL OK (hit)\n"));
		return 0;
	    }
	    MPRINTF(("DoMacro SYMBOL FAIL\n"));
	    return 1;

	case MACRO_ALT:
	    /* process list of alternatives */
	    VMPRINTF(("DoMacro ALT\n"));
	    /* scan list */
	    for(i=0;i<MP->Part.Alt.Cnt;i++) {
		LexN = 0;
		if (TryMacro(MP->Part.Alt.List[i],&LexN) != NULL) {
		    break;
		}
	    }
	    if (i >= MP->Part.Alt.Cnt) {
		MPRINTF(("DoMacro ALT FAIL\n"));
		return 1;
	    }
	    tmp = DoMacro(MP->Part.Alt.List[i],Tail);
	    /* advance tail if needed */
	    while (Tail && Tail->Assoc != NULL)
		Tail = Tail->Assoc;
	    if (tmp) {
		MPRINTF(("DoMacro ALT (DoMacro) FAIL\n"));
		return 1;
	    }

	    /* see if we should perform recursion */
	    VMPRINTF(("DoMacro ALT %d recurse check\n",i));
	    while (MP->Recurse && MP->Hit <= 0) {
		MP->Hit++;
		tmp = DoMacro(MP,Tail);
		/* advance tail if needed */
		while (Tail && Tail->Assoc != NULL)
		    Tail = Tail->Assoc;
		if (tmp) {
		    MP->Hit--;
		    break;
		};
		MP->Hit--;
	    }

	    MPRINTF(("DoMacro ALT %d DoMacro OK\n",i));
	    return 0;

	case MACRO_LIST:
	    /* process list */
	    VMPRINTF(("DoMacro LIST\n"));
	    /* scan list */
	    for(i=0;i<MP->Part.List.Cnt;i++) {
		tmp = DoMacro(MP->Part.List.List[i],Tail);
		/* advance tail if needed */
		while (Tail && Tail->Assoc != NULL)
		    Tail = Tail->Assoc;
		if (tmp) {
		    MPRINTF(("DoMacro LIST FAIL\n",i));
		    return 1;
		}
	    }
	    MPRINTF(("DoMacro LIST OK\n",i));
	    return 0;

	default:
	    MPRINTF(("DoMacro unknown %d\n",MP->Code));
	    return 1;
    }
}

/*
 * Process a macro which has three parts
 *	- type notation
 *	- value notation
 *	- embeded definition
 * We only handle simply value notations and do not handle embedded defintions.
 */
static SNODE *UseMacro(TypPtr)
SNODE *TypPtr;
{
    static char *mod = "UseMacro";
    SNODE *NewOne;
    MPARSE *VPtr, *TPtr;

    MPRINTF(("In UseMacro\n"));

    /* check type */
    if ((TPtr = TypPtr->Desc.Macro.TypPart) == NULL) {
	ParseErr(mod,1,"No type for macro");
	return NULL;
    }

    /* first generate type result */
    if ((VPtr = TypPtr->Desc.Macro.ValPart) == NULL) {
	ParseErr(mod,1,"No value for macro");
	return NULL;
    }
    if (VPtr->Code != MACRO_VALUE || !VPtr->Part.Value.IsValue) {
	ParseErr(mod,1,"Value of macro too complicated for this version");
	return NULL;
    }

    NewOne = MakeSNODE();
    /* make sure we put name in new type information */
    if (TypPtr->Name)
	NewOne->Name = CopyStr(TypPtr->Name);

    /* update the local value if set */
    if (VPtr->Part.Value.Type != NULL) {
	NewOne->Code = VPtr->Part.Value.Type->Code;
	NewOne->Desc = VPtr->Part.Value.Type->Desc;
    }
    else {
	ParseErr(mod,1,"No type for macro found");
	return NULL;
    }

    /* see if we are already in macro, if so complain */
    if (InMacro)
	ParseErr(mod,2,"Multiple macro invocations not supported");

    InMacro++;
    if (DoMacro(TPtr,NewOne)) {
	/* it failed...eat until assignment */
	for(;;) {
	    int lex;
	    lex = GetTok();
	    if (lex == LEXEOF || lex == LEXASSIGN) break;
	    EatTok();
	}
    }
    InMacro--;

    /* unwind macro types */
    while (MacroTypes != NULL) {
	register TNAME *ptr;
	ptr = MacroTypes;
	MacroTypes = ptr->Next;
	FreeTNAME(ptr);
    }
	

#if	DODEBUG
    if (SymblDebug>3) SymDump("MACRO",NewOne,0);
    MPRINTF(("UseMacro done\n"));
#endif	DODEBUG
    return NewOne;
}

/*
 * Add a macro symbol to the list and return a valid pointer or NULL
 */
static MPARSE *MacroAdd(Ptr,flg)
MPARSE *Ptr;
int flg;
{
    static char mod[] = "MacroAdd";
#define MAXPARTS 200
    static MPARSE *Parts[MAXPARTS];
    static int PartCnt;
#define MAXALTS 200
    static MPARSE *Alts[MAXALTS];
    static int AltCnt;
    MPARSE *NewOne;

    /* see if we are initializing if so handle and return */
    if (Ptr == NULL && flg == 0) {
	AltCnt = PartCnt = 0;
	return NULL;
    }

    /* see if we need to take the current sequence list and build A node */
    if (flg != 0 && PartCnt > 0) {
	if (PartCnt > 1) {
	    /* yes */
	    register int i;
	    NewOne = MakeMPARSE();
	    NewOne->Code = MACRO_LIST;
	    NewOne->Part.List.Cnt = PartCnt;
	    i = sizeof(MPARSE *) * PartCnt;
	    NewOne->Part.List.List = (MPARSE **) Malloc(i,1);
	    for(i=0;i<PartCnt;i++)
		NewOne->Part.List.List[i] = Parts[i];
	}
	else {
	    /* no, just  use the pointer */
	    NewOne = Parts[0];
	}

	/* add to alternative list */
	if (AltCnt >= MAXALTS-1) {
	    ParseErr(mod,2,"Too many macro alternativess (max %d)",MAXALTS);
	    return NULL;
	}
	Alts[AltCnt++] = NewOne;
	PartCnt = 0;
	Parts[0] = NULL;
    }

    if (Ptr != NULL) {
	/* add the new piece to the current list */
	if (PartCnt >= MAXPARTS-1) {
	    ParseErr(mod,2,"Too many macro components (max %d)",MAXPARTS);
	    return NULL;
	}
	Parts[PartCnt++] = Ptr;
	return Ptr;
    }
    else {
	/* finish up */
	if (AltCnt == 1)
	    return Alts[0];
	else if (AltCnt > 1) {
	    register int i;
	    NewOne = MakeMPARSE();
	    NewOne->Code = MACRO_ALT;
	    NewOne->Part.Alt.Cnt = AltCnt;
	    i = sizeof(MPARSE *) * AltCnt;
	    NewOne->Part.Alt.List = (MPARSE **) Malloc(i,1);
	    for(i=0;i<AltCnt;i++)
		NewOne->Part.Alt.List[i] = Alts[i];
	    return NewOne;
	}
	else if (PartCnt == 1)
	    return Parts[0];
	return NULL;
    }
}

/*
 * quick and dirty recursive tree search for a macro symbol
 */
static MPARSE *FindMacroSymbol(Tree,Nam,SkipP)
MPARSE *Tree;
char *Nam;
MPARSE *SkipP;
{
    register MPARSE *tmp;

    if (Tree == NULL)
	return NULL;
    if (SkipP != NULL && Tree == SkipP)
	return NULL;

    if (Tree->Code == MACRO_SYMBOL) {
	/* check for match */
	if (Nam == Tree->Symbol)
	    return Tree;
    }
    else if (Tree->Code == MACRO_ALT) {
	/* check alternative list */
	register int i;
	for(i=0;i<Tree->Part.Alt.Cnt;i++) {
	    if (SkipP != NULL && Tree->Part.Alt.List[i] == SkipP)
		continue;
	    tmp = FindMacroSymbol(Tree->Part.Alt.List[i],Nam,SkipP);
	    if (tmp != NULL) return tmp;
	}
    }
    else if (Tree->Code == MACRO_LIST) {
	/* check sequence list */
	register int i;
	for(i=0;i<Tree->Part.List.Cnt;i++) {
	    if (SkipP != NULL && Tree->Part.Alt.List[i] == SkipP)
		continue;
	    tmp = FindMacroSymbol(Tree->Part.List.List[i],Nam,SkipP);
	    if (tmp != NULL) return tmp;
	}
    }

    return NULL;
}

/*****************************************************************
 *			PARSER
 *****************************************************************
 */

/* cache of compressed strings "empty", "value", "type" */
static char *EMPTYstr = NULL;
static char *VALUEstr = NULL;
static char *TYPEstr = NULL;

/*
 * parse a macro symbol, one of:
 *      empty
 *	type
 *      type ( name )
 *	value ( macrotype )
 *	value ( name macrotype )
 *	value ( VALUE macrotype )
 *	"string"
 */
static MPARSE *P_MacroSymbol()
{
    static char mod[] = "P_MacroSymbol";
    MPARSE *NewOne;
    int lex;
    TNAME *TNam;

    PPRINTF((">>Parse MacroSymbol\n"));

    NewOne = MakeMPARSE();

    lex = GetTok();
    if (lex == LEXCSTRING) {
	/* lexeme */
	NewOne->Code = MACRO_LEX;
	NewOne->Part.Lexeme = CopyStr(CurToken);
	EatTok();
	return NewOne;
    }
    else if (lex == LEXLIDENT) {
	if (CurToken == EMPTYstr) {
	    NewOne->Code = MACRO_EMPTY;
	    EatTok();
	    return NewOne;
	}
	else if (CurToken == VALUEstr) {
	    NewOne->Code = MACRO_VALUE;
	    EatTok();
	    /* get value type information */
	    if (PChk(mod,LEXLEFTPAREN)) {
		ParseErr(mod,1,"Incomplete macro value specification");
		FreeMPARSE(NewOne);
		return NULL;
	    }

	    lex = GetTok();
	    /* value (VALUE type) */
	    if (lex == LEXVALUE) {
		NewOne->Part.Value.IsValue = 1;
		EatTok();
		if (lex == LEXUIDENT) {
		    NewOne->Part.Value.TName = CopyStr(CurToken);
		    NewOne->Part.Value.Type = TypeBind(CurToken,0);
		    EatTok();
		}
		else
		    NewOne->Part.Value.Type = P_Type();
	    }
	    /* value (id type) */
	    else if (lex == LEXLIDENT) {
		NewOne->Part.Value.Name = CopyStr(CurToken);
		EatTok();
		lex = GetTok();
		if (lex == LEXUIDENT) {
		    NewOne->Part.Value.TName = CopyStr(CurToken);
		    NewOne->Part.Value.Type = TypeBind(CurToken,0);
		    EatTok();
		}
		else
		    NewOne->Part.Value.Type = P_Type();
	    }
	    /* value (type) */
	    else {
		if (lex == LEXUIDENT) {
		    NewOne->Part.Value.TName = CopyStr(CurToken);
		    NewOne->Part.Value.Type = TypeBind(CurToken,0);
		    EatTok();
		}
		else
		    NewOne->Part.Value.Type = P_Type();
	    }

	    if (PChk(mod,LEXRIGHTPAREN)) {
		FreeMPARSE(NewOne);
		return NULL;
	    }
	    return NewOne;
	}
	else if (CurToken == TYPEstr) {
	    NewOne->Code = MACRO_TYPE;
	    EatTok();
	    lex = GetTok();
	    if (lex == LEXLEFTPAREN) {
		if (PChk(mod,LEXLEFTPAREN)) {
		    ParseErr(mod,1,"Incomplete macro type specification");
		    FreeMPARSE(NewOne);
		    return NULL;
		}
		lex = GetTok();
		/* type (TYPE type) */
		if (lex == LEXTYPE) {
		    NewOne->Part.Type.IsType = 1;
		    EatTok();
		    lex = GetTok();
		    if (lex == LEXUIDENT) {
			NewOne->Part.Type.TName = CopyStr(CurToken);
			NewOne->Part.Type.Type = TypeBind(CurToken,0);
			EatTok();
		    }
		    else
			NewOne->Part.Type.Type = P_Type();
		}
		/* type (type) */
		else {
		    if (lex == LEXUIDENT) {
			NewOne->Part.Type.TName = CopyStr(CurToken);
			NewOne->Part.Type.Type = TypeBind(CurToken,0);
			EatTok();
		    }
		    else
			NewOne->Part.Type.Type = P_Type();
		}
		if (PChk(mod,LEXRIGHTPAREN)) {
		    FreeMPARSE(NewOne);
		    return NULL;
		}
		return NewOne;
	    }
	}
	else {
	    ParseErr(mod,1,"Unknown macro reference '%s'",CurToken);
	    EatTok();
	    FreeMPARSE(NewOne);
	    return NULL;
	}
    }
    else if (lex == LEXUIDENT) {
	/* symbol */
	NewOne->Code = MACRO_SYMBOL;
	NewOne->Symbol = CopyStr(CurToken);
	EatTok();
	return NewOne;
    }
    else {
	FreeMPARSE(NewOne);
	/* fall thru */
    }
    return NULL;
}

/*
 * parse named numbered list. The INTEGER record type is provided.
 */
static P_NamedNumberList(Base)
register SNODE *Base;
{
    static char mod[] = "P_NamedNumberList";
    int lex,nextlex;
    long val;
    char *nam;
    SNODE *NewOne;
    register SNODE *Tmp;

    PPRINTF((">>Parse Named Number List BEGIN\n"));

    if (PChk(mod,LEXLEFTCURLY)) return;
    for(;;) {
	lex = GetTok();
	nextlex = PeekTok();
	if (lex != LEXLIDENT || nextlex != LEXLEFTPAREN)
	    break;
	/* we have a name */
	nam = CopyStr(CurToken);
	EatTok();
	if (PChk(mod,LEXLEFTPAREN)) break;
	lex = GetTok();
	if (lex == LEXHYPEN) {
	    EatTok();
	    if (PNum(mod,&val)) break;
	    val = -val;
	}
	else if (lex == LEXNUMBER) {
	    if (PNum(mod,&val)) break;
	}
	else
	    break;
	if (PChk(mod,LEXRIGHTPAREN)) break;
	/* we have a named and a value */
	NewOne = MakeSNODE();
	NewOne->Code = SYM_VALINTEGER;
	NewOne->Desc.ValInt.val = val;
	NewOne->Desc.ValInt.id = nam;

	if (Base->Code == SYM_INTEGER) {
	    if (Base->Desc.TypInt.Min > val)
		Base->Desc.TypInt.Min = val;
	    if (Base->Desc.TypInt.Max < val)
		Base->Desc.TypInt.Max = val;
	    Base->Desc.TypInt.HasRange = 1;
	    /* put it into list sorted from low to high */
	    if ((Tmp = Base->Desc.TypInt.NamedList) == NULL)
		Base->Desc.TypInt.NamedList = NewOne;
	    else if (val < Tmp->Desc.ValInt.val) {
		NewOne->Desc.ValInt.vnext = Tmp;
		Base->Desc.TypInt.NamedList = NewOne;
	    }
	    else {
		for(;Tmp->Desc.ValInt.vnext;Tmp = Tmp->Desc.ValInt.vnext) {
		    if (val < Tmp->Desc.ValInt.val) {
			/* new value is before next value */
			NewOne->Desc.ValInt.vnext = Tmp->Desc.ValInt.vnext;
			Tmp->Desc.ValInt.vnext = NewOne;
			Tmp = NULL;
		    }
		}
		if (Tmp != NULL) {
		    /* add to end */
		    Tmp->Desc.ValInt.vnext = NewOne;
		}
	    }
	}
	else {
	    ParseErr(mod,1,"internal error -- Non-integer named list");
	    EatTok();
	}
	lex = GetTok();
	if (lex != LEXCOMMA) break;
	EatTok();
    }
    if (PChk(mod,LEXRIGHTCURLY)) return;
    PPRINTF((">>Parse Named Number List END\n"));
}

/*
 * SEQUENCE Element Type List
 */
static SNODE *P_EleTypeList()
{
    static char mod[] = "P_EleTypeList";
    int lex;
    SNODE *Head,*Tail,*NewOne,*Tmp;
    char *nam;

    PPRINTF((">>Parse Element Type List BEGIN\n"));

    Head = NULL;
    if (PChk(mod,LEXLEFTCURLY)) return Head;
    for(;;) {
	lex = GetTok();
	if (lex == LEXRIGHTCURLY) break;
	if (lex == LEXLIDENT) {
	    nam = CopyStr(CurToken);
	    EatTok();
	    lex = GetTok();
	    Tmp = P_Type();
	}
	else {
	    nam = NULL;
	    Tmp = P_Type();
	}
	/* add it */
	NewOne = MakeSNODE();
	NewOne->Code = SYM_SEQ;
	NewOne->Desc.Seq.Sname = nam;
	NewOne->Desc.Seq.TypPtr = Tmp;
	if (Head == NULL)
	    Head = Tail = NewOne;
	else {
	    Tail->Desc.Seq.SNext = NewOne;
	    Tail = NewOne;
	}
	/* handle comma if there */
	lex = GetTok();
	if (lex != LEXCOMMA) break;
	EatTok();
    }
    PChk(mod,LEXRIGHTCURLY);
    PPRINTF((">>Parse Element Type List END\n"));
    return Head;
}

/*
 * CHOICE alternative type list
 */
static SNODE *P_AltTypeList()
{
    static char mod[] = "P_AltTypeList";
    int lex;
    SNODE *Head,*Tail,*NewOne,*Tmp;
    char *nam;

    PPRINTF((">>Parse Alternate Type List BEGIN\n"));

    Head = NULL;
    if (PChk(mod,LEXLEFTCURLY)) return Head;
    for(;;) {
	lex = GetTok();
	if (lex == LEXRIGHTCURLY) break;
	if (lex == LEXLIDENT) {
	    nam = CopyStr(CurToken);
	    EatTok();
	    lex = GetTok();
	    Tmp = P_Type();
	}
	else {
	    nam = NULL;
	    Tmp = P_Type();
	}
	/* add it */
	NewOne = MakeSNODE();
	NewOne->Code = SYM_CHOICE;
	NewOne->Desc.Choice.Cname = CopyStr(nam);
	NewOne->Desc.Choice.TypPtr = Tmp;
	if (Head == NULL)
	    Head = Tail = NewOne;
	else {
	    Tail->Desc.Choice.CNext = NewOne;
	    Tail = NewOne;
	}
	/* handle comma if there */
	lex = GetTok();
	if (lex != LEXCOMMA) break;
	EatTok();
    }
    PChk(mod,LEXRIGHTCURLY);
    PPRINTF((">>Parse Alternate Type List END\n"));
    return Head;
}

/*
 * parse a macro alternative list
 */
static MPARSE *P_MacroAltList()
{
    static char mod[] = "P_MacroAltList";
    int lex, nextlex, dosplit;
    MPARSE *Tmp;

    dosplit = 0;
    MacroAdd(NULL,0);

    for(;;) {
	lex = GetTok();
	if (lex == LEXCSTRING || lex == LEXLIDENT || lex == LEXUIDENT) {
	    /* get the symbol */
	    if ((Tmp = P_MacroSymbol()) == NULL)
		return NULL;
	    /* add it */
	    if ((Tmp = MacroAdd(Tmp,dosplit)) == NULL)
		return NULL;
	    dosplit = 0;
	}
	else {
	    ParseErr(mod,1,"Unexpected part in MACRO list %s",StrTok(lex));
	    EatTok();
	    break; /* huh? */
	}
	lex = GetTok();
	if (lex == LEXALTERNATE) {
	    dosplit++;
	    EatTok();
	    continue;
	}
	/* see if it is time to terminate */
	if (lex == LEXEND) break;
	nextlex = PeekTok();
	if (nextlex == LEXNOTATION) break;
	if (lex == LEXUIDENT && nextlex == LEXASSIGN) break;
    }

    /* finish up */
    Tmp = MacroAdd(NULL,1);
    return Tmp;
}

/*
 * parse object identifier list
 */
static SNODE *P_ObjIDList()
{
    static char mod[] = "P_ObjIDList";
    int lex;
    SNODE *Head,*Tail,*NewOne;

    Head = NULL;

    if (PChk(mod,LEXLEFTCURLY)) return Head;
    for(;;) {
	lex = GetTok();
	if (lex == LEXRIGHTCURLY) break;
	if (lex == LEXLIDENT) {
	    NewOne = MakeSNODE();
	    NewOne->Code = SYM_VALOBJID;
	    NewOne->Desc.ValObjId.idstr = CopyStr(CurToken);
	    EatTok();
	    /* check for identifier value */
	    lex = GetTok();
	    if (lex == LEXLEFTPAREN) {
		EatTok();
		if (PNum(mod,&NewOne->Desc.ValObjId.idnum)) continue;
		if (PChk(mod,LEXRIGHTPAREN)) continue;
	    }
	    else
		NewOne->Desc.ValObjId.idnum = -1;
	}
	else if (lex == LEXNUMBER) {
	    NewOne = MakeSNODE();
	    NewOne->Code = SYM_VALOBJID;
	    PNum(mod,&NewOne->Desc.ValObjId.idnum);
	}
	else {
	    ParseErr(mod,1,"Expected %s or %s, got %s",
		     StrTok(LEXNUMBER),StrTok(LEXLIDENT),StrTok(lex));
	    EatTok();
	    break;
	}
	/* add this new one to the list */
	if (Head == NULL)
	    Head = Tail = NewOne;
	else {
	    Tail->Desc.ValObjId.idnext = NewOne;
	    Tail = NewOne;
	}
    }
    PChk(mod,LEXRIGHTCURLY);
    return Head;
}

/*
 * parse supporting macro production
 */
static P_MacroSupport(typptr,valptr)
MPARSE *typptr,*valptr;
{
    static char mod[] = "P_MacroSupport";
    int lex;
    char *SymName;
    int cnt;
    MPARSE *Head, *Tmp, *SymTop;

    PPRINTF((">>Parse MacroSupport\n"));

    lex = GetTok();
    if (lex != LEXUIDENT) {
	PChk(mod,LEXUIDENT);
	return;
    }
    SymName = CopyStr(CurToken);
    EatTok();
    if (PChk(mod,LEXASSIGN)) return;

    /* parse the alternate list */
    Head = P_MacroAltList();

    /* see if it is self referring */
    Tmp = FindMacroSymbol(Head,SymName,NULL);
    if (Tmp != NULL) {
	/* refers to self */
	SymTop = FindMacroSymbol(typptr,SymName,NULL);
	if (SymTop == NULL) SymTop = FindMacroSymbol(valptr,SymName,NULL);
	Tmp->Part.SymTop = SymTop;
	if (SymTop != NULL) SymTop->Recurse = 1;
    }

    /* patch it into the macro list */
    for(cnt=0;;cnt++) {
	Tmp = FindMacroSymbol(typptr,SymName,Tmp);
	if (Tmp == NULL) Tmp = FindMacroSymbol(valptr,SymName,Tmp);
	if (Tmp == NULL) break;
	/* copy it back */
	Tmp->Code = Head->Code;
	Tmp->Part = Head->Part;
    }
    if (cnt<=0)
	ParseErr(mod,1,"Unknown macro symbol %s",SymName);
    UnMakeStr(SymName);
}

/*
 * parse a macro type or value notation definition
 */
static MPARSE *P_MacroNotation(isvalue)
int isvalue;
{
    static char mod[] = "P_MacroNotation";
    MPARSE *Tmp;

    PPRINTF((">>Parse MacroNotation BEGIN\n"));

    if (PChk(mod,isvalue ? LEXVALUE : LEXTYPE)) return NULL;
    if (PChk(mod,LEXNOTATION)) return NULL;
    if (PChk(mod,LEXASSIGN)) return NULL;

    Tmp =  P_MacroAltList();

    PPRINTF((">>Parse MacroNotation BEGIN\n"));
    return Tmp;
}

/*
 * parse a value
 */
static SNODE *P_Value(Ptr)
SNODE *Ptr;
{
    static char mod[] = "P_Value";
    int lex, nextlex;
    SNODE *NewOne, *Tmp, *IdTyp;

    PPRINTF((">>Parse Value BEGIN\n"));

    IdTyp = NULL;
    NewOne = NULL;
    lex = GetTok();

    switch(Ptr->Code) {
	case SYM_INTEGER:
	    /* signed number or identifier */
	    nextlex = PeekTok();
	    if (lex == LEXHYPEN && nextlex == LEXNUMBER) {
		EatTok();
		NewOne = MakeSNODE();
		NewOne->Code = SYM_VALINTEGER;
		NewOne->Desc.ValInt.val = - myatoi(CurToken);
		EatTok();
	    }
	    else if (lex == LEXNUMBER) {
		NewOne = MakeSNODE();
		NewOne->Code = SYM_VALINTEGER;
		NewOne->Desc.ValInt.val = myatoi(CurToken);
		EatTok();
	    }
	    else if (lex == LEXHSTRING) {
		NewOne = MakeSNODE();
		NewOne->Code = SYM_VALINTEGER;
		NewOne->Desc.ValInt.val = atoh(CurToken);
		EatTok();
	    }
	    else if (lex == LEXLIDENT) {
		/* see if it is a named type of the integer */
		for(Tmp = Ptr->Desc.TypInt.NamedList;Tmp != NULL;
		    Tmp = Tmp->Desc.ValInt.vnext) {
		    if (Tmp->Desc.ValInt.id != NULL)
			if (Tmp->Desc.ValInt.id == CurToken) {
			    NewOne = MakeSNODE();
			    NewOne->Code = SYM_VALINTEGER;
			    NewOne->Desc.ValInt.val = Tmp->Desc.ValInt.val;
			    NewOne->Desc.ValInt.id = CopyStr(CurToken);
			    EatTok();
			    break;
			}
		}
		if (NewOne != NULL) break;
		/* do normal checks */
		/* if lower identifier, see if a defined value */
		if (lex == LEXLIDENT)
		    IdTyp = ValueBind(CurToken,Ptr,1);
		if (IdTyp != NULL) {
		    if (IdTyp->Code == SYM_VALINTEGER) {
			NewOne = IdTyp;
			EatTok();
		    }
		    /* else fall thru and error */
		}
		else {
		    /* force a value binding */
		    NewOne = ValueBind(CurToken,Ptr,0);
		    EatTok();
		}
	    }
	    else {
		ParseErr(mod,1,"Value does not match INTEGER type");
		EatTok();
	    }
	    break;
	case SYM_OSTRING:
	    if (lex == LEXCSTRING) {
		NewOne = MakeSNODE();
		NewOne->Code = SYM_VALOSTRING;
		NewOne->Desc.ValStr = CopyStr(CurToken);
		EatTok();
	    }
	    else if (lex == LEXHSTRING) {
		NewOne = MakeSNODE();
		NewOne->Code = SYM_VALOSTRING;
		NewOne->Desc.ValStr = CopyStr(CurToken);
		EatTok();
	    }
	    else if (lex == LEXLIDENT) {
		/* if lower identifier, see if a defined value */
		if (lex == LEXLIDENT)
		    IdTyp = ValueBind(CurToken,Ptr,1);
		if (IdTyp != NULL) {
		    if (IdTyp->Code == SYM_VALOSTRING) {
			NewOne = IdTyp;
			EatTok();
		    }
		    /* else fall thru and error */
		}
		else {
		    /* force a value binding */
		    NewOne = ValueBind(CurToken,Ptr,0);
		    EatTok();
		}
	    }
	    else {
		ParseErr(mod,1,"Value does not match OCTET STRING type");
		EatTok();
	    }
	    break;
	case SYM_NULL:
	    if (lex == LEXNULL) {
		NewOne = MakeSNODE();
		NewOne->Code = SYM_VALINTEGER;
		EatTok();
	    }
	    else if (lex == LEXLIDENT) {
		/* if lower identifier, see if a defined value */
		if (lex == LEXLIDENT)
		    IdTyp = ValueBind(CurToken,Ptr,1);
		if (IdTyp != NULL) {
		    if (IdTyp->Code == SYM_VALNULL) {
			NewOne = IdTyp;
			EatTok();
		    }
		    /* else fall thru and error */
		}
		else {
		    /* force a value binding */
		    NewOne = ValueBind(CurToken,Ptr,0);
		    EatTok();
		}
	    }
	    else {
		ParseErr(mod,1,"Value does not match NULL type");
		EatTok();
	    }
	    break;
	case SYM_OBJID:
	    if (lex == LEXLEFTCURLY)
		NewOne = P_ObjIDList();
	    else if (lex == LEXLIDENT) {
		/* if lower identifier, see if a defined value */
		if (lex == LEXLIDENT)
		    IdTyp = ValueBind(CurToken,Ptr,1);
		if (IdTyp != NULL) {
		    if (IdTyp->Code == SYM_VALOBJID) {
			NewOne = IdTyp;
			EatTok();
		    }
		    /* else fall thru and error */
		}
		else {
		    /* force a value binding */
		    NewOne = ValueBind(CurToken,Ptr,0);
		    EatTok();
		}
	    }
	    else {
		ParseErr(mod,1,"Value does not match OBJECT IDENTIFIER type");
		EatTok();
	    }
	    break;
	case SYM_CHOICE:
	    /* find if any choices match */
	    /* if lower identifier, see if a defined value */
	    if (lex == LEXLIDENT)
		IdTyp = ValueBind(CurToken,Ptr,1);
	    Tmp = FindValueType(Ptr,lex,CurToken,IdTyp);
	    if (Tmp != NULL) {
		/* a match, use it */
		NewOne = P_Value(Tmp);
	    }
	    else if (lex == LEXLIDENT) {
		/* force a value binding */
		NewOne = ValueBind(CurToken,Ptr,0);
		EatTok();
	    }
	    else {
		ParseErr(mod,1,"Value does not match any CHOICEs");
		EatTok();
	    }
	    break;
	case SYM_DELAY:
	    if (lex == LEXLIDENT) {
		/* force a value binding */
		NewOne = ValueBind(CurToken,Ptr,0);
		EatTok();
	    }
	    else {
		ParseErr(mod,1,"Value using delayed TYPE. Please re-write");
		EatTok();
	    }
	    break;
	default:
	    ParseErr(mod,1,"Invalid TYPE %d",Ptr->Code);
#if	DODEBUG
	    if (SymblDebug) SymDump(mod,Ptr,0);
#endif	DODEBUG
	    EatTok();
	    break;
    }

    PPRINTF((">>Parse Value END\n"));
    return NewOne;
}

/*
 * Parse a size or range restriction on the type
 */
static SNODE *P_TypeRange(Ptr,New)
SNODE *Ptr;
int New;
{
    static char mod[] = "P_TypeRange";
    SNODE *NewOne;
    int lex;

    PPRINTF((">>Parse TypeRange BEGIN\n"));

/* range inside of range check needed */

    NewOne = Ptr;

    switch(Ptr->Code) {
	case SYM_INTEGER:
	    lex = GetTok();
	    if (lex == LEXLEFTPAREN) {
		/* has a scalar range */
		if (New) {
		    NewOne = MakeSNODE();
		    NewOne->Name = CopyStr(Ptr->Name);
		    NewOne->Code = SYM_INTEGER;
		}
		if (PChk(mod,LEXLEFTPAREN)) break;
		if (PNum(mod,&NewOne->Desc.TypInt.Min)) break;
		if (PChk(mod,LEXRANGE)) break;
		/* handle MAX */
/*** FIX ****/
		lex = GetTok();
		if (lex == LEXUIDENT && strcmp(CurToken,"MAX") == 0) {
		    NewOne->Desc.TypInt.Max = -1;
		    EatTok();
		}
		else {
		    if (PNum(mod,&NewOne->Desc.TypInt.Max)) break;
		}
		if (PChk(mod,LEXRIGHTPAREN)) break;
		NewOne->Desc.TypInt.HasRange = 1;
	    }
	    else if (lex == LEXLEFTCURLY) {
		/* has a named number list */
		if (New) {
		    NewOne = MakeSNODE();
		    NewOne->Name = CopyStr(Ptr->Name);
		    NewOne->Code = SYM_INTEGER;
		}
		P_NamedNumberList(NewOne);
	    }
	    break;
	case SYM_OSTRING:
	    lex = GetTok();
	    if (lex == LEXLEFTPAREN) {
		/* has a size modifier */
		if (New) {
		    NewOne = MakeSNODE();
		    NewOne->Name = CopyStr(Ptr->Name);
		    NewOne->Code = SYM_OSTRING;
		}
		if (PChk(mod,LEXLEFTPAREN)) break;
		if (PChk(mod,LEXSIZE)) break;
		if (PChk(mod,LEXLEFTPAREN)) break;
		if (PNum(mod,&NewOne->Desc.TypOStr.MinSiz)) break;
		NewOne->Desc.TypOStr.MaxSiz = NewOne->Desc.TypOStr.MinSiz;
		lex = GetTok();
		if (lex == LEXRANGE) {
		    if (PChk(mod,LEXRANGE)) break;
		    if (PNum(mod,&NewOne->Desc.TypOStr.MaxSiz)) break;
		}
		if (PChk(mod,LEXRIGHTPAREN)) break;
		if (PChk(mod,LEXRIGHTPAREN)) break;
		NewOne->Desc.TypOStr.HasSize = 1;
	    }
	    break;
	default:
	    /* no range or size */
	    break;
    }
    PPRINTF((">>Parse TypeRange END\n"));
    return NewOne;
}

/*
 * Parse any Type
 */
static SNODE *P_Type()
{
    static char mod[] = "P_Type";
    int lex,nextlex,nogood;
    SNODE *NewOne;


    PPRINTF((">>Parse AnyType BEGIN\n"));

    /* no type */
    NewOne = NULL;

    /* load up a token */
    lex = GetTok();
    switch(lex) {
	/* BOOLEAN not supported */
	case LEXINTEGER:
	    VPPRINTF(("AnyType INTEGER\n"));
	    EatTok();
	    NewOne = MakeSNODE();
	    NewOne->Code = SYM_INTEGER;
	    NewOne = P_TypeRange(NewOne,0);
	    break;
	/* BIT STRING not supported */
	case LEXOCTET:
	    nextlex = PeekTok();
	    if (nextlex == LEXSTRING) {
		VPPRINTF(("AnyType OCTET STRING\n"));
		EatTok();
		EatTok();
		NewOne = MakeSNODE();
		NewOne->Code = SYM_OSTRING;
		NewOne = P_TypeRange(NewOne,0);
	    }
	    break;
	case LEXNULL:
	    VPPRINTF(("AnyType NULL\n"));
	    EatTok();
	    NewOne = MakeSNODE();
	    NewOne->Code = SYM_NULL;
	    break;
	case LEXSEQUENCE:
	    VPPRINTF(("AnyType SEQUENCE\n"));
	    EatTok();
	    lex = GetTok();
	    if (lex == LEXOF) {
		/* sequence of */
		EatTok();
		NewOne = MakeSNODE();
		NewOne->Code = SYM_SEQOF;
		NewOne->Desc.TypSeqOf = P_Type();
	    }
	    else {
		/* plain or expanded sequence */
		lex = GetTok();
		if (lex == LEXLEFTCURLY) {
		    NewOne = P_EleTypeList();
		}
		else {
		    NewOne = MakeSNODE();
		    NewOne->Code = SYM_SEQ;
		}
	    }
	    break;
	/* SET and SET OF not supported */
	case LEXCHOICE:
	    VPPRINTF(("AnyType CHOICE\n"));
	    EatTok();
	    NewOne = P_AltTypeList();
	    break;
	/* selection (lower case ident) followed by "<" not supported */
	case LEXLEFTBRACK:
	    VPPRINTF(("AnyType '['\n"));
	    EatTok();
	    /* handle class */
	    lex = GetTok();
	    switch (lex) {
		case LEXAPPLICATION:
		    EatTok();
		    break;
		default:
		    break;
	    }
	    /* handle class number */
	    lex = GetTok();
	    if (lex != LEXNUMBER) {
		ParseErr(mod,1,"Expected CLASS NUMBER, got %s",StrTok(lex));
		EatTok();
		break;
	    }
	    EatTok();
	    if (PChk(mod,LEXRIGHTBRACK)) break;
	    /* handle implicit */
	    lex = GetTok();
	    if (lex == LEXIMPLICIT) {
		EatTok();
	    }
	    /* make sure we do not have a recursive tag info */
	    lex = GetTok();
	    if (lex == LEXLEFTBRACK) {
		ParseErr(mod,1,"Secondary tag information");
		EatTok();
		break;
	    }
	    /* parse type */
	    NewOne = P_Type();
	    /* drop tag information since it is builtin */
	    break;
	/* ANY not supported */
	case LEXOBJECT:
	    nextlex = PeekTok();
	    if (nextlex == LEXIDENTIFIER) {
		EatTok();
		EatTok();
		NewOne = MakeSNODE();
		NewOne->Code = SYM_OBJID;
	    }
	    break;
	case LEXUIDENT:
	    /* UsefulType is an module defined type */
	    /* DefinedType is an externally defined type */
	    /* CharacterSetString is an internally defined type
	     * which we do not support at the moment
	     */
	    VPPRINTF(("AnyType UIDENT '%s'\n",CurToken));
	    /* find type in type list and return */
	    NewOne = TypeBind(CurToken,0);
	    if (NewOne->Code == SYM_MACRO) {
		PPRINTF(("AnyType using MACRO '%s'\n",CurToken));
		EatTok();
		NewOne = UseMacro(NewOne);
	    }
	    else {
		if (NewOne->Code == SYM_DELAY) {
		    PPRINTF(("AnyType using delay type '%s'\n",CurToken));
		    EatTok();
		}
		else {
		    PPRINTF(("AnyType using type '%s'\n",CurToken));
		    EatTok();
		    NewOne = P_TypeRange(NewOne,1);
		}
	    }
	    break;
	default:
	    ParseErr(mod,1,"Expected TYPE, got %s",StrTok(lex));
	    return NULL;
    }

    if (NewOne == NULL) {
	ParseErr(mod,1,"Expected TYPE, got %s",StrTok(lex));
    }

    PPRINTF((">>Parse AnyType END\n"));
    return NewOne;
}

/*
 * parse a macro body
 *	typeproduction valueproduction supportproduction*
 */
static SNODE *P_MacroBody()
{
    int lex,nextlex;
    MPARSE *TypPart, *ValPart;
    SNODE *NewOne;

    PPRINTF((">>Parse MacroBody\n"));
    TypPart = P_MacroNotation(0);
    ValPart = P_MacroNotation(1);
    for(;;) {
	lex = GetTok();
	nextlex = PeekTok();
	if (lex != LEXUIDENT || nextlex != LEXASSIGN)
	    break;
	P_MacroSupport(TypPart,ValPart);
    }

    /* make a macro symbol type */
    NewOne = MakeSNODE();
    NewOne->Code = SYM_MACRO;
    NewOne->Desc.Macro.TypPart = TypPart;
    NewOne->Desc.Macro.ValPart = ValPart;


#if	DODEBUG
    if(MacroDebug) MacroDump(TypPart,"TYPE",0);
    if(MacroDebug) MacroDump(ValPart,"VALUE",0);
    PPRINTF((">>Parse MacroBody END\n"));
#endif	DODEBUG

    return NewOne;
}

/*
 * value assignment
 *	name type "::=" value
 */
static P_ValueAssign()
{
    static char mod[] = "P_ValueAssign";
    register SNODE *TypPtr, *ValPtr;
    char *nam;
    register VNAME *VNam;
    int lex;

    PPRINTF((">>Parse ValueAssign BEGIN\n"));

    lex = GetTok();
    if (lex != LEXLIDENT) {
	ParseErr(mod,1,"Expected VALUE name, got %s",StrTok(lex));
	EatTok();
	return;
    }
    nam = CopyStr(CurToken);
    PPRINTF((">>VALUE '%s'\n",nam));
    EatTok();

    /* look up the value name */
    VNam = FindValue(nam,NULL);
    if (VNam != NULL && VNam->ValPtr->Code != SYM_VALDELAY)
	ParseErr(mod,1,"Redefinition of value %s on line %d",
		 nam,VNam->ValPtr->Desc.ValDelay.LineNum);

    /* parse the type */
    TypPtr = P_Type();
#if	DODEBUG
    if (SymblDebug>4) SymDump("VALUE P_Type",TypPtr,0);
#endif	DODEBUG

    /* error recovery */
    for(;;) {
	lex = GetTok();
	if (lex == LEXASSIGN || lex == LEXEOF || lex == LEXEND)
	    break;
	PError(mod,LEXASSIGN,lex);
	EatTok();
    }
    if (PChk(mod,LEXASSIGN)) return;

    /* if we do not have a type then fudge it */
    if (TypPtr == NULL) {
	lex = GetTok();
	if (lex == LEXLEFTCURLY) {
	    TypPtr = MakeSNODE();
	    TypPtr->Code = SYM_OBJID;
	}
	else {
	    TypPtr = MakeSNODE();
	    TypPtr->Code = SYM_INTEGER;
	}
    }

    /* parse the value */
    ValPtr = P_Value(TypPtr);
#if	DODEBUG
    if (SymblDebug>4) SymDump("VALUE P_Value",ValPtr,0);
#endif	DODEBUG

    /*
     * update value information if we got one
     */
    if (ValPtr != NULL) {
	/* lookup the value name */
	VNam = FindValue(nam,NULL);
	if (VNam == NULL) {
	    /* not defined, add new type */
	    AddValue(nam,TypPtr,ValPtr);
	}
	else {
	    /* is defined, either delay or replacement value */
	    if (VNam->ValPtr->Code == SYM_VALDELAY) {
		/* delay binding */
		/* copy value information */
		VNam->ValPtr->Code = ValPtr->Code;
		VNam->ValPtr->Desc = ValPtr->Desc;
		/* free temporary type node */
		FreeSNODE(ValPtr);
	    }
	    else {
		/* replacement */
		/* free old value */
		FreeSNODE(VNam->ValPtr);
		/* use new value information */
		VNam->ValPtr = ValPtr;
	    }
	}
    }

    PPRINTF((">>Parse ValueAssign END\n"));
}

/*
 * type assignment
 *	name "::=" type
 */
static P_TypeAssign()
{
    static char mod[] = "P_TypeAssign";
    register SNODE *TypPtr;
    register TNAME *TNam;
    register char *nam;
    int lex;

    PPRINTF((">>Parse TypeAssign BEGIN\n"));

    lex = GetTok();
    if (lex != LEXUIDENT) {
	ParseErr(mod,1,"Expected TYPE name, got %s",StrTok(lex));
	EatTok();
	return;
    }
    nam = CopyStr(CurToken);
    PPRINTF((">>TYPE '%s'\n",nam));
    EatTok();

    PChk(mod,LEXASSIGN);

    /* look up the type name */
    TNam = FindType(nam,NULL);
    if (TNam != NULL && TNam->TypPtr->Code != SYM_DELAY)
	ParseErr(mod,1,"Redefinition of type %s on line %d",
		 nam,TNam->TypPtr->Desc.Delay.LineNum);

    /* parse the type */
    TypPtr = P_Type();
#if	DODEBUG
    if (SymblDebug>4) SymDump("TYPE P_Type",TypPtr,0);
#endif	DODEBUG

    /*
     * update type information if we got one
     */
    if (TypPtr != NULL) {
	/* lookup the type name */
	TNam = FindType(nam,NULL);
	if (TNam == NULL) {
	    /* not defined, add new type */
	    AddType(nam,TypPtr);
	}
	else {
	    /* is defined, either delay or replacement type */
	    if (TNam->TypPtr->Code == SYM_DELAY) {
		/* delay binding */
		/* copy type information */
		TNam->TypPtr->Code = TypPtr->Code;
		TNam->TypPtr->Desc = TypPtr->Desc;
		/* free temporary type node */
		FreeSNODE(TypPtr);
	    }
	    else {
		/* replacement */
		/* free old type */
		FreeSNODE(TNam->TypPtr);
		/* use new type information */
		TNam->TypPtr = TypPtr;
	    }
	}
    }

    PPRINTF((">>Parse TypeAssign END\n"));
}

/*
 * parse a macro definition
 *	macroname MACRO ::= BEGIN macrobody END
 */
static P_MacroDefine()
{
    int lex;
    char *nam;
    static char mod[] = "P_MacroDefine";
    SNODE *tptr;
    register char *p;

    PPRINTF((">>Parse MacroDefine BEGIN\n"));

    lex = GetTok();
    if (lex != LEXUIDENT) {
	PError(mod,LEXUIDENT,lex);
	EatTok();
	return;
    }
    nam = CopyStr(CurToken);
    PPRINTF((">>MACRO '%s'\n",nam));
    /* macro names are all uppercase */
    for (p=nam;*p;p++) {
	if (islower(*p)) {
	    ParseErr(mod,0,"Macro name '%s' is not all UPPERCASE letters",nam);
	    break;
	}
    }
    EatTok();
    if (PChk(mod,LEXMACRO)) return;
    /* error recovery */
    for(;;) {
	lex = GetTok();
	if (lex == LEXASSIGN || lex == LEXEOF || lex == LEXEND)
	    break;
	PError(mod,LEXASSIGN,lex);
	EatTok();
    }
    if (PChk(mod,LEXASSIGN)) return;
    if (PChk(mod,LEXBEGIN)) return;
    tptr = P_MacroBody();
    if (PChk(mod,LEXEND)) return;

    AddType(nam,tptr);

    PPRINTF((">>Parse MacroDefine END\n"));
}


/* parse a module body
 * ModuleBody:
 *	{TypeAssignment|{ValueAssignment|MacroAssignment}}*
 */
static P_ModuleBody()
{
    int lex,nextlex;
    static char mod[] = "P_ModuleBody";

    PPRINTF((">>Parse ModuleBody BEGIN\n"));

    /* 0 or more productions of TypeAssignment or ValueAssignment */
    for(;;) {
	/* see if we need to resolve some delay bindings */
	if (TDelayList || VDelayList)
	    ResolveDelay(0);
	/* process current token */
	lex = GetTok();
	if (lex == LEXUIDENT || lex == LEXLIDENT) {
	    nextlex = PeekTok();
	    if (nextlex == LEXASSIGN)
		P_TypeAssign();
	    else {
		if (nextlex == LEXMACRO)
		    P_MacroDefine();
		else
		    P_ValueAssign();
	    }
	    continue;
	}
	if (lex == LEXEND || lex == LEXEOF)
	    break;	/* end of file or module */
	ParseErr(mod,1,"Expected %s or %s, got %s",
		 StrTok(LEXUIDENT),StrTok(LEXLIDENT),StrTok(lex));
	/* eat the token */
	EatTok();
    }

    PPRINTF((">>Parse ModuleBody END\n"));
}

/*
 * parse imports list
 * 	IMPORTS {names+ FROM modname}+ ;
 */
static P_Imports()
{
    static char mod[] = "P_Imports";
    register int lex;
    register INODE *ImportList,*tmp;
    MNODE *mptr;

    PPRINTF((">>Parse IMPORTS BEGIN\n"));

    PChk(mod,LEXIMPORTS);
    for(;;) {
	/* see if we doing the right thing */
	lex = GetTok();
	if (lex != LEXUIDENT && lex != LEXLIDENT)
	    break;
	ImportList = NULL;
	for(;;) {
	    lex = GetTok();
	    if (lex != LEXUIDENT && lex != LEXLIDENT)
		break;
	    /* add name to temporary import list */
	    tmp = MakeINODE();
	    tmp->Name = CopyStr(CurToken);
	    tmp->Next = ImportList;
	    ImportList = tmp;
	    EatTok();
	    /* process commas */
	    lex = GetTok();
	    if (lex != LEXCOMMA)
		break;
	    EatTok();
	}

	/* FROM */
	if (PChk(mod,LEXFROM)) break;
	/* module */
	lex = GetTok();
	if (lex != LEXUIDENT) {
	    PError(mod,LEXUIDENT,lex);
	    EatTok();
	    break;
	}
	/* check module */
	mptr = FindModContext(CurToken);
	if (mptr == NULL) {
	    /* no such module, complain */
	    ParseErr(mod,0,"can not find IMPORT FROM module '%s'",CurToken);
	}
	else {
	    /* add types and Values to current context */
	    for(tmp = ImportList;tmp != NULL;tmp = tmp->Next) {
		if (isupper(*tmp->Name)) {
		    TNAME *tp;
		    tp = FindType(tmp->Name,mptr);
		    if (tp == NULL)
			ParseErr(mod,0,
				 "IMPORT of type '%s' from '%s' not found",
				 tmp->Name,CurToken);
		    else
			AddType(tp->Name,tp->TypPtr);
		}
		else {
		    VNAME *newtp,*tp;
		    tp = FindValue(tmp->Name,mptr);
		    if (tp == NULL)
			ParseErr(mod,0,
				 "IMPORT of Value '%s' from '%s' not found",
				 tmp->Name,CurToken);
		    else {
			newtp = AddValue(tp->Name,tp->TypPtr,tp->ValPtr);
			/* import tree pointer TOO! */
			newtp->TreePtr = tp->TreePtr;
		    }
		}
	    }
	}
	/* free list up */
	while (ImportList != NULL) {
	    tmp = ImportList;
	    ImportList = tmp->Next;
	    UnMakeStr(tmp->Name);
	    FreeINODE(tmp);
	}
	EatTok();
	/* see if we are done */
	lex = GetTok();
	if (lex == LEXSEMI)
	    break;
    }
    PChk(mod,LEXSEMI);

    PPRINTF((">>Parse IMPORTS END\n"));
}

/*
 * parse exports list (ignored)
 * 	EXPORTS names+ ;
 */
static P_Exports()
{
    static char mod[] = "P_Exports";
    register int lex;
    register INODE *tmp;

    PPRINTF((">>Parse Exports BEGIN\n"));

    PChk(mod,LEXEXPORTS);
    for(;;) {
	lex = GetTok();
	if (lex != LEXUIDENT && lex != LEXLIDENT)
	    break;
	tmp = MakeINODE();
	tmp->Name = CopyStr(CurToken);
	tmp->Next = CurContext->Exports;
	CurContext->Exports = tmp;
	EatTok();
	/* see if we are done */
	lex = GetTok();
	if (lex != LEXCOMMA)
	    break;
	EatTok();
    }
    PChk(mod,LEXSEMI);

    PPRINTF((">>Parse Exports END\n"));
}

/* parse a module
 * ModuleDefintions:
 *	name DEFINITTIONS ::= BEGIN {Exports|Imports}* ModuleBody END
 */
static P_Module()
{
    static int modulenum = 0;
    int lex,nextlex;
    static char mod[] = "P_Module";
    char tmpbuf[32], *nam;

    PPRINTF((">>Parse Module BEGIN\n"));

    /* see if we are a module */
    lex = GetTok();
    nextlex = PeekTok();
    if (lex == LEXUIDENT) {
	if (nextlex == LEXDEFINITIONS) {
	    /* this is a real module definition */
	    nam = CopyStr(CurToken);
	    PPRINTF((">>MODULE %s\n",nam));
	    PushModule(nam);
	    EatTok();

	    if (PChk(mod,LEXDEFINITIONS)) return;
	    if (PChk(mod,LEXASSIGN)) return;
	    if (PChk(mod,LEXBEGIN)) return;

	    /* process imports and exports */
	    for(;;) {
		lex = GetTok();
		if (lex == LEXIMPORTS) {
		    P_Imports();
		}
		else if (lex == LEXEXPORTS) {
		    P_Exports();
		}
		else
		    break;
	    }
	    /* process body */
	    P_ModuleBody();
	    PopModule();
	    PChk(mod,LEXEND);
	    PPRINTF((">>Parse Module END '%s'\n",nam));
	    return;
	}
    }

    /* looks like we are not a real module...make one up */
    sprintf(tmpbuf,"UNNAMED MODULE#%d",modulenum++);
    PushModule(MakeStr(tmpbuf));
    PPRINTF((">>MODULE %s\n",tmpbuf));
    P_ModuleBody();
    PopModule();
    PPRINTF((">>Parse Module END '%s'\n",tmpbuf));
}

/* parse a set of modules...
 *	modules+
 */
static P_Modules()
{
    int lex;

    PPRINTF((">>Parse Modules BEGIN\n"));
    for (;;) {
	lex = GetTok();
	if (lex == LEXEOF) break;
	P_Module();
    }
    PPRINTF((">>Parse Modules END\n"));
}


/*****************************************************************
 *		MAIN ENTRY POINT
 *****************************************************************
 */
/*
 * set up memory and initialize things
 */
static setup()
{
    /* create hash table for strings */
    if (HashStrTable == NULL) {
	int len = HASHSTRTABSIZ * sizeof(HASHSTR *);
	HashStrTable = (HASHSTR **) Malloc(len,0);
    }
    /* init reserved word hash */
    if (MaxRWordLen == 0)
	InitRWords();

    /* init special macro tokens string into table */
    if (EMPTYstr == NULL) {
	EMPTYstr = MakeStr("empty");
	VALUEstr = MakeStr("value");
	TYPEstr = MakeStr("type");
    }

    /* init special object type strings */
    if (OBJECTTYPEstr == NULL) {
	OBJECTTYPEstr = MakeStr("OBJECT-TYPE");
	SYNTAXstr = MakeStr("SYNTAX");
	NETADDRstr = MakeStr("NetworkAddress");
	IPADDRstr = MakeStr("IpAddress");
	COUNTERstr = MakeStr("Counter");
	GAUGEstr = MakeStr("Gauge");
	TIMETICKSstr = MakeStr("TimeTicks");
	OPAQUEstr = MakeStr("Opaque");
	DESCRIPTIONstr = MakeStr("DESCRIPTION");
    }

    /* initialize tree top */
    if (treetop == NULL) {
	register struct tree *np, *tp;

	/* create unlabeled root node */
	treetop = np = MakeTREE();
	np->label = " <root> ";
	np->subid = -1;
	tp = np;


	/* link in children in numeric order */

	/* .ccitt(0) */
	np = MakeTREE();
	np->label = MakeStr("ccitt");
	np->subid = 0;
	tp->child_list = np;	/* link as first child */
	tp = np;

	/* .iso(1) */
	np = MakeTREE();
	np->label = MakeStr("iso");
	np->subid = 1;
	tp->next_peer = np;
	tp = np;

	/* .joint-iso-ccitt(2) */
	np = MakeTREE();
	np->label = MakeStr("joint-iso-ccitt");
	np->subid = 2;
	tp->next_peer = np;
	tp = np;
    }
}

/*
 * release memory in opposite order of creation
 */
static teardown()
{
    register MNODE *mcp;
#if	MEMACCT
    int MemCnt;
#endif

    /* clear out remaining module information (plus type and value) */
    while ((mcp = TopContext) != NULL) {
	register TNAME *tptr;
	register VNAME *vptr;
	TopContext = mcp->Next;
	/* free types */
	while ((tptr = mcp->Types) != NULL) {
	    mcp->Types = tptr->Next;
	    UnMakeStr(tptr->Name);
	    FreeTNAME(tptr);
	}
	/* free values */
	while ((vptr = mcp->Values) != NULL) {
	    mcp->Values = vptr->Next;
	    UnMakeStr(vptr->Name);
	    FreeVNAME(vptr);
	}
	UnMakeStr(mcp->Name);
#if	MEMACCT
	MemCntMNODE--;
	RelCntMNODE++;
#endif
	Free(mcp);
    }

    /* clear out exported symbol records */
    ScanRelSNODE(0);

    /* clear out macros */
    while (MacroHead != NULL) {
	FreeMPARSE(MacroHead);
    }

    /* zap special object type strings */
    if (OBJECTTYPEstr != NULL) {
	UnMakeStr(OBJECTTYPEstr);	OBJECTTYPEstr = NULL;
	UnMakeStr(SYNTAXstr);		SYNTAXstr = NULL;
	UnMakeStr(NETADDRstr);		NETADDRstr = NULL;
	UnMakeStr(IPADDRstr);		IPADDRstr = NULL;
	UnMakeStr(COUNTERstr);		COUNTERstr = NULL;
	UnMakeStr(GAUGEstr);		GAUGEstr = NULL;
	UnMakeStr(TIMETICKSstr);	TIMETICKSstr = NULL;
	UnMakeStr(OPAQUEstr);		OPAQUEstr = NULL;
	UnMakeStr(DESCRIPTIONstr);	DESCRIPTIONstr = NULL;
    }

    /* zap special macro strings */
    if (EMPTYstr != NULL) {
	UnMakeStr(EMPTYstr);		EMPTYstr = NULL;
	UnMakeStr(VALUEstr);		VALUEstr = NULL;
	UnMakeStr(TYPEstr);		TYPEstr = NULL;
    }

    /* remove hash table for strings and deallocate strings */
    if (HashStrTable != NULL) {
	register int i;
	/* free hash entries and strings */
	for(i=0;i<HASHSTRTABSIZ;i++) {
	    register struct HashStrEntry *Ptr;
	    while (Ptr = HashStrTable[i]) {
		HashStrTable[i] = Ptr->Next;
		if (Ptr->Cnt <= 0) {
#if	MEMACCT
		    MemCntSTR--;
		    RelCntSTR++;
#endif
		    Free(Ptr);
		}
	    }
	}
	/* free table */
	Free(HashStrTable);		HashStrTable = NULL;
    }

#if	MEMACCT
    printf("Count of Malloc %d (%d)\n",MemCntMalloc,RelCntMalloc);
    MemCnt = 0;
    printf("Count of TokLoad %d (%d)\n",MemCntTokLoad,RelCntTokLoad);
    MemCnt += MemCntTokLoad;
    printf("Count of SNODE %d (%d)\n",MemCntSNODE,RelCntSNODE);
    MemCnt += MemCntSNODE;
    printf("Count of MPARSE %d (%d)\n",MemCntMPARSE,RelCntMPARSE);
    MemCnt += MemCntMPARSE;
    printf("Count of DNODE %d (%d)\n",MemCntDNODE,RelCntDNODE);
    MemCnt += MemCntDNODE;
    printf("Count of MNODE %d (%d)\n",MemCntMNODE,RelCntMNODE);
    MemCnt += MemCntMNODE;
    printf("Count of TNAME %d (%d)\n",MemCntTNAME,RelCntTNAME);
    MemCnt += MemCntTNAME;
    printf("Count of VNAME %d (%d)\n",MemCntVNAME,RelCntVNAME);
    MemCnt += MemCntVNAME;
    printf("Count of INODE %d (%d)\n",MemCntINODE,RelCntINODE);
    MemCnt += MemCntINODE;
    printf("Count of TREE %d (%d)\n",MemCntTREE,RelCntTREE);
    MemCnt += MemCntTREE;
    printf("Count of strings %d (%d)\n",MemCntSTR,RelCntSTR);
    MemCnt += MemCntSTR;
    printf("Count of enums %d (%d)\n",MemCntENUMS,RelCntENUMS);
    MemCnt += MemCntENUMS;
    printf("Difference in Malloc %d\n",MemCntMalloc-MemCnt);
/* have a memory leak in macro parse alt and list lists */
#endif	MEMACCT
#if	HASHSTRACCT
    HashStats();
#endif	HASHSTRACCT
}

struct tree *
read_mib(filename)
    char *filename;
{

    setup();

    /* read the file */
    if (OpenTok(filename) < 0)
	return NULL;
    P_Modules();
    CloseTok();

#if	DODEBUG
    if (TreeDebug) DumpTree(treetop,0);
#endif	DODEBUG

    teardown();
#if	0
    return treetop;
#else	0
    /* for compatibility */
    return treetop->child_list;
#endif	0
}
