/* BEAST - Bedevilled Audio System
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bstinstrumenteditor.h"
#include	"bstdialogs.h"
#include	"gtkitemfactory.h"


/* --- signals --- */
enum
{
  SIGNAL_LAST
};


/* --- CList columns --- */
enum
{
  CLIST_COL_GUID,
  CLIST_COL_NAME,
  CLIST_COL_SOURCE,
  CLIST_COL_BLURB,
  CLIST_N_COLS
};
static gchar *clist_titles[CLIST_N_COLS] =
{
  "Guid",
  "Name",
  "Source",
  "Blurb",
};
static guint clist_width[CLIST_N_COLS] =
{
  30,
  80,
  50,
  100,
};


/* --- prototypes --- */
static void	bst_instrument_editor_init		  (BstInstrumentEditor		*instrument_editor);
static void	bst_instrument_editor_class_init	  (BstInstrumentEditorClass	*class);
static void	bst_instrument_editor_destroy		  (GtkObject			*object);
static gint	bst_instrument_editor_darea_expose_event  (GtkWidget			*widget,
							   GdkEvent			*event,
							   BstInstrumentEditor		*inst_ed);
static void	bst_instrument_editor_force_combo_text	  (BstInstrumentEditor		*inst_ed);
static void	bst_instrument_editor_instrument_selected (BstInstrumentEditor		*insted,
							   gint				 row,
							   gint				 column,
							   GdkEventButton		*event);
static void	bst_instrument_editor_update_samples	  (BstInstrumentEditor		*inst_ed,
							   BseSample			*selected_sample,
							   gboolean			 rebuild_check);
static void	bst_instrument_editor_update_instrument	  (BstInstrumentEditor		*inst_ed,
							   guint			 row);
static void	bst_instrument_editor_refresh_fields	  (BstInstrumentEditor		*inst_ed);
static void	bst_instrument_editor_apply_fields	  (BstInstrumentEditor		*inst_ed);
     
/* ok, need to state that somewhere...
 * *_reload():
 * - rebuild the whole instrument list, refresh the display via instrument_selected()
 * *_update_instrument():
 * - update a single clist_instruments item
 * *_update_samples_cb():
 * - (called from bse) rebuild the list_samples,
 * *_update_samples():
 * - check for list_samples<->BseSample inconsistency, if so rebuild
 * - select a sample
 * *_refresh_fields():
 * - refresh the various fields in the dialog
 * *_apply_fields():
 * - apply all of the dialogs fields
 * - refresh all fields
 * - refresh current instrument
 */

/* -- variables --- */
static GtkWindowClass	*parent_class = NULL;
static guint		 instrument_editor_signals[SIGNAL_LAST] = { 0 };
static gchar		*bst_istrument_editor_factories_path = "<BstInstrumentEditor>";


/* --- menus --- */
#define INED_OP(instrument_editor_op)	  \
  ((GtkItemFactoryCallback) bst_instrument_editor_clop), \
  (BST_INSTRUMENT_EDITOR_OP_ ## instrument_editor_op)

/* instrument list popup
 */
static GtkItemFactoryEntry ilist_popup_entries[] =
{
  { "/Instrument Menu", "",		INED_OP (NONE),		"<Title>" },
  { "/---",		"",		INED_OP (NONE),		"<Separator>" },
  { "/New Instrument",	"<control>N",	INED_OP (INSTR_NEW),	"<Item>" },
  { "/Delete Instrument", "",		INED_OP (INSTR_DEL),	"<Item>" },
};
static guint n_ilist_popup_entries = (sizeof (ilist_popup_entries) /
				      sizeof (ilist_popup_entries[0]));


/* --- functions --- */
GtkType
bst_instrument_editor_get_type (void)
{
  static GtkType instrument_editor_type = 0;
  
  if (!instrument_editor_type)
    {
      GtkTypeInfo instrument_editor_info =
      {
	"BstInstrumentEditor",
	sizeof (BstInstrumentEditor),
	sizeof (BstInstrumentEditorClass),
	(GtkClassInitFunc)  bst_instrument_editor_class_init,
	(GtkObjectInitFunc) bst_instrument_editor_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      
      instrument_editor_type = gtk_type_unique (gtk_window_get_type (), &instrument_editor_info);
    }
  
  return instrument_editor_type;
}

static void
bst_instrument_editor_class_init (BstInstrumentEditorClass *class)
{
  GtkObjectClass *object_class;
  GtkWindowClass *window_class;
  
  object_class = (GtkObjectClass*) class;
  window_class = (GtkWindowClass*) class;
  
  parent_class = gtk_type_class (gtk_window_get_type ());
  
  /* gtk_object_class_add_signals (object_class, instrument_editor_signals, SIGNAL_LAST);
   */
  
  object_class->destroy = bst_instrument_editor_destroy;

  class->factories_path = bst_istrument_editor_factories_path;
  class->ilist_popup_factory = NULL;
}

GtkWidget*
bst_instrument_editor_new (BseSong *song)
{
  BstInstrumentEditor *instrument_editor;
  GtkWidget *widget;
  gchar *string;
  
  g_return_val_if_fail (song != NULL, NULL);
  
  instrument_editor = gtk_type_new (bst_instrument_editor_get_type ());

  instrument_editor->song = song;
  
  string = g_strconcat (song->name, " [Instruments]", NULL);
  gtk_window_set_title (GTK_WINDOW (instrument_editor), string);
  g_free (string);
  string = NULL;
  
  widget = GTK_WIDGET (instrument_editor);
  
  return widget;
}

static void
bst_instrument_editor_update_samples_cb (gpointer inst_ed)
{
  bst_instrument_editor_update_samples (inst_ed, NULL, TRUE);
}

static void
bst_instrument_editor_destroy (GtkObject *object)
{
  BstInstrumentEditor *inst_ed;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (BST_IS_INSTRUMENT_EDITOR (object));
  
  inst_ed = BST_INSTRUMENT_EDITOR (object);
  
  /*  bse_sample_list_remove_cb (bst_instrument_editor_update_samples_cb, inst_ed);
   */

  gtk_object_unref (BST_INSTRUMENT_EDITOR_CLASS (GTK_OBJECT (inst_ed)->klass)->ilist_popup_factory);
  
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
bst_instrument_editor_init (BstInstrumentEditor *inst_ed)
{
  BstInstrumentEditorClass *class;
  GtkWidget *main_vbox;
  GtkWidget *vbox;
  GtkWidget *vbox_l;
  GtkWidget *vbox_r;
  GtkWidget *hbox;
  GtkWidget *big_hbox;
  GtkWidget *frame;
  GtkWidget *any;
  guint i;

  inst_ed->current_row = -1;
  inst_ed->block_apply = FALSE;

  class = BST_INSTRUMENT_EDITOR_CLASS (GTK_OBJECT (inst_ed)->klass);

  /* Pattern view popup
   */
  if (!class->ilist_popup_factory)
    {
      class->ilist_popup_factory =
	gtk_item_factory_new (gtk_menu_get_type (),
			      bst_istrument_editor_factories_path,
			      NULL);
      gtk_signal_connect (GTK_OBJECT (class->ilist_popup_factory),
			  "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed),
			  &class->ilist_popup_factory);
      gtk_item_factory_create_items (GTK_ITEM_FACTORY (class->ilist_popup_factory),
				     n_ilist_popup_entries,
				     ilist_popup_entries,
				     class);
    }
  else
    gtk_object_ref (class->ilist_popup_factory);
  gtk_window_add_accelerator_table (GTK_WINDOW (inst_ed),
				    GTK_ITEM_FACTORY (class->ilist_popup_factory)->table);

  /*  bse_sample_list_add_cb (bst_instrument_editor_update_samples_cb, inst_ed);
   */
  
  gtk_widget_set (GTK_WIDGET (inst_ed),
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::title", "InstrumentEditor",
		  "GtkWindow::window_position", GTK_WIN_POS_NONE,
		  "GtkWindow::allow_shrink", FALSE,
		  "GtkObject::signal::map", bst_instrument_editor_reload, NULL,
		  "GtkWindow::allow_grow", TRUE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 0,
		    "GtkContainer::border_width", 0,
		    "GtkWidget::parent", inst_ed,
		    "GtkWidget::visible", TRUE,
		    NULL);
  vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 5,
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  
  /* Instruments List
   */
  inst_ed->clist_instruments = (GtkCList*) gtk_clist_new_with_titles (CLIST_N_COLS, clist_titles);
  gtk_widget_set (GTK_WIDGET (inst_ed->clist_instruments),
		  "GtkWidget::width", 300,
		  "GtkWidget::height", 120,
		  "GtkObject::object_signal::select_row", bst_instrument_editor_instrument_selected, inst_ed,
		  "GtkWidget::visible", TRUE,
		  "GtkWidget::parent", vbox,
		  NULL);
  gtk_clist_set_policy (inst_ed->clist_instruments, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_clist_set_selection_mode (inst_ed->clist_instruments, GTK_SELECTION_BROWSE);
  gtk_clist_column_titles_passive (inst_ed->clist_instruments);
  for (i = 0; i < CLIST_N_COLS; i++)
    gtk_clist_set_column_width (inst_ed->clist_instruments, i, clist_width[i]);
  
  bst_new_hseperator (main_vbox);
  
  /* name/blurb entries
   */
  hbox = bst_new_hbox (main_vbox, FALSE, 0, 5, FALSE, 2,
		       vbox_l = bst_new_vbox (NULL, TRUE, 0, 0, FALSE, 0,
					      NULL),
		       vbox_r = bst_new_vbox (NULL, TRUE, 0, 0, TRUE, 0,
					      NULL),
		       NULL);
  bst_new_label (vbox_l, "Name:");
  inst_ed->entry_name = (GtkEntry*) bst_new_entry (vbox_r, GTK_WIDGET (inst_ed));
  bst_new_label (vbox_l, "Blurb:");
  inst_ed->entry_blurb = (GtkEntry*) bst_new_entry (vbox_r, GTK_WIDGET (inst_ed));
  gtk_signal_connect_object (GTK_OBJECT (inst_ed->entry_name),
			     "changed",
			     GTK_SIGNAL_FUNC (bst_instrument_editor_apply_fields),
			     GTK_OBJECT (inst_ed));
  gtk_signal_connect_object (GTK_OBJECT (inst_ed->entry_blurb),
			     "changed",
			     GTK_SIGNAL_FUNC (bst_instrument_editor_apply_fields),
			     GTK_OBJECT (inst_ed));
  
  /* sample list
   */
  hbox = bst_new_hbox (main_vbox, FALSE, 0, 5, TRUE, 2,
		       vbox_l = bst_new_vbox (NULL, TRUE, 0, 0, FALSE, 0,
					      NULL),
		       vbox_r = bst_new_vbox (NULL, TRUE, 0, 0, TRUE, 0,
					      NULL),
		       NULL);
  bst_new_label (vbox_l, "Sample:");
  any =
    gtk_widget_new (gtk_combo_get_type (),
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::parent", vbox_r,
		    NULL);
  inst_ed->entry_combo = (GtkEntry*) GTK_COMBO (any)->entry;
  inst_ed->list_samples = (GtkList*) GTK_COMBO (any)->list;
  gtk_list_set_selection_mode (inst_ed->list_samples, GTK_SELECTION_BROWSE);
  gtk_signal_connect_object (GTK_OBJECT (inst_ed->entry_combo),
			     "changed",
			     GTK_SIGNAL_FUNC (bst_instrument_editor_force_combo_text),
			     GTK_OBJECT (inst_ed));
  gtk_signal_connect_object (GTK_OBJECT (inst_ed->list_samples),
			     "selection_changed",
			     GTK_SIGNAL_FUNC (bst_instrument_editor_force_combo_text),
			     GTK_OBJECT (inst_ed));
  gtk_signal_connect_object (GTK_OBJECT (inst_ed->list_samples),
			     "selection_changed",
			     GTK_SIGNAL_FUNC (bst_instrument_editor_apply_fields),
			     GTK_OBJECT (inst_ed));
  
  /* main field hbox
   */
  big_hbox = bst_new_hbox (main_vbox, FALSE, 5, 5, FALSE, 0,
			   vbox = bst_new_vbox (NULL, FALSE, 2, 0, FALSE, 0,
						NULL),
			   NULL);
  
  /* flags
   */
  frame = bst_new_vframe (vbox, "Flags", TRUE /* FALSE */, 0,
			  inst_ed->toggle_interpolation = (GtkToggleButton*) bst_new_toggle_os (NULL, "Interpolation", TRUE, bst_instrument_editor_apply_fields, inst_ed),
			  inst_ed->toggle_polyphony = (GtkToggleButton*) bst_new_toggle_os (NULL, "Polyphony", FALSE, bst_instrument_editor_apply_fields, inst_ed),
			  inst_ed->toggle_sequencer_sustain = (GtkToggleButton*) bst_new_toggle_os (NULL, "Sequencer Sustain", TRUE, bst_instrument_editor_apply_fields, inst_ed),
			  NULL);
  // gtk_box_set_child_packing (GTK_BOX (vbox), frame, FALSE, TRUE, 0, GTK_PACK_END);

  /* tuning
   */
  frame = bst_new_hframe (vbox, "Tuning", FALSE, 1,
			  vbox_l = bst_new_vbox (NULL, TRUE, 0, 0, FALSE, 0,
						 NULL),
			  vbox_r = bst_new_vbox (NULL, TRUE, 0, 0, TRUE, 0,
						 NULL),
			  NULL);
  gtk_container_border_width (GTK_CONTAINER (GTK_BIN (frame)->child), 5);
  bst_new_label (vbox_l, "Volume:");
  any = bst_new_spinner_os (vbox_r, &inst_ed->adj_volume,
			   BSE_DFL_INSTRUMENT_VOLUME, BSE_MIN_VOLUME, BSE_MAX_VOLUME, 5,
			   NULL, NULL);
  gtk_widget_set_usize (any, 55, -1);
  bst_new_label (vbox_l, "Balance:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_balance,
		     0, BSE_MIN_BALANCE, BSE_MAX_BALANCE, 5,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Transpose:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_transpose,
		     0, BSE_MIN_TRANSPOSE, BSE_MAX_TRANSPOSE, 5,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Fine-Tune:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_fine_tune,
		     0, BSE_MIN_FINE_TUNE, BSE_MAX_FINE_TUNE, 5,
		     bst_instrument_editor_apply_fields, inst_ed);
  
  /* envelope
   */
  bst_new_vframe (big_hbox, "Envelope", TRUE, 1,
		  frame = bst_new_vframe (NULL, NULL, TRUE, 1,
					  inst_ed->darea = bst_new_darea_s (NULL, TRUE, -1, 40, NULL, bst_instrument_editor_darea_expose_event, inst_ed),
					  NULL),
		  bst_new_hbox (NULL, FALSE, 0, 5, TRUE, 2,
				vbox_l = bst_new_vbox (NULL, TRUE, 0, 0, FALSE, 0,
						       NULL),
				vbox_r = bst_new_vbox (NULL, TRUE, 0, 0, TRUE, 0,
						       NULL),
				NULL),
		  NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_border_width (GTK_CONTAINER (frame), 5);
  bst_new_label (vbox_l, "Delay Time:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_delay_time,
		     0, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME, 10,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Attack Time:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_attack_time,
		     0, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME, 10,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Attack Level:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_attack_level,
		     0, BSE_MIN_ENV_LEVEL, BSE_MAX_ENV_LEVEL, 5,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Decay Time:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_decay_time,
		     0, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME, 10,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Sustain Level:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_sustain_level,
		     0, BSE_MIN_ENV_LEVEL, BSE_MAX_ENV_LEVEL, 5,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Sustain Time:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_sustain_time,
		     0, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME, 10,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Release Level:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_release_level,
		     0, BSE_MIN_ENV_LEVEL, BSE_MAX_ENV_LEVEL, 5,
		     bst_instrument_editor_apply_fields, inst_ed);
  bst_new_label (vbox_l, "Release Time:");
  bst_new_spinner_os (vbox_r, &inst_ed->adj_release_time,
		     0, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME, 10,
		     bst_instrument_editor_apply_fields, inst_ed);

  /* fill UI
   */
  bst_instrument_editor_update_samples (inst_ed, NULL, TRUE);
}

static gint
bst_instrument_editor_darea_expose_event (GtkWidget	      *widget,
					  GdkEvent	      *event,
					  BstInstrumentEditor *inst_ed)
{
  GtkDrawingArea *darea;
  GdkDrawable *drawable;
  GdkGC *black_gc;
  GdkGC *white_gc;
  guint max_width;
  guint max_height;
  guint total_time;
  guint sustain_time;
  guint descent;
  gdouble x_factor;
  gdouble y_factor;
  gdouble offset;
  gdouble len;

  g_return_val_if_fail (widget != NULL, TRUE);
  g_return_val_if_fail (GTK_IS_DRAWING_AREA (widget), TRUE);
  g_return_val_if_fail (inst_ed != NULL, TRUE);
  g_return_val_if_fail (BST_IS_INSTRUMENT_EDITOR (inst_ed), TRUE);

  darea = GTK_DRAWING_AREA (widget);
  drawable = widget->window;
  white_gc = widget->style->white_gc;
  black_gc = widget->style->black_gc;
  max_width = widget->allocation.width;
  max_height = widget->allocation.height;

  if (inst_ed->toggle_sequencer_sustain->active)
    sustain_time = BSE_MAX_ENV_TIME;
  else
    sustain_time = inst_ed->adj_sustain_time->value;
  total_time = MAX (1,
		    inst_ed->adj_delay_time->value +
		    inst_ed->adj_attack_time->value +
		    inst_ed->adj_decay_time->value +
		    sustain_time +
		    inst_ed->adj_release_time->value);

  /* draw background
   */
  gdk_draw_rectangle (drawable, white_gc,
		      TRUE,
		      0,
		      0,
		      max_width,
		      max_height);

  /* calc intermediates
   * and draw bottom line
   */
  descent = max_height / 10;
  max_height -= descent + 1;
  gdk_draw_line (drawable, black_gc,
		 0,
		 max_height,
		 max_width,
		 max_height);
  max_width -= 1;
  x_factor = max_width * 1.0 / total_time;
  y_factor = max_height * 1.0 / BSE_MAX_ENV_LEVEL;

  offset = 0;
  len = inst_ed->adj_delay_time->value * x_factor;

  /* draw attack
   */
  offset += len;
  len = inst_ed->adj_attack_time->value * x_factor;
  gdk_draw_line (drawable, black_gc,
		 0.5 + offset,
		 max_height,
		 0.5 + offset + len,
		 max_height - inst_ed->adj_attack_level->value * y_factor);

  /* draw decay
   */
  offset += len;
  len = inst_ed->adj_decay_time->value * x_factor;
  gdk_draw_line (drawable, black_gc,
		 0.5 + offset,
		 max_height - inst_ed->adj_attack_level->value * y_factor,
		 0.5 + offset + len,
		 max_height - inst_ed->adj_sustain_level->value * y_factor);

  /* draw sustain
   */
  offset += len;
  len = sustain_time * x_factor;
  gdk_draw_line (drawable, black_gc,
		 0.5 + offset,
		 max_height - inst_ed->adj_sustain_level->value * y_factor,
		 0.5 + offset + len,
		 max_height - inst_ed->adj_release_level->value * y_factor);

  /* draw release
   */
  offset += len;
  len = inst_ed->adj_release_time->value * x_factor;
  gdk_draw_line (drawable, black_gc,
		 0.5 + offset,
		 max_height - inst_ed->adj_release_level->value * y_factor,
		 0.5 + offset + len /* max_width - 1 */,
		 max_height);

  gdk_flush ();

  return TRUE;
}

static void
bst_instrument_editor_force_combo_text (BstInstrumentEditor *inst_ed)
{
  gchar *text = NULL;

  g_return_if_fail (inst_ed != NULL);
  g_return_if_fail (BST_IS_INSTRUMENT_EDITOR (inst_ed));

  if (inst_ed->list_samples->selection)
    {
      BseSample *sample;

      sample = gtk_object_get_user_data (inst_ed->list_samples->selection->data);

      text = bse_sample_get_name (sample);
    }
  
  if (text)
    {
      gtk_signal_handler_block_by_func (GTK_OBJECT (inst_ed->entry_combo),
					GTK_SIGNAL_FUNC (bst_instrument_editor_force_combo_text),
					inst_ed);
      
      gtk_entry_set_text (inst_ed->entry_combo, text);
      
      gtk_signal_handler_unblock_by_func (GTK_OBJECT (inst_ed->entry_combo),
					  GTK_SIGNAL_FUNC (bst_instrument_editor_force_combo_text),
					  inst_ed);
    }
}

static void
bst_instrument_editor_update_instrument (BstInstrumentEditor *inst_ed,
					 guint		      row)
{
  BseInstrument *instrument;

  g_return_if_fail (inst_ed != NULL);
  g_return_if_fail (BST_INSTRUMENT_EDITOR (inst_ed));
  
  instrument = gtk_clist_get_row_data (inst_ed->clist_instruments, row);
  if (instrument)
    {
      gchar buffer[32];
  
      sprintf (buffer, "%04u", bse_instrument_get_guid (instrument));
      gtk_clist_set_text (inst_ed->clist_instruments,
			  row, 0,
			  buffer);
      gtk_clist_set_text (inst_ed->clist_instruments,
			  row, 1,
			  bse_instrument_get_name (instrument));
      gtk_clist_set_text (inst_ed->clist_instruments,
			  row, 2,
			  (instrument->type == BSE_INSTRUMENT_SAMPLE ?
			   instrument->sample->name :
			   (gpointer) bse_instrument_type_get_nick (instrument->type)));
      gtk_clist_set_text (inst_ed->clist_instruments,
			  row, 3,
			  bse_instrument_get_blurb (instrument));
    }
}

void
bst_instrument_editor_reload (BstInstrumentEditor *inst_ed)
{
  BseInstrument *selected_instrument;
  GList *list;

  g_return_if_fail (inst_ed != NULL);
  g_return_if_fail (BST_INSTRUMENT_EDITOR (inst_ed));
  g_return_if_fail (!inst_ed->block_apply);

  selected_instrument = gtk_clist_get_row_data (inst_ed->clist_instruments,
						inst_ed->current_row);
  inst_ed->current_row = -1;

  gtk_clist_freeze (inst_ed->clist_instruments);
  gtk_clist_clear (inst_ed->clist_instruments);

  for (list = inst_ed->song->instruments; list; list = list->next)
    {
      BseInstrument *instrument;
      gchar *text[16] = { 0 };
      gint row;

      instrument = list->data;

      row = gtk_clist_append (inst_ed->clist_instruments, text);
      gtk_clist_set_row_data (inst_ed->clist_instruments, row, instrument);
      bst_instrument_editor_update_instrument (inst_ed, row);
      if (selected_instrument == instrument)
	gtk_clist_select_row (inst_ed->clist_instruments, row, -1);
    }
  if (!inst_ed->song->instruments)
    bst_instrument_editor_instrument_selected (inst_ed, -1, -1, NULL);

  gtk_clist_thaw (inst_ed->clist_instruments);
}

static void
bst_instrument_editor_update_samples (BstInstrumentEditor *inst_ed,
				      BseSample		  *selected_sample,
				      gboolean		   rebuild)
{
  GList *cnode;

  g_return_if_fail (inst_ed != NULL);
  g_return_if_fail (BST_IS_INSTRUMENT_EDITOR (inst_ed));

  if (rebuild)
    {
      GList *list;
      GList *sample_list;

      sample_list = bse_sample_list_all ();

      rebuild = FALSE;
      cnode = inst_ed->list_samples->children;
      for (list = sample_list; list; list = list->next)
	{
	  if (!cnode ||
	      gtk_object_get_user_data (cnode->data) != list->data)
	    {
	      rebuild = TRUE;
	      break;
	    }
	  cnode = cnode->next;
	}
      if (cnode != NULL)
	rebuild = TRUE;
      
      if (rebuild)
	{
	  gtk_list_clear_items (inst_ed->list_samples, 0, -1);
	  
	  for (list = sample_list; list; list = list->next)
	    {
	      BseSample *sample;
	      GtkWidget *item;
	      
	      sample = list->data;
	      
	      item = gtk_list_item_new_with_label (bse_sample_get_name (sample));
	      gtk_widget_set (item,
			      "GtkWidget::visible", TRUE,
			      "GtkObject::user_data", sample,
			      NULL);
	      list->data = item;
	    }
	  
	  gtk_list_append_items (inst_ed->list_samples, sample_list);
	}
      else
	g_list_free (sample_list);
    }
      
  for (cnode = inst_ed->list_samples->children; cnode; cnode = cnode->next)
    {
      if (gtk_object_get_user_data (cnode->data) == selected_sample)
	{
	  gtk_list_select_child (inst_ed->list_samples, cnode->data);
	  break;
	}
    }
}

static void
bst_instrument_editor_instrument_selected (BstInstrumentEditor		*inst_ed,
					   gint				 row,
					   gint				 column,
					   GdkEventButton		*event)
{
  g_return_if_fail (inst_ed != NULL);
  g_return_if_fail (BST_INSTRUMENT_EDITOR (inst_ed));
  g_return_if_fail (!inst_ed->block_apply);

  inst_ed->current_row = row;

  if (row >= 0 &&
      gtk_clist_row_is_visible (inst_ed->clist_instruments, row) != GTK_VISIBILITY_FULL)
    gtk_clist_moveto (inst_ed->clist_instruments, row, -1, -1, -1);

  bst_instrument_editor_refresh_fields (inst_ed);

  if (event &&
      event->type == GDK_BUTTON_PRESS &&
      event->button == 3)
    {
      BstInstrumentEditorClass *class;

      class = BST_INSTRUMENT_EDITOR_CLASS (GTK_OBJECT (inst_ed)->klass);

      gtk_item_factory_popup (GTK_ITEM_FACTORY (class->ilist_popup_factory),
			      inst_ed,
			      event->x_root,
			      event->y_root,
			      event->button,
			      event->time);
    }
}

static void
bst_instrument_editor_refresh_fields (BstInstrumentEditor *inst_ed)
{
  BseInstrument *instrument;
  gchar *text;
  
  g_return_if_fail (inst_ed != NULL);
  g_return_if_fail (BST_INSTRUMENT_EDITOR (inst_ed));
  g_return_if_fail (!inst_ed->block_apply);
  
  instrument = gtk_clist_get_row_data (inst_ed->clist_instruments,
				       inst_ed->current_row);

  inst_ed->block_apply = TRUE;

  /* set text fields
   */
  text = instrument ? instrument->name : "";
  if (!g_str_equal (gtk_entry_get_text (inst_ed->entry_name), text))
    gtk_entry_set_text (inst_ed->entry_name, text);
  text = instrument ? instrument->blurb : "";
  if (!g_str_equal (gtk_entry_get_text (inst_ed->entry_blurb), text))
    gtk_entry_set_text (inst_ed->entry_blurb, text);
  
  /* set selected sample
   * FIXME: eval instrument type
   */
  bst_instrument_editor_update_samples (inst_ed, instrument ? instrument->sample : NULL, FALSE);
  
  /* set toggles
   */
  gtk_toggle_button_set_state (inst_ed->toggle_interpolation,
			       instrument ? instrument->interpolation : TRUE);
  gtk_toggle_button_set_state (inst_ed->toggle_polyphony,
			       instrument ? instrument->polyphony : FALSE);
  gtk_toggle_button_set_state (inst_ed->toggle_sequencer_sustain,
			       instrument ? instrument->sustain_time == BSE_NO_ENV_TIME : TRUE);
  
  /* set adjustments
   */
  gtk_adjustment_set_value (inst_ed->adj_volume,
			    instrument ? instrument->volume : BSE_DFL_INSTRUMENT_VOLUME);
  gtk_adjustment_set_value (inst_ed->adj_balance,
			    instrument ? instrument->balance : 0);
  gtk_adjustment_set_value (inst_ed->adj_transpose,
			    instrument ? instrument->transpose : 0);
  gtk_adjustment_set_value (inst_ed->adj_fine_tune,
			    instrument ? instrument->fine_tune : 0);
  
  /* set envelope
   */
  gtk_adjustment_set_value (inst_ed->adj_delay_time,
			    instrument ? instrument->delay_time : 0);
  gtk_adjustment_set_value (inst_ed->adj_attack_time,
			    instrument ? instrument->attack_time : 32);
  gtk_adjustment_set_value (inst_ed->adj_attack_level,
			    instrument ? instrument->attack_level : BSE_MAX_ENV_LEVEL);
  gtk_adjustment_set_value (inst_ed->adj_decay_time,
			    instrument ? instrument->decay_time : 32);
  gtk_adjustment_set_value (inst_ed->adj_sustain_level,
			    instrument ? instrument->sustain_level : BSE_MAX_ENV_LEVEL / 2);
  gtk_adjustment_set_value (inst_ed->adj_release_level,
			    instrument ? instrument->release_level : BSE_MAX_ENV_LEVEL / 2);
  gtk_adjustment_set_value (inst_ed->adj_sustain_time,
			    instrument ? instrument->sustain_time : 64);
  gtk_adjustment_set_value (inst_ed->adj_release_time,
			    instrument ? instrument->release_time : 32);
  
  if (inst_ed->darea)
    gtk_widget_queue_draw (inst_ed->darea);
  
  inst_ed->block_apply = FALSE;
}

static void
bst_instrument_editor_apply_fields (BstInstrumentEditor *inst_ed)
{
  BseSample *sample;
  BseInstrument *instrument;

  g_return_if_fail (inst_ed != NULL);
  g_return_if_fail (BST_IS_INSTRUMENT_EDITOR (inst_ed));

  if (inst_ed->block_apply)
    return;
  
  instrument = gtk_clist_get_row_data (inst_ed->clist_instruments,
				       inst_ed->current_row);
  if (!instrument ||
      !inst_ed->list_samples->selection)
    return;

  inst_ed->block_apply = TRUE;

  sample = gtk_object_get_user_data (inst_ed->list_samples->selection->data);

  /* text entries
   */
  bse_instrument_set_name (instrument, gtk_entry_get_text (inst_ed->entry_name));
  bse_instrument_set_blurb (instrument, gtk_entry_get_text (inst_ed->entry_blurb));

  /* sample
   */
  bse_instrument_set_sample (instrument, sample);

  /* toggles
   */
  instrument->interpolation = inst_ed->toggle_interpolation->active;
  instrument->polyphony = inst_ed->toggle_polyphony->active;
  if (inst_ed->toggle_sequencer_sustain)
    instrument->sustain_time = BSE_NO_ENV_TIME;

  /* adjustments
   */
  instrument->volume = inst_ed->adj_volume->value;
  instrument->balance = inst_ed->adj_balance->value;
  instrument->transpose = inst_ed->adj_transpose->value;
  instrument->fine_tune = inst_ed->adj_fine_tune->value;

  /* envelope
   */
  instrument->delay_time = inst_ed->adj_delay_time->value;
  instrument->attack_time = inst_ed->adj_attack_time->value;
  instrument->attack_level = inst_ed->adj_attack_level->value;
  instrument->decay_time = inst_ed->adj_decay_time->value;
  instrument->sustain_level = inst_ed->adj_sustain_level->value;
  instrument->release_level = inst_ed->adj_release_level->value;
  instrument->sustain_time = inst_ed->adj_sustain_time->value;
  instrument->release_time = inst_ed->adj_release_time->value;

  bse_song_instrument_changed (instrument->song, instrument);

  inst_ed->block_apply = FALSE;

  bst_instrument_editor_update_instrument  (inst_ed, inst_ed->current_row);

  bst_instrument_editor_refresh_fields (inst_ed);

  inst_ed->block_apply = FALSE;
}

void
bst_instrument_editor_clop (BstInstrumentEditorClass   *class,
			    BstInstrumentEditorClassOps ined_op)
{
  GtkWidget *widget;
  BstInstrumentEditor *inst_ed;

  g_return_if_fail (class != NULL);
  g_return_if_fail (BST_IS_INSTRUMENT_EDITOR_CLASS (class));
  g_return_if_fail (ined_op < BST_INSTRUMENT_EDITOR_OP_LAST);

  widget = gtk_get_event_widget (gtk_get_current_event ());
  g_return_if_fail (widget != NULL);
  widget = gtk_widget_get_toplevel (widget);
  g_return_if_fail (widget != NULL);

  /* if we got called from the popup, find out about the invoking inst_ed
   */
  if (GTK_IS_MENU (widget))
    widget = gtk_object_get_user_data (GTK_OBJECT (widget));

  g_return_if_fail (BST_IS_INSTRUMENT_EDITOR (widget));

  inst_ed = BST_INSTRUMENT_EDITOR (widget);

  gtk_widget_ref (GTK_WIDGET (inst_ed));

  switch (ined_op)
    {
      BseInstrument *instrument;
      BseSample *sample;

    case BST_INSTRUMENT_EDITOR_OP_INSTR_NEW:
      sample = bse_get_zero_sample ();
      instrument = bse_song_sample_instrument_new (inst_ed->song, sample);
      bse_sample_unref (sample);
      bst_instrument_editor_reload (inst_ed);
      break;

    default:
      g_warning ("BstInstrumentEditorClassOps: operation `%d' unhandled\n", ined_op);
      break;
    }

  gtk_widget_unref (GTK_WIDGET (inst_ed));
}
