/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1998 Olaf Hoehmann and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	"bseinstrument.h"
#include	"bsemem.h"
#include	"bseprivate.h"



/* --- prototypes --- */

/* --- functions --- */
BseInstrument*
bse_instrument_type_sample_new (BseSong		       *song,
				BseSample	       *sample)
{
  register BseInstrument *instrument;
  
  g_return_val_if_fail (song != NULL, NULL);
  g_return_val_if_fail (sample != NULL, NULL);
  
  instrument = bse_instrument_chunk_new0 ();
  
  instrument->type = BSE_INSTRUMENT_SAMPLE;
  instrument->song = song;
  instrument->name = g_strdup ("");
  instrument->blurb = g_strdup ("");
  instrument->deferred_sample_name = NULL;
  instrument->deferred_sample_path = NULL;
  instrument->sample = sample;
  bse_sample_ref (sample);
  instrument->interpolation = TRUE;
  
  instrument->polyphony = FALSE;
  instrument->volume = BSE_DFL_INSTRUMENT_VOLUME;
  instrument->balance = 0;
  instrument->transpose = 0;
  instrument->fine_tune = 0;
  
  instrument->delay_time = 0;
  instrument->attack_time = 0;
  instrument->attack_level = BSE_MAX_ENV_LEVEL;
  instrument->decay_time = 0;
  instrument->sustain_level = BSE_MAX_ENV_LEVEL;
  instrument->sustain_time = BSE_NO_ENV_TIME;
  instrument->release_level = 0;
  instrument->release_time = 0;
  
  bse_song_instrument_changed (instrument->song, instrument);
  
  return instrument;
}

void
bse_instrument_delete (BseInstrument *instrument)
{
  g_return_if_fail (instrument != NULL);
  
  g_dataset_destroy (instrument);
  
  if (instrument->sample)
    {
      bse_sample_unref (instrument->sample);
      instrument->sample = NULL;
    }
  
  g_free (instrument->name);
  g_free (instrument->blurb);
  
  bse_instrument_chunk_free (instrument);
}

guint
bse_instrument_get_guid (BseInstrument *instrument)
{
  gint i;
  
  g_return_val_if_fail (instrument != NULL, 0);
  
  i = g_list_index (instrument->song->instruments, instrument);
  if (i < 0)
    return 0;
  else
    return i + 1;
}

void
bse_instrument_handle_changed (BseSong		*song,
			       BseInstrument	*instrument)
{
  g_return_if_fail (song != NULL);
  g_return_if_fail (instrument != NULL);
  g_return_if_fail (instrument->song == song);
  
  instrument->volume = CLAMP (instrument->volume, BSE_MIN_VOLUME, BSE_MAX_VOLUME);
  instrument->balance = CLAMP (instrument->balance, BSE_MIN_BALANCE, BSE_MAX_BALANCE);
  instrument->transpose = CLAMP (instrument->transpose, BSE_MIN_TRANSPOSE, BSE_MAX_TRANSPOSE);
  instrument->fine_tune = CLAMP (instrument->fine_tune, BSE_MIN_FINE_TUNE, BSE_MAX_FINE_TUNE);
  
  instrument->delay_time = CLAMP (instrument->delay_time, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME);
  instrument->attack_time = CLAMP (instrument->attack_time, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME);
  instrument->attack_level = CLAMP (instrument->attack_level, BSE_MIN_ENV_LEVEL, BSE_MAX_ENV_LEVEL);
  instrument->decay_time = CLAMP (instrument->decay_time, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME);
  instrument->sustain_level = CLAMP (instrument->sustain_level, BSE_MIN_ENV_LEVEL, BSE_MAX_ENV_LEVEL);
  if (instrument->sustain_time != BSE_NO_ENV_TIME)
    instrument->sustain_time = CLAMP (instrument->sustain_time, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME);
  instrument->release_level = CLAMP (instrument->release_level, BSE_MIN_ENV_LEVEL, BSE_MAX_ENV_LEVEL);
  instrument->release_time = CLAMP (instrument->release_time, BSE_MIN_ENV_TIME, BSE_MAX_ENV_TIME);
}

void
bse_instrument_set_sample (BseInstrument  *instrument,
			   BseSample      *sample)
{
  g_return_if_fail (instrument != NULL);
  g_return_if_fail (sample != NULL);

  if (sample != instrument->sample)
    {
      bse_sample_unref (instrument->sample);
      instrument->sample = sample;
      bse_sample_ref (instrument->sample);

      bse_song_instrument_changed (instrument->song, instrument);
    }
}

void
bse_instrument_set_name (BseInstrument	  *instrument,
			 const gchar	  *name)
{
  g_return_if_fail (instrument != NULL);
  g_return_if_fail (name != NULL);
  
  if (!name)
    name = "";
  
  g_free (instrument->name);
  instrument->name = g_strdup (name);
  
  bse_song_instrument_changed (instrument->song, instrument);
}

gchar*
bse_instrument_get_name (BseInstrument	  *instrument)
{
  g_return_val_if_fail (instrument != NULL, NULL);
  
  return instrument->name;
}

void
bse_instrument_set_blurb (BseInstrument	  *instrument,
			  const gchar	  *blurb)
{
  g_return_if_fail (instrument != NULL);
  g_return_if_fail (blurb != NULL);
  
  if (!blurb)
    blurb = "";
  
  g_free (instrument->blurb);
  instrument->blurb = g_strdup (blurb);
  
  bse_song_instrument_changed (instrument->song, instrument);
}

gchar*
bse_instrument_get_blurb (BseInstrument	  *instrument)
{
  g_return_val_if_fail (instrument != NULL, NULL);
  
  return instrument->blurb;
}
