#!/usr/bin/env python
#
# Copyright 2005 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

from gnuradio import gr, gru, blks
from gnuradio import usrp
from gnuradio import eng_notation
from gnuradio.eng_option import eng_option
from optparse import OptionParser

import random
import time
import struct
import sys
import os

# from current dir
from transmit_path import transmit_path
from receive_path import receive_path

#print os.getpid()
#raw_input('Attach and press enter')


# Linux specific...
def open_tun_interface(tun_device_filename):
    from fcntl import ioctl
    
    TUNMODE = 0x0001
    TUNSETIFF = 0x400454ca

    tun = os.open(tun_device_filename, os.O_RDWR)
    ifs = ioctl(tun, TUNSETIFF, struct.pack("16sH", "gr%d", TUNMODE))
    ifname = ifs[:16].strip("\x00")
    return (tun, ifname)
    

# /////////////////////////////////////////////////////////////////////////////
#                             the flow graph
# /////////////////////////////////////////////////////////////////////////////

class my_graph(gr.flow_graph):

    def __init__(self, rx_subdev_spec, tx_subdev_spec,
                 bitrate, decim_rate, interp_rate, spb, bt, rx_callback, log_p):

        gr.flow_graph.__init__(self)
        self.txpath = transmit_path(self, tx_subdev_spec, bitrate, interp_rate,
                                    spb, bt, log_p)
        self.rxpath =  receive_path(self, rx_subdev_spec, bitrate, decim_rate,
                                    spb, rx_callback, log_p)


# /////////////////////////////////////////////////////////////////////////////
#                                   main
# /////////////////////////////////////////////////////////////////////////////

def main():

    def send_pkt(payload='', eof=False):
        return fg.txpath.send_pkt(payload, eof)

    def rx_callback(ok, payload):
        if options.verbose:
            print "Rx: ok = %r  len(payload) = %4d" % (ok, len(payload))
        if ok:
            os.write(tun_fd, payload)

    MTU = 1500
    
    parser = OptionParser (option_class=eng_option)
    parser.add_option("-f", "--freq", type="eng_float", default=423.1e6,
                       help="set Tx and Rx frequency to FREQ [default=%default]", metavar="FREQ")
    parser.add_option("-r", "--bitrate", type="eng_float", default=None,
                      help="specify bitrate.  spb and interp will be derived.")
    parser.add_option("-g", "--gain", type="eng_float", default=27,
                      help="set rx gain")
    parser.add_option("-R", "--rx-subdev-spec", type="subdev", default=None,
                      help="select USRP Rx side A or B")
    parser.add_option("-T", "--tx-subdev-spec", type="subdev", default=None,
                      help="select USRP Tx side A or B")
    parser.add_option("-S", "--spb", type="int", default=None, help="set samples/baud [default=%default]")
    parser.add_option("-d", "--decim", type="intx", default=None,
                      help="set fpga decim rate to DECIM [default=%default]")
    parser.add_option("-i", "--interp", type="intx", default=None,
                      help="set fpga interpolation rate to INTERP [default=%default]")
    parser.add_option("", "--bt", type="float", default=0.3, help="set bandwidth-time product [default=%default]")
    parser.add_option("","--log", action="store_true", default=False,
                      help="enable diagnostic logging")
    parser.add_option("","--tun-device-filename", default="/dev/net/tun",
                      help="path to tun device file [default=%default]")
    parser.add_option("-v","--verbose", action="store_true", default=False)
    (options, args) = parser.parse_args ()

    if len(args) != 0:
        parser.print_help()
        sys.exit(1)

    if options.freq < 1e6:
        options.freq *= 1e6

    # build the graph
    fg = my_graph(options.tx_subdev_spec, options.rx_subdev_spec,
                  options.bitrate, options.decim, options.interp,
                  options.spb, options.bt, rx_callback, options.log)

    if fg.txpath.bitrate() != fg.rxpath.bitrate():
        print "WARNING: Transmit bitrate = %sb/sec, Receive bitrate = %sb/sec" % (
            eng_notation.num_to_str(fg.txpath.bitrate()),
            eng_notation.num_to_str(fg.rxpath.bitrate()))
             
    print "bitrate: %sb/sec" % (eng_notation.num_to_str(fg.txpath.bitrate()),)
    print "spb:     %3d" % (fg.txpath.spb(),)
    print "interp:  %3d" % (fg.txpath.interp(),)

    fg.txpath.set_freq(options.freq)
    fg.rxpath.set_freq(options.freq)
    fg.rxpath.set_gain(options.gain)
    print "Rx gain_range: ", fg.rxpath.subdev.gain_range(), " using", fg.rxpath.gain

    (tun_fd, tun_ifname) = open_tun_interface(options.tun_device_filename)

    print "Allocated tun interface: %s" % (tun_ifname,)
    print "You must ifconfig it and set up your routing now." 

    fg.start()  # Start flow graph

    # main loop that blocks reading packets from the tun interface

    while 1:
        payload = os.read(tun_fd, MTU)
        if not payload:
            send_pkt(eof=True)
            break
        if options.verbose:
            print "Tx: len(payload) = %4d" % (len(payload),)
        send_pkt(payload)

    fg.stop()     # tell flow graph to stop
    fg.wait()     # wait for it to finish
                

if __name__ == '__main__':
    try:
        main()
    except KeyboardInterrupt:
        pass
