/*	symbols.c - symbols (labels) printing and handling, NeXT disassembler.

	Copyright (C) 1989 by Bill Spitzak
	See Copyright notice in makefile

	See also symtab.c for the routines which sort symbols.
*/

#include "dis.h"

/*	Print an assembler directive that will produce an arbitrary symbol
	table entry.  This is printed when we don't know anything better
	to do with a symbol.		*/
printstab(struct nlist *s) {
    struct nte {unsigned char type; char *symbol;};
    static struct nte nametable[] = {
	{N_UNDF,	"N_UNDF"},
	{N_ABS,		"N_ABS"},
	{N_INDR,	"N_INDR"},
	{N_SECT,	"N_SECT"},
	{N_GSYM,	"N_GSYM"},
	{N_FNAME,	"N_FNAME"},
	{N_FUN,		"N_FUN"},
	{N_STSYM,	"N_STSYM"},
	{N_LCSYM,	"N_LCSYM"},
	{N_RSYM,	"N_RSYM"},
	{N_SLINE,	"N_SLINE"},
	{N_SSYM,	"N_SSYM"},
	{N_SO,		"N_SO"},
	{N_LSYM,	"N_LSYM"},
	{N_SOL,		"N_SOL"},
	{N_PSYM,	"N_PSYM"},
	{N_ENTRY,	"N_ENTRY"},
	{N_LBRAC,	"N_LBRAC"},
	{N_RBRAC,	"N_RBRAC"},
	{N_BCOMM,	"N_BCOMM"},
	{N_ECOMM,	"N_ECOMM"},
	{N_ECOML,	"N_ECOML"},
	{N_LENG,	"N_LENG"},
	{N_PC,		"N_PC"},
	{0,0}};
    struct nte *n;
    sprint(s->section ? ".stabd " : ".stabs ");
    if (s->name) fprint("\"%s\",",s->name); else sprint("0,");
    for (n=nametable; n->type != (s->type&-2) && n->symbol; n++);
    if (n->symbol) {
	sprint(n->symbol); if (s->type&1) sprint("+N_EXT");
	}
    else fprint("0x%2x",s->type);
    fprint(",%d",s->desc);
    if (!s->section) fprint(",%d",s->value);
    }

/*	Print a line containing a symbol.  If the symbol could properly be
	imbedded in a tab before disassembled code, return TRUE.  Otherwise
	return false to force the disassembler to start a new line after it.
*/
printsymbol(struct nlist *s) {
    extern struct nlist *sourcesymtab;
    extern char *symnametable;
    extern flag noextern;
    if (noextern && s->type==(N_UNDF|1) && !s->value) return(FALSE);
    if (s->section) startline(s->value);
    else startlinenoaddress();
    switch (s->type&-2) {
    case N_UNDF :
	if (s->value) fprint(".comm %s, %d",s->name,s->value);
	else fprint(".globl %s",s->name);
	break;
    case N_ABS :
	if (s->type&1) sprint(".globl ");
	fprint("%s = %x",s->name,s->value);
	break;
    case N_INDR:
	if (s->type&1) {
	    fprint(".globl %s",s->name);
	    startlinenoaddress();
	    }
	fprint("%s = %s",s->name,symnametable+s->value);
	break;
    case N_SECT:
	if (s->type&1) {
	    fprint(".globl %s",s->name);
	    startline(s->value);
	    }
	fprint(s->name ? s->name : "D%x", s->value);
	cprint(':');
	return(!s->name || strlen(s->name)<7);
	break;
    case N_SLINE:
	fprint("| line %d",s->desc);
	break;
    case N_SO:
	fprint("| source file \"%s\"",s->name);
	break;
    case N_SOL:
	fprint("| include file \"%s\"",s->name);
	break;
    default:
	printstab(s);
	break;
	}
    return(FALSE);
    }

/*	This is the usual method to create a disassembly symbol.  For
	now it is unlikely we will produce any other types of symbols
	than these static labels.  Make sure the name is in static
	storage! */
struct nlist *
createlabel(unsigned address,unsigned sn,unsigned type,char *name) {
    struct nlist *s;
    s = malloc(sizeof(struct nlist));
    s->name = name;
    s->type = N_SECT;
    if (!sn) sn=1;
    s->section = sn;
    s->desc = type;
    s->value = address;
    addsymbol(s);
    return(s);
    }
