/*------------------------------------------------------------------------------
	Executive.h				Copyright (c) 1990, Marc A. Davidson

	REVISIONS
	Date			Who		Modification
	24-Sept-90	MAD		Created (simple synchronous routines)
	5-Oct-90		MAD		Added cthread routines and queues for executing 
								asynchronous commands.
------------------------------------------------------------------------------*/
# import <objc/Object.h>
# import <cthreads.h>

# define CSTR	const char *

@interface Executive:Object
 /*
  * Executive allows the execution of shell commands from a program, providing
  * both synchronous and asynchronous execution of commands.  Other features
  * include the ability to direct command output to another object through the
  * popen(3) mechanism and displaying of errors through a standard panel for
  * consistency in error reporting throughout an application.
  */
{	id					target;
	SEL				action;
	double			period;					/* period at which entry gets called */
	/*
	 * internal instance variables
	 * @public is set for these since the internal commands need to acces
	 * them via self->xxx.
	 */
@public
	int				curCmdId;				/* current command id in use */
	int				numExecuting;			/* number of commands currently executing */
	mutex_t			runningLock;			/* for access to running queue */
	mutex_t			doneLock;				/* for access to done queue */
	id					running;					/* queue for commands that are executing */
	id					done;						/* queue for commands that are done */
}

+ new;
 /*
  * Allocates a new Executive object with the default period for updating during
  * asynchronous commands.
  */

+ newPeriod:(double)p;
 /*
  * Allocates a new Executive object with the specified period for updating
  * during asynchronous commands.
  */

- target;
- setTarget:anObject;
- (SEL) action;
- setAction:(SEL)aSelector;
 /*
  * Sets the target and action of the Executive for commands that are executed
  * asynchronously.
  */

- setPeriod:(double)p;
- (double)period;
 /*
  * Sets and queries the period instance variable.
  */

- (int) execute:(CSTR)command;
- (int) execute:(CSTR)command async:(BOOL)async;
- (int) execute:(CSTR)command environs:(CSTR)environs async:(BOOL)async;
 /*
  * The execute suite of methods execute a given command either synchronously or
  * asynchonously.  If the command is executed synchronously the result of the
  * system() call is returned by the method.  Otherwise, the result returned is
  * a unique identifier for this command (a small positive integer value). The
  * target is notified when the command completes.  The action that the target
  * implements should be a method that takes two arguments: the command number
  * (as returned by execute:async: or execute:async:environs:) and the result of
  * the system() call (i.e. commandComplete:(int)cmdNum withResult:(int)result).
  *
  *	SEE ALSO:
  *		system(3) for result codes returned.
  */

- (int)pipe:(CSTR)command to:anObject :(SEL)aSelector async:(BOOL)async;
- (int)pipe:(CSTR)command environs:(CSTR)environs to:anObject :(SEL)aSelector
	async:(BOOL)async;
 /*
  * The pipe methods execute a popen() call that will send the results of
  * reading a pipe to the specified object with a selector that takes either
  * one or two arguments.  If the pipe is executed asynchronously, the selector
  * should be for a method that takes two arguments, that of the command number
  * and one line that was read from the pipe
  * (i.e. processFrom:(int)cmdId line:(const char *)line).
  * When the asynchronous command completes it notifies the target of the object
  * (see above under execute: for notifying behavior).
  * If the pipe is executed synchronously, the selector method should accept one
  * argument, that of the line read (i.e. processLine:), complete with newline.
  * The result returned is either 0 or a unique identifier for the asynchronous
  * command (a small positive integer value, same as execute: above).
  */

- showError:(int)err;
- showError:(int)err while:(CSTR)doingWhat;
- showError:(int)err while:(CSTR)doingWhat on:(CSTR)fname;
- showError:(int)err while:(CSTR)doingWhat on:(CSTR)fname using:(CSTR)prog;
 /*
  * showError allows the application to have a regular set of error-reporting
  * abilities in various degrees of granularity.
  * The area in the upper portion of the NXRunAlertPanel panel will be the text
  * (by named parameter) "<prog> error"
  * The lower part will have the text "Error while <doingWhat> <fname>
  * (error code <err>)"
  *
  * All of the methods eventually call showError:while:on:using:.  If any of the
  * more general methods are called, they substitute defaults for the missing
  * parameters.
  *
  * The defaults for these methods are:
  *
  *		doingWhat	defaults to "executing a command"
  *		fname			defaults to "on a file"
  *		prog			defaults to "File"
  */

- showFError:(CSTR)fname;
 /*
  * showFError is used when a file error occurs during a system call.  It takes the
  * file name given and presents it with the text of the error recorded in the
  * errno global variable.
  */

@end

