 /*
  * Khoros: $Id: lvsamp.c,v 1.1 1991/05/10 15:41:54 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: lvsamp.c,v 1.1 1991/05/10 15:41:54 khoros Exp $";
#endif

 /*
  * $Log: lvsamp.c,v $
 * Revision 1.1  1991/05/10  15:41:54  khoros
 * Initial revision
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvsamp.c
 >>>>
 >>>>      Program Name: vsamp
 >>>>
 >>>> Date Last Updated: Thu Mar  7 15:56:37 1991 
 >>>>
 >>>>          Routines: lvsamp - the library call for vsamp
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvsamp - library call for vsamp
*
* Purpose:
*    
*    Sample an image along an arbitrary line.
*    
*    
* Input:
*    
*    image          pointer to image to be sampled
*    
*    line           pointer to image pointer  to  receive  the  output
*                   image
*    
*    x1             X coordinate of start point
*    
*    y1             Y coordinate of start point
*    
*    x2             X coordinate of end point
*    
*    y2             Y coordinate of end point
*    
*    n              number of samples to  generate  between  the  end-
*                   points
*    
*    
* Output:
*    
*    line           xvimage pointer indicated by line is  attached  to
*                   the new image
*    
*    
*
* Written By: Richard Krukar
*    
*    6-Feb-91 Scott Wilson - Cleaned up  for  installation,  completed
*    documentation.
*    
*    8-Feb-91 Scott Wilson - changed length to  n,  added  pixel  size
*    stuff.
*    
*    10-Feb-91 Scott Wilson -re-wrote the entire inner loop - the ori-
*    ginal had bugs and did not handle the boundary conditions.
*    
*    20-Feb-91 Scott Wilson - corrected pixel size computation bug.
*    
*    
****************************************************************/


/* -library_def */
int lvsamp(image,line,x1,y1,x2,y2,n)
struct xvimage *image,**line;
int x1,y1,x2,y2,n;
/* -library_def_end */

/* -library_code */
{
    int i,rows,cols;
    float x_delta,y_delta;
    float *i_data,*line_dat;
    struct xvimage *l;
    int yl,yh,xl,xh;
    float yn,xn;
    float z1,z2,z3,z4,za,zb,zp;
    float a,b;

    /* Check image characteristics */
    if (image->data_storage_type != VFF_TYP_FLOAT)
      {
        fprintf(stderr,"lvsamp: Input image must be of type FLOAT\n");
        return(0);
      }
    if (image->num_data_bands != 1 || image->map_scheme != VFF_MS_NONE)
      {
        fprintf(stderr,"lvsamp: Input image must be single band, no map!\n");
        return(0);
      }

    rows = image->col_size;
    cols = image->row_size;

    /* Check line endpoints to make sure they are in bounds */
    if (x1 < 0 || x1 > cols-1 ||
        y1 < 0 || y1 > rows-1 ||
        x2 < 0 || x2 > cols-1 ||
        y2 < 0 || y2 > rows-1)
      {
        fprintf(stderr,"lvsamp: line endpoints must lie within image boundaries!\n");
        return(0);
      }
    if (x1 == x2 && y1 == y2)
      {
        fprintf(stderr,"lvsamp: line endpoints must not be the same point!\n");
        return(0);
      }

    /* Make output image */
    *line = createimage(1,n,VFF_TYP_FLOAT,
                        1,1,"created by vsamp", 
                        0,0,VFF_MS_NONE,VFF_MAPTYP_NONE,
                        VFF_LOC_IMPLICIT,0);
    if (*line == NULL)
      {
        fprintf(stderr,"lvsamp: Unable to allocate memory for line image\n");
        return(0);
      }

    l = *line;
    line_dat = (float *)(l->imagedata);
    bzero(line_dat,n*sizeof(float));

    /* Compute sampling intervals */
    x_delta = (float)(x2-x1)/(float)(l->row_size-1);
    y_delta = (float)(y2-y1)/(float)(l->row_size-1);

    /* Set new pixel sizes */
    a = (x2-x1)*image->pixsizx;
    b = (y2-y1)*image->pixsizy;
    l->pixsizx = sqrt((double)(a*a+b*b))/(float)(l->row_size-1);
    l->pixsizy = l->pixsizx;

    i_data = (float *)(image->imagedata);
    for(i=0; i<(l->row_size)-1; i++)
      {
        /* Compute coordinates of i'th sample point */
        xn = x1 + i*x_delta;
        yn = y1 + i*y_delta;
        /* Compute coordinates of pixels bounding the box the sample
           point lies in */
        xl = floor((double)xn);
        xh = xl+1;
        yl = floor((double)yn);
        yh = yl+1;
        if (xl == cols-1 && yl == rows-1)
          {
            /* Desired point is corner value - just return it */
            *line_dat = i_data[PIXEL(xl,yl,rows,cols)];
          }
        else if (xl == cols-1 && yl != rows-1)
          {
            /* If were at the right hand edge then interpolate only
               on the y values.
            z1 = i_data[PIXEL(xl,yl,rows,cols)];
            z2 = i_data[PIXEL(xl,yh,rows,cols)];
            zp = z1*(yh-yn)+z2*(yn-yl);
            *line_dat = zp;
          }
        else if (xl != cols-1 && yl == rows-1)
          {
            /* Were on the bottom edge, so interpolate only on the
               x values */
            z1 = i_data[PIXEL(xl,yl,rows,cols)];
            z2 = i_data[PIXEL(xh,yl,rows,cols)];
            zp = z1*(xh-xn)+z2*(xn-xl);
            *line_dat = zp;    
          }
        else
          {
            /* If the containing box is fully in bounds then interpolate the 
               four pixel values to find the intermediate value */
            z1 = i_data[PIXEL(xl,yl,rows,cols)];
            z2 = i_data[PIXEL(xh,yl,rows,cols)];
            z3 = i_data[PIXEL(xh,yh,rows,cols)];
            z4 = i_data[PIXEL(xl,yh,rows,cols)];
            za = z1*(yh-yn)+z4*(yn-yl);
            zb = z2*(yh-yn)+z3*(yn-yl);
            zp = za*(xh-xn)+zb*(xn-xl);
            *line_dat = zp;
          }
        line_dat++;
      }
    *line_dat = i_data[PIXEL(x2,y2,rows,cols)];
    return(1);
}
/* -library_code_end */
