/*
 *  $Id: main.c,v 1.1 1998/12/02 23:37:08 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <linux/module.h>	/* include only once in a logical module */
#include <fist.h>
#include <cryptfs.h>



// cryptfs_interpose(inode_t *hidden_inode, inode_t **result, super_block_t *super)
int
cryptfs_interpose(dentry_t *hidden_dentry, dentry_t *this_dentry, super_block_t *sb, int flag)
{
    inode_t *hidden_inode = hidden_dentry->d_inode;
    int err = 0;
    inode_t *new_inode;

    print_entry_location();

    ASSERT(hidden_inode != NULL);
    ASSERT(this_dentry->d_inode == NULL);

    /* check that the lower file system didn't cross a mount point */
    if (hidden_inode->i_dev != stohs(sb)->s_dev) {
	dput(hidden_dentry);
	err = -EXDEV;
	goto out;
    }

    /*
     * iget will eventually call our read_inode, and that will create
     * the link between our inode and hidden_inode.
     *
     * XXX: fix assumption that hidden and this inode numbers are same
     */
    new_inode = iget(sb, hidden_inode->i_ino);
    if (!new_inode) {
	dput(hidden_dentry);
	err = -EACCES;		/* should be impossible??? */
	goto out;
    }

    /* only (our) lookup wants to do a d_add */
    if (flag)
	d_add(this_dentry, new_inode);
    else
	d_instantiate(this_dentry, new_inode);

    if (dtopd(this_dentry) == NULL) {
	dtohd(this_dentry) = hidden_dentry;
    } else {
	ASSERT(dtohd(this_dentry) == hidden_dentry);
    }

    /*
     * We allocated our new inode above, as a side effect of calling iget
     * which is done in a VFS generic code.   Linking our inode to the
     * hidden one is done when iget calls the generic read_inode,
     * which calls our read_inode.
     */

    // all well, update inode modes */
    fist_copy_attr_all(new_inode, hidden_inode);

out:
    print_exit_status(err);
    return err;
}


#ifdef FIST_DEBUG
/* find hidden dentry given this cryptfs dentry */
dentry_t *
__cryptfs_hidden_dentry(char *file, char *func, int line, dentry_t *this_dentry)
{
    dentry_t *hidden_dentry;

    ASSERT(this_dentry != NULL);
    ASSERT(this_dentry->d_op != NULL);
    ASSERT(this_dentry->d_op == &cryptfs_dops);
    ASSERT(this_dentry->d_sb->s_op == &cryptfs_sops);
    if (this_dentry->d_inode)
	ASSERT(this_dentry->d_inode->i_op == &cryptfs_iops);

    hidden_dentry = dtohd(this_dentry);
    ASSERT(hidden_dentry != NULL);
    return hidden_dentry;
}
#endif /* FIST_DEBUG */


STATIC dentry_t *
cryptfs_parse_options(super_block_t *sb, char *options)
{
    dentry_t *hidden_root = ERR_PTR(-EINVAL);
    print_entry_location();

    while (*options) {
	if (!strncmp("dir=", options, 4)) {
	    fist_dprint(4, "cryptfs: using directory: %s\n", options + 4);
	    /* note: the name passed need not be encoded */
	    hidden_root = lookup_dentry(options+4, NULL, LOOKUP_FOLLOW);
	    fist_dprint(6, "parse_options: new s_root, inode: 0x%x, 0x%x\n",
			(int) hidden_root, (int) hidden_root->d_inode);
	} else {
	    printk(KERN_WARNING "cryptfs: unrecognized options '%s'\n", options);
	}
	while (*options && *options != ',')
	    options++;
    }
    print_exit_location();
    return hidden_root;
}


super_block_t *
cryptfs_read_super(super_block_t *sb, void *raw_data, int silent)
{
    super_block_t *ret_sb = NULL;
    dentry_t *hidden_root;

    print_entry_location();

    MOD_INC_USE_COUNT;
    lock_super(sb);
    if (!raw_data) {
	printk(KERN_WARNING "cryptfs_read_super: missing data argument\n");
	goto out_module;
    }
    /*
     * Allocate superblock private data
     */
    stopd(sb) = kmalloc(sizeof(struct cryptfs_sb_info), GFP_KERNEL);
    stohs(sb) = NULL;

    hidden_root = cryptfs_parse_options(sb, raw_data);
    if (IS_ERR(hidden_root)) {
	printk(KERN_WARNING "cryptfs_read_super: lookup_dentry failed (err = %ld)\n", PTR_ERR(hidden_root));
	goto out_free;
    }
    if (!hidden_root->d_inode) {
	printk(KERN_WARNING "cryptfs_read_super: no directory to interpose on\n");
	goto out_free;
    }
    stohs(sb) = hidden_root->d_sb;

    sb->s_op = &cryptfs_sops;
    /*
     * we can't use d_alloc_root if we want to use
     * our own interpose function unchanged,
     * so we simply replicate *most* of the code in d_alloc_root here
     */
    sb->s_root = d_alloc(NULL, &(const struct qstr) { "/", 1, 0 });
    if (IS_ERR(sb->s_root)) {
	printk(KERN_WARNING "cryptfs_read_super: iget failed\n");
	goto out_dput;
    }

    sb->s_root->d_op = &cryptfs_dops;
    sb->s_root->d_sb = sb;
    sb->s_root->d_parent = sb->s_root;
    if (cryptfs_interpose(hidden_root, sb->s_root, sb, 0))
	goto out_dput2;
#if 0
    /*
     * interpose always increments the hidden inode's refcount by one
     * but in this function it was already done by the lookup_dentry
     * above (called from parse_options), which eventually calls iget() on
     * hidden_inode, thus increasing the refcount once too many.
     */
    iput(hidden_root->d_inode);
#endif

    ret_sb = sb;
    fist_print_dentry(__FUNCTION__ "OUT hidden_dentry", hidden_root);
    goto out;

 out_dput2:
    dput(sb->s_root);
 out_dput:
    dput(hidden_root);
 out_free:
    kfree_s(stopd(sb), sizeof(struct cryptfs_sb_info));
    stopd(sb) = NULL;
 out_module:
    MOD_DEC_USE_COUNT;
 out:
    unlock_super(sb);
    print_exit_location();
    return ret_sb;
}


/*----*/

STATIC struct file_system_type cryptfs_fs_type =
{
    "cryptfs",			/* name */
    0,				/* flags */
    cryptfs_read_super,		/* read_super function */
    NULL			/* next file_system_type */
};


int
init_cryptfs_fs(void)
{
    return register_filesystem(&cryptfs_fs_type);
}


/* Every kernel module contains stuff like this. */
#ifdef MODULE
int
init_module(void)
{
    printk("Inserting module cryptfs version $Id: main.c,v 1.1 1998/12/02 23:37:08 ezk Exp $\n");
    return init_cryptfs_fs();
}

void
cleanup_module(void)
{
    printk("Removing module cryptfs version $Id: main.c,v 1.1 1998/12/02 23:37:08 ezk Exp $\n");
    unregister_filesystem(&cryptfs_fs_type);
}

void
fist_mod_dec_use_count(void)
{
    MOD_DEC_USE_COUNT;
}
#endif /* MODULE */

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
