
/* beginning of ecashlib.h */


/****************************************************************
*								*
* ecash.h							*
*								*
* Version history:						*
*  960123 -	Ian Goldberg <iang@cs.berkeley.edu>		*
*  960129 -	Felix Croes <felix@digicash.com>		*
*  960327 -	Bryce Wilcox <bryce@digicash.com>		*
*		Felix Croes <felix@digicash.com>		*
*		Marcel van der Peijl <bigmac@digicash.com>	*
*		Branko Lankester <branko@digicash.com>		*
*		Niels Ferguson <niels@digicash.com>		*
*  960612 -     Bryce Wilcox <bryce@digicash.com>               *
*								*
*								*
*								*
* Copyright (c) 1996 DigiCash					*
*								*
* "DigiCash" and "Ecash" are trademarks.			*
*								*
* This file is published for informational purposes only.	*
* DigiCash makes no claims about its suitability for any	*
* particular purpose.						*
*								*
****************************************************************/




/* beginning of eecdef.h */

/************************************************************************
*									*
* eecdef.h								*
*									*
* Common macros and typedefs that need to be included into each         *
* Ecash(tm) program.                                                    *
*									*
************************************************************************/

/* 
 * Here is our platform specific stuff.  Hopefully this will work
 * on your platform just by #define'ing the proper flag.  If not, let
 * us know.  If there is not a flag here for your platform then 
 * hopefully you can do without one!  (This is, after all, a pure 
 * ANSI C library plus support for Windows DLL's.)
 */


/* 
 * Platform-identification flags:
 *
 * "_Windows" -- Windows 16 or 32 bit or Windows NT (the only reason for this
 *	flag is to enable Windows DLL's-- see below.)
 *
 * "__alpha" -- DEC Alpha CPU architecture (the only reason for this flag is 
 *	to correctly identify a 32-bit data type (an Alpha "int")-- see below.)
 */

#ifdef _WINDOWS
#include <windows.h>

#ifdef WIN32
#define EC_EXPORT __declspec(dllexport)
#else
#define EC_EXPORT WINAPI
#endif

/*
 * Define "EC_EXPORT" to be a keyword your platform needs to make a 
 * function available for linking.  To make a Windows DLL, for  
 * example, define it to be "WINAPI".  
 */


#else /* #ifdef _WINDOWS */
#define EC_EXPORT
#endif


#ifndef EC_EXPORT
#define EC_EXPORT
#endif


typedef long Long;
typedef unsigned long ULong;
typedef short Short;
typedef unsigned short UShort;
typedef int Int;
typedef unsigned int UInt;
typedef char Char;
typedef unsigned char Byte;


#undef EC_MAXLONG
#define EC_MAXLONG (2 << --(sizeof(Long) * 8))L
#undef EC_MINLONG
#define EC_MINLONG ++(-1 * (2 << --(sizeof(Long) * 8)))L

#undef EC_MAXULONG
#define EC_MAXULONG ((2 << (sizeof(Long) * 8)) -1)L

#undef EC_MAXSHORT
#define EC_MAXSHORT (2 << --(sizeof(Short) * 8))L
#undef EC_MINSHORT
#define EC_MINSHORT ++(-1 * (2 << --(sizeof(Short) * 8)))L

#undef EC_MAXUSHORT
#define EC_MAXUSHORT ((2 << (sizeof(Short) * 8)) -1)L

#undef EC_MAXINT
#define EC_MAXINT (2 << --(sizeof(Int) * 8))L
#undef EC_MININT
#define EC_MININT ++(-1 * (2 << --(sizeof(Int) * 8)))L

#undef EC_MAXUNIT
#define EC_MAXUINT ((2 << (sizeof(Int) * 8)) -1)L

#undef EC_MAXCHAR
#define EC_MAXCHAR (2 << --(sizeof(Char) * 8))L
#undef EC_MINCHAR
#define EC_MINCHAR ++(-1 * (2 << --(sizeof(Char) * 8)))L

#undef EC_MAXBYTE
#define EC_MAXBYTE (2 << (sizeof(Byte) * 8)L



typedef short Int16;
typedef unsigned short UInt16;


#ifdef __alpha
typedef int Int32;
typedef unsigned int UInt32;
#else
typedef long Int32;
typedef unsigned long UInt32;
#endif


#undef EC_MAXINT16
#define EC_MAXINT16 (Int16)32768
#undef EC_MININT16
#define EC_MININT16 (Int16)-32767

#undef EC_MAXUINT16
#define EC_MAXUINT16 (UInt16)65535

#undef EC_MAXINT32
#define EC_MAXINT32 (Int32)2147483648L
#undef EC_MININT32
#define EC_MININT32 (Int32)-2147483647L

#undef EC_MAXUINT32
#define EC_MAXUINT32 (UInt32)4294967295L




#define EC_MINTID_INVALID 0L /* it is supposed to be an error if you ask for
	an EC_MintID and you get this. */
#define EC_MINTID_UNKNOWN 1L

#define EC_CURRID_INVALID 0L /* it is supposed to be an error if you ask for
	an EC_CurrencyID and you get this. */
#define EC_CURRID_UNKNOWN 1L

#define EC_MINPAYMENTS_INVALID EC_MAXUINT16 /* it is supposed to be an error
	if you ask for "minimum payments" and get this. */

#define EC_ACCID_NONE "There is no EC_AccountID present.  This probably indicates an error in the program."

#define EC_MINTINFO_NONE "There is no information about this mint available yet.  This is probably because your Ecash(tm) client has not communicated with the mint yet."

	/*!! note that the base_magnitude, base_factor, and granularity are 
	just hardcoded here for now.  Should be fixed-- either transmit this 
	data in the setup message or have a table mapping mintID/currency 
	pairs onto base_magnitude, base_factor, granularity triplets. */

#define EC_COINAGE_BASE_MAGNITUDE_DEFAULT -2
#define EC_COINAGE_BASE_FACTOR_DEFAULT 1
#define EC_COINAGE_GRANULARITY_DEFAULT -2



/* end of eecdef.h */




/*
 * About Classes:
 *
 * The ecashlib is best understood as being made up of a set
 * of object oriented classes.  Each class is a conceptual
 * unit designed to perform a specific function in the
 * ecashlib.  Each class has some private data members which
 * contain all of the data that the class uses.  Each class
 * also has some public methods which can be called by other
 * classes or by the Ecash application.
 * 
 * (In fact, many classes also have protected methods which 
 * are available only to ecashlib classes, and private
 * methods, which are available only to the class itself.
 * But Ecash application programmers will not need to know
 * about those methods.)
 *
 * The struct which contains a classes private data is named
 * "EC_Foo".  The methods of a class are named 
 * "EC_foo_func()".  Each method (except for the 
 * "EC_foo_new()" method) takes as its first argument a
 * "this pointer"-- a pointer to the instance of class Foo
 * which is to execute the method.  
 *	Objects (class instances) are created either by 
 * calling "EC_foo_new()" or by calling a "get" method which
 * gets an instance of the class.  For example calling 
 * "EC_info_get_coinage()" creates an instance of EC_Coinage
 * and passes back a pointer to it.
 *	Objects are destroyed by calling "EC_foo_free()".  This
 * is like a C++ destructor.  It performs clean-up 
 * operations, frees any memory allocations or other locks 
 * that the object holds and frees the structure which 
 * contains the object's private data.
 *
 * Although this object-oriented programming methodology may
 * be unfamiliar to you at first, it should become intuitive
 * and valuable as soon as you begin to use it.
 */


/*
 * Conventions:
 *
 *
 *	  Names
 *
 * All global symbols and data types are prefixed with "EC_" to 
 * prevent namespace conflicts.
 *
 * 
 * For functions:	EC_classabbreviation_function_name
 * For data types:	EC_DataTypeName
 * For constants:	EC_CONSTANT_NAME
 * For variables:	var_name
 *				 
 * Each function name includes the name of the class to which that 
 * function belongs.  Functions which have no class (the
 * initialization functions and a couple of others) use the name 
 * "main" for their class name.
 * 
 * Variable names have special conventions.  (Note that in this 
 * API variable names are only found as parameter names in function 
 * declarations.)
 *  1. Pointer variables have "_ptr" appended to their name if they are 
 * going to be assigned to a _new_ object or if they are going to have 
 * free called upon them.  Else, they have "_lnk" appended to their 
 * name.
 *  2. NULL-terminated Char * variables have "_str" in their name.
 *  3. If a parameter is a pointer pointer and it is used to pass data
 * _out_ of the function, then it has "_out" appended to its name.  Note
 * that anytime the data being passed out is a pointer, it is the caller's
 * responsibility to free that pointer once the caller is finished with it.
 *
 *
 *	 Parameters
 *
 * Functions which belong to a class (except for the 'new' functions)
 * always have as their first parameter a pointer to an instance of 
 * that class.
 *
 * Pointer parameters which are used to pass data out of the function
 * always occur after other parameters.
 *
 * On some platforms functions which are exported from the ecashlib
 * will need a keyword in the declaration (e.g. "far", "loadds", 
 * "pascal").  To handle this, each function declared in the API 
 * header files will be prefixed with a macro "EC_EXPORT" which will be
 * #define'd to the appropriate keywords for that platform.  Each
 * application function that is to be called by the ecashlib must be
 * pre-fixed by the macro "CALLBACK" which will be similarly
 * #define'd.
 */


/*
 * What version of the API is this?  The major number is the
 * version of the API.  The minor number represents a revision
 * of the API which is not a version change.
 * Format: major * 0x0100 + minor
 */
#define EC_API_VERSION 0x0100




		/* primitives */

	/* Int, Short, Long, Int32, Int16, Byte */

/*
 * eecdef.h defines the following:
 * 
 * Int -- a primitive datatype which is at least as long as a Short
 *	and at most as long as a Long.
 * UInt -- same as above, but unsigned.
 * Short -- a primitive datatype which is at least 16 bits.
 * UShort -- same as above, but unsigned.
 * Long -- a primitive datatype which is at least 32 bits and is at
 *	least as long as a Short.
 * ULong -- same as above, but unsigned.
 * Int32 -- a primitive datatype which is exactly 32 bits long.
 * UInt32 -- same as above but unsigned.
 * Int16 -- a primitive datatype which is exactly 16 bits long.
 * UInt16 -- same as above but unsigned.
 * Byte -- a primitive datatype which is exactly 8 bits long and
 *	unsigned.
 */



	/* ecashlib header files */

/* beginning of epocktid.h */


	/* ecashlib public headers */

/* beginning of epocktid.fh */


	/* EC_PocketID */

/*
 * An EC_PocketID consists of an EC_AccountID and a 
 * NULL-terminated string which points to the directory on 
 * the local file system wherein the ecash files for this
 * pocket are (going to be) kept.
 */

typedef struct EC_PocketID_s EC_PocketID;



/* end of epocktid.fh */





/* beginning of eaccid.fh */


	/* EC_AccountID */

/*
 * An EC_AccountID consists of an EC_MintID and a 
 * NULL-terminated string which uniquely names the account
 * at that mint.
 *
 * notes:
 *	If you want to converts EC_AccountIDs into a portable 
 *	format, a good convention to follow is to concatenate a 
 *	string representation of the EC_MintID, a colon (':'), 
 *	and the account name into a single string.
 *
 *	Currently an EC_MintID is defined as an unsigned integer,
 *	so the first part of this string should be that integer 
 *	represented as a string of decimal numerals.
 *
 *	Current convention is to use the user's e-mail address as
 *	the account name.
 */


typedef struct EC_AccountID_s EC_AccountID;



/* end of eaccid.fh */




/* beginning of eerrno.fh */


	/* EC_Errno */

/*
 * Every function except for the "Get data" functions returns
 * an EC_Errno.  Also each action handler has an EC_Errno which is a 
 * part of its state, and is accessible via EC_acth_get_errno().
 */


	/* non error */
/*
 *	EC_ERR_INVALID
 * You shouldn't ever see an EC_Errno in this state.  It means the 
 * EC_Errno hasn't been properly initialized.
 *
 *	EC_ERR_NONE
 * No error.  Some functions which need to return a boolean value use 
 * this to represent a "true" return value.
 *
 *	EC_ERR_NONE_2
 * No error.  This is returned by some functions to indicate that a 
 * certain condition occurred during its operation, which was not 
 * an error condition.  Some functions which need to return a boolean
 * value use this to represent a "false" return value.
 *
 *	EC_ERR_NONE_3
 * No error.  This is returned by some functions to indicate that a 
 * certain condition occurred during its operation, which was not 
 * an error condition.
 */

	/* exception from user, application, network or foreign 
	ecash application */
/*
 *	EC_ERR_ACTION_ABORTED_REWOUND
 * A complex action was aborted via a call to its action handler's 
 * "EC_acth_abort()" function.  The action was not yet complete, so 
 * the state of the ecashlib was rewound, making it as if the action
 * had never been initiated.
 *
 *	EC_ERR_ACTION_ABORTED_FF
 * A complex action was aborted via a call to its action handler's 
 * "EC_acth_abort()" function.  The action was already complete, so
 * the state of the ecashlib was "fast forwarded", making it as if
 * the action completed normally, except for the lack of some
 * auxiliary information.
 *
 *	EC_ERR_ACTION_REJECTED
 * A complex action was rejected by the mint or by the other party to 
 * the transaction.  It is likely that the rejecting party sent a
 * communication describing the reason for rejection.  See 
 * EC_acth_get_communication().
 */

	/* error on the part of the application */
/*
 *	EC_ERR_BAD_VALUE
 * A value was encountered which was of the proper form, but which did
 * not cause the desired result.  This is typically bad user input.  
 * This error is never fatal.
 *
 *	EC_ERR_ILLEGAL_OPERATION
 * A condition was encountered which should never occur in a bug-free
 * application.  Typically this is caused by the application passing 
 * an illegal value to the ecashlib, such as a NULL pointer where a 
 * non-NULL pointer was expected or vice versa.
 *
 *	EC_ERR_NULL_THIS
 * The 'this' argument to a function was a NULL pointer.
 *
 *	EC_ERR_CALLBACK
 * A callback method failed.
 *
 *	EC_ERR_ACTH_MANAGEMENT
 * An ecashlib action handler was mis-used, e.g. calling 
 * EC_acth_set_msg() when the action handler does not require any 
 * message.  These errors are rarely fatal.
 */

	/* system failure/resource depletion */
/*
 *	EC_ERR_OUT_OF_MEM
 * A call to the "malloc" callback returned a NULL pointer.
 */

	/* error on the part of ecashlib */
/*
 *	EC_ERR_INTERNAL
 * An unknown error has occurred within the ecashlib.  This could be 
 * caused by a particularly destructive error on the part of the 
 * application, or it could be caused by an error on the part of the 
 * ecashlib.  Running out of memory can also generate EC_ERR_INTERNAL.
 * These errors may be fatal.
 *
 *	EC_ERR_NOT_IMPLEMENTED
 * The ecashlib was asked to do something that it _will_ do
 * when it is finished, but that it doesn't do now.
 */



	/* Standard return values */

/*
 * Every function which returns an EC_Errno may return each
 * of these:
 *	EC_ERR_NONE: meaning the function completed successfully.
 *	EC_ERR_NULL_THIS: meaning that the first argument to the
 *		function, which should have pointed to an instance of 
 *		the function's class, was NULL.  (This value is never 
 *		returned by 'main' functions which have no 'this'
 *		parameter.)
 *	EC_ERR_CALLBACK: meaning that the function made a call to
 *		a callback function and received a 'failure' return
 *		value.
 *	EC_ERR_OUT_OF_MEM: meaning that somewhere down the call
 *		stack a call was made to the "malloc" callback and a
 *		NULL pointer was returned from that call.
 *	EC_ERR_INTERNAL: meaning that an internal error occurred
 *		in the ecashlib.
 *	EC_ERR_NOT_IMPLEMENTED: meaning that the function
 *		tried to do something that is not yet
 *		implemented.  This should go away in final
 *		revision.
 */


typedef enum
{
		/* non-error */
	EC_ERR_INVALID = -1,

	EC_ERR_NONE = 0,
	EC_ERR_NONE_2 = 1,
	EC_ERR_NONE_3 = 2,

		/* exception from user or foreign ecash application */
	EC_ERR_ACTION_ABORTED_REWOUND = 16,
	EC_ERR_ACTION_ABORTED_FF = 17,
	EC_ERR_ACTION_REJECTED = 18,

		/* error on the part of the application */
	EC_ERR_BAD_VALUE = 32,
	EC_ERR_ILLEGAL_OPERATION = 33,
	EC_ERR_NULL_THIS = 34,
	EC_ERR_CALLBACK = 35,
	EC_ERR_ACTH_MANAGEMENT = 36,

		/* system failure/resource depletion */
	EC_ERR_OUT_OF_MEM = 48,

		/* error on the part of ecashlib */
	EC_ERR_INTERNAL = 64,
	EC_ERR_NOT_IMPLEMENTED = 65
} EC_Errno;



/* end of eerrno.fh */













	/* EC_PocketID functions */

/*
 * Create a new EC_PocketID.
 *
 * parameters:
 *	accID_ptr: the EC_AccountID
 *      base_dir_str: a string pointing to the base directory
 *
 * return values:
 *      pointer to the new EC_PocketID on success, NULL pointer on failure
 */
  EC_PocketID *EC_pocketID_new(const EC_AccountID *accID_lnk, 
	const char *base_dir_str);


/*
 * Free an EC_PocketID.
 *
 * return values:
 *	standard EC_Errno
 */
  EC_Errno EC_pocketID_free(EC_PocketID *this_pocketID_ptr);


/*
 * Duplicate an EC_PocketID.
 *
 * return values:
 *      pointer to the new EC_PocketID on success, NULL pointer on failure
 */
  EC_PocketID *EC_pocketID_dup(const EC_PocketID *this_pocketID_lnk);


/*
 * Get the contents of an EC_PocketID.
 */
  EC_AccountID *EC_pocketID_get_accID(
	const EC_PocketID *this_pocketID_lnk);

  char *EC_pocketID_get_base_dir(const EC_PocketID *this_pocketID_lnk);



/* end of epocktid.h */




/* beginning of eaccid.h */


	/* ecashlib public headers */


/* beginning of emintid.fh */




	/* EC_MintID */

/*
 * An EC_MintID is an unsigned 32-bit int which uniquely
 * identifies an ecash mint.  The list of mint ID's already taken is
 * maintained by DigiCash.  One mint ID which is defined in the
 * ecashlib header files is EC_MINTID_INVALID.
 */

typedef UInt32 EC_MintID;


/* end of emintid.fh */








	/* EC_AccountID functions */

/*
 * Create a new EC_AccountID.
 *
 * parameters:
 *	mintID: the ID uniquely identifying the mint
 *	acc_name_str: a string (usually an e-mail address) uniquely (within
 *		the context of the mint) identifying the account
 *
 * return values:
 *	pointer to the new EC_AccountID on success, NULL pointer on failure
 */
  EC_AccountID *EC_accID_new(EC_MintID mintID, const Char *acc_name_str);


/*
 * Free the EC_AccountID.
 *
 * return values:
 *	standard EC_Errno
 */
  EC_Errno EC_accID_free(EC_AccountID *this_accID_ptr);


/*
 * Duplicate an EC_AccountID.
 *
 * return values:
 *	pointer to the new EC_AccountID on success, NULL pointer on failure
 */
  EC_AccountID *EC_accID_dup(const EC_AccountID *this_accID_lnk);


/*
 * Get the contents of an EC_AccountID.
 */
  EC_MintID EC_accID_get_mintID(const EC_AccountID *this_accID_lnk);

  Char *EC_accID_get_acct_name(const EC_AccountID *this_accID_lnk);



/* end of eaccid.h */




/* beginning of eerrno.h */






/* end of eerrno.h */




/* beginning of erecord.h */


	/* ecashlib public headers */

/* beginning of eaddr.fh */


	/* EC_Address */

/*
 * An EC_Address contains all the information necessary for the
 * application to deliver a message to another ecash application.
 * The contents and the format are invisible to ecashlib since
 * ecashlib does not do any networking.  That is: the ecashlib never
 * creates or reads an EC_Address-- it just passes them from
 * application code to application code.
 *
 * The formats for EC_Address (e.g. TCP/IP, e-mail, and future 
 * networking protocols) will be developed by DigiCash in 
 * conjunction with application developers.
 */

typedef struct EC_Address_s EC_Address;



/* end of eaddr.fh */




/* beginning of erecord.fh */



	/* EC_Record, EC_PaymentRecord and EC_Booking */

/*
 * The ecashlib keeps a list of records.  There are two types of 
 * records: coin payment records, and bookings-- or records of things
 * that happened to the account.
 */


typedef struct EC_RecordList_s EC_RecordList;

typedef struct EC_Record_s EC_Record;

typedef struct EC_Booking_s EC_Booking;

typedef struct EC_PaymentRecord_s EC_PaymentRecord;


typedef enum
{
	EC_TRAN_STATE_SUCCESSFUL = 0,
	EC_TRAN_STATE_UNFINISHED = 1,
	EC_TRAN_STATE_REJECTED = 2,
	EC_TRAN_STATE_CANCELLED = 3,
	EC_TRAN_STATE_ERROR = 4
} EC_TransactionState;


typedef enum
{
	EC_RTYPE_PREC = 0,
	EC_RTYPE_BOOKING = 1
} EC_RecordType;


typedef enum
{
	EC_BOOKINGTYPE_WITHDRAWAL = 0,
	EC_BOOKINGTYPE_DEPOSIT = 1,
	EC_BOOKINGTYPE_ACCEPT_PAYMENT = 2
} EC_BookingType;



/* end of erecord.fh */




/* beginning of epocket.fh */


	/* EC_Pocket */
/*
 * An EC_Pocket is an object responsible for all of the
 * ecash operations having to do with a given account.  
 * (Note that a specific account is tied to a specific 
 * issuer.)
 *
 * When you deposit cash from that EC_Pocket, it goes into 
 * that EC_Account.  When you accept a payment using that
 * EC_Pocket, the money is cleared using that account.
 *
 * For now, when you withdraw cash from that account, it 
 * goes into that EC_Pocket.  In the future it will be 
 * possible to withdraw cash from an account into one of
 * multiple EC_Pockets, but for now the relationship 
 * between EC_Pockets and bank accounts is one-to-one.
 *
 * If you want to deal with multiple accounts, you make
 * multiple EC_Pockets.
 */


typedef struct EC_Pocket_s EC_Pocket;



/* end of epocket.fh */




/* beginning of eamount.fh */



	/* EC_Amount */

/*
 * An EC_Amount is a signed 32-bit integer.  It represents an amount
 * of money in units of the coinage's base value.
 */


typedef Int32 EC_Amount;




/* end of eamount.fh */












	/* EC_RecordList functions */


/* Read the next record from a record list. */
  EC_Record *EC_rlist_read_record(EC_RecordList *rlist_lnk);

/* Free a record list. */
  EC_Errno EC_rlist_free(EC_RecordList *rlist_ptr);

/*
 * Delete all but the last num entries in a record list.
 * Note that once a record has been deleted you can no longer cancel
 * or re-send that payment.  On the other hand it may be wasteful of
 * resources to keep _all_ records forever.  The application should 
 * never call EC_rlist_snip() on records that it might want to
 * cancel or resend, but it _should_ call EC_rlist_snip() on
 * records that it will never want to cancel or resend.
 */
  EC_Errno EC_rlist_snip(EC_RecordList *rlist_lnk, UInt32 num);



	/* EC_Record functions */


/* Free a record. */
  EC_Errno EC_rec_free(EC_Record *record_ptr);

/* Find out what kind of record this is. */
  EC_RecordType EC_rec_get_type(const EC_Record *record_lnk);

/*
 * Get the EC_PaymentRecord from this record.  Returns NULL if
 * called on an EC_Record which is not a payment record.
 */
  EC_PaymentRecord *EC_rec_get_prec(const EC_Record *record_lnk);

/*
 * Get the EC_Booking from this record.  Returns NULL if called
 * on an EC_Record which is not a booking.
 */
  EC_Booking *EC_rec_get_booking(const EC_Record *record_lnk);



	/* EC_PaymentRecord functions. */

  EC_Errno EC_prec_free(EC_Record *record_ptr);

/* Get the payer's address from a payment record. */
  EC_Address *EC_prec_payer_address(const EC_Record *record_lnk);

/* Get the Account ID of the recipient from a payment record. */
  EC_AccountID *EC_prec_recip_ID(const EC_Record *record_lnk);

/* Get the amount of the payment from a payment record. */
  EC_Amount EC_prec_amount(const EC_Record *record_lnk);

/*
 * Get the time of the payment from a payment record.
 * The return value is an unsigned 32-bit integer representing
 * seconds since 00:00:00 UTC Jan. 1, 1970.
 */
  UInt32 EC_prec_time(const EC_Record *record_lnk);

/* Get the description of the payment from a payment record. */
  char *EC_prec_desc(const EC_Record *record_lnk);

/* Get the state of a payment from a payment record. */
  EC_TransactionState EC_prec_get_state(const EC_PaymentRecord *prec_lnk);



	/* EC_Booking functions */

  EC_Errno EC_book_free(EC_Booking *book_ptr);

/* Get booking type. */
  EC_BookingType EC_book_get_type(const EC_Booking *book_lnk);

/* Get the amount. */
  EC_Amount EC_book_amount(const EC_Booking *book_lnk);

/*
 * Get the time the event was initiated.
 * The return value is an unsigned 32-bit integer representing
 * seconds since 00:00:00 UTC Jan. 1, 1970.
 */
  UInt32 EC_book_start_time(const EC_Booking *book_lnk);

/*
 * Get the time that the event was completed.
 * The return value is an unsigned 32-bit integer representing
 * seconds since 00:00:00 UTC Jan. 1, 1970.
 */
  UInt32 EC_book_end_time(const EC_Booking *book_lnk);

/* Get the description of the booking. */
  char *EC_book_desc(const EC_Booking *book_lnk);

/* Get the state of a booking. */
  EC_TransactionState EC_book_get_state(const EC_Booking *book_lnk);



/* end of erecord.h */




/* beginning of eacth.h */




	/* ecashlib public headers */

/* beginning of emsg.fh */



	/* EC_Msg */

/* An EC_Msg contains information that is to be passed from client
 * to client or client to server or server to client.  Usually
 * the contents of an EC_Msg are not important to the application--
 * it is just a chunk of data with a length.
 */

typedef struct EC_Msg_s EC_Msg;



	/* EC_MsgType */

/* There are currently three kinds of messages from the application's
 * point of view: payments, payment requests, and "other"
 * (none-of-the-above).  Currently all synchronous messages fall into
 * the "other" category.
 */

typedef enum
{
	EC_MSGTYPE_INVALID = -1,
	EC_MSGTYPE_OTHER = 0,
	EC_MSGTYPE_PAYMENT = 1,
	EC_MSGTYPE_PAYREQ = 2
} EC_MsgType;



/* end of emsg.fh */




/* beginning of einfo.fh */


	/* EC_Info */

/* EC_Info contains information about an account: account ID,
 * information about the user, the bank, balance, currency, etc.
 * It provides a snap-shot of the account at the time that
 * the EC_Info was created.
 */

typedef struct EC_Info_s EC_Info;




/* end of einfo.fh */




/* beginning of eacth.fh */



	/* EC_ActionHandler */

/*
 * An EC_ActionHandler is responsible for managing the interaction
 * between the application and the ecashlib during a "complex 
 * action".
 *
 * There is exactly one EC_ActionHandler per complex action.  It is
 * created in response to the "EC_acc_begin" call that initiated the
 * complex action.
 */



typedef struct EC_ActionHandler_s EC_ActionHandler;



	/* EC_StateType */

/*
 * Each action handler has an EC_StateType which, along with EC_Errno, 
 * reflects its current state.  
 *
 *	EC_STATE_NEED_SEND_MSG
 * Indicates that the application needs to send a message in order 
 * for the action handler to continue.  The application can find out 
 * what message needs to be sent and what address to send it to be 
 * calling EC_acth_get_message().
 *	
 *	EC_STATE_NEED_RECEIVE_MSG
 * Indicates that the action handler needs a message in order to 
 * continue.
 * 
 *	EC_STATE_DONE
 * Indicates that the action for this action handler is done.
 * The action handler's EC_Errno is set to indicate whether this
 * state was reached by successful completion of the action or
 * by some exceptional condition.
 */


typedef enum
{
	EC_STATE_DONE = 0,
	EC_STATE_NEED_SEND_MSG = 1,
	EC_STATE_NEED_RECEIVE_MSG = 2,
	EC_STATE_INVALID = 3
} EC_StateType;



/* end of eacth.fh */









	


	


	/* Action handler functions */

/* 
 * Note the absense of a public "EC_acth_new()" function.  Instead
 * use "EC_pocket_begin_foo()" to begin an action that will be handled
 * by the resulting EC_ActionHandler.
 */


/*
 * Free the action handler.
 * If this handler's state is not EC_STATE_DONE, then the 
 * EC_ActionHandler will _not_ be freed and EC_acth_free will return 
 * an EC_ERR_ACTH_MANAGEMENT.
 */
  EC_Errno EC_acth_free(EC_ActionHandler *this_acth_ptr);


/*
 * Abort the current action.
 * If this handler's state is EC_STATE_DONE, then this will return 
 * EC_ERR_ACTH_MANAGEMENT.  There are two possible results from an
 * abort.  One: the state of the client was re-wound so that it is
 * as if the aborted action never happened.  Two: the action was too 
 * far along for that, and it is as if the action was completed.
 * (In the second case, aborting the action caused the action to cease
 * prematurely, after the necessary information had been transmitted, 
 * but before some auxiliary information was transmitted.)
 * After the EC_acth_abort_action() is called, the acth's EC_ERRNO
 * will reflect one of the two results.
 */
  EC_Errno EC_acth_abort_action(EC_ActionHandler *this_acth_lnk);


/*
 * Get a communication which was passed to the ecashlib by another 
 * ecash application via an EC_Msg.  If the last EC_Msg that was
 * processed by this action handler contained a communication then
 * this call will return that communication.
 */
  EC_Errno EC_acth_get_communication(
	const EC_ActionHandler *this_acth_lnk, UInt32 *comm_size_out, 
	Byte **comm_ptr_out);



	/* interactions between the action handler and the application */

/*
 * Give a message which has been received to the action handler.
 * The "sender_addr" parameter is optional-- a NULL pointer may
 * be passed instead of the sender's address.
 * Note:  The sender-address is only stored in records-- it is not
 * acted upon by ecashlib.
 * Note: this is only for messages that the action handler is
 * requesting.  If the ecashlib is not in state 
 * EC_STATE_NEED_RECEIVE_MSG, then calling EC_acth_process_msg()
 * generates an EC_ERR_ACTH_MANAGEMENT.
 */
  EC_Errno EC_acth_process_msg(EC_ActionHandler *this_acth_lnk,
	const EC_Msg *msg_lnk, const EC_Address *sender_addr_lnk);

/*
 * Get the action handler's error state.
 * The return value is the EC_Errno which is contained by the 
 * action handler as a part of its state.
 */
  EC_Errno EC_acth_get_errno(const EC_ActionHandler *this_acth_lnk);

  EC_StateType EC_acth_get_state(const EC_ActionHandler *this_acth_lnk);


/* Get a copy of the message to be sent. */
  EC_Msg *EC_acth_get_msg(EC_ActionHandler *this_acth_lnk);

/*
 * Get the address of an Ecash application to which a message
 * should be sent, or from which a message should be received.
 */
  EC_Address *EC_acth_get_address(EC_ActionHandler *this_acth_lnk);

/*
 * Get info about the account that this action handler is uses.
 * info_ptr_out: This EC_Info is frozen when the action is completed or
 * aborted, yielding a snap-shot of the state of the account at that 
 * moment.  Calling EC_acth_get_info() before that time is equivalent
 * to calling EC_pocket_get_info().
 */
  EC_Info *EC_acth_get_info(const EC_ActionHandler *this_acth_lnk);


/* end of eacth.h */




/* beginning of emsg.h */


	/* ecashlib public headers */


/* beginning of ecoinage.fh */


	/* ecashlib public includes */



	/* EC_CurrencyID */

/*
 * An EC_CurrencyID uniquely identifies a currency.  The list of
 * currencies is maintained by DigiCash.  One currency ID
 * which is defined in the ecashlib header files is
 * EC_CURRID_INVALID.
 */


typedef UInt32 EC_CurrencyID;



	/* EC_Coinage */

/*
 * An EC_Coinage is a data type which contains some information
 * about the ecash.
 *
 * An EC_Coinage has four qualities:
 * 1.  An issuer.
 * 2.  A currency.  Ecash is exchangeable at the issuer for this
 *	currency.
 * 3.  A base value.  This is the smallest value that can be
 *	exchanged in ecash of this kind.
 * 4.  A granularity.  This is the degree of precision that is
 *	commonly used when representing amounts of this kind of ecash
 *	to a user.
 *
 *
 * The way this is implemented is as follows:
 * 1. Each EC_Coinage has an EC_MintID to identify the issuer.
 * 2. Each EC_Coinage has an EC_CurrencyID to identify the currency.
 * 3. Each EC_Coinage has a pair of 32-bit numbers to identify the
 *	base value.  The first of these numbers is signed and identifies
 *	the order of magnitude (power of ten) of the base value.  (For
 *	example -2 to indicate that the base value is on the order of
 *	0.01.)
 *	The second of these numbers is unsigned and indicates the
 *	factor which determines the base value within that order of
 *	magnitude.  (In the example, a factor of 1 would mean that the
 *	base value is 0.01, and a factor of 5 would mean that the base
 *	value is 0.05.)
 * 4. Each EC_Coinage has a signed 32-bit number to identify the
 *	granularity.  (For example, -2 to indicate that a precision of
 *	0.01 is used when displaying amounts in this EC_Coinage.)
 */


typedef struct EC_Coinage_s EC_Coinage;



/* end of ecoinage.fh */


















	/* EC_Msg Functions */


/*
 * Create a new EC_Msg.
 * parameters:
 *	msg_size: the size of the msg_contents data.
 *	msg_contents: a pointer to a block of a data of size msg_size
 *		containing the message contents.  EC_msg_new will make a
 *		copy of msg_contents, so you may free msg_contents after
 *		calling EC_msg_new().
 */
  EC_Msg *EC_msg_new(UInt32 msg_size, const Byte *msg_contents_ptr);

/* Free an EC_Msg. */
  EC_Errno EC_msg_free(EC_Msg *this_msg_ptr);

/*
 * Make a copy of a message.
 *
 * return values:
 *	pointer to the new EC_Msg on success, NULL pointer on failure.
 */
  EC_Msg *EC_msg_dup(const EC_Msg *this_msg_lnk);

/*
 * Get the information about an EC_Msg that you need to know
 * in order to transmit the EC_Msg to another ecash
 * application.
 */
  UInt32 EC_msg_get_size(const EC_Msg *this_msg_lnk);

  Byte *EC_msg_get_contents(const EC_Msg *this_msg_lnk);

/* Get information about the type of message. */
  EC_MsgType EC_msg_get_type(const EC_Msg *this_msg_lnk);



	/* EC_Msg Functions for asynchronous messages */

/*
 * "Asynchronous messages" are those messages which arrive as the 
 * start of a new conversation rather than as the continuation of
 * an existing conversation.  Currently all and only asynchronous 
 * messages are payments and payment requests.
 */


/*
 * Get the payment info from a payment EC_Msg.
 *
 * parameters:
 *	coinage_ptr_out: returns the coinage of the payment
 *	amount_out: returns the amount of the payment
 *	desc_str_out: returns the description string of the payment
 *	accID_out: returns the account ID that the payment is made to
 *
 * return values:
 *	standard EC_Errno
 *
 * notes:
 *	EC_msg_get_pay_info returns an error if called on an EC_Msg which 
 *	is not a payment.
 */
  EC_Errno EC_msg_get_pay_info(const EC_Msg *this_msg_lnk, 
	EC_Coinage **coinage_ptr_out, EC_Amount *amount_out,
	char **desc_str_out, EC_AccountID *accID_out);

/*
 * Get the payment-request info from a payment-request EC_Msg.
 * Returns an error if called on an EC_Msg which is not a payment 
 * request.
 *
 * parameters:
 *	pocket_lnk: This is a pointer to a pocket which you are considering 
 *		using to pay.  Thus if the payment request includes requests 
 *		for multiple coinages EC_msg_get_payreq_info will return info 
 *		about the request which this pocket is capable of satisfying.
 *	accID_ptr_out: This is the accountID that the payment should be 
 *		made to.
 *	addr_ptr_out: This is the address the the payment should be made to.
 *		If there is no address (the EC_Address * is a NULL pointer),
 *		you are supposed to use the address from which the payment
 *		request originated.
 *	amount_out: the amount of the payment
 *	description_str_out: the description of the payment
 *		*!note it is not clear right now whether your payment must
 *		have a description string identical to this one, or whether
 *		you may put whatever description string you want in your
 *		payment.
 *
 * return values:
 *	standard EC_Errno
 */
  EC_Errno EC_msg_get_payreq_info(const EC_Msg *this_msg_lnk, 
	EC_Pocket *pocket_lnk, EC_AccountID **accID_ptr_out,
	EC_Address **addr_ptr_out, EC_Amount *amount_out,
	char **desc_str_out);



/* end of emsg.h */




/* beginning of epocket.h */


	/* public ecashlib includes */




/* beginning of eprotocl.fh */



	/* EC_Protocol */

/*
 * The protocol.  Currently, there is only one: the ecash 2.x
 * online cash protocol.
 */

typedef enum
{
	EC_PROTOCOL_INVALID = 0,
	EC_PROTOCOL_ECASH_ONLINE_COINS = 2
} EC_Protocol;



/* end of eprotocl.fh */


















	/* EC_Pocket functions */


/*
 * Create a new EC_Pocket.
 *
 * parameters:
 *	pocketID_lnk: the EC_PocketID for this pocket
 *	*!b the pocket ID must be known _before_ EC_pocket_begin_open_acc()
 *	*!b the pocket ID consists of an EC_AccountID and a "base_dir" string
 *	mint_addr_lnk: address of the mint for this pocket
 *	acct_info_str: info about bank account for this pocket
 *	rnd_data_str: random data used for recovery
 *	pocket_password_str: a password to encrypt the data in the files.  If
 *		the files already exist then this must the password that was
 *		used to encrypt them.  Else, EC_pocket_new() will fail and
 *		return a NULL pointer.  If the files don't already exist
 *		then this will be the password used to encrypt them.
 *		*! Note that currently only the sec key is encrypted...
 *	acct_password_str: the password used to protect the account at the
 *		mint.  It is used to seed the generation of the RSA key pair.
 *		This argument is only necessary when creating the RSA key
 *		pair.  That is: the first time that EC_pocket_new() is
 *		called on a given EC_PocketID.  On successive calls of
 *		EC_pocket_new(), the RSA key pair will be read from
 *		the pocket's persistent storage.
 *
 * notes:
 *	*!note:  For now only a single pocket can exist per base_dir,
 *	and only a single EC_Pocket can be instantiated at a time.
 *	If the EC_AccountID does not correspond to the EC_AccountID
 *	stored in the files in base_dir, or if you try to new a second
 *	EC_Pocket when one already exists, you will get an error.
 *
 *	*!note:  For now, only a single EC_Pocket can exist per bank
 *	account.  If you try to use the same bank account from different
 *	hard-drives or different base_dirs, you may experience problems
 *
 *	*!design:  If we evolve some better (i.e. more abstract) way to
 *	interface ecashlib with permanent storage, EC_PocketID will lose
 *	the base_dir member at that same time.
 *
 *	A failure return value can result if ecashlib is unable to open
 *	or create the directory specified in the pocketID_lnk.
 *
 * return values:
 *	pointer to the new EC_Pocket on success, NULL pointer on failure
 */
  EC_Pocket *EC_pocket_new(const EC_PocketID *pocketID_lnk, 
	const EC_Address *mint_addr_lnk, const Char *acct_info_str, 
	const Char *pocket_password_str, const Char *rnd_data_str,
	const Char *acct_password_str);


/*
 * Free an EC_Pocket.
 *
 * return values:
 *	standard EC_Errno
 */
  EC_Errno EC_pocket_free(EC_Pocket *this_pocket_ptr);


/*
 * Set a different password for this pocket.
 *
 * parameters:
 *	new_password_str: new password
 *	old_password_str: old password
 *	
 * return values:
 *	standard EC_Errno
 *	EC_ERR_BAD_VALUE: old_password_str wasn't the right old password.
 */
  EC_Errno EC_pocket_password(EC_Pocket *this_pocket_lnk,
	const Char *new_password_str, const Char *old_password_str);


/*
 * Get information about a pocket.
 *
 * return values:
 *	pointer to the new EC_Info on success, NULL pointer on failure
 */
  EC_Info *EC_pocket_get_info(const EC_Pocket *this_pocket_lnk);


/*
 * Get a record list for a pocket.
 *
 * return values:
 *	pointer to the new EC_RecordList on success, NULL pointer on failure
 */
  EC_RecordList *EC_pocket_get_rlist(const EC_Pocket *this_pocket_lnk);



	/* Pocket "Complex Action" functions */

/*
 * These are Account functions which initiate an complex action--an
 * action that cannot be completed without some networking services
 * being performed by the application on behalf of the ecashlib.
 * When an "EC_pocket_begin_foo()" function is called an EC_ActionHandler 
 * will be created to handle the complex action.
 */


/*
 * Open the account for this pocket.
 *
 * parameters:
 *	acct_password_str: password needed to open the account at the mint
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 *
 * notes:
 *	If this pocket already has an open account, 
 *	EC_pocket_begin_open_account() will return a NULL pointer.
 *!! Shouldn't it return an EC_ActionHandler which will enter an error
 * state?
 *	Also note that the issuer may refuse to open an account for this
 *	EC_AccountID for some reason.
 */
  EC_ActionHandler *EC_pocket_begin_open_account(
	EC_Pocket *this_pocket_lnk, const Char *acct_password_str);


/*
 * Withdraw cash.
 *
 * parameters:
 *	amount: amount to withdraw
 *	min_payments: after withdrawal at least this many payments can
 *		can be made before 
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
  EC_ActionHandler *EC_pocket_begin_withdrawal(
	EC_Pocket *this_pocket_lnk, EC_Amount amount, UInt32 min_payments);


/*
 * Deposit cash.
 *
 * parameters:
 *	amount: amount to deposit
 *	min_payments: after deposit at least this many payments can
 *		can be made before needing a withdrawal
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
  EC_ActionHandler *EC_pocket_begin_deposit(EC_Pocket *this_pocket_lnk,
	EC_Amount amount, UInt32 min_payments);


/*
 * Make payment.
 *
 * parameters:
 *	protocol_lnk: protocol to use for payment
 *	amount: amount to pay
 *	description_str: payment description or NULL
 *	recip_accID_lnk, recip_addr_lnk: account ID and address of recipient
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
  EC_ActionHandler *EC_pocket_begin_make_payment(
	EC_Pocket *this_pocket_lnk,
	EC_Protocol protocol_lnk, EC_Amount amount,
	const Char *description_str, const EC_AccountID *recip_accID_lnk,
	const EC_Address *recip_addr_lnk);


/*
 * Accept incoming payment.
 *
 * parameters:
 *	payment: the EC_Msg which contains the payment
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
  EC_ActionHandler *EC_pocket_begin_accept_payment(
	EC_Pocket *this_pocket_lnk,
	const EC_Msg *payment_lnk);


/*
 * Request payment.
 *
 * parameters:
 *	protocol_lnk: protocol to use for payment
 *	amount: amount to request
 *	description_str: payment description or NULL
 *	requestee_accID_lnk, requestee_addr_lnk: ID and address of
 *		requestee
 *	return_addr_lnk: the address which the payment should be sent
 *		to if different from the account's normal address.  This
 *		is an optional parameter which will be ignored if a NULL
 *		pointer or the null EC_Address is passed.
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
  EC_ActionHandler *EC_pocket_begin_request_payment(
	EC_Pocket *this_pocket_lnk,
	EC_Protocol protocol_lnk, EC_Amount amount,
	const Char *description_str, const EC_AccountID *requestee_accID_lnk,
	const EC_Address *requestee_addr_lnk, 
	const EC_Address *return_addr_lnk);


/*
 * Cancel a payment from a record.
 *
 * parameters:
 *	payment_lnk: a record of the payment
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
  EC_ActionHandler *EC_pocket_begin_cancel_payment(
	const EC_Pocket *this_pocket_lnk,
	const EC_PaymentRecord *payment_lnk);


/*
 * Resend a payment from a record.
 *
 * parameters:
 *	payment_lnk: a record of the payment
 *
 * return values:
 *	pointer to the new EC_ActionHandler on success, NULL pointer on
 *		failure
 */
  EC_ActionHandler *EC_pocket_begin_resend_payment(
	const EC_Pocket *this_pocket_lnk,
	const EC_PaymentRecord *payment_lnk);



/* end of epocket.h */




/* beginning of einfo.h */


	/* ecashlib public headers */







/* beginning of eaccstat.fh */



	/* EC_AccountStatus */

/*
 * An EC_Info has an EC_AccountStatus, indicating the 
 * status of the EC_Account.
 *
 * It should be understood that this information may be 
 * out-of-date by the time the ecash application sees it-- 
 * if the issuer disabled the account, the application 
 * won't know about it until the next time you try to use 
 * it!
 */

typedef enum
{
	EC_ACCOUNTSTATUS_OK = 0, /* the account is OK */
	EC_ACCOUNTSTATUS_UNKNOWN = 1, /* the status of the
		account is even more unknown than usual. */
		/*! What is this value used for ? */
	EC_ACCOUNTSTATUS_UNOPENED = 2, /* the account 
		has not been opened at the issuer */
	EC_ACCOUNTSTATUS_DISABLED = 3, /* the account has been
		disabled by the issuer */
	EC_ACCOUNTSTATUS_INVALID = 4 /* this is returned in case of error */
} EC_AccountStatus;



/* end of eaccstat.fh */













	/* EC_Info functions */


/* 
 * Note the lack of a public "EC_info_new()" function.  Instead use
 * EC_pocket_get_info() or EC_acth_get_info().
 */


/*
 * Free account information structure. 
 *
 * return values:
 *	standard EC_Errno
 */
  EC_Errno EC_info_free(EC_Info *this_info_ptr);

/*
 * Get the account ID.
 *
 * return values:
 *	pointer to the EC_AccountID on success, NULL on failure
 */
  EC_AccountID *EC_info_get_accID(const EC_Info *this_info_lnk);

/*
 * Get the account info.
 *
 * return values:
 *	NULL-terminated string containing the account info on success,
 *		NULL Char pointer on failure
 */
  Char *EC_info_get_account_info(const EC_Info *this_info_lnk);

/*
 * Get the account status.
 *
 * return values:
 *	EC_AccountStatus on success, undefined on failure
 */
  EC_AccountStatus EC_info_get_acct_status(const EC_Info *this_info_lnk);

/*
 * Get the mint ID.
 *
 * return values:
 *	EC_MintID on success, undefined on failure
 */
  EC_MintID EC_info_get_mintID(const EC_Info *this_info_lnk);

/*
 * Get the mint info.
 *
 * return values:
 *	NULL-terminated string containing the mint info on success, NULL Char
 *		pointer on failure
 */
  Char *EC_info_get_mint_info(const EC_Info *this_info_lnk);

/*
 * Get the mint address.
 *
 * return values:
 *	pointer to the EC_Address on success, NULL pointer on failure
 */
  EC_Address *EC_info_get_mint_addr(const EC_Info *this_info_lnk);

/*
 * Get the coinage for this account.
 *
 * return values:
 *	pointer to the EC_Coinage on success, NULL pointer on failure
 */
  EC_Coinage *EC_info_get_coinage(const EC_Info *this_info_lnk);

/*
 * Get the amount in cash for this account.
 *
 * return values:
 *	EC_Amount on success, undefined on failure
 */
  EC_Amount EC_info_get_cash(const EC_Info *this_info_lnk);

/*
 * Get the amount stored at the mint for this account.
 *
 * return values:
 *	EC_Amount on success, undefined on failure
 */
  EC_Amount EC_info_get_balance(const EC_Info *this_info_lnk);

/*
 * Get the amount in expired cash for this account.
 *
 * return values:
 *	EC_Amount on success, undefined on failure
 */
  EC_Amount EC_info_get_exp_cash(const EC_Info *this_info_lnk);

/*
 * Get the number of payments guaranteed to be payable using the
 * current coins.
 *
 * return values:
 *	minimum guaranteed payments on success, 
 *		EC_MINPAYMENTS_INVALID on failure
 */
  UInt16 EC_info_get_min_payments(const EC_Info *this_info_lnk);

/*
 * Get the recovery string used to recover this pocket's lost
 * coins in the case of a disaster.
 *
 * return values:
 *	the recovery string on success, NULL pointer on failure
 *
 * notes:
 *	The recovery_str member of an EC_Info is only valid after a 
 *	successful call to this EC_Pocket's EC_pocket_new() and before this 
 *	EC_Pocket leaves memory-- the recovery_str is not stored in this 
 *	EC_Pocket's persistent storage!  Whenever the recovery_str member 
 *	is invalid it is set to a NULL pointer.
 */
  Char *EC_info_get_recovery_string(const EC_Info *this_info_lnk);


/* end of einfo.h */




/* beginning of ecoinage.h */


	/* ecashlib public headers */









	/* EC_Coinage functions */


/* 
 * Note the absence of a public "EC_coinage_new()" function.  Instead
 * use EC_info_get_coinage().
 */


/*
 * Free an EC_Coinage.
 * return values:
 *	standard EC_Errno
 */
  EC_Errno EC_coinage_free(EC_Coinage *this_coinage_ptr);

/*
 * Get the mint ID of a coinage.
 * return values:
 *	The mint ID on success, EC_MINTID_INVALID on failure.
 */
  EC_MintID EC_coinage_get_mintID(const EC_Coinage *this_coinage_lnk);

/*
 * Get the currency ID of a coinage.
 * return values:
 *	The currency ID on success, EC_CURRID_INVALID on failure.
 */
  EC_CurrencyID EC_coinage_get_currID(const EC_Coinage *this_coinage_lnk);

/*
 * Get the magnitude of the base value of a coinage.
 * return values:
 *	The magnitude of the base value on success, EC_MININT32 on failure.
 */
  Int32 EC_coinage_get_base_magn(const EC_Coinage *this_coinage_lnk);

/*
 * Get the factor of the base value of a coinage.
 * return values:
 *	The magnitude of the base value on success, EC_MAXUINT32 on failure.
 */
  UInt32 EC_coinage_get_base_factor(const EC_Coinage *this_coinage_lnk);

/*
 * Get the granularity of a coinage.
 * return values:
 *	The granularity of the coinage on success, EC_MININT32 on failure.
 */
  Int32 EC_coinage_get_granularity(const EC_Coinage *this_coinage_lnk);



/* end of ecoinage.h */




/* beginning of eaddr.h */

	/* ecashlib public headers */









	/* EC_Address functions */

/*
 * Create a new EC_Address.
 *
 * parameters:
 *	addr_size: the size of the addr_contents data.
 *	addr_contents: a pointer to a block of a data of size 
 *		addr_size containing the address contents.  EC_addr_new 
 *		will make a copy of addr_contents, so you may free 
 *		addr_contents after calling EC_addr_new().
 */
  EC_Address *EC_addr_new(UInt32 addr_size, const Byte *addr_contents);


/*
 * Free an EC_Address.
 *
 * return values:
 *	standard EC_Errno
 */
  EC_Errno EC_addr_free(EC_Address *this_addr_ptr);


/*
 * Duplicate an EC_Address.
 *
 * return values:
 *	pointer to the new EC_Address on success, NULL on failure.
 */
  EC_Address *EC_addr_dup(const EC_Address *this_addr_lnk);


/*
 * Get the contents of an EC_Address.
 */
  UInt32 EC_addr_get_size(const EC_Address *this_addr_lnk);

  Byte *EC_addr_get_contents(const EC_Address *this_addr_lnk);


/* end of eaddr.h */




/* beginning of eamount.h */






/* end of eamount.h */




/* beginning of emintid.h */






/* end of emintid.h */




/* beginning of eprotocl.h */










/* beginning of eprotocl.h */




/* beginning of eloglevl.h */



/* beginning of eloglevl.fh */


/* The logging levels used by the log callback function. */
typedef enum 
{
	EC_LOGLEVEL_DEBUG = 0, /* debugging info */
	EC_LOGLEVEL_INFO = 1, /* general information */
	EC_LOGLEVEL_NOTICE = 2, /* notices */
	EC_LOGLEVEL_WARNING = 3, /* warnings */
	EC_LOGLEVEL_ERROR = 4 /* errors */
} EC_LogLevel;



/* end of eloglevl.fh */






/* end of eloglevl.h */






	/* General Ecashlib functions */

/*
 * Initialize the package by specifying ANSI C-style memory
 * allocation, reallocation and deallocation, as well as a 
 * log function and a yield function.  The log function is 
 * called with two parameters, a string to log and a logging
 * level.
 *
 * The malloc and realloc functions must conform to 
 * ANSI C3.159-1989.
 *
 * The realloc function must handle a NULL pointer 
 * "data_ptr" parameter by allocating a new block of the 
 * appropriate size and returning a pointer to it.
 *
 * If a NULL pointer is passed instead of the "malloc", 
 * "realloc" or "free" function pointer then ecashlib will 
 * try to use a standard function in place of each such 
 * function.  This will not work on all platforms.
 *! Why not?  malloc, realloc and free are ANSI C.
 *
 * The "yield" function will get called frequently when 
 * ecashlib is doing computation-intensive functions.  If
 * a NULL pointer is passed instead of a yield function 
 * pointer, it will not be called.
 *
 * return values:
 *	standard EC_Errno
 */
  EC_Errno EC_main_init(void *malloc_cb_initt(size_t size), 
	void *realloc_cb_initt(void *data_ptr, size_t new_size), 
	void free_cb_initt(void *data_ptr), 
	void yield_cb_initt(int this_is_the_last_yield),
	void log_cb_initt(EC_LogLevel level, const Char *text_str));

/*
 * What version of the ecashlib is this?
 * The return value is of the format "xx.yyy.zzz {keywords ...}"
 * where xx is the major version number, yyy is the minor version
 * number (which involves a change to the API) and zzz is the
 * patchlevel.  Finally optional keywords may follow.
 */
  Char *EC_main_get_libver(void);

/*
 * Clean_up will cause ecashlib to clean up its files and internal 
 * memory allocation.  It also unlocks any files or callbacks that
 * it may have locked.  Call this as the last thing, after you have 
 * freed all EC_ data-structures that you were responsible for.
 */
  EC_Errno EC_main_cleanup(void);



	/* EC_Coinage string conversion functions. */

/* These are just for convenience.  You may safely ignore them. */

/*
 * Convert a string to a currency ID.
 *
 * parameters:
 *	name_str: a string identifying the currency.  This can be any of 
 *		several strings such as the 3-letter abbreviation (e.g. "USD"), 
 *		the full name (e.g. "United States Dollar") and perhaps some 
 *		alternate names (e.g. "U.S. Dollar" or "US$").  The list of 
 *		which alternate names are understood by this function is 
 *		maintained by DigiCash.
 *
 * return values:
 *	the currency ID on success, EC_CURRID_INVALID on failure
 */
  EC_CurrencyID EC_main_str_to_curr(const Char *name_str);

/*
 * Convert a currency ID to the currency's abbreviation.
 *
 * parameters:
 *	currID: the currency
 *	abbr: the 3-letter abbreviation for that currency.  The list
 *		of currencies and their 3-letter abbreviations is maintained
 *		by DigiCash.
 */
  EC_Errno EC_main_curr_to_abbr(EC_CurrencyID currID, Char abbr_buf[4]);

/*
 * Convert a currency ID to the currency's name.
 *
 * parameters:
 *	currID: the currency
 *
 * return values:
 *	the full name for that currency on success (the list of currencies 
 *		and their names is maintained by DigiCash.), NULL pointer on
 *		failure
 */
  Char *EC_main_curr_to_name(EC_CurrencyID currID);

/*
 * Convert an EC_Amount and EC_Coinage to its string representation.
 *
 * parameters:
 *	amount: the amount
 *	coinage: the coinage of ecash that the amount is in
 *
 * return values:
 *	the string representation of that amount and coinage on success,
 *		NULL pointer on failure
 */
  Char *EC_main_value_to_str(EC_Amount amount, 
	const EC_Coinage *coinage_lnk);



/* end of ecashlib.h */


