/*-
 * Copyright (c) 1993, Trusted Information Systems, Incorporated
 * All rights reserved.
 *
 * Redistribution and use are governed by the terms detailed in the
 * license document ("LICENSE") included with the toolkit.
 */

/*
 *	Author: Marcus J. Ranum, Trusted Information Systems, Inc.
 *	Includes SSL tunneling support by Tim Adam (tma@osa.com.au)
 *	To use as an SSL plug, include a line like
 *	plug-gw: port 4430 10.* -plug-to https -ssl
 *	Meaning listen on port 4430 for connections from 10.*
 *	which are interpreted as ssl security handoff proxy requests.
 */
static	char	RcsId[] = "$Header: /usr/home/rick/fwtk2.0/fwtk/plug-gw/RCS/plug-gw.c,v 1.6 1996/09/05 19:36:29 rick Exp $";

/* #define	BINDDEBUG */
/* #define	BINDDEBUGPORT 6565 */

#include	<stdio.h>
#include	<string.h>
#include	<syslog.h>
#include	<sys/types.h>
#include	<sys/time.h>
#include	<sys/socket.h>
#include	<netdb.h>
#include	<netinet/in.h>

#include	"firewall.h"

static	char		rhost[512];
static	char		raddr[512];
static	time_t		ontime;
static	time_t		offtime;
static	int		portid = -1;
static	int		timeout = PROXY_TIMEOUT;
static	char		**validdests = (char **)0;
static	Cfg		*confp;

main(ac,av)
int	ac;
char	*av[];
{
	char		**xav = av;
	Cfg		*cfp;
	Cfg		*cf;
	char		*srvnam;

#ifndef	LOG_DAEMON
	openlog("plug-gw",LOG_PID);
#else
	openlog("plug-gw",LOG_PID|LOG_NDELAY,LFAC);
#endif

	if (ac > 2 && !strcmp(av[1], "-daemon")) {
		int port = str_to_port(av[2]);
		portid = port;

		ac -= 2;
		av += 2;

		syslog(LLEV,"Starting daemon mode on port %d",port);
		do_daemon(port);
	}

#ifdef	BINDDEBUG
	/*
	useful for debugging. if you define this it will bind the
	port and listen directly (circumventing inetd) so you can
	run the program under a debugger
	*/
	debugbind();
#endif

	time(&ontime);

	/* if called with an argument, it is the connecting service name */
	if(ac > 1)
		srvnam = av[1];
	else {
		char *p = (char *)strrchr(xav[0], '/');
		srvnam = xav[0];
		if (p)
			srvnam = &p[1];
	}

	if (portid == -1 && srvnam)
		portid = str_to_port(srvnam);

	if(peername(0,rhost,raddr,sizeof(rhost))) {
		syslog(LLEV,"cannot get remote host: %m");
		exit(1);
	}


	if((cfp = cfg_read("plug-gw")) == (Cfg *)-1)
		exit(1);

	confp = cfp;

	if((cf = cfg_get("timeout",cfp)) != (Cfg *)0) {
		if(cf->argc != 1) {
			syslog(LLEV,"fwtkcfgerr: timeout must have one parameter, line %d",cf->ln);
			exit(1);
		}
		timeout = atoi(cf->argv[0]);
	}


	for(cf = cfg_get("port",cfp); cf != (Cfg *)0; cf = cfg_get("port",(Cfg *)0)) {
		int	x;

		if(cf->argc < 2) {
			syslog(LLEV,"fwtkcfgerr: missing parameter, line %d",cf->ln);
			continue;
		}

		/* see if ports match */
		if(strcmp(cf->argv[0],"any")) {
			if(portid == -1)
				continue;

			if(portid != str_to_port(cf->argv[0]))
				continue;
		}


		/* if ports match, now see if calling host matches */
		for(x = 1; x < cf->argc - 1; x++) {
			if (cf->argv[x][0] == '-')
				break;	/* end of host list */
			if(hostmatch(cf->argv[x],raddr))
				acceptrule(portid,cf,cf->argc,cf->argv);
		}
	}

	if(portid == -1)
		syslog(LLEV,"deny host=%s/%s",rhost,raddr);
	else
		syslog(LLEV,"deny host=%s/%s service=%s",rhost,raddr,av[1]);
	exit(1);
}



static void
accept_setdest(v, c)
char	*v;
int	c;
{
	int	dests = 0;

	if (validdests == (char **)0)
		validdests = (char **)malloc(sizeof(char *) * 2);
	else {
		for(dests = 0; validdests[dests] != (char *)0; dests++)
			;
		validdests = (char **)realloc(validdests,(dests + 2) * sizeof(char *));
	}
	if (validdests == (char **)0)
		return;
	validdests[dests] = v;
	validdests[dests + 1] = (char *)0;
}

int baddest(fd, dest)
int	fd;
char	*dest;
{
	syslog(LLEV,"deny host=%s/%s destination=%.256s",rhost,raddr,dest);
	return 0;
}

acceptrule(p,c,ac,av)
int	p;
Cfg	*c;
int	ac;
char	*av[];
{
	fd_set		redy;
	fd_set		iced;
	int		ib = 0;
	int		ob = 0;
	int		x;
	int		serfd;
	int		privport = 0;
static	char		buf[1024 * 4];
	void		(*op)();
	char		*dhost = NULL;
	char		hostport[1024 * 4];
	char		*ptr;
	int		state = 0;
	int		ssl_plug = 0;

	struct timeval	timo;

	if(c->flags & PERM_DENY) {
		if (p == -1)
			syslog(LLEV,"deny host=%s/%s port=any",rhost,raddr);
		else
			syslog(LLEV,"deny host=%s/%s port=%d",rhost,raddr,p);
		exit(1);
	}

	for (x = 2; x < ac; x++) {
		if(av[x][0] == '-')
			break;
	}

	for (; x < ac; x++) {
		if(!strcmp(av[x],"-plug-to")) {
			if (++x >= ac) {
				syslog(LLEV,"fwtkcfgerr: -plug-to takes an argument, line %d",c->ln);
				exit (1);
			}
			dhost = av[x];
			continue;
		}

		if (!strcmp(av[x],"-privport")) {
			privport = 1;
			continue;
		}

		if (!strcmp(av[x], "-port")) {
			if (++x >= ac) {
				syslog(LLEV,"fwtkcfgerr: -port takes an argument, line %d",c->ln);
				exit(1);
			}
			portid = str_to_port(av[x]);
			continue;
		}

		if (!strcmp(av[x], "-ssl")) {
			ssl_plug = 1;
			continue;
		}

		/* options that take parameters and lists */
		op = NULL;
		if (!strcmp(av[x],"-dest"))
			op = accept_setdest;
		if (op == NULL) {
			syslog(LLEV,"fwtkcfgerr: bad option line %d: %s",c->ln, av[x]);
			exit(1);
		}
		if (++x >= ac) {
			syslog(LLEV,"fwtkcfgerr: malformed line %d: missing option",c->ln);
			exit(1);
		}
		if (!strcmp(av[x],"{")) {
			while (1) {
				if (++x >= ac) {
					syslog(LLEV,"fwtkcfgerr: malformed line %d: missing closing brace",c->ln);
					exit(1);
				}
				if (!strcmp(av[x], "}"))
					break;
				(*op)(av[x], c->ln);
			}
		} else
			(*op)(av[x],c->ln);
	}
	if (ssl_plug) {
		/*
		 * Read CONNECT line and skip any headers
		 * up to a blank line (consisting of \r\n or just \n)
		 *
		 * state is:
		 * 0 when reading first line,
		 * 1 when reading later lines
		 * 2 when at the end of a line,
		 * 3 after a blank line or buffer overflow
		 */
		 ptr = buf;
		 while (state < 3 && read(0, ptr, 1) > 0) {
			/* Just ingore these so that \r\n is a valid EOL */
			if (*ptr == '\r')
				continue;
			switch (state) {
			    case 0:
				if (*ptr == '\n')
					/* found end of first line */
					state = 2;
				/* don't overflow buffer */
				if (++ptr >= buf + sizeof(buf))
					state = 3;
				break;
			    case 1: /* in later line */
				if (*ptr == '\n')
					/* now at an end of line */
					state = 2;
				break;
			    case 2: /* alread at an end of line */
				if (*ptr == '\n')
					/* reached end of headers */
					state = 3;
				else
					/* start a new line */
					state = 1;
				break;
			}
		}
		if (sscanf(buf, "CONNECT %s HTTP/1.0", hostport) != 1) {
			strcpy(buf, "HTTP/1.0 400 Bad request\r\n\r\n");
			write(0, buf, strlen(buf));
			exit(1);
		}
		if ((ptr = strtok(hostport, ":")) != NULL) {
			dhost = ptr;
			ptr += strlen(ptr) + 1;
			portid = atoi(ptr);
		} else {
			portid = -1;
		}
	}

	if (dhost == NULL) {
		syslog(LLEV,"fwtkcfgerr: not sure what host to connect to, line %d",c->ln);
		exit(1);
	}
	if (validdests != (char **) 0) {
		char 	**xp;

		for (xp = validdests; *xp != (char *)0; xp++) {
			if (**xp == '!' && hostmatch(*xp + 1, dhost)) {
				return(baddest(0, dhost));
			} else {
				if (hostmatch(*xp, dhost))
					break;
			}
		}
		if (*xp == (char *)0)
			return(baddest(0, dhost));
	}

	if (portid == -1) {
		syslog(LLEV,"fwtkcfgerr: not sure what port to connect to, line %d",c->ln);
		exit(1);
	}

	if((serfd = conn_server(dhost,portid,privport,(char *)0)) < 0) {
		syslog(LLEV,"cannot connect to server %s/%d: %m",dhost,portid);
		exit(1);
	}

	syslog(LLEV,"connect host=%s/%s destination=%s/%d",rhost,raddr,dhost,portid);

	if (state > 0) {
		strcpy(buf, "HTTP/1.0 200 Connection established\r\n");
		strcat(buf, "Proxy-agent: plug-gw/ssl\r\n\r\n");
		write(0, buf, strlen(buf));
	}

	FD_ZERO(&iced);
	FD_SET(0,&iced);
	FD_SET(serfd,&iced);

	while(1) {
		(void)bcopy(&iced,&redy,sizeof(fd_set));

		timo.tv_sec = timeout;
		timo.tv_usec = 0;
		if(select(serfd + 2,&redy,(fd_set *)0,(fd_set *)0,&timo) <= 0) {
			syslog(LLEV,"Network timeout signal after %d seconds",timeout);
			break;
		}

		if(FD_ISSET(0,&redy)) {
			if((x = read(0,buf,sizeof(buf))) <= 0)
				break;
			if(write(serfd,buf,x) != x)
				break;
			ob += x;
		}
		if(FD_ISSET(serfd,&redy)) {
			if((x = read(serfd,buf,sizeof(buf))) <= 0)
				break;
			if(write(0,buf,x) != x)
				break;
			ib += x;
		}
	}
	time(&offtime);
	syslog(LLEV,"disconnect host=%s/%s destination=%s/%d in=%d out=%d duration=%d",rhost,raddr,dhost,portid,ib,ob,offtime - ontime);
	exit(0);
}



#ifdef	BINDDEBUG
debugbind()
{
	struct	sockaddr_in	mya;
	int	x;
	int	nread;

	if((x = socket(AF_INET,SOCK_STREAM,0)) < 0) {
		perror("socket");
		exit(1);
	}
	mya.sin_family = AF_INET;
	bzero(&mya.sin_addr,sizeof(mya.sin_addr));
	mya.sin_port = htons(BINDDEBUGPORT);
	if(bind(x,(struct sockaddr *)&mya,sizeof(mya))) {
		perror("bind");
		exit(1);
	}
	if(listen(x,1) < 0) {
		perror("listen");
		exit(1);
	}
	if((nread = accept(x,0,0)) < 0) {
		perror("accept");
		exit(1);
	}
	close(0);
	dup(nread);
	close(1);
	dup(nread);
}
#endif
