%{
/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * parser for sfc configuration file
 *
 * $Id: config.y,v 1.38 1995/09/12 15:07:44 robby Rel $
 *
 */

#ifndef lint
static char rcsid[] = "$Id: config.y,v 1.38 1995/09/12 15:07:44 robby Rel $";
#endif

#ifdef YYDEBUG
extern int yydebug = 1;
#endif

#define STRSIZE 255

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <limits.h>

#include <linux/ip.h>
#include <linux/netdevice.h>
#include <linux/sf_kernel.h>

#include "sf_rc.h"
#include "sf_global.h"
#include "sf_config.h"

static int	yylineno = 1;		/* actual line number, used for error messages */
static char	act_proto_name[5];	/* needed to determine correct port with getservbyname */

static struct sf_fw *fw = NULL;		/* needed to construct new firewall entry */
static int	first_new_addr = 0;	/* needed to parse hostaddr */
static struct in_addr  inaddr;		/* used to convert ulong to in_addr */
static int	dynamic = 0;		/* is the current rule dynamic */

static struct notification *not = NULL;	/* needed to construct new notification entry */
static struct let_if_chain *iftmp = NULL;	/* needed to construct if entries */
static struct let_if_chain *lettmp = NULL;	/* needed to construct let entries */

char msgtemp[1000];			/* needed to build message string */

static int    var_free = 0;			/* first free entry in varnames */

struct operand_type {
  int val;
  int type;
};

%}

%union {
  unsigned long         number;
  long			integer;
  char                  string[STRSIZE];
  unsigned short        ip_portno;        /* port number, network byte order */
  struct operand_type	operand;          /* to parse variables and constants */
}

%type <ip_portno>       portaddr
%type <string>          operator
%type <string>          lq
%type <integer>		qualifier
%type <integer>		timeoutval
%type <integer>		dynamicaddr
%type <string>		mailaddress
%type <operand>		const_or_var
%type <integer>		numberorspoof

%token	T_REJECT T_BLOCK T_ACCEPT T_SETUP T_RULES T_END
%token	T_WITH T_COMMA T_COLON T_SEMICOLON T_BACKSLASH T_DOTDOT
%token	T_NET_UNREACHABLE T_HOST_UNREACHABLE T_PROTO_UNREACHABLE
%token	T_PORT_UNREACHABLE
%token	T_ALL T_FROM T_TO T_PORT T_MASK
%token  T_ICMP T_IGMP T_TCP T_UDP T_RIP
%token	T_IPOPT_RR T_IPOPT_TS T_IPOPT_SECURITY T_IPOPT_LSRR T_IPOPT_SATID
%token	T_IPOPT_SSRR
%token	T_ICMP_ECHOREPLY T_ICMP_DEST_UNREACH T_ICMP_SOURCE_QUENCH
%token	T_ICMP_REDIRECT T_ICMP_ECHO T_ICMP_TIME_EXCEEDED T_ICMP_PARAMETERPROB
%token	T_ICMP_TIMESTAMP T_ICMP_TIMESTAMPREPLY T_ICMP_INFO_REQUEST
%token	T_ICMP_INFO_REPLY T_ICMP_ADDRESS T_ICMP_ADDRESSREPLY 
%token	T_IGMP_HOST_MEMBERSHIP_QUERY T_IGMP_HOST_MEMBERSHIP_REPORT
%token  T_IGMP_HOST_LEAVE_MESSAGE
%token	T_NOTIFICATION T_NOTIFICATION_LEVEL T_LEVEL 
%token	T_MAIL T_SPY T_SYSLOG T_EXEC T_IF T_THEN T_ENDIF T_RELEVEL T_PARAN
%token	T_EQUAL T_NEQUAL T_GREATER T_SMALLER T_ROUTE T_VIA
%token	T_OPTIONS T_INTERNALNETS T_DEVICE T_INSIDE T_OUTSIDE
%token  T_LET T_ASSIGNMENT T_PLUS T_MINUS T_TTL
%token  T_CURPROTO T_SOURCEHOST T_SOURCENET T_DESTHOST T_DESTNET T_TIMEOUT
%token  T_MAIL_DEFAULT T_MESSAGE T_SPOOF
%token  <string> IP_ADDRESS IDENTIFIER STRING
%token  <integer> NUMBER 

%%

config			: /* Empty */
			| setup_section
			  rules_section
			  notification_section
			  T_END
				{
#ifdef SF_DEBUG
  printf("end.\n");
#endif
				}
			| error {YYABORT;}
			;

setup_section		: T_SETUP
				{
#ifdef SF_DEBUG
  printf("\nsetup\n");
#endif
				}
			  internal_statement
			  mail_statement
			;

mail_statement		: T_MAIL_DEFAULT STRING T_SEMICOLON
				{
#ifdef SF_DEBUG
  printf("mail_default %s;\n",$2);
#endif
  sf_mailaddr = malloc(strlen($2)+1);
  if (sf_mailaddr == NULL) { sferror("Not enough memory"); return -1; }
  strcpy(sf_mailaddr,$2+1);
  sf_mailaddr[strlen(sf_mailaddr)-1]=0;
				}
			;

rules_section		: T_RULES
				{
#ifdef SF_DEBUG
  printf("\nrules\n");
#endif
				}
			  statements
;

statements 		: statement
			| statement statements
			;
	
statement		: rule_statement T_SEMICOLON
                                {
#ifdef SF_DEBUG
  printf(";\n");
#endif
                                  act_proto_name[0] = 0;
                                }
			;

rule_statement		: action ipheader source destination notification_level
			;

action			: reject
			| block
			| accept
			;

reject			: T_REJECT 
                                {
#ifdef SF_DEBUG
  printf("reject");
#endif
  fw = init_new_entry();
  if (fw == NULL) return -1;
  fw->fw_rc = SF_RC_RHOST;
                                }
			| T_REJECT T_WITH 
                                {
#ifdef SF_DEBUG
  printf("reject\twith ");
#endif
  fw = init_new_entry();
  if (fw == NULL) return -1;
                                }
			  icmp_reject_option
			;

icmp_reject_option	: T_NET_UNREACHABLE
                                {
#ifdef SF_DEBUG
  printf("icmp_net_unreachable\n");
#endif
  fw->fw_rc = SF_RC_RNET;
                                }
			| T_HOST_UNREACHABLE
                                {
#ifdef SF_DEBUG
  printf("icmp_host_unreachable\n");
#endif
  fw->fw_rc = SF_RC_RHOST;
                                }
			| T_PROTO_UNREACHABLE
                                {
#ifdef SF_DEBUG
  printf("icmp_protocol_unreachable\n");
#endif
  fw->fw_rc = SF_RC_RPROTO;
                                }
			| T_PORT_UNREACHABLE
                                {
#ifdef SF_DEBUG
  printf("icmp_port_unreachable\n");
#endif
  fw->fw_rc = SF_RC_RPORT;
                                }
			;

block			: T_BLOCK
                                {
#ifdef SF_DEBUG
  printf("block");
#endif
  fw = init_new_entry();
  if (fw == NULL) return -1;
  fw->fw_rc = SF_RC_BLOCK;
                                }
			;

accept			: T_ACCEPT
                                {
#ifdef SF_DEBUG
  printf("accept");
#endif
  fw = init_new_entry();
  if (fw == NULL) return -1;
  fw->fw_rc = SF_RC_ACCEPT;
                                }
			;

ipheader		: protocol
			| T_OPTIONS 
                                {
#ifdef SF_DEBUG
  printf("\toptions\t");
#endif
                                }
			  ip_options 
                                {
#ifdef SF_DEBUG
  printf("\n");
#endif
                                }
			  protocol
			;

ip_options		: ip_option
			| ip_options T_COMMA 
                                {
#ifdef SF_DEBUG
  printf(",\n\t\t");
#endif
                                }
			  ip_option
			;

ip_option		: T_IPOPT_RR		/* record packet route	*/
                                {
#ifdef SF_DEBUG
  printf("record_route");
#endif
  fw->fw_flags |= SF_FW_OPT_RR;
                                }
			| T_IPOPT_TS		/* timestamp 		*/
                                {
#ifdef SF_DEBUG
  printf("timestamp");
#endif
  fw->fw_flags |= SF_FW_OPT_TS;
                                }
			| T_IPOPT_SECURITY	/* provide s,c,h,tcc 	*/
                                {
#ifdef SF_DEBUG
  printf("security");
#endif
  fw->fw_flags |= SF_FW_OPT_SEC;
                                }
			| T_IPOPT_LSRR		/* loose source route 	*/
                                {
#ifdef SF_DEBUG
  printf("loose_source_route");
#endif
  fw->fw_flags |= SF_FW_OPT_LSR;
                                }
			| T_IPOPT_SATID		/* satnet id 		*/
                                {
#ifdef SF_DEBUG
  printf("sat_id");
#endif
  fw->fw_flags |= SF_FW_OPT_SATID;
                                }
			| T_IPOPT_SSRR		/* strict source route 	*/
                                {
#ifdef SF_DEBUG
  printf("strict_source_route");
#endif
  fw->fw_flags |= SF_FW_OPT_SSR;
                                }
			| T_TTL lq NUMBER       /* time to live         */
                                {
#ifdef SF_DEBUG
  printf("time_to_live %s %li",$2,$3);
#endif
  fw->ttl = (__u8)$3;
  if (strcmp($2,"=") == 0) fw->fw_flags |= SF_FW_TTL_EQUAL;
  if (strcmp($2,"<") == 0) fw->fw_flags |= SF_FW_TTL_LESS;
  if (strcmp($2,">") == 0) fw->fw_flags |= SF_FW_TTL_GREATER;
  if (strcmp($2,"!=") == 0) fw->fw_flags |= SF_FW_TTL_NOTEQUAL;
                                }
			;

protocol		: all		/* all protocols 			*/
			| icmp		/* internet control message protocol	*/
			| igmp		/* internet group multicast protocol	*/
			| tcp		/* transmission control protocol	*/
			| udp		/* user datagram protocol		*/
			| rip		/* routing information protocol 	*/
			| current	/* current protocol (dynamic rule) */
			| NUMBER	/* any protocol (see RFC 1340) */
				{
#ifdef SF_DEBUG
  printf("\tprot %i\n",$1);
#endif
  if ( ($1>=0) && ($1<=255) )
    { fw->fw_flags |= SF_FW_CHECK_PROTOCOL;
      fw->protocol = (__u8)$1;
    }
   else
    sferrorl("Protocol number must be between 0 an 255");
				}
			;

all			: T_ALL
                                {
#ifdef SF_DEBUG
  printf("\tall\n");
#endif
  /* all is 0000 => do nothing */
                                }
			;

icmp			: T_ICMP
                                {
#ifdef SF_DEBUG
  printf("\ticmp\n");
#endif
  fw->fw_flags |= SF_FW_CHECK_PROTOCOL;
  fw->protocol = IPPROTO_ICMP;
                                }
			| T_ICMP 
				{
#ifdef SF_DEBUG
  printf("\ticmp\t");
#endif
  fw->fw_flags |= SF_FW_CHECK_PROTOCOL;
  fw->protocol = IPPROTO_ICMP;
                                }
			  icmp_types
                                {
#ifdef SF_DEBUG
  printf("\n");
#endif
                                }
			;

icmp_types		: icmp_type
			| icmp_type T_COMMA 
                                {
#ifdef SF_DEBUG
  printf(",\n\t\t");
#endif
                                }
			  icmp_types
			;

icmp_type		: T_ICMP_ECHOREPLY	/* Echo Reply			*/
                                {
#ifdef SF_DEBUG
  printf("icmp_echo_reply");
#endif
  fw->fw_flags |= SF_ICMP_ECHOREPLY;
                                }
			| T_ICMP_DEST_UNREACH	/* Destination Unreachable	*/
                                {
#ifdef SF_DEBUG
  printf("icmp_destination_unreachable");
#endif
  fw->fw_flags |= SF_ICMP_DEST_UNREACH;
                                }
			| T_ICMP_SOURCE_QUENCH	/* Source Quench                */
                                {
#ifdef SF_DEBUG
  printf("icmp_source_quench");
#endif
  fw->fw_flags |= SF_ICMP_SOURCE_QUENCH;
                                }
			| T_ICMP_REDIRECT	/* Redirect (change route)      */
                                {
#ifdef SF_DEBUG
  printf("icmp_redirect");
#endif
  fw->fw_flags |= SF_ICMP_REDIRECT;
                                }
			| T_ICMP_ECHO		/* Echo Request                 */
	                	{
#ifdef SF_DEBUG
  printf("icmp_echo");
#endif
  fw->fw_flags |= SF_ICMP_ECHO;
                                }
                        | T_ICMP_TIME_EXCEEDED  /* Time Exceeded		*/
                                {
#ifdef SF_DEBUG
  printf("icmp_time_exceeded");
#endif
  fw->fw_flags |= SF_ICMP_TIME_EXCEEDED;
                                }
                        | T_ICMP_PARAMETERPROB  /* Parameter Problem		*/
                                {
#ifdef SF_DEBUG
  printf("icmp_parameter_problem");
#endif
  fw->fw_flags |= SF_ICMP_PARAMETERPROB;
                                }
                        | T_ICMP_TIMESTAMP      /* Timestamp Request		*/
                                {
#ifdef SF_DEBUG
  printf("icmp_timestamp");
#endif
  fw->fw_flags |= SF_ICMP_TIMESTAMP;
                                }
                        | T_ICMP_TIMESTAMPREPLY /* Timestamp Reply		*/
                                {
#ifdef SF_DEBUG
  printf("icmp_timestamp_reply");
#endif
  fw->fw_flags |= SF_ICMP_TIMESTAMPREPLY;
                                }
                        | T_ICMP_INFO_REQUEST   /* Information Request		*/
                                {
#ifdef SF_DEBUG
  printf("icmp_info_request");
#endif
  fw->fw_flags |= SF_ICMP_INFO_REQUEST;
                                }
                        | T_ICMP_INFO_REPLY     /* Information Reply		*/
                                {
#ifdef SF_DEBUG
  printf("icmp_info_reply");
#endif
  fw->fw_flags |= SF_ICMP_INFO_REPLY;
                                }
                        | T_ICMP_ADDRESS        /* Address Mask Request		*/
                                {
#ifdef SF_DEBUG
  printf("icmp_address");
#endif
  fw->fw_flags |= SF_ICMP_ADDRESS;
                                }
                        | T_ICMP_ADDRESSREPLY   /* Address Mask Reply		*/
                                {
#ifdef SF_DEBUG
  printf("icmp_adress_reply");
#endif
  fw->fw_flags |= SF_ICMP_ADDRESSREPLY;
                                }
                        ;

igmp                    : T_IGMP
                                {
#ifdef SF_DEBUG
  printf("\tigmp\n");
#endif
  fw->fw_flags |= SF_FW_CHECK_PROTOCOL;
  fw->protocol = IPPROTO_IGMP;
                                }
                        | T_IGMP
                                {
#ifdef SF_DEBUG
  printf("\tigmp\t");
#endif
  fw->fw_flags |= SF_FW_CHECK_PROTOCOL;
  fw->protocol = IPPROTO_IGMP;
                                }
                          igmp_types
                                {
#ifdef SF_DEBUG
  printf("\n");
#endif
                                }
                        ;

igmp_types              : igmp_type
                        | igmp_types T_COMMA
                                {
#ifdef SF_DEBUG
  printf(",\n\t\t");
#endif
                                }
                          igmp_type
                        ;

igmp_type               : T_IGMP_HOST_MEMBERSHIP_QUERY  /* From RFC1112
        */
                                {
#ifdef SF_DEBUG
  printf("igmp_host_membership_query");
#endif
  fw->fw_flags |= SF_IGMP_HOST_MEMBERSHIP_QUERY;
                                }
                        | T_IGMP_HOST_MEMBERSHIP_REPORT /* Ditto
        */
                                {
#ifdef SF_DEBUG
  printf("igmp_host_membership_report");
#endif
  fw->fw_flags |= SF_IGMP_HOST_MEMBERSHIP_REPORT;
                                }
                        | T_IGMP_HOST_LEAVE_MESSAGE     /* An extra BSD seems to send   */
                                {
#ifdef SF_DEBUG
  printf("igmp_host_leave_message");
#endif
  fw->fw_flags |= SF_IGMP_HOST_LEAVE_MESSAGE;
                                }
                        ;

tcp                     : T_TCP
                                {
#ifdef SF_DEBUG
  printf("\ttcp\n");
#endif
  strcpy(act_proto_name,"tcp");
  fw->fw_flags |= SF_FW_CHECK_PROTOCOL;
  fw->protocol = IPPROTO_TCP;
                                }
                        ;

udp                     : T_UDP
                                {
#ifdef SF_DEBUG
  printf("\tudp\n");
#endif
  strcpy(act_proto_name,"udp");
  fw->fw_flags |= SF_FW_CHECK_PROTOCOL;
  fw->protocol = IPPROTO_UDP;
                                }
                        ;

rip                     : T_RIP
                                {
#ifdef SF_DEBUG
  printf("\trip\n");
#endif
  fw->fw_flags |= SF_FW_PROT_RIP;
                                }
                        | T_RIP
                                {
#ifdef SF_DEBUG
  printf("\trip\t");
#endif
  fw->fw_rip_idx = sf_addr_free;
  fw->fw_flags |= SF_FW_PROT_RIP;
                                }
                          hostaddresses
                                {
#ifdef SF_DEBUG
  printf("\n");
#endif
  fw->fw_rip_cnt = sf_addr_free - fw->fw_rip_idx;
                                }
			| T_RIP T_INSIDE
				{
#ifdef SF_DEBUG
  printf("\trip inside\n");
#endif
  fw->fw_flags |= SF_FW_PROT_RIP;
  fw->fw_rip_idx = 1; /* internalnets */
  fw->fw_rip_cnt = sf_addr[0].addr;
				}
			| T_RIP T_OUTSIDE
				{
#ifdef SF_DEBUG
  printf("\trip outside\n");
#endif
  fw->fw_flags |= SF_FW_PROT_RIP;
  fw->fw_rip_idx = 1; /* internalnets */
  fw->fw_rip_cnt = sf_addr[0].addr;
  fw->fw_flags |= SF_RIP_ADDR_NEG; /* outside */
				}
                        ;

current			: T_CURPROTO
				{
#ifdef SF_DEBUG
  printf("\tcurrentprotocol\n");
#endif
  if (!dynamic)
    { sferrorl("`currentprotocol' allowed in dynamic rule only, error"); }
  fw->fw_flags |= SF_FW_PROT_CURRENT;
				}
			;

source                  : /* Empty */
                        | T_FROM
                                {
#ifdef SF_DEBUG
  printf("\tfrom\t");
#endif
  fw->fw_src_idx = sf_addr_free;
                                }
                          addresses
                                {
#ifdef SF_DEBUG
  printf("\n");
#endif
  fw->fw_src_cnt = sf_addr_free - fw->fw_src_idx;
                                }
			| T_FROM T_INSIDE 
				{
#ifdef SF_DEBUG
  printf("\tfrom\tinside\n");
#endif
  fw->fw_src_idx = 1; /* internalnets */
  fw->fw_src_cnt = sf_addr[0].addr;
				}
                        | T_FROM T_INSIDE T_PORT portaddr
                                {
  int i;
#ifdef SF_DEBUG
  printf("\tfrom\tinside port %hu\n",$4);
#endif
  fw->fw_src_idx = sf_addr_free;
  fw->fw_src_cnt = sf_addr[0].addr;
  for (i=1 ; i <= sf_addr[0].addr ; i++)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = sf_addr[i].addr;
      sf_addr[sf_addr_free].mask = sf_addr[i].mask;
      sf_addr[sf_addr_free].port = $4;
      sf_addr[sf_addr_free].prend = $4;
      sf_addr_free++;
    }
                                }
                        | T_FROM T_INSIDE T_PORT portaddr T_DOTDOT portaddr
                                {
  int i;
#ifdef SF_DEBUG
  printf("\tfrom\tinside port %hu..%hu\n",$4,$6);
#endif
  fw->fw_src_idx = sf_addr_free;
  fw->fw_src_cnt = sf_addr[0].addr;
  for (i=1 ; i <= sf_addr[0].addr ; i++)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = sf_addr[i].addr;
      sf_addr[sf_addr_free].mask = sf_addr[i].mask;
      sf_addr[sf_addr_free].port = $4;
      sf_addr[sf_addr_free].prend = $6;
      sf_addr_free++;
    }
                                }
			| T_FROM T_OUTSIDE
				{
#ifdef SF_DEBUG
  printf("\tfrom\toutside\n");
#endif
  fw->fw_src_idx = 1; /* internalnets */
  fw->fw_src_cnt = sf_addr[0].addr;
  fw->fw_flags |= SF_FW_SRC_NEG; /* don't match if address is internal */
				}
                        | T_FROM T_OUTSIDE T_PORT portaddr
                                {
  int i;
#ifdef SF_DEBUG
  printf("\tfrom\toutside port %hu\n",$4);
#endif
  fw->fw_src_idx = sf_addr_free;
  fw->fw_src_cnt = sf_addr[0].addr;
  for (i=1 ; i <= sf_addr[0].addr ; i++)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = sf_addr[i].addr;
      sf_addr[sf_addr_free].mask = sf_addr[i].mask;
      sf_addr[sf_addr_free].port = $4;
      sf_addr[sf_addr_free].prend = $4;
      sf_addr_free++;
    }
  fw->fw_flags |= SF_FW_SRC_NEG; /* don't match if address is internal */
                                }
                        | T_FROM T_OUTSIDE T_PORT portaddr T_DOTDOT portaddr
                                {
  int i;
#ifdef SF_DEBUG
  printf("\tfrom\toutside port %hu..%hu\n",$4,$6);
#endif
  fw->fw_src_idx = sf_addr_free;
  fw->fw_src_cnt = sf_addr[0].addr;
  for (i=1 ; i <= sf_addr[0].addr ; i++)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = sf_addr[i].addr;
      sf_addr[sf_addr_free].mask = sf_addr[i].mask;
      sf_addr[sf_addr_free].port = $4;
      sf_addr[sf_addr_free].prend = $6;
      sf_addr_free++;
    }
  fw->fw_flags |= SF_FW_SRC_NEG; /* don't match if address is internal */
                                }
			| T_FROM dynamicaddr
                                {
#ifdef SF_DEBUG
  printf("\tfrom\tdynamic: %li\n",$2);
#endif
  if (!dynamic)
    { sferrorl("Dynamic addresses allowed in dynamic rules only, error"); }
  fw->fw_src_idx = $2;
  fw->fw_src_cnt = 0;
                                }

                 ;

destination             : /* Empty */
                        | T_TO
                                {
#ifdef SF_DEBUG
  printf("\tto\t");
#endif
  fw->fw_dst_idx = sf_addr_free;
                                }
                          addresses
                                {
#ifdef SF_DEBUG
  printf("\n");
#endif
  fw->fw_dst_cnt = sf_addr_free - fw->fw_dst_idx;
                                }
                        | T_TO T_INSIDE
                                {
#ifdef SF_DEBUG
  printf("\tto\tinside\n");
#endif
  fw->fw_dst_idx = 1; /* internalnets */
  fw->fw_dst_cnt = sf_addr[0].addr;
                                }
			| T_TO T_INSIDE T_PORT portaddr
				{
  int i;
#ifdef SF_DEBUG
  printf("\tto\tinside port %hu\n",$4);
#endif
  fw->fw_dst_idx = sf_addr_free;
  fw->fw_dst_cnt = sf_addr[0].addr;
  for (i=1 ; i <= sf_addr[0].addr ; i++)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = sf_addr[i].addr;
      sf_addr[sf_addr_free].mask = sf_addr[i].mask;
      sf_addr[sf_addr_free].port = $4;
      sf_addr[sf_addr_free].prend = $4;
      sf_addr_free++;
    } 
				}
			| T_TO T_INSIDE T_PORT portaddr T_DOTDOT portaddr
                                {
  int i;
#ifdef SF_DEBUG
  printf("\tto\tinside port %hu..%hu\n",$4,$6);
#endif
  fw->fw_dst_idx = sf_addr_free;
  fw->fw_dst_cnt = sf_addr[0].addr;
  for (i=1 ; i <= sf_addr[0].addr ; i++)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = sf_addr[i].addr;
      sf_addr[sf_addr_free].mask = sf_addr[i].mask;
      sf_addr[sf_addr_free].port = $4;
      sf_addr[sf_addr_free].prend = $6;
      sf_addr_free++;
    }
                                }
                        | T_TO T_OUTSIDE
                                {
#ifdef SF_DEBUG
  printf("\tto\toutside\n");
#endif
  fw->fw_dst_idx = 1; /* internalnets */
  fw->fw_dst_cnt = sf_addr[0].addr;
  fw->fw_flags |= SF_FW_DST_NEG; /* don't match if address is internal */
                                }
                        | T_TO T_OUTSIDE T_PORT portaddr
                                {
  int i;
#ifdef SF_DEBUG
  printf("\tto\toutside port %hu\n",$4);
#endif
  fw->fw_dst_idx = sf_addr_free;
  fw->fw_dst_cnt = sf_addr[0].addr;
  for (i=1 ; i <= sf_addr[0].addr ; i++)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = sf_addr[i].addr;
      sf_addr[sf_addr_free].mask = sf_addr[i].mask;
      sf_addr[sf_addr_free].port = $4;
      sf_addr[sf_addr_free].prend = $4;
      sf_addr_free++;
    }
  fw->fw_flags |= SF_FW_DST_NEG; /* don't match if address is internal */
				}
                        | T_TO T_OUTSIDE T_PORT portaddr T_DOTDOT portaddr
                                {
  int i;
#ifdef SF_DEBUG
  printf("\tto\toutside port %hu..%hu\n",$4,$6);
#endif
  fw->fw_dst_idx = sf_addr_free;
  fw->fw_dst_cnt = sf_addr[0].addr;
  for (i=1 ; i <= sf_addr[0].addr ; i++)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = sf_addr[i].addr;
      sf_addr[sf_addr_free].mask = sf_addr[i].mask;
      sf_addr[sf_addr_free].port = $4;
      sf_addr[sf_addr_free].prend = $6;
      sf_addr_free++;
    }
  fw->fw_flags |= SF_FW_DST_NEG; /* don't match if address is internal */
				}
			| T_TO dynamicaddr
				{
#ifdef SF_DEBUG
  printf("\tto\tdynamic: %li\n",$2);
#endif
  if (!dynamic)
    { sferrorl("Dynamic addresses allowed in dynamic rules only, error"); }
  fw->fw_dst_idx = $2;
  fw->fw_dst_cnt = 0;
				}
                        ;

dynamicaddr		: T_SOURCEHOST {$$ = SF_SOURCEHOST;}
			| T_SOURCENET  {$$ = SF_SOURCENET;}
			| T_DESTHOST   {$$ = SF_DESTHOST;}
			| T_DESTNET    {$$ = SF_DESTNET;}
			;

addresses               : address
                        | addresses T_COMMA
                                {
#ifdef SF_DEBUG
  printf(",\n\t\t");
#endif
                                }
                          address
                        ;

address                 : hostaddr
                                {
  int i;
  for (i=first_new_addr; i < sf_addr_free; i++)
    { sf_addr[i].port  = SF_ALL_PORTS;  /* no port specified => match all ports */
      sf_addr[i].prend = USHRT_MAX;
#ifdef SF_DEBUG
      inaddr.s_addr = sf_addr[i].addr;
      printf("%s",inet_ntoa(inaddr));
      inaddr.s_addr = sf_addr[i].mask;
      printf(" mask %s",inet_ntoa(inaddr));
      if (i != (sf_addr_free - 1)) printf(",\n\t\t");
#endif
    }
                                }
                        | T_PORT portaddr
                                {
  if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
    { sf_addr_error(); return -1; }
  sf_addr[sf_addr_free].addr = 0;
  sf_addr[sf_addr_free].mask = SF_IGNORE_ADDRESS;
  sf_addr[sf_addr_free].port = $2;
  sf_addr[sf_addr_free].prend = $2; /* single port specified */
  sf_addr_free++;
#ifdef SF_DEBUG
  printf("port %hu",$2);
#endif
                                }
                        | T_PORT portaddr T_DOTDOT portaddr
                                {
  if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
    { sf_addr_error(); return -1; }
  sf_addr[sf_addr_free].addr = 0;
  sf_addr[sf_addr_free].mask = SF_IGNORE_ADDRESS;
  sf_addr[sf_addr_free].port = $2;
  sf_addr[sf_addr_free].prend = $4;
  sf_addr_free++;
#ifdef SF_DEBUG
  printf("port %hu..%hu",$2,$4);
#endif
                                }
                        | hostaddr T_PORT portaddr
                                {
  int i;
  for (i=first_new_addr; i < sf_addr_free; i++)
    { sf_addr[i].port = $3;  
      sf_addr[i].prend = $3;
#ifdef SF_DEBUG
      inaddr.s_addr = sf_addr[i].addr;
      printf("%s",inet_ntoa(inaddr));
      inaddr.s_addr = sf_addr[i].mask;
      printf(" mask %s port %hu",inet_ntoa(inaddr),$3);
      if (i != (sf_addr_free - 1)) printf(",\n\t\t");
#endif
    }
                                }
                        | hostaddr T_PORT portaddr T_DOTDOT portaddr
                                {
  int i;
  for (i=first_new_addr; i < sf_addr_free; i++)
    { sf_addr[i].port = $3;
      sf_addr[i].prend = $5;
#ifdef SF_DEBUG
      inaddr.s_addr = sf_addr[i].addr;
      printf("%s",inet_ntoa(inaddr));
      inaddr.s_addr = sf_addr[i].mask;
      printf(" mask %s port %hu..%hu",inet_ntoa(inaddr),$3,$5);
      if (i != (sf_addr_free - 1)) printf(",\n\t\t");
#endif
    }
                                }
                        ;

hostaddr		: hostaddr_2 
				{ 
  int i;
  for (i=first_new_addr ; i < sf_addr_free ; i++)
    { /* calculate mask */
      inaddr.s_addr = sf_addr[i].addr;
      if (inet_lnaof(inaddr) == 0) /* address is net address */
        { if ((sf_addr[i].addr&0xFF) <= 0x7F) /* class A */
	    { sf_addr[i].mask = inet_addr("255.0.0.0"); continue; }
          if ((sf_addr[i].addr&0xFF) <= 0xBF) /* class B */
	    { sf_addr[i].mask = inet_addr("255.255.0.0"); continue; }
          if ((sf_addr[i].addr&0xFF) <= 0xDF) /* class C */
	    { sf_addr[i].mask = inet_addr("255.255.255.0"); continue; }
        }
      sf_addr[i].mask = inet_addr("255.255.255.255"); 
    }
				}
			| hostaddr_2 T_MASK IP_ADDRESS
				{ 
  unsigned long msk;
  int i;
  msk = inet_addr($3);
  if ((signed long int)msk == -1)
    { char errstr[80];
      sprintf(errstr,"Invalid mask %s",$3);
      sferrorl(errstr);
      msk = inet_addr("255.255.255.255");
    }
  for (i=first_new_addr ; i < sf_addr_free ; i++)
    sf_addr[i].mask = msk; 
				}
			;

hostaddr_2		: IP_ADDRESS 
				{ 
  /* convert to unsigned long, network byte order */
  if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
    { sf_addr_error(); return -1; }
  first_new_addr = sf_addr_free;
  sf_addr[sf_addr_free].addr = inet_addr($1); 
  if (   ( strcmp($1,"255.255.255.255") != 0)
      && ((signed long int)sf_addr[sf_addr_free].addr == -1))
    { char errstr[80];
      sprintf(errstr,"Error converting ip_address %s",$1);
      sferrorl(errstr);
      break;
    } 
  sf_addr_free++;
				}
                        | STRING 
				{ 
  /* convert to ip address */ 
  char name[255];
  struct hostent *hostptr;
  char **listptr;
  struct in_addr *ptr;

  strcpy(name,$1+1);
  name[strlen(name)-1]=0;
  first_new_addr = sf_addr_free;
  if ((hostptr=gethostbyname(name)) == NULL)
    { char errstr[80];
      sprintf(errstr,"Unknown host %s",$1);
      sferrorl(errstr);
      break;
    }
  if (hostptr->h_addrtype != AF_INET)
    { char errstr[80];
      sprintf(errstr,"Unknown address type received for host %s",$1);
      sferrorl(errstr);
      break;
    }
  listptr = hostptr->h_addr_list;
  while ( (ptr = (struct in_addr *) *listptr++) != NULL)
    { if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
        { sf_addr_error(); return -1; }
      sf_addr[sf_addr_free].addr = ptr->s_addr;
      sf_addr_free++;
    }
				} 
			;

portaddr		: NUMBER { $$ = (unsigned short)$1; }
			| IDENTIFIER 
				{ 
  /* convert service to port number */ 
  struct servent *servptr;
  if ((servptr=getservbyname($1,act_proto_name))==NULL)
    { char errstr[80];
      sprintf(errstr,"unknown service %s for protocol %s",$1,act_proto_name);
      sferrorl(errstr);
      $$ = 0;
      break;
    }
  $$ = (unsigned short)ntohs(servptr->s_port);
				}
			;

notification_level	: T_NOTIFICATION_LEVEL NUMBER
				{
#ifdef SF_DEBUG
  printf("\tnotification_level %li",$2);
#endif
  if ($2 > 0)
    { fw->level.num = (int) $2;
      fw->fw_flags |= SF_FW_LOG;
    }
  if ($2 < 0)
    { sferrorl("Notification level must not be negative"); }
                                }
			;

notification_section	: /* Empty */
			| T_NOTIFICATION 
				{
#ifdef SF_DEBUG
  printf("\nnotification\n");
#endif
  dynamic = 1;
                                }
			  levels
				{
  dynamic = 0;
				}
			;

levels			: level
			| levels level
			;

numberorspoof		: NUMBER { $$=$1; }
			| T_SPOOF { $$=SPOOF_LEVEL; }	

level			: T_LEVEL numberorspoof T_COLON
				{
  struct notification *ntmp;
#ifdef SF_DEBUG
  printf("level %li :\n",$2);
#endif
  if ( ($2<=0) && ($2!=SPOOF_LEVEL) )
    sferrorl("Level number must be greater than zero");
  not = init_new_notification_entry();
  if (not == NULL) return -1;
  not->n_level = (int) $2;

  /* order chain by level */
  if (notify == NULL) /* first entry in notification chain */
    { notify = not; break; }
  if (notify->n_level == not->n_level)
    { char errstr[80];
      sprintf(errstr,"Duplicate level %i",not->n_level);
      sferrorl(errstr);
    }
  if (notify->n_level >= not->n_level)
    { not->n_next = notify; notify = not; break; }
  ntmp = notify;
  while ( (ntmp->n_next != NULL) && (ntmp->n_next->n_level < not->n_level) )
    { ntmp = ntmp->n_next; }
  if ((ntmp->n_next != NULL) && (ntmp->n_next->n_level == not->n_level))
    { char errstr[80];
      sprintf(errstr,"Duplicate level %i",not->n_level);
      sferrorl(errstr);
    }
  not->n_next = ntmp->n_next;
  ntmp->n_next = not;
                                }
			  entries
			;

entries			: entry
			| entries entry
			| error
				{
  struct let_if_chain *iftmp2;
  /* clean up if_chains in construction */
  while (iftmp != NULL)
    { /* insert actual notification structure into primary chain
         to be found when cleaning up */
      if (iftmp->then != NULL)
        {
          not->n_next = notify;
          notify = not;
          /* let not point to previous notification structure */
          not = iftmp->then;
        }
      /* restore itfmp pointer */
      iftmp2 = iftmp -> next;
      /* clean up if chain */
      if (not->n_let_if == iftmp) { not->n_let_if = NULL; free(iftmp); }
       else
        { struct let_if_chain *iftmp3;
          iftmp3 = not->n_let_if;
          while (iftmp3->next != iftmp) { iftmp3 = iftmp3 -> next; }
          iftmp3->next = NULL;
          free(iftmp);
        }
      iftmp = iftmp2;
    }
  YYABORT;
				}
			;

entry			: /* empty */ T_SEMICOLON
			| event T_SEMICOLON
				{
#ifdef SF_DEBUG
  printf(";\n");
#endif
				}
			;

event			: message
			| log
			| exec
			| mail
			| spy
			| if_statement 
			| relevel
			| reconfigure
                        | let 
			;

message			: T_MESSAGE 
				{
#ifdef SF_DEBUG
  printf("\tmessage");
#endif
  msgtemp[0]=0;
  if (not->n_message != NULL) 
    { strcpy(msgtemp,not->n_message);
      free(not->n_message);
    }
				}
			  messagetext
				{
  not->n_message = malloc(strlen(msgtemp)+1);
  if (not->n_message == NULL) {sferror("Not enough memory"); return -1; }
  strcpy(not->n_message,msgtemp);
				}
			;

messagetext		: messagestring
			| messagetext messagestring
			;

messagestring		: STRING
				{
  int msglen = 0;
#ifdef SF_DEBUG
  printf("\n\t %s",$1);
#endif
  if ( (msglen = strlen(msgtemp)) > 0 )
    { msgtemp[msglen] = '\n';
      msgtemp[msglen+1] = 0;
    }
  if ( strlen($1)+strlen(msgtemp)-1 > sizeof(msgtemp) )
    sferrorl("Message too long");
   else
    { strcat(msgtemp,$1+1); /* copy without leading " */
      msgtemp[strlen(msgtemp)-1]=0; /* delete trailing " */
    }
				}
			;

reconfigure		: 
				{
#ifdef SF_DEBUG
  printf("\t");
#endif
				}
			  rule_statement timeoutval
				{
  struct rule_chain *ruletmp, *ruletmp2;
#ifdef SF_DEBUG
  printf(" timeout %li",$3);
#endif

  ruletmp = malloc(sizeof(struct rule_chain));
  if (ruletmp == NULL) { sferror("Not enough memory"); return -1; }
  ruletmp->next = NULL;

  ruletmp->rule_id = fw;
  ruletmp->timeout = $3;

  if (not->n_rule == NULL)
    { not->n_rule = ruletmp; }
   else
    { ruletmp2 = not->n_rule;
      while (ruletmp2->next != NULL) {ruletmp2 = ruletmp2->next;}
      ruletmp2->next = ruletmp;
    }
				}
			;

timeoutval		: /* empty */ {$$ = 0;}
			| T_TIMEOUT NUMBER {$$ = (long) $2;}
			;

log			: T_SYSLOG 
				{
#ifdef SF_DEBUG
  printf("\tsyslog");
#endif
  not->n_syslog = 1;
                                }
			;

exec			: T_EXEC STRING
				{
  struct exec_chain *exectmp, *exectmp2;
#ifdef SF_DEBUG
  printf("\texec %s",$2);
#endif
  exectmp = malloc(sizeof(struct exec_chain));
  if (exectmp == NULL) { sferror("Not enough memory"); return -1; }
  exectmp->next = NULL;

  exectmp->exec_command=malloc(strlen($2)+1);
  if (exectmp->exec_command == NULL) {sferror("Not enough memory"); return -1; }
  /* copy string without leading " */
  strcpy(exectmp->exec_command,$2+1);
  /* delete trailing " */
  exectmp->exec_command[strlen(exectmp->exec_command)-1] = 0;

  if (not->n_exec == NULL)
    { not->n_exec = exectmp; }
   else
    { exectmp2 = not->n_exec;
      while (exectmp2->next != NULL) {exectmp2 = exectmp2->next;}
      exectmp2->next = exectmp;
    }
                                }
			; 

mail			: T_MAIL mailaddress
				{
#ifdef SF_DEBUG
  printf("\tmail %s",$2);
#endif
  if (not->n_mail != NULL) 
    { sferrorl("Only one mail statement is allowed"); return -1; }
    
  not->n_mail = malloc(strlen($2)+1);
  if (not->n_mail == NULL) {sferror("Not enough memory"); return -1; }
  strcpy(not->n_mail,$2);
                                }
			;

mailaddress		: STRING
				{
  if (strlen($1) >= STRSIZE)
  { sferrorl("String too long in mail statement");
    $1[STRSIZE-1] = 0;
  }
  strcpy($$,$1+1);
  $$[strlen($$)-1]=0;
				}
			| /* Empty */
				{
  strcpy($$,sf_mailaddr);
				}
			;

spy			: T_SPY
				{
#ifdef SF_DEBUG
  printf("\tspy");
#endif
  if (not->n_level != SPOOF_LEVEL)
    not->n_spy = 1;
  else
    sferrorl("`spy' is not allowed in level spoof");
                                }
			;

relevel			: T_RELEVEL NUMBER
				{
#ifdef SF_DEBUG
  printf("\trelevel %li",$2);
#endif
  if (not->n_level != SPOOF_LEVEL)
    {
      not->n_relevel = 1;
      not->n_relevel_val.num = (int) $2;
    }
  else
    sferrorl("`relevel' is not allowed in level spoof");
                                }
			;

if_statement		: T_IF 
				{
  struct let_if_chain *iftmp2;
#ifdef SF_DEBUG
  printf("\tif");
#endif
  iftmp2 = iftmp; 
  iftmp = malloc(sizeof(struct let_if_chain));
  if (iftmp == NULL) { sferror("Not enough memory"); return -1; }
  iftmp->statement_type = IF_STATEMENT;
  iftmp->then = NULL;
  iftmp->next = iftmp2; /* save old iftmp in order to handle
                           nested ifs correctly */

  if (not->n_let_if == NULL)
    { not->n_let_if = iftmp; }
   else
    { iftmp2 = not->n_let_if;
      while (iftmp2->next != NULL) {iftmp2 = iftmp2->next;}
      iftmp2->next = iftmp;
    }

                                }
			  optparan condition optparan T_THEN 
				{
#ifdef SF_DEBUG
  printf("then\n");
#endif
  /* handle nested ifs */
  iftmp->then = not; /* save parent notification structure */
  not = init_new_notification_entry();
  if (not == NULL) return -1;
  not->n_level = iftmp->then->n_level;
                                }
			  entries 
			  T_ENDIF
				{
  struct notification *ntmp;
  struct let_if_chain *iftmp2;
#ifdef SF_DEBUG
  printf("\tendif");
#endif
  ntmp = not;
  not = iftmp->then; /* restore pointer to parent notification structure */
  iftmp->then = ntmp;

  iftmp2 = iftmp;
  iftmp = iftmp2->next;
  iftmp2->next = NULL;
                                }
			;

optparan		: /* empty */ 
			| T_PARAN
			;

const_or_var		: IDENTIFIER qualifier
				{
  int varno;
#if SF_DEBUG
  printf(" %s:%li ",$1,$2);
#endif
  varno = find_varname($1);
  if (varno == -1) return -1;
  $$.type = $2;
  $$.val = varno;
				}
			| NUMBER
				{
#if SF_DEBUG
  printf(" %li ",$1);
#endif
  $$.type = OPERAND_CONST;
  $$.val = $1;
				}
			;

condition		: const_or_var lq 
				{
#ifdef SF_DEBUG
  printf("%s",$2);
#endif
  iftmp->operand1 = $1.val;
  iftmp->optyp1 = $1.type;
  if (strcmp($2,"=") == 0)  { iftmp->operator = OP_EQUAL; }
  if (strcmp($2,"<") == 0)  { iftmp->operator = OP_LESS; }
  if (strcmp($2,">") == 0)  { iftmp->operator = OP_GREATER; }
  if (strcmp($2,"!=") == 0) { iftmp->operator = OP_NOTEQUAL; }
                                }
			  const_or_var
				{
  iftmp->operand2 = $4.val;
  iftmp->optyp2 = $4.type;
				}
			;

lq			: T_EQUAL {strcpy($$,"=");}
			| T_SMALLER {strcpy($$,"<");}
			| T_GREATER {strcpy($$,">");}
			| T_NEQUAL {strcpy($$,"!=");}
			;

let			: T_LET IDENTIFIER qualifier T_ASSIGNMENT
				{
  struct let_if_chain *lettmp2;
#ifdef SF_DEBUG
  printf("\tlet %s:%li := ",$2,$3);
#endif
  lettmp = malloc(sizeof(struct let_if_chain));
  if (lettmp == NULL) { sferror("Not enough memory"); return -1; }
  lettmp->statement_type = LET_STATEMENT;
  lettmp->next = NULL;

  lettmp->variable = find_varname($2);
  lettmp->vartyp = $3;
  if (lettmp->variable == -1) return -1;

  if (not->n_let_if == NULL)
    { not->n_let_if = lettmp; }
   else
    { lettmp2 = not->n_let_if;
      while (lettmp2->next != NULL) {lettmp2 = lettmp2->next;}
      lettmp2->next = lettmp;
    }
                                }
			  expression timeoutval
				{
#ifdef SF_DEBUG
  printf(" timeout %li",$7);
#endif
  lettmp->timeout = $7;
				}
			;

expression		: IDENTIFIER qualifier
				{
#ifdef SF_DEBUG
  printf("%s:%li",$1,$2);
#endif
  lettmp->operand1 = find_varname($1);
  if (lettmp->operand1 == -1) return -1;
  lettmp->optyp1   = $2;
  lettmp->operand2 = 0;
  lettmp->optyp2   = OPERAND_NOT_USED;
  lettmp->operator = OP_NOP;
	       			}
			| NUMBER
                                {
#ifdef SF_DEBUG
  printf("%li",$1);
#endif
  lettmp->operand1 = (int) $1;
  lettmp->optyp1   = OPERAND_CONST;
  lettmp->operand2 = 0;
  lettmp->optyp2   = OPERAND_NOT_USED;
  lettmp->operator = OP_NOP;
                                }
			| IDENTIFIER qualifier operator NUMBER
                                {
#ifdef SF_DEBUG
  printf("%s:%li %s %li",$1,$2,$3,$4);
#endif
  lettmp->operand1 = find_varname($1);
  if (lettmp->operand1 == -1) return -1;
  lettmp->optyp1   = $2;
  lettmp->operand2 = (int) $4;
  lettmp->optyp2   = OPERAND_CONST;
  if (strcmp($3,"-") == 0) 
    { lettmp->operator = OP_MINUS; }
   else
    { lettmp->operator = OP_PLUS; }
                                }
			| IDENTIFIER qualifier operator IDENTIFIER qualifier
                                {
#ifdef SF_DEBUG
  printf("%s:%li %s %s:%li",$1,$2,$3,$4,$5);
#endif
  lettmp->operand1 = find_varname($1);
  if (lettmp->operand1 == -1) return -1;
  lettmp->optyp1   = $2;
  lettmp->operand2 = find_varname($4);
  if (lettmp->operand2 == -1) return -1;
  lettmp->optyp2   = $5;
  if (strcmp($3,"-") == 0)
    { lettmp->operator = OP_MINUS; }
   else
    { lettmp->operator = OP_PLUS; }
                                }
			;

operator		: T_PLUS {strcpy($$,"+");}
			| T_MINUS {strcpy($$,"-");}
			;

qualifier		: /* empty */ {$$ = OPERAND_VAR;}
			| T_COLON T_SOURCEHOST {$$ = OPERAND_VAR_SOURCEHOST;}
			| T_COLON T_DESTHOST {$$ = OPERAND_VAR_DESTHOST;}
			;

internal_statement	: /* empty */
				{
  sf_addr[0].addr = 0; /* number of internalnets */
  sf_addr_free = 1;
				}
			| T_INTERNALNETS
                                {
#ifdef SF_DEBUG
  printf("internalnets\n\t");
#endif
  if (sf_addr_free != 0) 
    { sferror("Array sf_addr not empty"); return -1; }
  if (sf_addr_free >= SF_ADDRESS_CNT_MAX)
    { sf_addr_error(); return -1; }
  sf_addr_free++; /* sf_addr[0].addr contains number of internalnets-entries */
                                } 
			  hostaddresses T_SEMICOLON
				{
  /* save number of "inside nets" in sf_addr[0].addr, so it can easily
   * be determined by the filter */
  sf_addr[0].addr = sf_addr_free - 1;
#ifdef SF_DEBUG
  printf(";\t[%lu internalnet entries]\n",sf_addr[0].addr);
#endif
				}
			;

hostaddresses		: netaddr
			| hostaddresses T_COMMA
                                {
#ifdef SF_DEBUG
  printf(",\n\t");
#endif
                                }
			  netaddr
			;

netaddr			: hostaddr
                                {
  int i;
  for (i=first_new_addr; i < sf_addr_free; i++)
    { sf_addr[i].port  = SF_ALL_PORTS;
      sf_addr[i].prend = USHRT_MAX;
#ifdef SF_DEBUG
      inaddr.s_addr = sf_addr[i].addr;
      printf("%s",inet_ntoa(inaddr));
      inaddr.s_addr = sf_addr[i].mask;
      printf(" mask %s",inet_ntoa(inaddr));
      if (i != (sf_addr_free - 1)) printf(",\n\t");
#endif  
    } 
                                }

%%

#include "lex.yy.c"

void yyerror(char *s)     /* called by parser */
{ fprintf(stderr,"The parser detected a %s in line %i!\n",s,yylineno);
  parseerror = 1;
}

void sferrorl(char *s)    /* error message with line number */
{ fprintf(stderr,"%s in line %i!\n",s,yylineno);
  parseerror = 1;
}

void sferror(char *s)     /* error message without line number */
{ fprintf(stderr,"%s!\n",s); 
  parseerror = 1;
}

void sf_addr_error(void)
{ char errstr[80];
  sprintf(errstr,"Too many addresses, maximum is %i",SF_ADDRESS_CNT_MAX);
  sferror(errstr);
}


struct sf_fw *init_new_entry(void)
{ struct sf_fw *tmp;
  tmp = malloc(sizeof(struct sf_fw));
  if (tmp == NULL) { sferror("Not enough memory"); return NULL; }
  tmp->magic = SF_FW_MAGIC;
  tmp->fw_line = yylineno;
  tmp->fw_src_idx = 0;
  tmp->fw_src_cnt = 0;
  tmp->fw_dst_idx = 0;
  tmp->fw_dst_cnt = 0;
  tmp->fw_flags = 0;
  tmp->protocol = 0;
  tmp->ttl = 0;
  tmp->fw_rip_idx = 0;
  tmp->fw_rip_cnt = 0;
  tmp->rule_id.ptr = tmp;
  tmp->level.ptr = NULL;
  if (dynamic)
    { tmp->timeout = SF_RULE_DYNAMIC_INACTIVE; }
   else
    { tmp->timeout = SF_RULE_STATIC; }

  /* insert into rules list */
  tmp->fw_next = rules;
  rules=tmp;

  return tmp;
}

struct notification *init_new_notification_entry(void)
{ struct notification *tmp;
  tmp = malloc(sizeof(struct notification));
  if (tmp == NULL) { sferror("Not enough memory"); return NULL; }
  tmp->n_level   = 
  tmp->n_syslog  = 
  tmp->n_spy     = 
  tmp->n_relevel =  
  tmp->n_relevel_val.num = 0;
  tmp->n_message = NULL;
  tmp->n_next = NULL;
  tmp->n_exec = NULL;
  tmp->n_mail = NULL;
  tmp->n_let_if = NULL;
  tmp->n_rule = NULL;
  return tmp;
}

int find_varname(char *name)  /* translates variable name to number */
{ int i = 0;
  while (i < var_free)
    { if (strcmp(name, varnames[i]) == 0) return i; 
      i++;
    }

  /* variable name not in table, allocate new entry */
  if (var_free >= SF_VAR_CNT_MAX)
    { char errstr[80];
      sprintf(errstr,"Too many variables, maximum is %i",SF_VAR_CNT_MAX);
      sferror(errstr); 
      return -1; 
    }
  varnames[var_free]=malloc(strlen(name)+1);
  if (varnames[var_free]==NULL)
    { sferror("Not enough memory");
      return -1;
    }
  strcpy(varnames[var_free++], name);
  variables[i].value   =  /* initialize variable */
  variables[i].address = 
  variables[i].timeout = 0;
  sf_var_free++;
  return i; /* i == var_free - 1 */
}

