/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * In this file all data structures holding the configuration
 * for the firewell daemon are declared. All functions concerning 
 * the configuration are provided here.
 *
 * $Id: sf_config.h,v 1.17 1995/08/05 11:35:17 robby Rel $
 */

#ifndef _SF_CONFIG_H
#define _SF_CONFIG_H

/*
 * configuration data for the firewall daemon 
 */

/* chain containing all rules */
extern struct sf_fw *rules;

/* firewall entries point to the following array containing the ip addresses */
extern struct sf_address sf_addr[SF_ADDRESS_CNT_MAX];
extern int		sf_addr_free;                   /* first free entry in array */

/* name of firewall device in filesystem */
extern char		sf_devname[255];

/* default mail address */
extern char		*sf_mailaddr;

/* variables */
struct variable {
	struct variable		*next;
	long			value;
	unsigned long		address;
	long			timeout;
};

#define SF_VAR_CNT_MAX  200                             /* max. number of variables */
extern struct variable  variables[SF_VAR_CNT_MAX];      /* array to store variable values */
extern char		*varnames[SF_VAR_CNT_MAX];	/* variable names */
extern int 		sf_var_free;			/* first free entry in array */

/* notification structures */
extern struct notification	*notify;

#define SPOOF_LEVEL INT_MIN

struct notification {
	int			n_level;		/* notification level */
	struct notification	*n_next;		/* pointer to next entry */
	int			n_syslog:1,		/* make syslog entry */
				n_spy:1,		/* counter intelligence */ 
				n_relevel:1;		/* change notification level for rule */
        union {
	  int num;
	  struct notification *ptr;			/* new level */
        } n_relevel_val;
	char			*n_message;		/* log message */
	struct exec_chain	*n_exec;		/* commands to execute */
	char              	*n_mail;		/* addresses to send mail to */
	struct let_if_chain	*n_let_if;		/* set variables, if clauses */
	struct rule_chain	*n_rule;		/* dynamic rules */
};

struct exec_chain {
	struct exec_chain	*next;			/* pointer to next entry */
	char			*exec_command;		/* command line */
};

struct let_if_chain { /* if operand1 ? operand2 then */
		      /* let variable := operand1 ? operand2; */
	struct let_if_chain	*next;			/* pointer to next entry */

#define IF_STATEMENT		1
#define LET_STATEMENT		2
	int			statement_type;		/* if or let statement */

#define OPERAND_NOT_USED 	0
#define OPERAND_CONST    	1
#define OPERAND_VAR      	2
#define OPERAND_VAR_SOURCEHOST	3
#define OPERAND_VAR_DESTHOST	4
	int			vartyp:4,		/* type of dest. variable (used only in let statement) */
				optyp1:4, optyp2:4;	/* types of operands */
	int			variable;		/* variable to store result in (used only in let statement) */
	long			timeout;		/* given timeout value */
	int			operand1, operand2;	/* operands */

#define OP_NOP		0
#define OP_PLUS		1
#define OP_MINUS	2
#define OP_EQUAL        10
#define OP_LESS         11
#define OP_GREATER      12
#define OP_NOTEQUAL     13
	int			operator;		/* operator */
	struct notification	*then;			/* used only in if statement */
};

struct rule_chain {
	struct rule_chain	*next;			/* pointer to next entry */
	struct sf_fw		*rule_id;		/* number of rule */
	long			timeout;		/* timeout value */
};

/*
 * read config file
 */

extern int parseerror; /* parseerror == 1 => error occured while parsing */
extern int sf_read_config_file(char *);

/*
 * free allocated memory and initialize all data structures
 */

extern void sf_clean_up_data(void);

/* This function passed all configuration data to the filter.
 * It should be called after sf_read_config_file().
 * The return value is zero on success, negative otherwise.
 * The firewall device must exist and must be closed before calling 
 * the function.
 * This function calls sf_config_clear(), sf_config_add(), sf_config_addr().
 */

extern int sf_configure_filter(void);

/*
 * functions for configuring the firewall
 * The first parameter is the filehandle of the firewall device
 * which must be opened for writing before calling one of these
 * functions.
 * The return value is zero on success, negative otherwise.
 */
extern int sf_config_flush(int);		/* abort all active TCP connections
						   that are no longer allowed */
extern int sf_config_flush_all(int);		/* abort all active TCP connections */
extern int sf_config_clear(int);		/* clear all entries */
extern int sf_config_add(int,struct sf_fw *);	/* add entry to firewall */
extern int sf_config_replace(int,struct sf_fw *);/* replace entry identified by fw_id */
extern int sf_config_delete(int,unsigned long);	/* delete entry identified by fw_id */
extern int sf_config_addr(int);			/* update address array */

#endif /* _SF_CONFIG_H */

