/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * sf Firewall Daemon -- User Mode Logging Module
 *
 * $Id: sf_daemon.c,v 1.42 1995/09/12 15:08:59 robby Rel $
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <limits.h>
#include <netdb.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/vfs.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <linux/netdevice.h>

#include "sf_custom.h"
#include "sf_filter.h"
#include "sf_control.h"
#include "sf_global.h"
#include "sf_config.h"
#include "sf_rc.h"
#include "sf_daemon.h"
#include "sf_spy.h"
#include "sf_proto.h"

#undef LOG_DEBUG

/*
 * Public variables 
 */

int fdlock = 0;                         /* lock file handle */
int fddev = 0;                          /* device file handle */


/*
 * Static variables
 */

static FILE *fdlog         = NULL;      /* file pointer to above log file */
static char *last_log      = NULL;	/* last log message */ 
static char *last_msg      = NULL;      /* last user message */
static char num_last_log   = 0;		/* number of repeated messages */
static time_t time_log     = 0;         /* time of last log message */
static time_t ltime        = 0;         /* the current time */
static char lstime[20]	   = "";        /* the current time in ASCII */
static struct notification *new_note;   /* changing the notification level */
static int busy            = 0;         /* daemon is busy */
static int alarm_pending   = 0;         /* a timeout event is pending */
static int signal_pending  = 0;         /* a signal is pending */
static int low_disk        = 0;         /* notification has already been done */
static struct timeout *event_queue = NULL; /* events to be processed */
static char tmp[255];                   /* used for sprintf() */
static struct sf_fw *dynrules = NULL;   /* chain containing all dynamic rules */


static char *action[]   = {"reject(port unr)", 
                           "reject(protocol unr)",
                           "reject(host unr)",
                           "reject(net unr)",
                           "block",
                           "accept"};

/*
 * Default action for IP spoofing notification -- this case must be
 * treated specially, because spoofing is an implicit rule.
 */

static struct notification notification_spoof = {
  0,     /* n_level       */
  NULL,  /* n_next        */
  0,     /* n_syslog      */
  0,     /* n_spy         */
  0,     /* n_relevel     */
  {0},   /* n_relevel_val */
  SPOOF_MESSAGE,
  NULL,
  NULL,
  NULL,
  NULL 
};

struct sf_fw rule_spoof = {
  SF_FW_MAGIC,  /* magic   */
  NULL,         /* fw_next */
  0,            /* fw_line */
  {0},          /* rule_id */
  {&notification_spoof},
  SF_RULE_SPOOFING
};


/*
 * Definitions for the event manager: event queue and signal handlers
 */
                           
struct timeout {
        struct timeout *next;
        time_t time;
        void (*proc)(int, int);
        int param;
        int param2;
};

static void catch_alarm(int);
static void process_alarm(void);
static void process_signal(void);


/*
 * Close open file descriptors, etc. This function is called prior to exec()
 */

void prepare_for_exec(void)
{
  static int hasclosed = 0;

  if (hasclosed) 
    return;

  close(fdlock);
  close(fddev); 
  fclose(fdlog);
 
  hasclosed = 1;
}


/*
 * Get the current time, convert it to ASCII and store both
 * values in static variables. 
 */

static void get_time(void)
{
  /* Get current time */
  time(&ltime);

  /* Convert the current time to a string. Don't use ctime(), because
     the ctime() output is too verbose to be useful. */

  strftime(lstime, sizeof(lstime), TIME_FORMAT, localtime(&ltime));
}


/*
 * Connect to the operating system's log daemon and set up our log file
 */

static void connect_logs(char *config)
{
  openlog("firewall", LOG_CONS|LOG_PID, LOG_AUTHPRIV); /* no return code */

  if ((fdlog = fopen(LOG, "a")) == NULL) {
    perror("Cannot append to log file");
    exit(1);
  }

  /* The log file must not be fully buffered... We want to be able to read
   * it even if the system crashes.
   */
  setvbuf(fdlog, NULL, _IOLBF, 0);
  setvbuf(stderr, NULL, _IOLBF, 0);

  fprintf(fdlog, "%s daemon started, config: %s\n", lstime, config);
  
  /* all's well that ends well */
}


/*
 * Update the log file if there are log messages in the buffer
 */

static void update_log(void)
{
  if (num_last_log > 0) {
    fprintf(fdlog, "%s last message repeated %i times\n", lstime, num_last_log);
    num_last_log = 0;
  }
  if (last_log != NULL) {
    free(last_log);
    last_log = NULL;
  }
  if (last_msg != NULL) {
    free(last_msg);
    last_msg = NULL;
  }
}


/*
 * Flush the log from time to time. This function is called by the event mgr.
 */

static void flush_log(int dummy1, int dummy2)
{
  if ((last_log != NULL) && (num_last_log > 0) && (time_log + MAX_HOLD <= ltime))
    update_log();
}


/* 
 * This function is used to generate beautified log entries (multi line,
 * wrapping, etc.)
 */

void flogf(FILE *fd, char *time, char *message)
{
  int col, timew;

  fprintf(fd, "%s ", time);
  col = timew = strlen(time)+1;

  for (; *message != 0; message++) {

    if (*message == '\n') {
      fputc('\n', fd);
      for (col = 0; col < timew; col++)
        fputc(' ', fd);
    }
    else { 
      if (*message == ' ') {
        char *pos;

        pos = index(message+1, ' ');
        if (((pos != NULL) && (col+pos-message > 79)) ||
            ((pos == NULL) && (col+strlen(message) > 79))) {
          fputc('\n', fd);
          for(col = 0; col < timew-1; col++)
            fputc(' ',fd);
        }
      }

      if (col > 79) {
        fputc('\n', fd);
        for (col = 0; col < timew; col++) 
          fputc(' ',fd);
      }

      fputc(*message, fd);
      col++;
    }
  }

  fputc('\n', fd);
}  


/*
 * Log an error message to all available logs.
 */

static void log_error(char *msg)
{
  get_time();

  update_log();

  syslog(LOG_AUTHPRIV|LOG_ALERT, msg);

  flogf(fdlog, lstime, msg);
}


/*
 * Critical error: Try to log the error if possible, then stop.
 */

static void crit_error(char *msg)
{
  log_error(msg);

  fclose(fdlog);

  /* Well, maybe one shoudn't clutter the console... */
  fprintf(stderr, "\007Firewall daemon -- Fatal error: %s!\n", msg);

  /* The device will make sure all packets are blocked if the device is
     closed. To restore the `pass all' state, the user must unload the
     device module. Normally, the daemon should be restarted. */
  close(fddev);

  /* give up lock */
  flock(fdlock, LOCK_UN|LOCK_NB);
  close(fdlock);

  exit(1);
}


/*
 * Mail to addressee, the mail text is read from file descriptor fd
 */

FILE *mail(char *addressee)
{
  int filedes[2], pid;

  if (pipe(filedes) < 0)
    crit_error("Cannot create pipe!");

  if ((pid = fork()) < 0) { /* fork error */
    close(filedes[0]);
    close(filedes[1]);
    crit_error("Cannot fork()!");
  }

  if (pid == 0) { /* child */

    prepare_for_exec();

    /* redirect stdin */
    close(STDIN_FILENO); dup(filedes[0]); close(filedes[0]); close(filedes[1]);

    execl(MAIL, MAIL, addressee, NULL);

    perror("Cannot exec mail program");
    exit(1);
  }
  /* parent */
  close(filedes[0]);

  /* Write message text to pipe (= stdin of mail process) */
  return fdopen(filedes[1], "a");
}


/*
 * Fetch value of host addr(ess) specific variable.
 * The value is only valid if the timeout hasn't expired.
 */

static int fetch_var_value(int var, int addr)
{
  struct variable *vptr;

  for(vptr = variables[var].next;
      (vptr != NULL) && (vptr->address != addr);
      vptr = vptr->next) ;

  /* return 0 if variable doesn't exist or variable has expired */
  if ((vptr == NULL) || (vptr->timeout < ltime))
    return 0;

  return vptr->value;
}


/*
 * Get address of host addr(ess) specific variable.
 * If it is in the list, return immediatly.
 * If not, try to reclaim the storage of a timed-out variable.
 * If this fails, allocate a new one.
 */

static struct variable *fetch_var_addr(int var, int addr)
{
  struct variable *vptr;
  struct variable *timed_out = NULL;
  
  for (vptr = &(variables[var]);
       (vptr->next != NULL) && (vptr->next->address != addr);
       vptr = vptr->next) {

    if ((timed_out == NULL) && (vptr->next->timeout < ltime))
      timed_out = vptr->next; /* recycle this entry */
  }

  if (vptr->next != NULL)
    return vptr->next;

  if (timed_out != NULL) {
    timed_out->address = addr; /* reclaim storage */
    return timed_out;
  }

  /* else allocate new storage */
  vptr->next = (struct variable *) malloc(sizeof(struct variable));
  vptr = vptr->next;
  vptr->next    = NULL;
  vptr->address = addr;

  return vptr;
}


/*
 * Return the value of an operand, be it a const, a variable or
 * a qualified variable. All variables must be checked for expiration.
 */

static int fetch_op_value(int optype, int operand, struct iphdr *ip)
{
  switch (optype) {
    case OPERAND_CONST: 
         return operand;
    break; /* yes, I know we don't need this break... */
    case OPERAND_VAR:   
         if (variables[operand].timeout >= ltime)
           return variables[operand].value;
    break;
    case OPERAND_VAR_SOURCEHOST: 
         return fetch_var_value(operand, ip->saddr);
    break;   
    case OPERAND_VAR_DESTHOST:   
         return fetch_var_value(operand, ip->daddr);
    break;
  }

  return 0; /* default and OPERAND_NOT_USED */
}


/*
 * Figure out what class an address belongs to and return a mask
 */

static int get_class(int addr)
{
  if ((addr & 0xFF) <= 0x7F) /* class A */
    return inet_addr("255.0.0.0");
  if ((addr & 0xFF) <= 0xBF) /* class B */
    return inet_addr("255.255.0.0");
  if ((addr & 0xFF) <= 0xDF) /* class C */
    return inet_addr("255.255.255.0"); 
  
  /* huh? class D, E, or what? */
  return inet_addr("255.255.255.255");
}


/*
 * Add an address to the firewall address array using dynamic 
 * addresses (host, net) from the current ip packet
 */

static void add_address(long *idx, long *cnt, struct iphdr *ip)
{
  long i;
  long ptr = -1;
  long code = *idx;

  /* try to recycle free address entry */
  for (i = 1; i < sf_addr_free; i++) {
    if (sf_addr[i].prend == 0) {
      ptr = i;
      break;
    }
  }

  /* allocate a new entry */
  if (ptr == -1) {
    ptr = sf_addr_free;
    sf_addr_free++;
  }

  *idx = ptr;       /* point to newly allocated array entry */
  *cnt = 1;         /* which contains exactly one address  */

  sf_addr[*idx].port  = SF_ALL_PORTS; /* all ports match */
  sf_addr[*idx].prend = USHRT_MAX;

  switch (code) {
    case SF_SOURCEHOST : sf_addr[*idx].addr = ip->saddr;
                         sf_addr[*idx].mask = -1; /* all ones, match exactly */
    break;
    case SF_SOURCENET  : sf_addr[*idx].addr = ip->saddr;
                         sf_addr[*idx].mask = get_class(ip->saddr);
    break;
    case SF_DESTHOST   : sf_addr[*idx].addr = ip->daddr;
                         sf_addr[*idx].mask = -1; /* all ones, match exactly */
    break;
    case SF_DESTNET    : sf_addr[*idx].addr = ip->daddr;
                         sf_addr[*idx].mask = get_class(ip->daddr);
    break;
    default	       : /* this should not happen */
			 sf_addr[*idx].addr = 0;
			 sf_addr[*idx].mask = SF_IGNORE_ADDRESS;
    break;
  }
}


/*
 * Process timeout alarms. If the firewall is currently busy, delay
 * the procedure.
 */

static void catch_alarm(int sig) /* This is the SIGALRM signal catcher */
{
  /* re-set signal handler */
  signal(SIGALRM, catch_alarm);

  if (event_queue == NULL)
    return;

  if (busy)
    alarm_pending = 1;
  else 
    process_alarm();
} 

static void process_alarm(void) /* Process as many events as appropriate */
{
  struct timeout *event;

  alarm_pending = 0;

  get_time();

  event = event_queue;
  while ((event != NULL) && (event->time <= ltime)) {
    struct timeout *temp;

    /* call associated action */
    event->proc(event->param, event->param2);

    temp = event;
    event = event->next;
    free(temp);
  }
  event_queue = event;

  if (event_queue != NULL) 
    alarm(event_queue->time - ltime);
}


/*
 * Add event to the event queue and set alarm if needed
 */

time_t add_event(void (*proc)(int, int), int param, int param2, int timeout)
{
  struct timeout *new;

  if ((new = malloc(sizeof(struct timeout))) == NULL)
    crit_error("malloc: Not enough memory!");

  time(&ltime);
  
  new->proc   = proc;
  new->param  = param;
  new->param2 = param2;
  new->time   = ltime+timeout;

  if (event_queue == NULL) {
    new->next   = NULL;
    event_queue = new;
    alarm(timeout);
  }
  else {
    if (event_queue->time > new->time) {
      new->next   = event_queue;
      event_queue = new;
      alarm(timeout);
    }
    else {
      struct timeout *event = event_queue;

      while ((event->next != NULL) && (event->time <= new->time))
        event = event->next;
      new->next   = event->next;
      event->next = new; 
    } 
  }

  return new->time;
}


/*
 * Remove a dynamic rule. This function is invoked by the event manager.
 */

static void remove_rule(int id, int tid) 
{
  int rc;
  struct sf_fw *rule          = (struct sf_fw *) id;
  struct rule_chain *template = (struct rule_chain *) tid;

  /* first remove the rule from the filter */
  rc = sf_config_delete(fddev, id);

  /* remove the rule from the dynamic rule chain */
  if (dynrules == rule)
    dynrules = dynrules->fw_next;
  else {
    struct sf_fw *dyn;

    dyn = dynrules;
    while ((dyn != NULL) && (dyn->fw_next != rule))
      dyn = dyn->fw_next;
    if (dyn != NULL)
      dyn->fw_next = dyn->fw_next->fw_next;
  }

  update_log();
  if (rc == 0) {
    sprintf(tmp, "removed dynamic rule created %lis ago from template at line %i", 
            template->timeout, rule->fw_line);

    /* free dynamic entries in the address table */
    if (template->rule_id->fw_src_idx < 0)
      sf_addr[rule->fw_src_idx].prend = 0;
    if (template->rule_id->fw_dst_idx < 0)
      sf_addr[rule->fw_dst_idx].prend = 0;

    /* free tHe memory */
    free(rule);
  }
  else 
    sprintf(tmp, "ERROR removing dynamic rule created %lis ago from template at line %i", 
            template->timeout, rule->fw_line);

  flogf(fdlog, lstime, tmp);
}


/*
 * Execute the statements in a notifcation body.
 * This routine will be called recursively, because we have `if' 
 * statements.
 */

static void execute_notify(struct sf_fw *rule, char *message,
                           struct notification *note, struct iphdr *ip)
{
  union {
    struct exec_chain   *_exec;
    struct let_if_chain *_li;
    struct rule_chain   *_rule;
  } p;


  /* personal log message? */
  if (note->n_message != NULL) {
    if ((last_msg == NULL) || (strcmp(last_msg, note->n_message) != 0)) {
      if (last_msg != NULL)
        free(last_msg);
      last_msg = strdup(note->n_message);
      flogf(fdlog, lstime, note->n_message);
    }
  }


  /* syslog the message? */
  if (note->n_syslog)  {
    syslog(LOG_AUTHPRIV|LOG_CRIT, message);
    if (note->n_message != NULL)
      syslog(LOG_AUTHPRIV|LOG_CRIT, note->n_message);
  }


  /* change log level? */
  if (note->n_relevel)
    new_note = note->n_relevel_val.ptr;


  /* send mail? */
  if (note->n_mail != NULL) {
    FILE *fdmail;

    update_log();
    sprintf(tmp, "sending mail to %s", note->n_mail);
    flogf(fdlog, lstime, tmp);

    fdmail = mail(note->n_mail);
#ifdef PRIORITY
    fputs("Priority: " PRIORITY "\n", fdmail);
#endif
#ifdef REPLY_TO
    fputs("Reply-To: " REPLY_TO "\n", fdmail);
#endif
    fprintf(fdmail, "Subject: Notification, %s\n\n", lstime);
    if (note->n_message != NULL)
      fprintf(fdmail, "%s\n\n", note->n_message);
    fprintf(fdmail, "Triggered by:");
    flogf(fdmail,"",message);
    fclose(fdmail);
  }


  /* execute something? */
  for (p._exec = note->n_exec; p._exec != NULL; p._exec = p._exec->next) {
    int pid;

    update_log();
    sprintf(tmp, "executing sh: %s", p._exec->exec_command);
    flogf(fdlog, lstime, tmp);

    if ((pid = fork()) == 0) { /* child */

      prepare_for_exec();
     
      /* Note! The system() call is not a particular safe one.
         The user is advised to avoid using the exec-command
         facility... system() was chosen because some other user
         might need fancy shell features, redirection, etc. */ 

      if (system(p._exec->exec_command) < 0) {
        perror("Cannot exec external command");
        exit(1);
      }

      exit(0);
    }
    if (pid < 0)
      crit_error("Cannot fork()!");
  }


  /* start counter intelligence? */
  if (note->n_spy) 
    if (spy(ip,message,note->n_message,note->n_mail)) {
      update_log();
      sprintf(tmp, "started counter intelligence against host %s",
              inet_ntoa(*((struct in_addr *) &(ip->saddr))));
      flogf(fdlog, lstime, tmp);
    }
  

  /* assign variables or evaluate if condition? */
  for (p._li = note->n_let_if; p._li != NULL; p._li = p._li->next) {
    int op1, op2;

    /* fetch operands */
    op1 = fetch_op_value(p._li->optyp1, p._li->operand1, ip);
    op2 = fetch_op_value(p._li->optyp2, p._li->operand2, ip);
   
    if (p._li->statement_type == LET_STATEMENT) {
      struct variable *vptr;

      /* compute result of op1$op2 */
      if (p._li->operator == OP_MINUS)
        op2 = -1 * op2;

      op1 += op2;

      /* store result */
      if (p._li->vartyp == OPERAND_VAR) 
        variables[p._li->variable].value = op1;
      else {
        if (p._li->vartyp == OPERAND_VAR_SOURCEHOST)
          vptr = fetch_var_addr(p._li->variable, ip->saddr);
        else
          vptr = fetch_var_addr(p._li->variable, ip->daddr);

        vptr->value   = op1;
        vptr->timeout = (p._li->timeout == 0) ? INT_MAX : ltime+p._li->timeout;

        /* update total variable sum */
        if (variables[p._li->variable].timeout < ltime) 
          variables[p._li->variable].value = 0;
        variables[p._li->variable].value += op2;
      }

      variables[p._li->variable].timeout =
        (p._li->timeout == 0) ? INT_MAX : ltime+p._li->timeout;
    }

    else { /* if statement */
      int cond;

      cond = 0; /* condition is false by default */
      switch (p._li->operator) {
        case OP_EQUAL:    cond = op1 == op2;
        break;
        case OP_LESS:     cond = op1  < op2;
        break;
        case OP_GREATER:  cond = op1  > op2;
        break;
        case OP_NOTEQUAL: cond = op1 != op2;
        break;
      }

      if (cond && p._li->then)
        execute_notify(rule, message, p._li->then, ip); /* recursive call */
    }
  }


  /* dynamic rules? */
  for (p._rule = note->n_rule; p._rule != NULL; p._rule = p._rule->next) {
    struct sf_fw *new_rule;

    /* copy template for new rule */
    if ((new_rule = malloc(sizeof(struct sf_fw))) == NULL)
      crit_error("malloc: Not enough memory!");
    memcpy(new_rule, p._rule->rule_id, sizeof(struct sf_fw));

    /* add dynamic data */
    if (new_rule->fw_src_idx < 0)
      add_address(&(new_rule->fw_src_idx), &(new_rule->fw_src_cnt), ip);
    if (new_rule->fw_dst_idx < 0)
      add_address(&(new_rule->fw_dst_idx), &(new_rule->fw_dst_cnt), ip);
    if ((new_rule->fw_flags & SF_FW_PROT_CURRENT) == SF_FW_PROT_CURRENT) {
      new_rule->fw_flags &= ~SF_FW_PROT;
      new_rule->fw_flags |= SF_FW_CHECK_PROTOCOL;
      new_rule->protocol  = ip->protocol;
    }

    /* store new rule id */
    new_rule->rule_id.ptr = new_rule; 
    new_rule->fw_next     = NULL;

    sf_config_addr(fddev);
    sf_config_add(fddev, new_rule);

    /* add rule to dynamic rule chain */
    if (dynrules == NULL) 
      dynrules = new_rule;
    else {
      struct sf_fw *dyn;

      dyn = dynrules;
      while (dyn->fw_next != NULL)
        dyn = dyn->fw_next;  
      dyn->fw_next = new_rule;
    }

    update_log();

    /* if timeout != 0: make sure new rule will expire */
    if (p._rule->timeout != 0) {
      new_rule->timeout = 
        add_event(remove_rule, (int) new_rule, (int) p._rule, p._rule->timeout);
      
      sprintf(tmp, "created dynamic rule from template at line %i, timeout: %lis",
              new_rule->fw_line, p._rule->timeout);
    }
    else 
      sprintf(tmp, "created new rule from template at line %i, no timeout.",
              new_rule->fw_line); 
    
    flogf(fdlog, lstime, tmp);
  }
}


/*
 * Main loop: Read message buffer. The daemon will be terminated by
 * a signal, so we loop forever.
 */

static void packet_loop(void)
{
  struct iphdr  *ip;
  struct tcphdr *tcp;
  char cs[20], cd[20], rule_desc[20], tmp[255];
  char message[255], source_port[85], target_port[85], protocol_desc[15];
  int i, protocol_index;
  struct sf_proc log_data;
  struct sf_fw *rule;
  struct statfs statbuf;
  long free_blocks;


  while (1) {

    /* and timeout alarms to be processed */
    if (alarm_pending)
      process_alarm();

    if (signal_pending)
      process_signal();
   
    /* timeout alarms may do what they please during this phase only */ 
    busy = 0;
    if ((read(fddev, &log_data, sizeof(log_data))) < 0) 
      crit_error("Cannot read from firewall device!");

    get_time();

    /* check free disk space */
    if (statfs(LOG, &statbuf) < 0)
      crit_error("statfs: Cannot get free disk space!");
    else {
      free_blocks = statbuf.f_bavail != -1 ? statbuf.f_bavail : statbuf.f_blocks;
      free_blocks >>= 1; /* make it KB */
      if (free_blocks < DISK_MINFREE) {
        if (! low_disk) { /* notification done? */
          FILE *fdmail;

          low_disk = 1;
          log_error("Free disk space is low!");

          fdmail = mail(sf_mailaddr);
#ifdef PRIORITY
          fputs("Priority: " PRIORITY "\n", fdmail);
#endif
#ifdef REPLY_TO
          fputs("Reply-To: " REPLY_TO "\n", fdmail);
#endif
          fprintf(fdmail, "Subject: Firewall - DISK SPACE IS LOW!\n\nTime: %s\nSpace left on log device: %li KB\n",
                  lstime, free_blocks);
          fclose(fdmail);
        }
      }
      else if (free_blocks > 2*DISK_MINFREE) /* reset notification */
        low_disk = 0;
    }

    /* timeout alarms will have to wait until the buffer is processed */
    busy = 1;
    /* Examine message buffer */
    for (i = 0; i < log_data.num_entries; i++) {

      rule = log_data.data[i].rule.ptr;

      /* handle implicit rules */
      if (((int) rule == RULE_SPOOF_RECV) || ((int) rule == RULE_SPOOF_XMIT))
        rule = &rule_spoof;

      if ((rule == NULL) || (rule->level.ptr == NULL))
        continue; /* there might be an obsolete entry in the buffer */

      ip   = (struct iphdr *)  log_data.data[i].ip_packet;
      tcp  = (struct tcphdr *) ((unsigned long *)ip + ip->ihl);

      /* Compose the mandatory log entry.
         Log entries consist of:
         0) current date and time (not needed for syslog)
         1) filter function's rule number
         2) filter function's action: block/reject/accept
         3) packet source and destination
         4) packet type and port (if any)
       */

      *protocol_desc = 
      *source_port =
      *target_port = 0;

      if (ip->protocol < MAX_PROTOCOL) {
        protocol_index = ip->protocol;
        if ((ip->protocol == IPPROTO_UDP) ||
            (ip->protocol == IPPROTO_TCP)) {
          struct servent *serv_ent;
          char proto[4];

          if (ip->protocol == IPPROTO_UDP) 
            strcpy(proto, "udp");
          else
            strcpy(proto, "tcp");

          if ((serv_ent = getservbyport(tcp->th_sport, proto)) != NULL)
            sprintf(source_port, ":%s", serv_ent->s_name);
          else
            sprintf(source_port, ":%i", ntohs(tcp->th_sport));

          if ((serv_ent = getservbyport(tcp->th_dport, proto)) != NULL)
            sprintf(target_port, ":%s", serv_ent->s_name);
          else
            sprintf(target_port, ":%i", ntohs(tcp->th_dport));
        }
      }
      else {
        protocol_index = MAX_PROTOCOL+1;
        sprintf(protocol_desc, "%i", ip->protocol);
      }

      strcpy(cs, inet_ntoa(*((struct in_addr *) &(ip->saddr))));
      strcpy(cd, inet_ntoa(*((struct in_addr *) &(ip->daddr))));

      if (rule->timeout == SF_RULE_STATIC) 
        sprintf(rule_desc, "(s %i)", rule->fw_line);
      else if (rule->timeout == SF_RULE_SPOOFING)
        sprintf(rule_desc, "SPOOF: %s", log_data.data[i].devname);
      else 
        sprintf(rule_desc, "(d %i)", rule->fw_line);

      sprintf(message, "%s %s %s%s %s%s->%s%s",
              rule_desc,            
              action[log_data.data[i].action-SF_RC_MIN],
              protocol[protocol_index], protocol_desc,
              cs, source_port,
              cd, target_port);

#ifdef LOG_DEBUG
      flogf(stderr, lstime, message);
#endif

      /* Do some extra work to identify and kill repeated messages */

      if ((last_log != NULL) && (strcmp(message, last_log) == 0) && (num_last_log < MAX_REPEATS)) {
        /* increase the repeated messages counter */
        num_last_log++;
        /* make sure log file will be flushed sometime */
        if (num_last_log == 1)
          add_event(flush_log, 0, 0, MAX_HOLD);
      }
      else {
        /* Notify the user of repeated messages? */
        update_log();

        /* save a copy of the current message to identify repeated messages */
	last_log = strdup(message);
        /* save the time of this log access */
        time_log = ltime;

        /* write message to log file and flush the log file to disk */
        flogf(fdlog, lstime, last_log);
      }

      new_note = rule->level.ptr;
      execute_notify(rule, message, rule->level.ptr, ip);
      if (rule->level.ptr != new_note) {
        int i;

        rule->level.ptr = new_note;
        update_log();
        fprintf(fdlog, "%s changing notification level to ", lstime);
        if (new_note != NULL)  
          i = new_note->n_level;
        else { /* signal the change to the filter */
          i = 0;
          rule->fw_flags &= ~SF_FW_LOG;
          sf_config_replace(fddev, rule);
        }
        sprintf(tmp, "changing notification level to %i", i);
        flogf(fdlog, lstime, tmp);
      }
    }
  }
 
  fputs("sf_daemon: This should never happen!\n", stderr);
  exit(1);
}


/*
 * Clean up before terminating
 */

static void cleanup(int sig)
{
  close(fddev);

  update_log();

  get_time();
  
  fprintf(fdlog, "%s daemon terminated by signal %i\n", lstime, sig);
  fclose(fdlog);

  /* give up lock */
  flock(fdlock, LOCK_UN|LOCK_NB);
  close(fdlock);
 
  exit(0);  
}


/*
 * SIGUSR2: Fork process and write configuration data to named pipe.
 * This function fork()s and thereby provides a `snapshot'.
 */

static void process_signal(void) 
{
  signal_pending = 0;

  if (fork() == 0) { /* child */
    int fd, i;
    struct sf_fw *r;
    struct sf_fw rt;

    prepare_for_exec(); /* not really, but we want to close all files */

    /* open the firewall named pipe */
    if ((fd = open(NPIPE, O_WRONLY)) < 0)
      exit(1);
    /* write the address array */
    write(fd, &sf_addr_free, sizeof(sf_addr_free));
    write(fd, sf_addr,       sizeof(sf_addr));

    /* write all static rules */
    for (r = rules; r != NULL; r = r->fw_next) {
      memcpy(&rt, r, sizeof(rt));
      if (r->level.ptr != NULL)
        rt.level.num = r->level.ptr->n_level;

      write(fd, &r,  sizeof(r));
      write(fd, &rt, sizeof(rt));
    }
    write(fd, &r, sizeof(r));

    /* write all dynamic rules */
    for (r = dynrules; r != NULL; r = r->fw_next) {
      memcpy(&rt, r, sizeof(rt));
      if (r->level.ptr != NULL)
        rt.level.num = r->level.ptr->n_level;

      write(fd, &r,  sizeof(r));
      write(fd, &rt, sizeof(rt));
    }
    write(fd, &r, sizeof(r));


    /* write variables and values */
    write(fd, &sf_var_free, sizeof(sf_var_free));
    write(fd, variables,    sizeof(variables));
    for (i = 0; i < SF_VAR_CNT_MAX; i++) {
      struct variable *v;

      v = variables[i].next;
      while (v != NULL) {
        write(fd, &v, sizeof(v));
        write(fd, v,  sizeof(*v));
        v = v->next;
      }
      write(fd, &v, sizeof(v));
    }
    for (i = 0; i < SF_VAR_CNT_MAX; i++) {
      if (varnames[i] == NULL)
        write(fd, &(varnames[i]), sizeof(char));
      else 
        write(fd, varnames[i], strlen(varnames[i])+1);
    }

    /* close the firewall named pipe */
    close(fd);
    exit(0);    
  }
}

static void catch_usr2(int sig) /* The signal handler */
{
  /* re-set signal handler */
  signal(SIGUSR2, catch_usr2);

  if (busy)
    signal_pending = 1;
  else
    process_signal();
}


/*
 * Initialize the daemon 
 */

void init_log(void)
{
  /* Get the current time first and convert it to ASCII.
     Both values are stored in static variables, because
     the time is needed so very often. */
  get_time();

  if (sf_configure_filter() != 0) {
    fputs("Error configuring the kernel filter module!\n", stderr);
    exit(1);
  }
 
  if ((fddev = open(sf_devname,O_RDWR)) < 0) {
    perror("Cannot open the firewall device");
    exit(1);
  }
}


/*
 * Start the daemon 
 */

void start_log(char *config) 
{ 
  int i;
 
  /* Catching all signals is required, because we want to flush 
     the log file before terminating and record the time */

  for (i = 1; i < NSIG; i++)
    signal(i, cleanup);

  /* Important! Ignore the return code of any children. This
     process never wait()s for the termination of its children,
     so the children would end up as zombies using up the
     process table slots. Explicitly ignoring the SIGCHLD signal
     avoids the zombies */
  signal(SIGCHLD, SIG_IGN);

  /* Set alarm */
  signal(SIGALRM, catch_alarm);

  /* SIGUSR2: Write config to named pipe */
  signal(SIGUSR2, catch_usr2);

  /* Other signals to ignore */
  signal(SIGURG,  SIG_IGN);
  signal(SIGCONT, SIG_IGN);
  signal(SIGIO,   SIG_IGN);

  connect_logs(config);
  packet_loop();

  exit(1); 
}

