/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * sf firewall Counter Intelligence Module
 *
 * Some people consider this type of action unethical. However,
 * spy() will only be activated if explicitly requested by the 
 * `spy' keyword in the firewall configuration file. Thus the
 * decision is left to you.
 *
 * $Id: sf_spy.c,v 1.15 1995/08/08 13:42:14 robby Rel $
 */

#include <ctype.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <sys/ioctl.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "sf_custom.h"
#include "sf_global.h"
#include "sf_config.h"
#include "sf_daemon.h"
#include "sf_spy.h"

/*
 * Note: Since all actions are triggered automatically, you should
 *       think twice before adding anything to this module...
 *
 * To avoid a finger war, a spy() will only be done if there has not been
 * a spy() to this particular host for SPY_TIMEOUT (i.e. 5 minutes). While
 * the timeout cannot prevent annoying finger battles, one finger per five 
 * minutes will not escalate into a war. 
 */

#define MAX_EXEC_TIME 60  /* each indivual exec (e.g. finger) may take 1 minute */
#define MAX_TIME    5*60  /* this timeout applies to the spying process itself */
#define MAX_OUTPUT  8192  /* maximum amount of information from programs */


/*
 * This struct is used to remember which hosts we spied on lately
 */

struct spy_struc {
  struct spy_struc *next;
  time_t expires;
  unsigned long host;
};

static struct spy_struc *history = NULL;   /* root for the struct */
static int exec_pid = 0;                   /* remember pid to kill */


/*
 * Kill an exec'ed process if it lasts to long. Invoked by the event manager.
 */

void kill_exec(int sig)
{
  kill(exec_pid, SIGKILL);
}


/*
 * Exec a program and write stdout/stderr to fd. This function converts
 * control characters to avoid interpretation by the shell if a user
 * cat's the log file. It also limits the information to MAX_OUTPUT
 * bytes, inserts line feeds when needed and ignores empty lines.
 */

void fork_exec_print(FILE *fd[], int num_fd, char *arg0, char *arg1, char *arg2, char *arg3, char *arg4)
{
  int filedes[2], pid, got, i, j, t, c = 0;
  FILE *fdout;
  char buffer[77];

  if (pipe(filedes) < 0)
    return;

  pid = fork();
  if (pid < 0)
    return;

  if (pid == 0) { /* child */

    /* redirect stdout */
    close(STDOUT_FILENO); dup(filedes[1]); close(filedes[1]); close(filedes[0]);
    /* and stderr */
    close(STDERR_FILENO); dup(STDOUT_FILENO);

    execl(arg0, arg0, arg1, arg2, arg3, arg4, NULL);

    perror("Cannot exec program");
    exit(1);
  }
  /* parent */
  close(filedes[1]);

  /* ensure the child won't take forever */
  exec_pid = pid;
  signal(SIGALRM, kill_exec);
  alarm(MAX_EXEC_TIME);

  /* read output */
  fdout = fdopen(filedes[0], "r");
  got   = 0;
  while ((!feof(fdout)) && (got < MAX_OUTPUT)) {
    buffer[0] = 0;
    fgets(buffer, sizeof(buffer), fdout);

    if ((buffer[0] != 0) && (buffer[0] != '\n')) {
      for (j = 0; j < num_fd; j++)
        fprintf(fd[j], "  ");

      for (t = i = 0; i < strlen(buffer); i++) { /* inspired by safe_finger */
        c = buffer[i]; t++;
        if (isascii(c) && (isprint(c) || (c == '\n'))) { /* text */
          for (j = 0; j < num_fd; j++) {
            if (c == '\\')
              putc(c,fd[j]);
            putc(c,fd[j]);
          }
        } 
        else if (c == '\t') { /* tab */
          for (j = 0; j < num_fd; j++)
            putc(' ',fd[j]);
          for (; t % 8 != 0; t++)
            for (j = 0; j < num_fd; j++)
              putc(' ',fd[j]); 
        }
        else /* quote all other trash */
          for (j = 0; j < num_fd; j++)
            fprintf(fd[j],"\\%03o", c & 0377);
      }
      if (c != '\n')
        for (j = 0; j < num_fd; j++)
          putc('\n',fd[j]);

      got += i;
    }
  }
  fclose(fdout);

  /* cancel the alarm */
  alarm(0);
  
  if (!feof(fdout)) 
    kill(pid, SIGKILL);     /* kill process if it returns too much data */

  waitpid(pid, NULL, 0);    /* fetch return code and forget it */

  for (j = 0; j < num_fd; j++)
    fflush(fd[j]);
}


/*
 * `finger', `ruser' and `sfident' a host. This function also uses 
 * gethostbyaddr() and gethostbyname() to inverse map addresses. 
 */

void finger(struct iphdr *ip, char *msg, char *optmsg, char *mail_to)
{
  int i, j, num_fd = 1;
  FILE *fdarray[2];
  char target[30], buffer[50];
  time_t ltime;
  struct in_addr shost = *((struct in_addr *) &(ip->saddr));
  struct hostent *h_ent;

  time(&ltime);

  fdarray[0] = fopen(SPY_LOG, "a");
  if (fdarray[0] == NULL) 
    return;

  /* start a mail process if requested */
  if (mail_to != NULL) {
    num_fd = 2; fdarray[1] = mail(mail_to);
  }

  /* Note: You may easily add any number of file handles, as long as you
     increase the size of the fdarray. This allows flexible extensions of the 
     alarming scheme (we only use two descriptors) */

  target[0] = '@'; 
  strcpy(&(target[1]), inet_ntoa(shost));

  /* format the time */
  strftime(buffer, sizeof(buffer), TIME_FORMAT, localtime(&ltime));
 
  /* don't print subject line to log file */ 
  for (i = 1; i < num_fd; i++) {
#ifdef PRIORITY
    fputs("Priority: " PRIORITY "\n", fdarray[i]);
#endif
#ifdef REPLY_TO
    fputs("Reply-To: " REPLY_TO "\n", fdarray[i]);
#endif
    fprintf(fdarray[i], "Subject: Counter intelligence, %s\n\n", buffer);
  }
  for (i = 0; i < num_fd; i++) { 
    fprintf(fdarray[i], "FIREWALL COUNTER INTELLIGENCE REPORT, %s\n\n", buffer);

    if (optmsg != NULL)
      fprintf(fdarray[i], "%s\n\n", optmsg);

    fprintf(fdarray[i], "Triggered by: %s\n", msg);
    fprintf(fdarray[i], "Host address: %s\n", &(target[1]));
  }

  /* get host name */
  if ((h_ent = gethostbyaddr((char *) &shost, sizeof(shost), AF_INET)) == NULL) 
    for (i = 0; i < num_fd; i++)
      fprintf(fdarray[i], "Host name:    -- unknown --\n\n");
  else {
    for (i = 0; i < num_fd; i++)
      fprintf(fdarray[i], "Host name:    %s\n", h_ent->h_name);
    if (h_ent->h_aliases[0] != NULL) {
      for (j = 0; h_ent->h_aliases[j] != NULL; j++) {
        for (i = 0; i < num_fd; i++) {
          if (j == 0)
            fprintf(fdarray[i], "Alias:        ");
          else
            fprintf(fdarray[i], "              ");
          fprintf(fdarray[i], "%s\n", h_ent->h_aliases[j]);
        }
      }
    }

    /* inverse map the address */
    strcpy(buffer, h_ent->h_name);
    if ((h_ent = gethostbyname(buffer)) == NULL) /* name not found */
      for (i = 0; i < num_fd; i++)
        fprintf(fdarray[i], "WARNING:      The host name is not listed!\n");
    else {
      for (j = 0; h_ent->h_addr_list[j] != NULL; j++) 
        if (*((unsigned long *) h_ent->h_addr_list[j]) == *((unsigned long *) &shost))
          break;

      if (h_ent->h_addr_list[j] == NULL) /* address not found */
        for (i = 0; i < num_fd; i++)
          fprintf(fdarray[i], "WARNING:      The host address does not match the host name!\n");
    }
  }

  /* Try to get the user name via the ident protocol.
     The ident protocol (cf. RFC 1413) is only defined for TCP connections
     to the local host. We have to ensure that this is the case.

     We are doing the ident query first, because finger and rusers might
     trigger alarms at the other host themselves and we might not get usable
     output from finger and rusers.

     To find out whether an address is local, we loop over all available
     addresses.
  */
  if (ip->protocol == IPPROTO_TCP) {
    char sport[8], dport[8];
    struct tcphdr *tcp;
    int fd, pos, k;
    struct ifconf ifc;
    struct ifreq  ifr[MAX_IF];

    /* is the destination IP address local? */
    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) >= 0) {

      ifc.ifc_len = sizeof(ifr);
      ifc.ifc_req = ifr;

      if (ioctl(fd, SIOCGIFCONF, &ifc) >= 0) {

        for (pos = k = 0; pos < ifc.ifc_len; pos += sizeof(struct ifreq), k++) {

          if (ip->daddr == *((unsigned long *) &((((struct sockaddr_in *)
                            &(ifr[k].ifr_addr))->sin_addr.s_addr))))
            break;
        }

        if (pos < ifc.ifc_len) { /* Yes! Address is local. */

          char us[30];
          struct in_addr ourhost = *((struct in_addr *) &(ip->daddr));

          strcpy(us, inet_ntoa(ourhost));

          tcp = (struct tcphdr *) ((unsigned long *)ip + ip->ihl);
          sprintf(sport,"%i",ntohs(tcp->th_sport));
          sprintf(dport,"%i",ntohs(tcp->th_dport));
          for (i = 0; i < num_fd; i++)
            fprintf(fdarray[i],"\nidentd information:\n");
          fork_exec_print(fdarray, num_fd, SFIDENT, &(target[1]), sport, us, dport);
        }
      }
      close(fd);
    }
  } /* end of ident query */

  /* finger */
  for (i = 0; i < num_fd; i++)
    fprintf(fdarray[i],"\nfinger information:\n");
  fork_exec_print(fdarray, num_fd, FINGER, FINGER_ARG, target, NULL, NULL);

#ifdef SPY_USING_RUSERS
  /* rusers -- may be #undefined, because most hosts don't like rusers */
  for (i = 0; i < num_fd; i++)
    fprintf(fdarray[i],"\nrusers information:\n");
  fork_exec_print(fdarray, num_fd, RUSERS, RUSERS_ARG, &(target[1]), NULL, NULL);
#endif

  /* bye, and thanks for all the fish... */
  for (i = 0; i < num_fd; i++) {
    fprintf(fdarray[i],"\n");
    fclose(fdarray[i]);
  }
}


/*
 * Kill a process by PID. This function is called by the event manager.
 */

void kill_spy(int pid, int dummy)
{
  kill(pid, SIGKILL); /* ignore the error if the process has terminated before */
}


/*
 * Spy on a given host. If the host has been spied on in the last
 * TIMEOUT minutes, return. Else fork() a finger() process.
 */

int spy(struct iphdr *ip, char *msg, char *optmsg, char *mail_to)
{
  int pid;
  time_t ltime;
  struct spy_struc *h;
  unsigned long host = (unsigned long) ip->saddr;

  time(&ltime);

  /* spy()ed on this host in the past few minutes? */
  for (h = history; h != NULL; h = h->next)
    if ((h->host == host) && (h->expires > ltime))
      return 0;

  /* no, remember the host's name and the timeout */
  if (history == NULL) {
    if ((h = history = malloc(sizeof(struct spy_struc))) == NULL)
      return 0;
    h->next = NULL;
  }
  else {
    h = history;
    while ((h->next != NULL) && (h->expires > ltime))
      h = h->next;
    if (h->next == NULL) {
      if ((h->next = malloc(sizeof(struct spy_struc))) == NULL)
        return 0;
      h = h->next;
      h->next = NULL;
    }
  }

  h->expires = ltime+SPY_TIMEOUT;
  h->host    = host;

  /* fork process, because spying can take rather a long time */
  pid = fork();
  if (pid < 0)
    return 0;

  if (pid == 0) { /* child process */
   
    prepare_for_exec(); /* give up logfiles and the firewall device */

    /* finger, ruser and identd a host */
    finger(ip, msg, optmsg, mail_to);

    exit(0); 
  }

  /* Ensure the child process will be killed if it takes to long.
     This is brute force. But it is the only way if one of the gethostby... 
     calls hangs. After all, someone might have tampered with the DNS */
  add_event(kill_spy, pid, 0, MAX_TIME);

  return 1;
}
