/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * sf Firewall Daemon Startup/sfc User Control Module
 *
 * $Id: sfc.c,v 1.15 1995/09/12 15:08:59 robby Rel $
 */

#include <netdb.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <linux/netdevice.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/types.h>
#include "sf_custom.h"
#include "sf_global.h"
#include "sf_config.h"
#include "sf_daemon.h"
#include "sf_kernel.h"
#include "sf_rc.h"
#include "sf_proto.h"

static int  running(void);
static void showcfg(void);
static char *arg0;
static void print_addr(unsigned long, unsigned long, int, char *, int);


/*
 * Show the syntax.
 */

static void show_syntax(void)
{
  fprintf(stderr, "\nUsage: %s command, where command is one of:\n\n" \
    "start [config]         Start the firewall daemon and configure it.\n" \
    "                       Use `config' to specify a configuration file\n" \
    "                       (default: " DEFAULT_CONFIG")\n\n" \
    "stop                   Signal the firewall daemon until it stops.\n" \
    "                       Note! To disable the firewall, you must unload\n" \
    "                       the kernel filter module.\n\n" \
    "reconfig [config]      Stop the firewall daemon, and restart it with a new\n" \
    " [flush or flush_all]  configuration file (default: " DEFAULT_CONFIG ").\n" \
    "                       `reconfig' does NOT stop any existing TCP connections,\n" \
    "                       unless `flush' or `flush_all' is specified.\n" \
    "                       `flush' ends all existing TCP connections which are\n" \
    "                       not allowed under the new configuration.\n" \
    "                       `flush_all' ends all existing TCP connections.\n\n" \
    "checkconfig [config]   Check a configuration file for syntax errors.\n" \
    "                       (default: " DEFAULT_CONFIG ")\n\n" \
    "show                   Show the current configuration.\n\n", arg0);

  exit(1);
}


/*
 * killproc() - kill a running process 
 */

void killproc(void) 
{ 
  int pid;

  if (!(pid = running())) {
    fprintf(stderr, "The firewall daemon is not running!\n");
    exit(1);
  }
  kill(pid, SIGTERM); sleep(1);
  if (running()) {
    fputs("The daemon is still running, waiting 5 seconds more...\n", stderr);
    sleep(5);
    kill(pid, SIGTERM); sleep(3);
    if (running()) {
      fputs("The process is still running, sending SIGKILL.\n", stderr);
      kill(pid, SIGKILL);
    }
  } 
}


/*
 * main() - check command line arguments and process them 
 */

#define A_STOP      0x01
#define A_START     0x02
#define A_FLUSH     0x04
#define A_FLUSH_ALL 0x08
#define A_SHOW      0x10
#define A_CHECKCFG  0x20


int main(int argc, char **argv)
{
  int pid, fd, log;
  char buf[20];
  char *config = DEFAULT_CONFIG;
  int action = 0;
  struct passwd *pwd = NULL;

  memset(sf_addr, 0, sizeof(sf_addr));


  /* don't run daemon as a super user */
  if ((getuid() == 0) || (geteuid() == 0)) 
    if (!(pwd = getpwnam(UNPRIV_NAME)) || (pwd->pw_uid <= 0)) {
      perror("Please create an unprivileged user named `firewall'.");
      exit(1);
    }

  arg0 = argv[0];

  /* check arguments */
  if (argc < 2)
    show_syntax();

  if (strcmp("start", argv[1]) == 0) {
    if (argc > 3)
      show_syntax();
    else if (argc == 3)
      config = argv[2];
    action = A_START;
  }
  else if (strcmp("stop", argv[1]) == 0) {
    if (argc != 2)
      show_syntax();
    action = A_STOP;
  }
  else if (strcmp("reconfig", argv[1]) == 0) {
    action = A_STOP | A_START;
    if (argc > 4)
      show_syntax();
    else if (argc == 3) {
      if (strcmp(argv[2], "flush") == 0)
        action |= A_FLUSH;
      else if (strcmp(argv[2], "flush_all") == 0)
        action |= A_FLUSH_ALL;
      else
        config = argv[2];
    }
    else if (argc == 4) {
      if (strcmp(argv[2], "flush") == 0) {
        action |= A_FLUSH;
        config = argv[3];
      }
      else if (strcmp(argv[2], "flush_all") == 0) {
        action |= A_FLUSH_ALL;
        config = argv[3];
      }
      else if (strcmp(argv[3], "flush") == 0) {
        action |= A_FLUSH;
        config = argv[2];
      }
      else if (strcmp(argv[3], "flush_all") == 0) {
        action |= A_FLUSH_ALL;
        config = argv[2];
      }
      else
        show_syntax();
    }
  }
  else if (strcmp("show", argv[1]) == 0) {
    if (argc != 2)
      show_syntax();
    action = A_SHOW;
  }
  else if ((strcmp("checkconfig", argv[1]) == 0) || (strcmp("check", argv[1]) == 0)) {
    if (argc > 3)
      show_syntax();
    else if (argc == 3)
      config = argv[2];
    action = A_CHECKCFG;
  }
  else
    show_syntax();


  /* parse config file first */
  if ((action & (A_START|A_CHECKCFG)) != 0) {
    if (sf_read_config_file(config) != 0) {
      fprintf(stderr, "Error parsing the configuration file `%s'!\n", config);
      exit(1);
    }
    else if ((action & A_CHECKCFG) == A_CHECKCFG)
      fprintf(stderr, "Parsing `%s': OK\n", config);
  }


  /* kill the process if requested */
  if ((action & A_STOP) == A_STOP) {
    killproc();
    unlink(PID_FILE);
  }


  if ((action & A_SHOW) == A_SHOW) 
    showcfg();


  /* create a new process */
  if ((action & A_START) == A_START) {
    /* check whether process is already or still running */
    if ((pid = running()) != 0) {
      fprintf(stderr, "The firewall daemon is already running (pid %i)!\n", pid);
      exit(1);
    }

    /* create log files and change owner if possible or if needed */
    if ((log = open(LOG, O_CREAT|O_APPEND|O_RDWR, 0600)) < 0) {
      perror("Cannot open log file");
      exit(1);
    }
    if ((getuid() == 0) || (geteuid() == 0))
      fchown(log, pwd->pw_uid, pwd->pw_gid);
    close(log);

    if ((log = open(SPY_LOG, O_CREAT|O_APPEND|O_RDWR, 0600)) < 0) {
      perror("Cannot open log file");
      exit(1);
    }
    if ((getuid() == 0) || (geteuid() == 0))
      fchown(log, pwd->pw_uid, pwd->pw_gid);
    close(log);

    init_log();

    /* disconnect tcp sessions which are not allowed under the new confiuration */
    if ((action & A_FLUSH) == A_FLUSH) 
      sf_config_flush(fddev);
  
    /* disconnect all tcp sessions */
    if ((action & A_FLUSH_ALL) == A_FLUSH_ALL)
      sf_config_flush_all(fddev);

    /* create the /etc/firewall.pid file and lock it */
    if ((fd = creat(PID_FILE, 0600)) < 0) {
      perror("Cannot create pid file");
      exit(1);
    }

    if ((flock(fd, LOCK_EX|LOCK_NB)) < 0) {
      perror("Cannot create lock");
      exit(1);
    }


    if ((pid = fork()) == 0) { /* child */

      close(fd);

      fdlock = open(PID_FILE, O_RDWR);
      flock(fdlock, LOCK_EX); /* wait until parent has given up the lock */

      /* give up super user privileges */
      if ((getuid() == 0) || (geteuid() == 0)) {
        setgid(pwd->pw_gid);
        setuid(pwd->pw_uid); /* how sad - all priviliges lost forever */
      }

      /* become a daemon */
      setsid();   /* disconnect from the terminal */
      chdir("/"); /* just in case someone wants to unmount the current dir. */

      /* start the daemon - this call should not return */
      start_log(config);

      exit(1); /* but you never know */
    }
    if (pid < 0) {
      perror("Cannot fork()");
      exit(1);
    }

    /* record the child's pid */
    sprintf(buf, "%i\n", pid);
    if (write(fd, buf, strlen(buf)) < 0) {
      perror("Cannot write to lock file");
      exit(1);
    }
      
    /* give up lock so child will run */
    flock(fd, LOCK_UN);
    close(fd);
  }

  return 0;
}


/*
 * See if daemon is running and fetch the pid. This function uses
 * a pid file, which is locked by the daemon if it is alive.
 */

int running(void)
{
  struct stat statbuf;
  int fd, pid;
  FILE *fdf;

  /* does the pid file exist? */
  if (stat(PID_FILE, &statbuf) < 0)
    return 0; /* no, the daemon can't be running */

  /* is it a regular file? */
  if (!S_ISREG(statbuf.st_mode)) {
    fprintf(stderr, "The file %s is not a regular file. sfc cannot continue.\n", PID_FILE);
    exit(1);
  }

  /* open the file and lock it */
  if ((fd = open(PID_FILE, O_RDWR)) < 0) {
    fprintf(stderr, "Cannot open the log file %s. sfc cannot continue.\n", PID_FILE);
    exit(1);
  }

  /* get lock */
  if (flock(fd, LOCK_NB|LOCK_EX) >= 0) { /* ok - lock process cannot be running */
    flock(fd, LOCK_NB|LOCK_UN);
    close(fd);
    return 0;
  }
  close(fd);
   
  /* didn't get lock, so the daemon is alive (cross your fingers) */
  
  /* read pid from file */
  if ((fdf = fopen(PID_FILE, "r")) == NULL) {
    fprintf(stderr, "Cannot open the log file %s. sfc cannot continue.\n", PID_FILE);
    exit(1);
  }

  pid = 0; 
  fscanf(fdf, "%i", &pid);
  fclose(fdf);
  
  return pid;
}


/*
 * Data structures and functions for pointer conversion: daemon to sfc
 */

#define M 277 /* some prime number */

unsigned int hash(void *p) /* silly simple hash function */
{
  return ((unsigned) p) % M;
}

struct hashlist {
  void *old;
  void *new;
  struct hashlist *next;
};

struct hashlist ahash[M];

void *convertp(void *p) /* converts a pointer from old to new */
{
  struct hashlist *hl;

  if (p == NULL)
    return NULL;

  hl = &(ahash[hash(p)]);

  while ((hl != NULL) && (hl->old != p))
    hl = hl->next;

  if (hl == NULL) {
    fprintf(stderr, "Upps... Cannot convert this pointer!?!\n");
    exit(1);
  }

  return hl->new;
}

void addp(void *old, void *new) /* add pointer to the hash structure */
{
  unsigned int h;

  if (old == NULL)
    return;

  h = hash(old);
  if (ahash[h].old == NULL) {
    ahash[h].old = old;
    ahash[h].new = new;
  }
  else {
    struct hashlist *hl;

    hl = &(ahash[h]);
    while (hl->next != NULL)
      hl = hl->next;
    hl->next = malloc(sizeof(struct hashlist));
    hl       = hl->next;
    hl->old  = old;
    hl->new  = new;
    hl->next = NULL;
  }
} 

/*
 * Show the current configuration 
 */

static void showcfg(void)
{
  char buffer[255];
  struct sf_fw *rules, *dynrules, *r;
  struct sf_fw krule;
  int fd, result, nrule, pid, i;
  time_t ltime;

  if (!(pid = running())) {
    fprintf(stderr, "The firewall daemon is not running!\n");
    exit(1);
  }

  memset(ahash, 0, sizeof(ahash));

  /* First get the data from the daemon */
  kill(pid, SIGUSR2); /* signal the process to write its data to the pipe */

  if ((fd = open(NPIPE, O_RDONLY)) < 0) {
    perror("Cannot open the named pipe for reading!");
    exit(1);
  }

  /* get the address array */
  read(fd, &sf_addr_free, sizeof(sf_addr_free));
  read(fd, sf_addr,       sizeof(sf_addr));
  
  rules         = 
  dynrules      = NULL;

  /* get all static rules */
  read(fd, &r, sizeof(r));
  while (r != NULL) {
    if (rules == NULL) 
      rules          = malloc(sizeof(struct sf_fw));
    else {
      rules->fw_next = malloc(sizeof(struct sf_fw));
      rules          = rules->fw_next;
    }

    addp(r, rules);

    read(fd, rules, sizeof(*rules));
    read(fd, &r,    sizeof(r));
  }
  if (rules != NULL)
    rules->fw_next = NULL;

  /* get all dynamic rules */
  read(fd, &r, sizeof(r));
  while (r != NULL) {
    if (dynrules == NULL)
      dynrules          = malloc(sizeof(struct sf_fw));
    else {
      dynrules->fw_next = malloc(sizeof(struct sf_fw));
      dynrules          = dynrules->fw_next;
    }

    addp(r, dynrules);

    read(fd, dynrules, sizeof(*dynrules));
    read(fd, &r,       sizeof(r));
  }
  if (dynrules != NULL)
    dynrules->fw_next = NULL;

  /* read variables and values */
  read(fd, &sf_var_free, sizeof(sf_var_free));
  read(fd, variables,    sizeof(variables));

  for (i = 0; i < SF_VAR_CNT_MAX; i++) {
    struct variable *v, *w;

    w = &(variables[i]);

    read(fd, &v, sizeof(v));
    while (v != NULL) {
      w->next = malloc(sizeof(struct variable));
      w       = w->next;
      
      read(fd, w,  sizeof(*w));
      read(fd, &v, sizeof(v));
    }
    w->next = NULL;
  }

  for (i = 0; i < SF_VAR_CNT_MAX; i++) {
    char buffer[4096];
    char c, *bufp;

    bufp = buffer;
    do {
      read(fd, &c, sizeof(c));
      *bufp = c;
      bufp++;
    } while (c != 0);
    if (*buffer)
      varnames[i] = strdup(buffer);
    else
      varnames[i] = NULL;
  }

  close(fd);
    

  if ((fd = open(sf_devname, O_RDONLY)) < 0) {
    perror("Cannot open the device for reading!");
    exit(1);
  }

  printf("Active rules:\n");
  printf("-------------\n");

  nrule = 0;
  do {
    result = read(fd, &krule, sizeof(krule));
    if (result < 0) {
      perror("Error reading the device!");
      exit(1);
    }
    if (result > 0) {
      struct sf_fw *rule;
      int tcp_or_udp = 0;

      nrule++;
      rule = convertp(krule.rule_id.ptr);

      printf("\n%3i (line %3i) ", nrule, krule.fw_line);

      if (krule.timeout == SF_RULE_STATIC)
        printf("static"); 
      else {
        strftime(buffer, sizeof(buffer), TIME_FORMAT, localtime(&(rule->timeout)));
        printf("dynamic, timeout %s ", buffer); 
      }

      printf("\n    ");
      switch (krule.fw_rc) {
        case SF_RC_ACCEPT : printf("accept"); 
        break;
        case SF_RC_BLOCK  : printf("block"); 
        break;
        case SF_RC_RNET   : printf("reject (net unreachable)");
        break;
        case SF_RC_RHOST  : printf("reject (host unreachable)"); 
        break;
        case SF_RC_RPROTO : printf("reject (protocol unreachable)"); 
        break;
        case SF_RC_RPORT  : printf("reject (port unreachable)"); 
        break;
      } 
      printf(", notification level %i\n", ((krule.fw_flags & SF_FW_LOG) == 0) ? 0 : rule->level.num);

      if ((krule.fw_flags & SF_FW_PROT) != SF_FW_PROT_ALL) {
        if ((krule.fw_flags & SF_FW_PROT) == SF_FW_PROT_RIP) {
          printf("    protocol RIP"); 
          if (krule.fw_rip_cnt > 0) {
            printf(", ");
            if ((krule.fw_flags & SF_RIP_ADDR_NEG) != 0) {
              printf("not ");
              print_addr(krule.fw_rip_idx, krule.fw_rip_cnt, 
                         0, "                   not ", 0);
            }
            else
              print_addr(krule.fw_rip_idx, krule.fw_rip_cnt,
                         0, "                   ", 0);
          }
          else
            printf("\n");
        }
        else {
          if (krule.protocol == IPPROTO_ICMP) {
            printf("    protocol ICMP");
            if ((krule.fw_flags & SF_TYPE_MASK) != SF_ICMP_ALLTYPES) {
              if ((krule.fw_flags & SF_ICMP_ECHOREPLY) != 0)
                printf(" `echo reply'");
              if ((krule.fw_flags & SF_ICMP_DEST_UNREACH) != 0)
                printf(" `dest. unreachable'");
              if ((krule.fw_flags & SF_ICMP_SOURCE_QUENCH) != 0)
                printf(" `source quench'");
              if ((krule.fw_flags & SF_ICMP_REDIRECT) != 0)
                printf(" `redirect'");
              if ((krule.fw_flags & SF_ICMP_ECHO) != 0)
                printf(" `echo request'");
              if ((krule.fw_flags & SF_ICMP_TIME_EXCEEDED) != 0)
                printf(" `time exceeded'");
              if ((krule.fw_flags & SF_ICMP_PARAMETERPROB) != 0)
                printf(" `parameter problem'");
              if ((krule.fw_flags & SF_ICMP_TIMESTAMP) != 0)
                printf(" `timestamp request'");
              if ((krule.fw_flags & SF_ICMP_TIMESTAMPREPLY) != 0)
                printf(" `timestamp reply'");
              if ((krule.fw_flags & SF_ICMP_INFO_REQUEST) != 0)
                printf(" `information request'");
              if ((krule.fw_flags & SF_ICMP_INFO_REPLY) != 0)
                printf(" `information reply'");
              if ((krule.fw_flags & SF_ICMP_ADDRESS) != 0)
                printf(" `address mask request'");
              if ((krule.fw_flags & SF_ICMP_ADDRESSREPLY) != 0)
                printf(" `address mask reply'");
            }
            printf("\n");
          }
          else if (krule.protocol == IPPROTO_IGMP) {
            printf("    protocol IGMP");
            if ((krule.fw_flags & SF_TYPE_MASK) != SF_IGMP_ALLTYPES) {
              if ((krule.fw_flags & SF_IGMP_HOST_MEMBERSHIP_QUERY) != 0)
                printf(" `membership query'");
              if ((krule.fw_flags & SF_IGMP_HOST_MEMBERSHIP_REPORT) != 0)
                printf(" `membership report'");
              if ((krule.fw_flags & SF_IGMP_HOST_LEAVE_MESSAGE) != 0)
                printf(" `leave message'");
            }
            printf("\n");
          }
          else {
            printf("    protocol %i (", krule.protocol);
            if (krule.protocol < MAX_PROTOCOL)
              printf("%s)\n", protocol[krule.protocol]);
            else
              printf("Unknown)\n");

            if ((krule.protocol == IPPROTO_TCP) || (krule.protocol == IPPROTO_UDP))
              tcp_or_udp = 1;
          }
        }
      }

      if ((krule.fw_flags & SF_FW_CHECK_TTL) != 0) {
        printf("    ttl field ");
        if ((krule.fw_flags & SF_FW_TTL) == SF_FW_TTL_EQUAL)
          printf("-");
        else if ((krule.fw_flags & SF_FW_TTL) == SF_FW_TTL_LESS)
          printf("<");
        else if ((krule.fw_flags & SF_FW_TTL) == SF_FW_TTL_GREATER)
          printf(">");
        else if ((krule.fw_flags & SF_FW_TTL) == SF_FW_TTL_NOTEQUAL)
          printf("!=");
        else
          printf("?");
        printf(" %i\n", krule.ttl);
      }

      if ((krule.fw_flags & SF_FW_CHECK_OPT) != 0) {
        printf("    options: ");
        if ((krule.fw_flags & SF_FW_OPT_RR) != 0)
          printf("`record route' ");
        if ((krule.fw_flags & SF_FW_OPT_TS) != 0)
          printf("`time stamp' ");
        if ((krule.fw_flags & SF_FW_OPT_SEC) != 0)
          printf("`security' ");
        if ((krule.fw_flags & SF_FW_OPT_LSR) != 0)
          printf("`loose source route' ");
        if ((krule.fw_flags & SF_FW_OPT_SSR) != 0)
          printf("`strict source route' ");
        if ((krule.fw_flags & SF_FW_OPT_SATID) != 0)
          printf("`sat id'");
        printf("\n");
      }

      if (krule.fw_src_cnt) {
        if ((krule.fw_flags & SF_FW_SRC_NEG) != 0)
          printf("    not from ");
        else
          printf("    from     "); 
        print_addr(krule.fw_src_idx, krule.fw_src_cnt, tcp_or_udp, 
                   "             ", krule.protocol);
      }
      if (krule.fw_dst_cnt) {
        if ((krule.fw_flags & SF_FW_DST_NEG) != 0)
          printf("    not to   "); 
        else
          printf("    to       ");
        print_addr(krule.fw_dst_idx, krule.fw_dst_cnt, tcp_or_udp, 
                   "             ", krule.protocol);
      }
    }
  } while (result > 0);
  close(fd);

  time(&ltime);

  printf("\n\nVariables:\n");
  printf("----------\n\n");

  for (i = 0; i < sf_var_free; i++) {
 
    if (varnames[i] != NULL) { 
      time_t timeout;

      timeout = variables[i].timeout;
      printf("%s = %li", varnames[i], (timeout > ltime) ? variables[i].value : 0);
      if ((timeout > ltime) && (timeout < INT_MAX)) {
        strftime(buffer, sizeof(buffer), TIME_FORMAT, localtime(&timeout));
        printf(", timeout %s\n", buffer);
      }
      else
        printf("\n");

      if (variables[i].next != NULL) {
        struct variable *v;
 
        v = variables[i].next;
        while (v != NULL) {
          timeout = v->timeout;
          if (timeout > ltime) {
            printf("    host %-15s = %li", 
                   inet_ntoa(*((struct in_addr *) &(v->address))), v->value);
            if ((timeout > ltime) && (timeout < INT_MAX)) {
              strftime(buffer, sizeof(buffer), TIME_FORMAT, localtime(&timeout));
              printf(", timeout %s\n", buffer);
            }
            else
              printf("\n");
          }

          v = v->next;
        }
      }
      
      printf("\n");
    }
  }
  printf("\n");
}

static void print_addr(unsigned long index, unsigned long count, 
                       int tcp_or_udp, char *s, int protocol)
{
  struct in_addr inadr;
  unsigned long k;

  for (k = index; k < (index+count); k++) {
    if (k > index)
      printf(s);
    if (sf_addr[k].mask != SF_IGNORE_ADDRESS) {
      inadr.s_addr = sf_addr[k].addr; printf(inet_ntoa(inadr));
      inadr.s_addr = sf_addr[k].mask; printf(" mask %s ", inet_ntoa(inadr));
    }
    if (tcp_or_udp && (sf_addr[k].port != SF_ALL_PORTS)) {
      printf("port %hu", sf_addr[k].port);
      if (sf_addr[k].prend != sf_addr[k].port) 
        printf("..%hu", sf_addr[k].prend);
      else {
        struct servent *serv_ent;
        char proto[4];

        if (protocol == IPPROTO_UDP)
          strcpy(proto, "udp");
        else
          strcpy(proto, "tcp");

        if ((serv_ent = getservbyport(htons(sf_addr[k].port), proto)) != NULL)
          printf(" (%s)", serv_ent->s_name);
      }
    }
    printf("\n");
  }
}

