/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * In this file all data structures holding the configuration
 * for the firewall daemon are declared. All functions concerning
 * the configuration are provided here.
 *
 * $Id: sf_config.c,v 1.29 1995/08/05 11:50:15 robby Rel $
 */

/*
 * configuration data for the daemon 
 */

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <sys/types.h>

#include "sf_global.h"
#include "sf_config.h"
#include "sf_control.h"
#include "sf_daemon.h"
#include "sf_custom.h"

/* rules chain */
struct sf_fw *rules = NULL;

/* firewall entries point to the following array containing the ip addresses */
struct sf_address	sf_addr[SF_ADDRESS_CNT_MAX];
int		sf_addr_free = 0;		/* first free entry in array */

/* name of firewall device in filesystem */
char		sf_devname[255] = DEVICE;

/* default mail address(es) */
char		*sf_mailaddr = NULL;

/* notification structure */
struct notification	*notify = NULL;

struct variable variables[SF_VAR_CNT_MAX];      /* array to store variable values */
char            *varnames[SF_VAR_CNT_MAX];      /* variable names */
int		sf_var_free = 0;		/* first free entry in array */


/*
 * Convert numeric levels to pointers 
 */

int convert_levels_notification(struct notification *n)
{
  struct notification *note;
  struct let_if_chain *ifch;

  if (n->n_relevel && n->n_relevel_val.num) {
    for (note = notify;
         (note != NULL) && (note->n_level != n->n_relevel_val.num);
         note = note->n_next) ;

    if (note == NULL) {
      fprintf(stderr, "Relevel statement: New level %i does not exist!\n",
              n->n_relevel_val.num);
      return -1;
    }
    n->n_relevel_val.ptr = note;
  }

  for (ifch = n->n_let_if; ifch != NULL; ifch = ifch->next) {
    if ( (ifch->statement_type == IF_STATEMENT) &&
         (convert_levels_notification(ifch->then) < 0) )
      return -1;

    if (ifch->statement_type == CALL_STATEMENT) {
      for (note = notify;
           (note != NULL) && (note->n_level != ifch->operator);
           note = note->n_next) ;

      if (note == NULL) {
        fprintf(stderr, "Call statement: Level %i does not exist!\n",
              ifch->operator);
        return -1;
      }
      ifch->operator = 0;
      ifch->then = note;
    }
  }

  return 0;
}

int convert_levels(void) 
{
  struct notification *note;
  struct sf_fw *rule;

  /* Set pointers in the sf_fw rule structures */
  for (rule=rules; rule!=NULL; rule=rule->fw_next) {
    if (rule->level.num) {
      for (note = notify; 
         (note != NULL) && (note->n_level != rule->level.num); 
         note = note->n_next) ;

      if (note == NULL) {
        fprintf(stderr, "Notification level %i does not exist!\n", rule->level.num);
        return -1;
      }

      rule->level.ptr = note;

    }
  }

  /* Set pointers from notification structure to sf_fw rules 
     and from `then' statements to sf_fw rules */
  for (note = notify; note != NULL; note = note->n_next) 
    if (convert_levels_notification(note) < 0)
      return -1;

  /* check whether a non-default spoof level exists */
  if ((notify != NULL) && (notify->n_level == SPOOF_LEVEL))
    rule_spoof.level.ptr = notify;

  /* and a non-default oversized packets level */
  if ((notify != NULL) && (notify->n_level == OVERSIZED_LEVEL))
    rule_oversized.level.ptr = notify;

  return 0;
}


/*
 * Read config file
 */

extern FILE *yyin;  /* lex input file */
int parseerror = 0;

int sf_read_config_file(char *filename)
{ extern int yyparse(void);
  int yyrc;

  if ( (yyin = fopen(filename,"r")) == NULL )
    { char errstr[300];
      sprintf(errstr,"Error opening %s",filename);
      perror(errstr);
      return -1;
    }

  parseerror = 0;
  yyrc = yyparse();

  fclose(yyin);

  if (!parseerror) parseerror = convert_levels() != 0;

  if (yyrc || parseerror) return -1;

  return 0;

}

/* 
 * pass all configuration data to the filter
 */

int sf_configure_filter(void)
{ int fd;
  struct sf_fw *rule;

  /* open firewall device for writing */
  if ((fd = open(sf_devname, O_WRONLY)) < 0)
    { if (errno == ENODEV)
        fputs("The module sf.o is not loaded!\n", stderr);
      else
        perror("Cannot open the firewall device"); 
      return -1;
    }

  /* reset filter configuration */
  if (sf_config_clear(fd) != 0)
    { perror("Error clearing filter configuration");
      close(fd);
      return -1;
    }

  /* pass filter rules */
  for (rule = rules; rule != NULL; rule=rule->fw_next)
    { if (rule->timeout == SF_RULE_STATIC)
        { if (sf_config_add(fd,rule) != 0)
            { perror("Error adding rule");
              (void)sf_config_clear(fd);
              close(fd);
              return -1;
            }
        }
    }
  
  /* pass address array */
  if (sf_config_addr(fd) != 0)
    { perror("Error passing address array");
      (void)sf_config_clear(fd);
      close(fd);
      return -1;
    }

  close(fd);

  return 0;
  
}
 
/*
 * functions for configuring the firewall
 */

int sf_config_flush(int fd)		/* abort all active TCP connections 
					   that are no longer allowed */
{ struct sf_control control;

  control.magic   = SF_CONTROL_MAGIC;
  control.command = SF_COMMAND_FLUSH;

  if (write(fd, &control, sizeof(control)) > 0)
    { return 0; }
   else
    { return -1; }
}

int sf_config_flush_all(int fd)		/* abort all active TCP connections */
{ struct sf_control control;

  control.magic   = SF_CONTROL_MAGIC;
  control.command = SF_COMMAND_FLUSH_ALL;

  if (write(fd, &control, sizeof(control)) > 0)
    { return 0; }
   else
    { return -1; }
}

int sf_config_clear(int fd)			/* clear all entries */
{ struct sf_control control;
  
  control.magic   = SF_CONTROL_MAGIC;
  control.command = SF_COMMAND_FW_CLEAR;

  if (write(fd, &control, sizeof(control)) > 0)
    { return 0; }
   else
    { return -1; }
}

int sf_config_add(int fd, struct sf_fw *entry)	/* add entry to firewall */
{ struct sf_control *control;
  int rc;

  if ( (control = (struct sf_control *) malloc(sizeof(struct sf_control)+
                                         sizeof(struct sf_fw))) == NULL)
    { return -1; }

  control->magic   = SF_CONTROL_MAGIC;
  control->command = SF_COMMAND_FW_ADD;
  memcpy(control->data,entry,sizeof(struct sf_fw));

  rc = write(fd, control, sizeof(struct sf_control)+sizeof(struct sf_fw));  
  free(control);

  if (rc > 0)
    { return 0; }
   else
    { return -1; }
}

int sf_config_replace(int fd, struct sf_fw *entry) /* replace entry identified by fw_id */
{ struct sf_control *control;
  int rc;

  if ( (control = (struct sf_control *) malloc(sizeof(struct sf_control)+
                                         sizeof(struct sf_fw))) == NULL)
    { return -1; }

  control->magic   = SF_CONTROL_MAGIC;
  control->command = SF_COMMAND_FW_REPLACE;
  memcpy(control->data,entry,sizeof(struct sf_fw));

  rc = write(fd, control, sizeof(struct sf_control)+sizeof(struct sf_fw));
  free(control);

  if (rc > 0)
    { return 0; }
   else
    { return -1; }
}

int sf_config_delete(int fd,unsigned long id)	/* delete entry identified by fw_id */
{ struct sf_control *control;
  int rc;

  if ( (control = (struct sf_control *) malloc(sizeof(struct sf_control)+
                                         sizeof(id))) == NULL)
    { return -1; }

  control->magic   = SF_CONTROL_MAGIC;
  control->command = SF_COMMAND_FW_DELETE;
  memcpy(control->data,&id,sizeof(id));

  rc = write(fd, control, sizeof(struct sf_control)+sizeof(id));
  free(control);

  if (rc > 0)
    { return 0; }
   else
    { return -1; }
}

int sf_config_addr(int fd)			/* update address array */
{ struct sf_control *control;
  int rc;

  if ( (control = (struct sf_control *) malloc(sizeof(struct sf_control)+
                         sizeof(sf_addr))) == NULL)
    { return -1; }

  control->magic   = SF_CONTROL_MAGIC;
  control->command = SF_COMMAND_FW_ADDR;
  sf_addr[0].mask = sf_addr_free; /* sf_addr[0].mask is not used otherwise */
  memcpy(control->data,sf_addr,sizeof(sf_addr));

  rc = write(fd, control, sizeof(struct sf_control)+
                          sizeof(sf_addr));
  free(control);

  if (rc > 0)
    { return 0; }
   else
    { return -1; }
}

