/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * Firewall character device -- loadable kernel module
 *
 * $Id: sf_device.c,v 1.17 1995/08/05 14:57:57 robby Rel $
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/fcntl.h>
#include <linux/errno.h>
#include <linux/major.h>
#include <linux/netdevice.h>
#include <linux/firewall.h>
#include "sf_kernel.h"
#include "sf_filter.h"
#include "sf_device.h"
#include "sf_global.h"

/*
 * Used to get the current state in a `for' loop.
 */

static int           isopen = 0;
static struct sf_fw *last   = NULL;

/*
 * Read the firewall device. This function takes the file mode to distinguish
 * between a `get config' call and a `get buffer' call.
 */

static int read_firewall(struct inode * inode, struct file * file, char * buf, int count)
{
	int len; 
	unsigned long flags;

	if ((file->f_mode & 2) == 0) { /* read state */
	  if (count != sizeof(struct sf_fw))
	    return -EINVAL;

	  /* get the rules one by one */
	  last = (struct sf_fw *) 
	         ((last == NULL) ? sf_rule_first() : sf_rule_next());

	  if (last == NULL)
	    return 0;
	 
	  memcpy_tofs(buf, last, sizeof(struct sf_fw)); 

	  len = sizeof(struct sf_fw);
	} 
	else { /* normal operation */
	  if (count != sizeof(struct sf_proc))
	    return -EINVAL;

	  if (sf_fw_enabled <= 0)
	    return 0;

	  /* sleep until there is data to consume */
	  interruptible_sleep_on(&sf_log_entry);

	  /* we don't want to be interrupted... */
	  save_flags(flags); cli();

          /* consume data */
	  memcpy_tofs(buf, &sf_log_data, sizeof(struct sf_proc));
	  /* reset offset */
	  sf_log_data.num_entries = 0;

	  /* interrupts are ok again */
	  restore_flags(flags);

	  len = sizeof(struct sf_proc);
	}

	file->f_pos += len;
	return len;
}


/*
 * Write the firewall device -- used for reconfiguration.
 */

static int write_firewall(struct inode * inode, struct file * file, const char * buf, int count)
{
	int len;
        int (*sf_fw_chk_save)(struct iphdr *, struct device *, int);

        /* save filter routine and set it to `block all'
	   to avoid corruption of the configuration data */
        sf_fw_chk_save = sf_fw_chk;
        sf_fw_chk      = sf_fw_chk_block;

        len = sf_write_config(buf,count);

        /* restore filter routine */
        sf_fw_chk      = sf_fw_chk_save;

	if (len > 0)
	  file->f_pos += len;
	return len;
}


/*
 * Open the firewall device. There are three cases:
 * read/write: normal operation
 * write only: reconfiguration
 * read only:  read current state 
 */

static int open_firewall(struct inode * inode, struct file * file)
{
	if (MINOR(inode->i_rdev) != 0) 
	  return -ENODEV;

	if ((file->f_mode & 2) == 0) { /* read only */
	  if (isopen != 0)
	    return -EBUSY;
	  isopen = 1;
	  last   = NULL;
	}
	else {
          if (sf_fw_enabled != 0)
            return -EBUSY;

	  sf_log_data.num_entries = 0;

	  if ((file->f_mode & 1) == 1) { /* read/write requested */
	    sf_fw_chk = sf_check_packet; /* normal operation */
	  } else { /* write only */
		  sf_fw_chk = sf_fw_chk_block; /* device open for reconfiguration */
	  }

	  sf_fw_enabled++;
	}

	MOD_INC_USE_COUNT;
	return 0;
}


static void close_firewall(struct inode * inode, struct file * file)
{
	MOD_DEC_USE_COUNT;

	if ((file->f_mode & 2) == 0) {
	  isopen = 0;
	}
	else {
	  sf_fw_enabled--;

	  /* Always block all packets if the daemon died. The user's
	     manual gives reasons for this behavior */ 
	  sf_fw_chk = sf_fw_chk_block;
	}
}


/* this structure contains the device switching table */

static struct file_operations firewall_fops = {
	NULL,		/* lseek */
	read_firewall,	/* read */
	write_firewall,	/* write */
	NULL,		/* readdir */
	NULL,		/* select */
	NULL,		/* ioctl */
	NULL,		/* mmap */
	open_firewall,	/* open code */
	close_firewall,	/* release code */
	NULL		/* fsync */
};


int init_module(void)
{
	if (register_chrdev(FIREWALL_MAJOR,"firewall",&firewall_fops)) {
		printk("unable to get major %d for firewall devs\n", FIREWALL_MAJOR);
		return -EIO;
	}

	if (register_firewall(PF_INET, &sf_fw_ops)) {
		printk("unable to register firewall\n"); 
		return -EIO;
	}

	if ((dev_base == NULL) || (strcmp(dev_base->name,"lo") != 0)) {
		printk("loopback is not first device in chain\n");
		return -EIO;
	}

	sf_init();

	return 0;
}

void cleanup_module(void)
{
	if(MOD_IN_USE)
		printk("firewall: busy - remove delayed\n");
	else {
		/* let all packets pass if filter is not loaded */
		sf_fw_chk = sf_fw_chk_pass;	
		sf_del_all_timers();

		unregister_chrdev(FIREWALL_MAJOR,"firewall");
		/* may fail.. */ 
		unregister_firewall(PF_INET, &sf_fw_ops);  
	}
}

