/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/* 
 * This file is part of the SF firewall loadable driver module
 *
 * $Id: sf_filter.c,v 1.40 1995/09/12 15:08:59 robby Rel $
 *
 */

#include <stdio.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/icmp.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/ip.h>
#include <linux/malloc.h>
#include <linux/timer.h>
#include <net/ip.h>
#include <linux/igmp.h>
#include "sf_kernel.h"
#include "sf_rc.h"
#include "sf_global.h"
#include "sf_filter.h"
#include "sf_control.h"
#include "sf_tcp.h"
#include "sf_icmp.h"

/* loopback is the first device in the dev chain */
#define LOOPBACK_MASK (dev_base->pa_mask)
#define LOOPBACK_ADDR (dev_base->pa_addr) 

/* firewall enabled */
int sf_fw_enabled = 0;

/* firewall initialized */
int sf_fw_initialized = 0;

/* temporary storage for packets to be sent to the daemon */
struct sf_proc sf_log_data;

/* wait queue for synchronization between filter function and daemon */
struct wait_queue *sf_log_entry = NULL;

/* 
 * private data 
 */

#ifdef SF_DEBUG
#define PRINTIP(a)	printk("%lu.%lu.%lu.%lu",ntohl(a)>>24&0xFF,\
						 ntohl(a)>>16&0xFF,\
						 ntohl(a)>>8&0xFF,\
						 ntohl(a)&0xFF)  
#endif

#define MIN(a,b) ((a) < (b) ? (a) : (b))

/*
 * linear list to store the rules in
 */

struct sf_fw *rules = NULL;

/*
 * pointer to save position for sf_first_rule() and sf_next_rule()
 * (used by "sfc show")
 */

struct sf_fw *rule_first_next = NULL;

/* 
 * array to store the ip addresses of the rules 
 */

struct sf_address	sf_addr[SF_ADDRESS_CNT_MAX];
int			sf_addr_free = 0; /* first free entry in array */

/* returns 1 if addr matches one of the "internalnet" addresses */
int sf_inside(__u32 addr)
  { int i;
    for (i=1; i<=sf_addr[0].addr; i++)
      { if ( (sf_addr[i].addr & sf_addr[i].mask) == (addr & sf_addr[i].mask) )
	  return 1;
      }
    return 0;
  }

/* returns 1 if address matches */
int sf_addr_match(long idx, long cnt, __u32 addr, unsigned short port, int negative)
  { int i;

    for (i=idx; i<(idx+cnt); i++)
      { /* set addrmatch if address does not care or if address matches */
	/* address is to be ignored if mask == 0 */
	int addrmatch =  
          ( (sf_addr[i].addr & sf_addr[i].mask) == (addr & sf_addr[i].mask) );
	int portmatch = 1;

	/* reset portmatch if port cares but does not match */
	if (port && (sf_addr[i].port != SF_ALL_PORTS))
          if ( (port < sf_addr[i].port ) || (sf_addr[i].prend < port) )
	    portmatch = 0;

	/* If the negative flag is not set, we have got a match if at
             least once addrmatch and portmatch is set, so return 1. 
	   If the negative flag is set, the return value must be zero
	     to be good. It may only be zero, if the port matches (or
	     does not care) and none of the addresses match. As the 
	     negative flag is set only if the outside keyword is used,
	     all port fields contain the same value. So we can return
	     one if portmatch is zero for the first array entry.       */
	if ( (addrmatch && portmatch) 
	  || (negative && !portmatch) ) return 1;

      }
    return 0;
  }

/*
 * interface cache to determine if interface is connected to inside or outside
 */

#define SF_IF_CACHE_SIZE	199	/* must be a prime number */
struct sf_if_cache_entry {
	__u32 ifaddr;			/* address of interface */
	int inside;			/* 1 if interface is inside */
};
struct sf_if_cache_entry sf_if_cache[SF_IF_CACHE_SIZE];
#define SF_IF_CACHE_KEY(a)  ( (a) % SF_IF_CACHE_SIZE )

/* 
 * hash queues for fragmented ip datagrams 
 */

struct sf_fragment {
	struct sf_fragment *prev, *next;/* queue pointers */
	__u16 id;			/* identification field */
	__u32 src;			/* source address */
	__u32 dst;			/* destination address */
	struct timer_list timer; 	/* expiration of datagram */
};
#define SF_FRAG_HASH_SIZE	499	/* must be a prime number */
struct sf_fragment *sf_frag[SF_FRAG_HASH_SIZE];
#define SF_FRAG_KEY(a)	( (a) % SF_FRAG_HASH_SIZE )

/* packet is a fragment if "more fragments" bit is set => add to hash queue */
#define SF_CHKFRAG      if (ntohs(ip->frag_off) & IP_MF) sf_frag_insert(ip->id,ip->saddr,ip->daddr)

/* delete entry from hash queue */
static void sf_frag_expire(unsigned long arg)
  { struct sf_fragment *frag;
    unsigned long flags;

    frag = (struct sf_fragment *)arg;
#ifdef SF_DEBUG
    printk("sf_frag_expire: id = %u\n",ntohs(frag->id));
#endif
    del_timer(&frag->timer);
    save_flags(flags); cli();
    /* delete fragment from queue */
    if (frag->prev != NULL) { frag->prev->next = frag->next; }
     else sf_frag[SF_FRAG_KEY(frag->id)] = frag->next;
    if (frag->next != NULL) frag->next->prev = frag->prev;
    restore_flags(flags);
    kfree_s(frag,sizeof(struct sf_fragment));
  }

/* insert hash queue entry */
static void sf_frag_insert(__u16 id, __u32 saddr, __u32 daddr)
  { int key = SF_FRAG_KEY(id);
    struct sf_fragment *frag;
    unsigned long flags;

    /* allocate memory and initialize fragmentation hash queue entry */
    if ( (frag = 
          (struct sf_fragment *) kmalloc(sizeof(struct sf_fragment),GFP_ATOMIC)) == NULL)
      { printk("sf_frag_insert: out of memory!\n");
        return;
      }
    frag->id  = id;
    frag->src = saddr;
    frag->dst = daddr;
#ifdef SF_DEBUG_FRAG
    printk("sf_frag_insert: id = %u\n",ntohs(frag->id));
#endif

    /* initialize timer */
    init_timer(&frag->timer);
    frag->timer.expires = jiffies + IP_FRAG_TIME; /* same time as in reassembly */
    frag->timer.data = (unsigned long) frag;
    frag->timer.function = sf_frag_expire;
    add_timer(&frag->timer);

    /* insert as first element into hash queue */
    save_flags(flags); cli();
    if (sf_frag[key] != NULL) sf_frag[key]->prev = frag;
    frag->next = sf_frag[key];
    frag->prev = NULL;
    sf_frag[key] = frag;
    restore_flags(flags);
  }

/*
 * hash queues for tcp circuits
 */

/* An established TCP connection is identified */
/* by the source and dest addresses and ports. */
/* The client is the host that initiates the connection. */

#define SF_TCP_CLIENT_SYN	0x01	/* client has sent SYN */
#define SF_TCP_SERVER_SYN	0x02	/* server has sent SYN */
#define SF_TCP_CLIENT_FIN	0x04	/* client has sent FIN */
#define SF_TCP_SERVER_FIN	0x08	/* server has sent FIN */

#define SF_TCP_ESTABLISHED	0x03	/* connection is up */
#define SF_TCP_TERMINATED	0x0F	/* connection has been terminated */

#define SF_TCP_SYN_TO		( 120 * HZ)	/* connection establishment timeout */
#define SF_TCP_FIN_WAIT		(  10 * HZ)	/* wait after FIN (let ACKs pass) */

struct sf_tcp_connection {
	struct sf_tcp_connection *prev, *next;
	__u32 client_addr, server_addr;	
	__u16 client_port, server_port;
	int state;			/* simplified state of tcp connection */
	struct timer_list timer;
	int timer_set;			/* one, if a timer is set */
};

#define SF_TCP_HASH_SIZE	499	/* must be a prime number */
struct sf_tcp_connection *sf_tcp[SF_TCP_HASH_SIZE];
#define SF_TCP_KEY(a)  ( (a) % SF_TCP_HASH_SIZE )

/* delete entry from hash queue */
static void sf_tcp_delete(struct sf_tcp_connection *sftcp)
  { unsigned long flags;
 
    save_flags(flags); cli();
    if (sftcp->prev != NULL) { sftcp->prev->next = sftcp->next; }
     else sf_tcp[SF_TCP_KEY(sftcp->client_addr + sftcp->server_addr
        + (__u32)sftcp->client_port + (__u32)sftcp->server_port)] = sftcp->next;
    if (sftcp->next != NULL) sftcp->next->prev = sftcp->prev;

#ifdef SF_DEBUG
    printk("sf_tcp_delete: %p ",sftcp);
    PRINTIP(sftcp->client_addr);
    printk(":%u -> ",ntohs(sftcp->client_port));
    PRINTIP(sftcp->server_addr);
    printk(":%u\n",ntohs(sftcp->server_port));
#endif

    restore_flags(flags);
    kfree_s(sftcp,sizeof(struct sf_tcp_connection));
  }

static void sf_tcp_expire(unsigned long arg)
  { struct sf_tcp_connection *sftcp;

    sftcp = (struct sf_tcp_connection *)arg;

#ifdef SF_DEBUG
    printk("sf_tcp_expire: ");
    PRINTIP(sftcp->client_addr);
    printk(":%u -> ",ntohs(sftcp->client_port));
    PRINTIP(sftcp->server_addr);
    printk(":%u\n",ntohs(sftcp->server_port));
#endif

    del_timer(&sftcp->timer);

    /* delete tcp entry from queue */
    sf_tcp_delete(sftcp);
  }

/* insert hash queue entry */
static struct sf_tcp_connection *sf_tcp_insert(__u32 saddr, __u32 daddr, __u16 source, __u16 dest)
  { int key = SF_TCP_KEY(saddr + daddr + (__u32)source + (__u32)dest);
    struct sf_tcp_connection *sftcp;
    unsigned long flags;
    int sf_flush_all(void);

    /* allocate memory and initialize tcp hash queue entry */
    if ( (sftcp =
          (struct sf_tcp_connection *) kmalloc(sizeof(struct sf_tcp_connection),GFP_ATOMIC)) == NULL)
      { printk("sf_tcp_insert: out of memory!\n");
	/* Clean up the tcp hash queues. This will abort all TCP connections. */
	/* But if the hash queues occupy a lot of memory, it will help! */
	sf_flush_all(); 
        return NULL;
      }
    sftcp->client_addr = saddr;
    sftcp->server_addr = daddr;
    sftcp->client_port = source;
    sftcp->server_port = dest;
    sftcp->state = 0;
#ifdef SF_DEBUG
    printk("sf_tcp_insert: (%i) %p ",key,sftcp);
    PRINTIP(sftcp->client_addr);
    printk(":%u -> ",ntohs(sftcp->client_port));
    PRINTIP(sftcp->server_addr);
    printk(":%u\n",ntohs(sftcp->server_port));
#endif

    /* initialize timer */
    init_timer(&sftcp->timer);
    sftcp->timer_set = 0;

    /* insert as first element into hash queue */
    save_flags(flags); cli();
    if (sf_tcp[key] != NULL) sf_tcp[key]->prev = sftcp;
    sftcp->next = sf_tcp[key];
    sftcp->prev = NULL;
    sf_tcp[key] = sftcp;
    restore_flags(flags);
    return sftcp;
  }

/*
 * RIP
 */

#define SF_RIP_PORT 520

struct riphdr {
  __u8		command;		/* RIP command */
  __u8		version;		/* RIP version */
  __u16		zero1;			/* must be zero */
};

struct ripdata {
  __u16		af;			/* address family */
  __u16		zero1;			/* must be zero */
  __u32		addr;			/* IP address */
  __u32		zero2;			/* must be zero */
  __u32		zero3;			/* must be zero */
  __u32		metric;			/* metric */
};

int check_rip (struct sf_fw *rule, struct riphdr *rip, unsigned long riplength)
  { unsigned long len;
    struct ripdata *rd = (struct ripdata *)(rip + 1);

#ifdef SF_DEBUG
   /* printk("RIP command %u version %u length %lu\n",
	   (unsigned) rip->command,(unsigned) rip->version,riplength);*/
#endif

    for (len=riplength - sizeof(struct riphdr); len>=20; len-=20, rd++)
      {
#ifdef SF_DEBUG
/*	printk(" af %u addr ",ntohs(rd->af));
        PRINTIP(rd->addr);
	printk(" metric %lu\n",ntohl(rd->metric));*/
#endif
	if ( (ntohs(rd->af) == AF_INET)
          && (ntohs(rd->metric) < 16)
          && ( ( !(rule->fw_flags & SF_RIP_ADDR_NEG)
              && !sf_addr_match(rule->fw_rip_idx, rule->fw_rip_cnt, rd->addr, 0, 0) )
            || ( (rule->fw_flags & SF_RIP_ADDR_NEG)
	      && sf_addr_match(rule->fw_rip_idx, rule->fw_rip_cnt, rd->addr, 0, 1) ) ) )
	  return 0;
      }

    return 1;
  }

/*
 *	Returns 0 if packet should be dropped, 1 if it should be accepted,
 *	and a negative value if an ICMP unreachable packet should be sent.
 *
 *	opt is one of SF_STATE_FORWARD, TRANSMIT or RECEIVE.
 *	This call is done from task[0], so it must not sleep.  
 */

int sf_check_packet(struct iphdr *ip, struct device *rif, int opt)
{
	struct sf_fw implicit_rule;
	struct sf_fw *rule = &implicit_rule;
	struct tcphdr *tcp=(struct tcphdr *)((unsigned long *)ip+ip->ihl);
	struct udphdr *udp=(struct udphdr *)((unsigned long *)ip+ip->ihl);
	struct icmphdr *icmp=(struct icmphdr *)((unsigned long *)ip+ip->ihl);
	struct igmphdr *igmp=(struct igmphdr *)((unsigned long *)ip+ip->ihl);
	struct sf_tcp_connection *sftcp = NULL;
	int returncode = FW_BLOCK;

#ifdef SF_DEBUG
  if (ip->protocol == IPPROTO_ICMP) {
    printk("check proto %i ",ip->protocol);
    PRINTIP(ip->saddr);
    printk(":%u->",ntohs(tcp->source));
    PRINTIP(ip->daddr);
    printk(":%u ",ntohs(tcp->dest));
    if (opt==SF_STATE_RECEIVE) printk("r"); 
    if (opt==SF_STATE_TRANSMIT) printk("t");
    if (opt==SF_STATE_FORWARD) printk("f");
    printk("\n");
  }
#endif 

	/* let packet pass if firewall functions are disabled */
	if (sf_fw_enabled <= 0) 
	  return FW_BLOCK;

        /* check if packet is long enough to hold IP header */
        if ( ntohs(ip->tot_len) < (4 * ip->ihl) ) return FW_BLOCK;
        if ( ip->ihl < 5 ) return FW_BLOCK;

	/* if the device Is loopback, source and dest addr. must be loopback addr. */
	if ( !(rif->flags & IFF_LOOPBACK)
	  && ( ((ip->saddr & LOOPBACK_MASK) == 
			(LOOPBACK_ADDR & LOOPBACK_MASK) )
	    || ((ip->daddr & LOOPBACK_MASK) == 
			(LOOPBACK_ADDR & LOOPBACK_MASK) ) ) )
            { /* packet not allowed */
	      rule->fw_rc = FW_BLOCK;
              if (opt == SF_STATE_RECEIVE)
                { rule->rule_id.num = RULE_SPOOF_RECV; }
              else
                { rule->rule_id.num = RULE_SPOOF_XMIT; }
              goto informlog; /* skip all further examination but inform daemon */
	    }

	/* check local packets only once */
	if ( (opt == SF_STATE_RECEIVE) && (ip->saddr == ip->daddr) )
	  return FW_ACCEPT;

	/* check packet */

	/* check interface */
	{ int ifinside, ifkey;

	  /* determine if interface is connected to inside or to outside */
	  if ( sf_if_cache[ifkey = SF_IF_CACHE_KEY(rif->pa_addr)].ifaddr == rif->pa_addr )
	    { ifinside = sf_if_cache[ifkey].inside; }
	   else /* interface not in cache */
	    { 
#ifdef SF_DEBUG
	      printk("sf: caching interface ");
	      PRINTIP(rif->pa_addr);
	      printk(", cache entry %i\n",ifkey);
#endif
	      ifinside = sf_inside(rif->pa_addr);
	      sf_if_cache[ifkey].ifaddr = rif->pa_addr;
	      sf_if_cache[ifkey].inside = ifinside;
	    }

	  if ( !(rif->flags & IFF_LOOPBACK)
            && ( ( (opt == SF_STATE_RECEIVE)  
		&& !(IN_CLASSD(ntohl(ip->saddr)))
                && (ifinside != sf_inside(ip->saddr)) )
	      || ( ( (opt == SF_STATE_TRANSMIT)  
                  || (opt == SF_STATE_FORWARD) ) 
		&& !(IN_CLASSD(ntohl(ip->daddr)))
		&& (ifinside != sf_inside(ip->daddr)) ) ) )
	    { /* packet not allowed */
	      rule->fw_rc = FW_BLOCK;

#if 0
	      /* (bootp) - drop packets from/to 0.0.0.0/255.255.255.255 */
	      if ((ip->saddr == 0) || (ip->daddr == 0) ||
		  (ip->saddr == 0xffffffff) || (ip->daddr == 0xffffffff))
		return FW_BLOCK;
#endif

	      if (opt == SF_STATE_RECEIVE)
		{ rule->rule_id.num = RULE_SPOOF_RECV; }
	      else
		{ rule->rule_id.num = RULE_SPOOF_XMIT; }
	      goto informlog; /* skip all further examination but inform daemon */
	    }
	}

	/* If SF_STATE_FORWARD, the packet has already been examined */
	/* at receive time. So skip the examination now. */
	if (opt == SF_STATE_FORWARD) return FW_ACCEPT;

	/* check fragments */
	if (ntohs(ip->frag_off) & IP_OFFSET) /* fragment offset != 0 */
	  { /* this is an ip fragment but not the first one of the packet */
	    struct sf_fragment *frag = sf_frag[SF_FRAG_KEY(ip->id)];

#ifdef SF_DEBUG_FRAG
	    printk("sf_check_packet: frag_off %u  tot_len %u\n",
		   (ntohs(ip->frag_off)&IP_OFFSET)<<3,ntohs(ip->tot_len));
#endif

	    if ( ((ntohs(ip->frag_off)&IP_OFFSET) << 3) + 
                 (unsigned int)ntohs(ip->tot_len) > 65535 ) 
	      {
#ifdef SF_DEBUG_FRAG
	        printk("sf_check_packet: oversized packet detected!\n");
#endif
		rule->fw_rc = FW_BLOCK;
                rule->rule_id.num = RULE_OVERSIZED;
                goto informlog; /* skip all further examination but inform daemon */
	      }

	    while (frag != NULL)
	      { if ( (ip->id==frag->id) && (ip->saddr==frag->src) && (ip->daddr==frag->dst) )
                  { /* fragment belongs to an allowed packet */
		    /* Always let the timer delete the queue entry, because */
		    /* fragments need not arrive in order, thus there may still */
		    /* be fragments missing when the last fragment arrives. */
		    del_timer(&frag->timer);
		    frag->timer.expires = jiffies + IP_FRAG_TIME; /* same time as in reassembly */
		    frag->timer.data = (unsigned long) frag;
		    frag->timer.function = sf_frag_expire;
		    add_timer(&frag->timer);
		    return FW_ACCEPT;
		  }
		frag = frag->next;
	      }
	      /* fragment must not pass */
	      return FW_BLOCK;
	  }

        /* Now we know that the packet is not a subsequent fragment.
           So we can check, if the packet is long enough to hold an
           entire UDP, ICMP, IGMP (or TCP, see below) header */
        if (ip->protocol == IPPROTO_UDP)
          if (ntohs(ip->tot_len) < (4 * ip->ihl + sizeof(struct udphdr)))
            return FW_BLOCK;
        if (ip->protocol == IPPROTO_ICMP)
          if (ntohs(ip->tot_len) < (4 * ip->ihl + sizeof(struct icmphdr)))
            return FW_BLOCK;
        if (ip->protocol == IPPROTO_IGMP)
          if (ntohs(ip->tot_len) < (4 * ip->ihl + sizeof(struct igmphdr)))
            return FW_BLOCK;

	/* check tcp */
	if (ip->protocol == IPPROTO_TCP) 
	  { int key;
	    int reverse = 0; /* indicates direction of packet  */
			     /* (reverse==0 => client->server) */

            /* check if packet is long enough to contain the TCP header */
            if (ntohs(ip->tot_len) < (4 * ip->ihl + sizeof(struct tcphdr)))
              return FW_BLOCK; 

            key = SF_TCP_KEY(ip->saddr + ip->daddr + (__u32)tcp->source + (__u32)tcp->dest);

#ifdef SF_DEBUG
/*	    printk("TCP:");
	    PRINTIP(ip->saddr);
	    printk(":%u -> ",ntohs(tcp->source));
	    PRINTIP(ip->daddr);
	    printk(":%u\n",ntohs(tcp->dest));*/
#endif

	    sftcp = sf_tcp[key];
	    while (sftcp != NULL)
	      { if ( (sftcp->client_addr == ip->saddr) && 
                     (sftcp->client_port == tcp->source) &&
		     (sftcp->server_addr == ip->daddr) &&
		     (sftcp->server_port == tcp->dest) )
		  { reverse = 0; /* packet from client to server */
		    break;
		  }
		if ( (sftcp->client_addr == ip->daddr) &&
		     (sftcp->client_port == tcp->dest) &&
		     (sftcp->server_addr == ip->saddr) &&
		     (sftcp->server_port == tcp->source) )
		  { reverse = 1; /* packet from server to client */
		    break;
		  }
	        sftcp = sftcp->next;
	      }

#ifdef SF_DEBUG
            if (tcp->syn || tcp->fin || tcp->rst)
              { printk("sf: TCP %p ",sftcp);
                if (reverse) printk("s->c "); else printk("c->s ");
                if (tcp->syn) printk("SYN ");
                if (tcp->fin) printk("FIN ");
                if (tcp->ack) printk("ACK ");
                if (tcp->rst) printk("RST ");
                printk("\n");
              }
#endif

	    if (sftcp == NULL) /* connection not in hash queue */
	      { if (tcp->syn && !tcp->ack) goto checkrules; /* new connection */
#ifdef SF_DEBUG
                printk("connection not in hash queue\n");
#endif
	        return FW_BLOCK; /* block packet */
	      }

	    /* connection found in hash queue */

	    /* filter FTP PORT commands */

#define FTP_CONTROL_PORT	21
#define FTP_DATA_PORT		20

	    if ( (ntohs(sftcp->server_port)==FTP_CONTROL_PORT) && !reverse )
	      { /* FTP packet from client to server */
		char *ftpdata=(char *)((unsigned long *)tcp+tcp->doff);

		/* Format of port command is "PORT n1,n2,n3,n4,n5,n6" */
		/* where n1.n2.n3.n4 is the IP address                */
		/* and n5*256+n6 is the port number.		      */
		/* Convert the string into numbers		      */
		if (strncmp(ftpdata,"PORT",4) == 0)
		  { int i;
		    __u16 ftpport = 0;
		    char *pktend=(char *)ip+ntohs(ip->tot_len);
		   
		    /* Skip the ip address and use the packet source address */
		    /* instead. So it is not possible to redirect the data */
		    /* connection to another host. */ 
		    ftpdata += 5;
		    for (i=0; i<4; i++)
		      { 
			while ( (ftpdata < pktend) && 
				(*ftpdata >=48) && (*ftpdata <=57) )
			    ftpdata += 1;
			ftpdata += 1;
		      }
		    /* convert port number */
		    for (i=0; i<2; i++)
		      { int j = 0;
			ftpport <<= 8;
			while ( (ftpdata < pktend) &&
				(*ftpdata >=48) && (*ftpdata <=57) )
			  { j = j * 10 + *ftpdata - 48;
			    ftpdata += 1;
			  }
			ftpdata += 1;
			ftpport += (__u16)j;
		      }
		    ftpport = htons(ftpport);

#ifdef SF_DEBUG
		    printk("sf: FTP PORT ");
		    PRINTIP(ip->saddr);
		    printk(":%u\n",ntohs(ftpport));
#endif
		    sf_tcp_insert(ip->daddr, ip->saddr,
				  htons(FTP_DATA_PORT), ftpport);

		  }

	      }

	    if ( (sftcp->state >= SF_TCP_ESTABLISHED) && !tcp->fin && !tcp->rst && !tcp->syn)
	      { SF_CHKFRAG; /* check if packet is fragmented */
	        return FW_ACCEPT; /* packet is allowed */ 
	      }
	    if ( tcp->rst )
	      { if (sftcp->timer_set) del_timer(&sftcp->timer);
		sf_tcp_delete(sftcp); /* RST aborts an connection immediately */

		SF_CHKFRAG; /* check if packet is fragmented */
		return FW_ACCEPT;
              }
	    if ( tcp->syn )
	      { if ( sftcp->state == SF_TCP_TERMINATED )
		  { /* Somebody is trying to establish a connection that is  */
		    /* still in the 2 MSL wait state. This should not happen */
		    /* if the TCP protocol is implemented correctly. But if  */
		    /* it happens nevertheless, we must not allow the        */
		    /* connection without checking the rules.                */
#ifdef SF_DEBUG_TCP
		    printk("sf_check_packet: SYN && SF_TCP_TERMINATED\n");
#endif
		    if (sftcp->timer_set) del_timer(&sftcp->timer);
		    sf_tcp_delete(sftcp);
		    return FW_BLOCK; /* The sender is going to retry... */
		  }
		if ( sftcp->state >= SF_TCP_ESTABLISHED )
                  { /* This may be a resent SYN packet */
		    SF_CHKFRAG; /* check if packet is fragmented */
		    return FW_ACCEPT;
		  }
		if ( !reverse ) 
		  { sftcp->state = SF_TCP_CLIENT_SYN;
#ifdef SF_DEBUG
		    printk("sf_check_packet: tcp client sent SYN\n");
#endif
		    if (sftcp->timer_set) del_timer(&sftcp->timer);
		    sftcp->timer.expires = jiffies + SF_TCP_SYN_TO;
		    sftcp->timer.data = (unsigned long) sftcp;
		    sftcp->timer.function = sf_tcp_expire;
		    sftcp->timer_set = 1;
		    add_timer(&sftcp->timer); 
		  }
		else
		  { sftcp->state |= SF_TCP_SERVER_SYN;
#ifdef SF_DEBUG
		    printk("sf_check_packet: tcp connection established\n");
#endif
		    /* connection established */
		    if (sftcp->timer_set) /* SYN packet may pass more than once */
		      { del_timer(&sftcp->timer); 
		        sftcp->timer_set = 0;
		      } 
		  }
		SF_CHKFRAG; /* check if packet is fragmented */
		return FW_ACCEPT;
	      }
	    if ( tcp->fin )
	      { if ( reverse )
		  { sftcp->state |= SF_TCP_SERVER_FIN; }
		 else
		  { sftcp->state |= SF_TCP_CLIENT_FIN; }
		if ( sftcp->state == SF_TCP_TERMINATED )
		  { if (sftcp->timer_set) del_timer(&sftcp->timer); 
		    sftcp->timer.expires = jiffies + SF_TCP_FIN_WAIT;
		    sftcp->timer.data = (unsigned long) sftcp;
		    sftcp->timer.function = sf_tcp_expire;
		    sftcp->timer_set = 1;
		    add_timer(&sftcp->timer); 
#ifdef SF_DEBUG
                    printk("sf_check_packet: tcp connection terminated\n");
#endif
		  }
		SF_CHKFRAG; /* check if packet is fragmented */
		return FW_ACCEPT;
	      }

	    /* Only packets with !SYN, !FIN, !RST that belong to connections
               with state < SF_TCP_ESTABLISHED can arrive here. */

            /* This should never happen - but it does happen, e.g. when
               one side thinks the connection is still established and
               the other side tries to establish a new connection to the
               same port numbers. This can also happen if the connection
               has died on one end only (e.g. rebooted PC). There's no
               need to accept these strange packets, since the other
               side of the connection will not accept them.
               
               Maybe we should send a RST packet here? */

#ifdef SF_DEBUG_TCP
            printk("sf_check_packet: TCP connection in strange state:\n");
            if (reverse)  printk("s->c "); else printk("c->s ");
            if (tcp->syn) printk("SYN ");
            if (tcp->fin) printk("FIN ");
            if (tcp->ack) printk("ACK ");
            if (tcp->rst) printk("RST ");
            printk("\n");
#endif
	    return FW_BLOCK;

	  } /* check tcp */

      checkrules:
	/* check rules */
	{
	  for (rule = rules; rule != NULL; rule = rule->fw_next)
	    {
#ifdef SF_DEBUG 
	      printk("sf: rule %li, flags %lx, prot %i\n",rule->rule_id.num,rule->fw_flags,rule->protocol); 
#endif 

	      /* check if protocols match */
	      
	      if ( (rule->fw_flags & SF_FW_PROT) != SF_FW_PROT_ALL )
	        { 
		  if ( (rule->fw_flags & SF_FW_PROT) == SF_FW_CHECK_PROTOCOL )
		    { unsigned long flg = rule->fw_flags & SF_TYPE_MASK;
		      if ( rule->protocol != ip->protocol ) continue;
		      switch (ip->protocol) 
			{ case IPPROTO_ICMP:
			    if ( flg == SF_ICMP_ALLTYPES ) break;
		            switch (icmp->type) /* filter msg type */
		       	      { case ICMP_ECHOREPLY:
			  	  if (! (flg & SF_ICMP_ECHOREPLY) ) continue;
			  	  break;
				case ICMP_DEST_UNREACH:
			  	  if (! (flg & SF_ICMP_DEST_UNREACH) ) continue;
			  	  break;
				case ICMP_SOURCE_QUENCH:
			  	  if (! (flg & SF_ICMP_SOURCE_QUENCH) ) continue;
			  	  break;
				case ICMP_REDIRECT:
			  	  if (! (flg & SF_ICMP_REDIRECT) ) continue;
			  	  break;
				case ICMP_ECHO:
			  	  if (! (flg & SF_ICMP_ECHO) ) continue;
			  	  break;
				case ICMP_TIME_EXCEEDED:
			  	  if ( !(flg & SF_ICMP_TIME_EXCEEDED) ) continue;
			  	  break;
				case ICMP_PARAMETERPROB:
			  	  if ( !(flg & SF_ICMP_PARAMETERPROB) ) continue;
			  	  break;
				case ICMP_TIMESTAMP:
			  	  if ( !(flg & SF_ICMP_TIMESTAMP) ) continue;
			  	  break;
				case ICMP_TIMESTAMPREPLY:
			  	  if ( !(flg & SF_ICMP_TIMESTAMPREPLY) ) continue;
			  	  break;
				case ICMP_INFO_REQUEST:
			  	  if ( !(flg & SF_ICMP_INFO_REQUEST) ) continue;
			  	  break;
				case ICMP_INFO_REPLY:
			  	  if ( !(flg & SF_ICMP_INFO_REPLY) ) continue;
			  	  break;
				case ICMP_ADDRESS:
			  	  if ( !(flg & SF_ICMP_ADDRESS) ) continue;
			  	  break;
				case ICMP_ADDRESSREPLY:
			  	  if ( !(flg & SF_ICMP_ADDRESSREPLY) ) continue;
			  	  break;
				default: continue;
		              }
		    	    break;
		  	  case IPPROTO_IGMP:
			    if ( flg == SF_IGMP_ALLTYPES ) break;
                    	    switch (igmp->type) /* filter msg type */
                       	      { case IGMP_HOST_MEMBERSHIP_QUERY:
                          	  if ( !(flg & IGMP_HOST_MEMBERSHIP_QUERY) ) continue;
                          	  break;
				case IGMP_HOST_MEMBERSHIP_REPORT:
			  	  if ( !(flg & SF_IGMP_HOST_MEMBERSHIP_REPORT) ) continue;
			  	  break;
				case IGMP_HOST_LEAVE_MESSAGE:
			  	  if ( !(flg & IGMP_HOST_LEAVE_MESSAGE) ) continue;
			  	  break;
				default: continue;
		      	      }
			    break;
			  default: break;
			} /* ip->protocol */
		    }
		   else /* rule->fw_flags & SF_FW_PROT != SF_FW_CHECK_PROTOCOL */
		    { switch (rule->fw_flags & SF_FW_PROT)
			{ case SF_FW_PROT_RIP:
		            if ( (ip->protocol!=IPPROTO_UDP) || (udp->dest!=htons(SF_RIP_PORT)) )
		              continue;
			    if ( rule->fw_rip_cnt == 0 ) break;
		            if ( !check_rip(rule, (struct riphdr *)(udp + 1),
				            ntohs(ip->tot_len) - ip->ihl*4 - 8) )
		              continue;
		            break;
		          default: continue;
			}
		    } /* if */
		} /* if */

	      if (rule->fw_flags & SF_FW_CHECK_TTL)
		switch (rule->fw_flags & SF_FW_TTL)
		  { case SF_FW_TTL_EQUAL: 
		      if (ip->ttl != rule->ttl) continue;
		      break;
		    case SF_FW_TTL_LESS:
		      if (ip->ttl >= rule->ttl) continue;
		      break;
		    case SF_FW_TTL_GREATER:
		      if (ip->ttl <= rule->ttl) continue;
		      break;
		    case SF_FW_TTL_NOTEQUAL:
		      if (ip->ttl == rule->ttl) continue;
		      break;
		    default: continue;
		  }

	      if (rule->fw_flags & SF_FW_CHECK_OPT)
		{ unsigned char *buff = (unsigned char *)(ip + 1); /* start of options */
		  int done = 0;
		  int len = sizeof(struct iphdr);

		  while (!done && len < ip->ihl*4) /* end of header not yet reached */
		    switch (*buff)
		      { case IPOPT_END: /* pad bytes */
#ifdef SF_DEBUG
		          printk("option end\n");
#endif
			  done = 1; break; /* end of options */
			case IPOPT_NOOP:
#ifdef SF_DEBUG
			  printk("option noop\n");
#endif
			  buff++; len++; break;
			case IPOPT_SEC: /* security */
#ifdef SF_DEBUG
			  printk("option sec\n");
#endif
			  if (rule->fw_flags & SF_FW_OPT_SEC) 
			    { done = 2; break; }
			  buff+=11; len+=11; break;
			case IPOPT_LSRR:
#ifdef SF_DEBUG
			  printk("option lsr\n");
#endif
			  if (rule->fw_flags & SF_FW_OPT_LSR)
			    { done = 2; break; }
			  buff++; len+=*buff; buff+=(*buff-1); break;
			case IPOPT_SSRR:
#ifdef SF_DEBUG
			  printk("option ssr\n");
#endif
			  if (rule->fw_flags & SF_FW_OPT_SSR)
			    { done = 2; break; }
			  buff++; len+=*buff; buff+=(*buff-1); break;
			case IPOPT_RR:
#ifdef SF_DEBUG
			  printk("option rr\n");
#endif
			  if (rule->fw_flags & SF_FW_OPT_RR)
			    { done = 2; break; }
			  buff++; len+=*buff; buff+=(*buff-1); break;
			case IPOPT_SID:
#ifdef SF_DEBUG
			  printk("option sid\n");
#endif
			  if (rule->fw_flags & SF_FW_OPT_SATID)
			    { done = 2; break; }
			  buff+=4; len+=4; break;
			case IPOPT_TIMESTAMP:
#ifdef SF_DEBUG	
			  printk("option ts\n");
#endif
			  if (rule->fw_flags & SF_FW_OPT_TS)
			    { done = 2; break; }
			  buff++; len+=*buff; buff+=(*buff-1); break;
			default: /* unknown option */
#ifdef SF_DEBUG
			  printk("unknown option %i\n",*buff);
#endif
			  done = 1; break;
		      }
#ifdef SF_DEBUG
		  printk(" done = %i\n",done);
#endif
		  if (done != 2) continue; /* no option matched */
		}

	      /* check addresses and ports */
	      if (rule->fw_src_cnt) 
		{ int match = 0;
		  unsigned short port = 0;
		  if ( (ip->protocol==IPPROTO_TCP) || (ip->protocol==IPPROTO_UDP) )
		    port = ntohs(tcp->source);
		  match = sf_addr_match(rule->fw_src_idx,rule->fw_src_cnt,ip->saddr,port,rule->fw_flags & SF_FW_SRC_NEG);
		  if ( ( match &&  (rule->fw_flags & SF_FW_SRC_NEG) )
		    || (!match && !(rule->fw_flags & SF_FW_SRC_NEG) ) ) continue;
		}
	      if (rule->fw_dst_cnt)
                { int match = 0;
                  unsigned short port = 0;
                  if ( (ip->protocol==IPPROTO_TCP) || (ip->protocol==IPPROTO_UDP) )
                    port = ntohs(tcp->dest);
                  match = sf_addr_match(rule->fw_dst_idx,rule->fw_dst_cnt,ip->daddr,port,rule->fw_flags & SF_FW_DST_NEG);
                  if ( ( match &&  (rule->fw_flags & SF_FW_DST_NEG) )
                    || (!match && !(rule->fw_flags & SF_FW_DST_NEG) ) ) continue;
		}

#ifdef SF_DEBUG
	      printk("sf: packet matches rule\n");
#endif
	      break; /* the packet matches the rule */
	    }
	}

	if (rule == NULL) return FW_BLOCK; /* packet matches no rule */

	if (rule->fw_rc == FW_ACCEPT)
	  { /* if a tcp packet arrives here, it is initiating a new, legal connection */
	    if (ip->protocol == IPPROTO_TCP) 
	      { sftcp = sf_tcp_insert(ip->saddr, ip->daddr, tcp->source, tcp->dest);
		if (sftcp != NULL)
		  { sftcp->state = SF_TCP_CLIENT_SYN;
#ifdef SF_DEBUG
		    printk("sf_check_packet: tcp client sent SYN\n");
#endif
		    sftcp->timer.expires = jiffies + SF_TCP_SYN_TO;
		    sftcp->timer.data = (unsigned long) sftcp;
		    sftcp->timer.function = sf_tcp_expire;
		    add_timer(&sftcp->timer); 
		    sftcp->timer_set = 1;
		  }
	      }
	    SF_CHKFRAG; /* check if packet is fragmented */
	  }

	returncode = rule->fw_rc;

	if (rule->fw_rc == SF_RC_TREJECT) /* TCP reset / best reject*/
	  {
	    if (opt == SF_STATE_TRANSMIT)
	      returncode = FW_REJECT;
	    else 
              { /* choose best method */
	        if (ip->protocol == IPPROTO_UDP)
	          { /* send ICMP port unreach. */
                    send_icmp_error(rif, ip, ICMP_DEST_UNREACH, 
                                    ICMP_PORT_UNREACH);
		    returncode = FW_BLOCK;
		  }
		else if (ip->protocol == IPPROTO_TCP)
		  { /* send TCP reset */
	            send_tcp_reset(rif, ip, tcp);
	            returncode = FW_BLOCK; /* block packet */
                  }
		else /* simple reject */
		  returncode = FW_BLOCK;
	      }
	  }
        else if (rule->fw_rc == SF_RC_ECHO) 
          {
            if ((ip->protocol == IPPROTO_ICMP) && (icmp->type == ICMP_ECHO))
              icmp_reflect(rif, ip, icmp);
            returncode = FW_BLOCK; /* in any case, block the packet */
          }
	else if (rule->fw_rc < FW_REJECT) /* Filter ICMP options */
	  if (opt == SF_STATE_TRANSMIT) 
	    returncode = FW_REJECT;
	  else
	    {
	      /* Linux FW functions are limited -- do it yourself... */
	      switch (rule->fw_rc)
	        { 
	          case SF_RC_RNET:   send_icmp_error(rif, ip, ICMP_DEST_UNREACH,
					       ICMP_NET_UNREACH);
		  break;
		  case SF_RC_RPROTO: send_icmp_error(rif, ip, ICMP_DEST_UNREACH,
					       ICMP_PROT_UNREACH);
	          break;
		  case SF_RC_RPORT:  send_icmp_error(rif, ip, ICMP_DEST_UNREACH,
					       ICMP_PORT_UNREACH);
		  break;
	        }
	      returncode = FW_BLOCK;
	    }

	if ( !(rule->fw_flags & SF_FW_LOG) ) goto creturn; /* no notification */
      
informlog:
	/* send packet to daemon, append to message buffer */
	if (sf_log_data.num_entries < NUM_PROC_ENTRIES) {
		struct sf_proc_entry *current_p
				= &(sf_log_data.data[sf_log_data.num_entries]);
		
	  current_p->action   = rule->fw_rc;
	  current_p->rule.num = rule->rule_id.num;
	  strncpy(current_p->devname,rif->name,7);
	  current_p->devname[7]=0;
	  memcpy(current_p->ip_packet, ip, MIN(PROC_SIZE_IP, ntohs(ip->tot_len)));	  
	  sf_log_data.num_entries++;
	}
	else {
	  sf_log_data.lost_entries++;

	  /* wake up daemon */
	  wake_up_interruptible(&sf_log_entry);

	  /* drop packet silently if buffer is full */
	  return FW_BLOCK;
	}

	/* wake up daemon */
	wake_up_interruptible(&sf_log_entry);

creturn:
	/* return rule dependent return code */
	return returncode;
}

/* 
 * initialization and interface functions
 */

/* return pointer to first rule */
long sf_rule_first(void)
{ rule_first_next = rules;
  return (long) rule_first_next;
}

/* return pointer to next rule */
long sf_rule_next(void)
{ if (rule_first_next!=NULL) 
    { rule_first_next = rule_first_next->fw_next;
      return (long) rule_first_next;
    }
  else return 0;
}

int sf_init(void)
{ int i;
  unsigned long flags;

#ifdef SF_DEBUG
  printk("sf_init() called\n");
#endif

  save_flags(flags); cli();

  /* initialize rules list */
  rules = NULL;
  rule_first_next = rules;

  /* initialize fragment hash queues */
  for (i=0; i<SF_FRAG_HASH_SIZE; i++) sf_frag[i] = NULL; 

  /* initialize tcp hash queues */
  for (i=0; i<SF_TCP_HASH_SIZE; i++) sf_tcp[i] = NULL;

  sf_fw_initialized = 1;
  restore_flags(flags);

  return 0;
}

int sf_clear(void)
{ unsigned long flags;
  struct sf_fw *rule;
 
#ifdef SF_DEBUG
  printk("sf_clear() called\n");
#endif

  save_flags(flags); cli();

  while (rules != NULL)
    { rule = rules;
      rules = rules->fw_next;
      kfree_s(rule,sizeof(struct sf_fw));
    }

  restore_flags(flags); 
  return 0; 
}

int sf_add(struct sf_fw *rule)
{ unsigned long flags;

#ifdef SF_DEBUG
  printk("sf_add() called, adding rule %li\n",rule->rule_id.num);
#endif

  save_flags(flags); cli();

  rule->fw_next = rules;
  rules = rule;

  restore_flags(flags);
  return 0;
}

int sf_replace(struct sf_fw *rule)
{ unsigned long flags;
  struct sf_fw *rp1, *rp2;

#ifdef SF_DEBUG
  printk("sf_replace() called, replacing rule %li\n",rule->rule_id.num);
#endif

  save_flags(flags); cli();

  rp1 = rules;
  rp2 = NULL;
  while ( (rp1 != NULL) && (rp1->rule_id.num != rule->rule_id.num) )
    { rp2 = rp1;
      rp1 = rp1->fw_next;
    }
  if (rp1 == NULL) /* id not found */
    { restore_flags(flags);
      return -1; 
    }
  if (rp2 != NULL) rp2->fw_next = rule;
    else rules = rule;
  rule->fw_next = rp1->fw_next;
  kfree_s(rp1,sizeof(struct sf_fw));

  restore_flags(flags);
  return 0;
}

int sf_delete(unsigned long id)
{ unsigned long flags;
  struct sf_fw *rp1, *rp2;

#ifdef SF_DEBUG
  printk("sf_delete() called, deleting rule %lu\n",id);
#endif

  save_flags(flags); cli();

  rp1 = rules;
  rp2 = NULL;
  while ( (rp1 != NULL) && (rp1->rule_id.num != id) )
    { rp2 = rp1;
      rp1 = rp1->fw_next;
    }
  if (rp1 == NULL) /* id not found */
    { restore_flags(flags);
      return -1;
    }
  if (rp2 != NULL) rp2->fw_next = rp1->fw_next;
    else rules = rp1->fw_next;
  kfree_s(rp1,sizeof(struct sf_fw));

  restore_flags(flags);
  return 0;
}

int sf_flush(void)
{ struct sf_fw *rule;
  struct sf_tcp_connection *sftcp;
  unsigned long flags;
  int i;

  save_flags(flags); cli();

#ifdef SF_DEBUG
  printk("sf_flush() called\n");
#endif

  for (i=0; i<SF_TCP_HASH_SIZE; i++)
    { sftcp=sf_tcp[i];
      while (sftcp!=NULL)
        { unsigned short source = ntohs(sftcp->client_port);
	  unsigned short dest   = ntohs(sftcp->server_port);
          for (rule=rules; rule!=NULL; rule=rule->fw_next)
            { 
	      if (   (rule->protocol != IPPROTO_TCP) 
		  && ((rule->fw_flags & SF_FW_PROT) != SF_FW_PROT_ALL) )continue;
	      if (rule->fw_src_cnt)
                { int match = sf_addr_match(rule->fw_src_idx,rule->fw_src_cnt,
                    sftcp->client_addr,source,rule->fw_flags & SF_FW_SRC_NEG);
		  if ( ( match &&  (rule->fw_flags & SF_FW_SRC_NEG) )
		    || (!match && !(rule->fw_flags & SF_FW_SRC_NEG) ) ) continue;
	        }
              if (rule->fw_dst_cnt)
                { int match = sf_addr_match(rule->fw_dst_idx,rule->fw_dst_cnt,
                    sftcp->server_addr,dest,rule->fw_flags & SF_FW_DST_NEG);
                  if ( ( match &&  (rule->fw_flags & SF_FW_DST_NEG) )
                    || (!match && !(rule->fw_flags & SF_FW_DST_NEG) ) ) continue;
                }
	      break; /* rule matches */
            }
	  if ( (rule==NULL) || (rule->fw_rc!=FW_ACCEPT) )
	    { struct sf_tcp_connection *sftcp2;
              if (sftcp->timer_set) del_timer(&sftcp->timer);
	      sftcp2 = sftcp->next;
	      sf_tcp_delete(sftcp);
	      sftcp = sftcp2;
            }
           else
	    sftcp=sftcp->next;
        }
    }
  restore_flags(flags);
  return 0;
}

int sf_flush_all(void)
{ int i;
  struct sf_tcp_connection *sftcp;
  unsigned long flags;

  save_flags(flags); cli();

#ifdef SF_DEBUG
  printk("sf_flush_all() called\n");
#endif

  for (i=0;i<SF_TCP_HASH_SIZE;i++)
    { sftcp=sf_tcp[i];
      while (sftcp != NULL)
        { 
#ifdef SF_DEBUG
          printk("sf: aborting %p ",sftcp);
          PRINTIP(sftcp->client_addr);
	  printk(":%u -> ",ntohs(sftcp->client_port));
          PRINTIP(sftcp->server_addr);
          printk(":%u\n",ntohs(sftcp->server_port));
#endif
	  if (sftcp->timer_set) del_timer(&sftcp->timer);
	  sf_tcp[i]=sftcp->next;
	  kfree_s(sftcp,sizeof(struct sf_tcp_connection));
	  sftcp=sf_tcp[i];
        }
    }
  restore_flags(flags); 
  return 0;
}

void sf_del_all_timers(void)
{ struct sf_fragment *frag;
  struct sf_tcp_connection *tcp, *otcp;
  int i;
  unsigned long flags;

  save_flags(flags); cli();

  for (i=0; i<SF_FRAG_HASH_SIZE; i++)
    while ( (frag = sf_frag[i]) != NULL)
      { del_timer(&frag->timer);
	sf_frag[i]=frag->next;
	kfree_s(frag,sizeof(struct sf_fragment));
      }

  for (i=0; i<SF_TCP_HASH_SIZE; i++)
    { tcp = sf_tcp[i];
      while (tcp != NULL)
        { if (tcp->timer_set)
	    { otcp = tcp;
	      tcp = tcp->next;
	      del_timer(&otcp->timer);
	      sf_tcp_delete(otcp);
	    }
	   else tcp = tcp->next;
	}
    }
 
  restore_flags(flags);
}

int sf_write_config(const char *buffer, int len)
{
	struct sf_control *control;
	struct sf_fw *rule;
        int rc, i;

        control = (struct sf_control *) kmalloc(len, GFP_ATOMIC);

	memcpy_fromfs(control, buffer, len);

	if (control->magic != SF_CONTROL_MAGIC)
	  { kfree_s(control,len);
	    return -EINVAL;
	  }

	if (!sf_fw_initialized) sf_init();

	switch (control->command)
	{
	  case SF_COMMAND_DISABLE:	sf_fw_enabled = 0;
					rc = len;
					break;

	  case SF_COMMAND_ENABLE:	sf_log_data.num_entries = 0;
					sf_log_data.lost_entries = 0;
					sf_fw_enabled = 1;
					rc = len;
					break;

	  case SF_COMMAND_CLEAR_LOST:	sf_log_data.lost_entries = 0;
					rc = len;
					break;

	  case SF_COMMAND_FW_CLEAR:	if (sf_clear() != 0)
					  { rc = -1; 
					    break;
					  }
					rc = len;
					break;

	  case SF_COMMAND_FW_ADD:	if (len != (sizeof(struct sf_control)+sizeof(struct sf_fw)))
					  { rc = -EINVAL;
					    break;
					  }
					if ((rule = (struct sf_fw *) kmalloc(sizeof(struct sf_fw),GFP_ATOMIC)) == NULL)
					  { rc = -ENOMEM;
					    break;
					  }
					memcpy(rule,control->data,sizeof(struct sf_fw));
					if (sf_add(rule) != 0)
					  { rc = -1;
					    break;
					  }
					rc = len;
					break;

	  case SF_COMMAND_FW_REPLACE:	if (len != (sizeof(struct sf_control)+sizeof(struct sf_fw)))
					  { rc = -EINVAL;
					    break;
					  }
					if ((rule = (struct sf_fw *) kmalloc(sizeof(struct sf_fw),GFP_ATOMIC)) == NULL)
					  { rc = -ENOMEM;
					    break;
					  }
					memcpy(rule,control->data,sizeof(struct sf_fw));
					if (sf_replace(rule) != 0)
					  { rc = -1;
					    break;
					  }
					rc = len;
					break;

	  case SF_COMMAND_FW_DELETE:	if (len != (sizeof(struct sf_control)+sizeof(unsigned long)))
					  { rc = -EINVAL;
					    break;
					  }
					if (sf_delete(*(unsigned long *)(control->data)) != 0)
					  { rc = -1;
					    break;
					  }
					rc = len;
					break;

	  case SF_COMMAND_FW_ADDR:	
#ifdef SF_DEBUG
  printk("sf_addr: transmitted address array\n");
#endif
					if (len != (sizeof(struct sf_control)+sizeof(sf_addr)))
					  { rc = -EINVAL;
					    break;
					  }
					memcpy(sf_addr,control->data,sizeof(sf_addr));
					sf_addr_free = sf_addr[0].mask;

					/* initialize interface cache hash table */
					for (i=0; i<SF_IF_CACHE_SIZE; i++) 
					  sf_if_cache[i].ifaddr = 0;

					rc = len;
					break;

	  case SF_COMMAND_FLUSH:	if (sf_flush() != 0)
					  { rc = -1;
					    break;
					  }
					rc = len;
					break;

	  case SF_COMMAND_FLUSH_ALL:	if (sf_flush_all() != 0)
					  { rc = -1;
					    break;
					  }
					rc = len;
					break;


	  default:			rc = -EINVAL;
	}

	kfree_s(control,len);
	return rc;

}


int sf_forward_chk(struct firewall_ops *this, 
				   int pf, 
				   struct device *dev,
				   void *phdr,
                   void *arg)
{
	return sf_fw_chk(phdr,dev,SF_STATE_FORWARD); 
}


int sf_input_chk(struct firewall_ops *this, 
				 int pf,
				 struct device *dev,
				 void *phdr,
                 void *arg)
{
	return sf_fw_chk(phdr, dev, SF_STATE_RECEIVE); 
}


int sf_output_chk(struct firewall_ops *this, 
				 int pf,
				 struct device *dev,
				 void *phdr,
                 void *arg)
{
	return sf_fw_chk(phdr,dev,SF_STATE_TRANSMIT); 
}

struct firewall_ops sf_fw_ops = {
	NULL,
	sf_forward_chk,
	sf_input_chk,
	sf_output_chk,
	PF_INET,
	1               /* higher priority than std. firewall */ 
}; 

