/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * In this file all data structures are defined that are used
 * by the firewall daemon and by the filter function.
 *
 * $Id: sf_global.h,v 1.19 1995/09/12 15:08:59 robby Rel $
 *
 */

#ifndef _SF_GLOBAL_H
#define _SF_GLOBAL_H

/*
 *      Format of an IP firewall descriptor
 */

struct sf_fw 
{
#define SF_FW_MAGIC 0x524F524FL                 /* RORO */
        unsigned long magic;                    /* Magic value */

        struct sf_fw  *fw_next;                 /* Next firewall on chain */

	int fw_line;				/* Zeilennummer der Regel */

	union {
 	  long num;				/* identification of entry */
	  struct sf_fw *ptr;			/* filter uses num, daemon uses ptr */
	} rule_id;

	union {					/* union to save storage */
	  struct notification *ptr;		/* pointer to notification level */
	  int num;				/* notification level for rule */
	} level;				/* num is only valid during init */

#define SF_RULE_STATIC                  -1
#define SF_RULE_DYNAMIC_INACTIVE        -2
#define SF_RULE_SPOOFING                -3
#define SF_RULE_OVERSIZED		-4
	long timeout;				/* state of rule (active, ...) */

	int fw_rc;				/* return code on match (see sf_rc.h) */

#define SF_SOURCEHOST		-1
#define SF_SOURCENET		-2
#define SF_DESTHOST		-3
#define SF_DESTNET		-4
	long fw_src_idx, fw_src_cnt;   		/* Index to address array for */
        long fw_dst_idx, fw_dst_cnt;   		/*  source and dest. IP addresses */

	unsigned long fw_flags;			/* see below */
	__u8 ttl;				/* time to live value to compare with */
	__u8 protocol;				/* protocol */
	long fw_rip_idx, fw_rip_cnt;   		/* index to rip addresses */
};

/*
 *	Values for "flags" field:

 *       bit         meaning
 *        0           log packet (notification_level > 0)
 *        1           check ttl field
 *        2-3         ttl: 00 -> ttl = value
 *                         01 -> ttl < value
 *                         10 -> ttl > value
 *                         11 -> ttl != value
 *        4	      1 = match "from" if none of the specified 
 *			  addresses match (used for "outside")
 *	  5	      1 = match "to" if none of the specified
 *                        addresses match (used for "outside")
 *        6           option record_route
 *        7           option timestamp
 *        8           option security
 *        9           option loose_source_route
 *       10           option strict_source_route
 *       11           option sat_id
 *       12-15        protocol: 0000  all protocols
 *                              0001  rip
 *				1111  current protocol (dynamic rule)
 *       16-31        type (protocol dependent) 
 */

/* structure of address array entry */

#define SF_ADDRESS_CNT_MAX   340   /* 340 * 12 bytes = 4080 bytes */
struct sf_address {
	__u32		addr, mask;	/* ip address and mask (this is NOT the netmask) */
					/* (both in network byte order) */
	unsigned short  port, prend;    /* port, end of port range (host byte order) */
        /* if port  is 0 (i.e. SF_ALL_PORTS), all ports match */
        /* if prend is 0, the address entry is free */
};

#define SF_ALL_PORTS          0    /* match on all ports */
#define SF_IGNORE_ADDRESS     0    /* match on all ip addresses */
#define SF_SOURCEHOST        -1    /* match source address in dynamic rule */
#define SF_SOURCENET         -2    /* match source net in dynamic rule */
#define SF_DESTHOST          -3    /* match destination address in dynamic rule */
#define SF_DESTNET           -4    /* match destination net in dynamic rule */

#define SF_FW_LOG            0x0001  /* log packet */

#define SF_FW_CHECK_TTL      0x0002  /* check time to live field */
#define SF_FW_TTL            0x000E  /* mask for SF_FW_TTL */
#define SF_FW_TTL_EQUAL      0x0002  /* ttl = value */
#define SF_FW_TTL_LESS       0x0006  /* ttl < value */
#define SF_FW_TTL_GREATER    0x000A  /* ttl > value */
#define SF_FW_TTL_NOTEQUAL   0x000E  /* ttl != value */

#define SF_FW_SRC_NEG	     0x0010  /* from condition negated */
#define SF_FW_DST_NEG	     0x0020  /* to condition negated */

#define SF_FW_CHECK_OPT	     0x0FC0  /* check for options */
#define SF_FW_OPT_RR         0x0040  /* record route */
#define SF_FW_OPT_TS         0x0080  /* time stamp */
#define SF_FW_OPT_SEC        0x0100  /* security */
#define SF_FW_OPT_LSR        0x0200  /* loose source route */
#define SF_FW_OPT_SSR        0x0400  /* strict source route */
#define SF_FW_OPT_SATID      0x0800  /* sat id */

#define SF_FW_PROT           0xF000  /* mask to isolate protocol */
#define SF_FW_PROT_ALL       0x0000  /* matches all protocols */
#define SF_FW_PROT_RIP       0x1000  /* rip */
#define SF_FW_CHECK_PROTOCOL 0xE000  /* compare protocol field */
#define SF_FW_PROT_CURRENT   0xF000  /* match current protocol in dynamic rule */

/* definitions for protocol dependent type bits: */
#define SF_TYPE_MASK           0xFFFF0000  /* mask to isolate protcol dependent bits */

#define SF_ICMP_ALLTYPES       0x00000000
#define SF_ICMP_ECHOREPLY      0x00010000  /* Echo Reply                   */
#define SF_ICMP_DEST_UNREACH   0x00020000  /* Destination Unreachable      */
#define SF_ICMP_SOURCE_QUENCH  0x00040000  /* Source Quench                */
#define SF_ICMP_REDIRECT       0x00080000  /* Redirect (change route)      */
#define SF_ICMP_ECHO           0x00100000  /* Echo Request                 */
#define SF_ICMP_TIME_EXCEEDED  0x00200000  /* Time Exceeded                */
#define SF_ICMP_PARAMETERPROB  0x00400000  /* Parameter Problem            */
#define SF_ICMP_TIMESTAMP      0x00800000  /* Timestamp Request            */
#define SF_ICMP_TIMESTAMPREPLY 0x01000000  /* Timestamp Reply              */
#define SF_ICMP_INFO_REQUEST   0x02000000  /* Information Request          */
#define SF_ICMP_INFO_REPLY     0x04000000  /* Information Reply            */
#define SF_ICMP_ADDRESS        0x08000000  /* Address Mask Request         */
#define SF_ICMP_ADDRESSREPLY   0x10000000  /* Address Mask Reply           */

#define SF_IGMP_ALLTYPES               0x00000000
#define SF_IGMP_HOST_MEMBERSHIP_QUERY  0x00010000  /* From RFC1112 */
#define SF_IGMP_HOST_MEMBERSHIP_REPORT 0x00020000  /* Ditto */
#define SF_IGMP_HOST_LEAVE_MESSAGE     0x00040000  /* An extra BSD seems to send */ 

#define SF_RIP_ADDR_NEG        0x00010000  /* match rip-addr if none of the specified */
                                           /* addresses match (used for "outside") */

#endif /* _SF_GLOBAL_H */
