/*  ----------------------------------------------------------------------
    sf Firewall Software -- a TCP/IP packet filter for Linux
    Copyright (C) 1996 Robert Muchsel and Roland Schmid

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please address all correspondence concerning the software to 
    firewall-bugs@switch.ch.
    ----------------------------------------------------------------------  */

/*
 * Routines to send TCP reset for the SF firewall packet filter
 * 
 * $Id: sf_icmp.c,v 1.0 1996/08/28 14:37:44 robby Rel $
 *
 */

#include <linux/config.h>
#include <stdio.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/inet.h>
#include <linux/netdevice.h>
#include <linux/icmp.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/ip.h>
#include <linux/malloc.h>
#include <linux/timer.h>
#include "sf_kernel.h"
#include "sf_rc.h"
#include "sf_global.h"
#include "sf_filter.h"
#include "sf_control.h"
#include "sf_ip.h"
#include "sf_icmp.h"

void icmp_reflect(struct device *dev, struct iphdr *ip, struct icmphdr *icmp)
{
  struct sk_buff *skb;
  struct rtable *rt;
  struct icmphdr *icmp2;
  __u32 saddr, daddr;
  register const __u16 *sp;
  register __u32 csum;
  int i;

  int len = sizeof(struct iphdr) + ntohs(ip->tot_len) - (ip->ihl<<2);

  if (icmp->type != ICMP_ECHO) /* Cannot reply */
    return;

#ifdef SF_DEBUG
  printk("icmp_reflect\n");
#endif

  /* Swap ! */
  saddr = ip->daddr;
  daddr = ip->saddr;

  skb = dev_alloc_skb(len + dev->hard_header_len + 15); 
  if (skb == NULL)
    return;
  IS_SKB(skb);

  skb->link3 = NULL;
  skb->sk = NULL; 
  skb->when = jiffies;
  skb->pkt_type = PACKET_OTHERHOST;
  skb->csum = 0;
  
  skb->saddr = saddr; 
  skb->daddr = daddr;

  skb->ip_hdr = (struct iphdr *) skb_put(skb, len);
  memcpy (skb->ip_hdr, ip, sizeof(struct iphdr));
  /* reverse packet, strip options */
  skb->ip_hdr->saddr = saddr;
  skb->ip_hdr->daddr = daddr;
  skb->ip_hdr->ttl = 255;
  skb->ip_hdr->tot_len = htons(len);
  skb->ip_hdr->frag_off = 0;
  skb->ip_hdr->ihl = 5; 

  rt = ip_rt_route(daddr, 0);
  skb->localroute = 0;
  skb->raddr = rt ? rt->rt_gateway : daddr;

  /* Add the physical headers */
  ip_mksend(rt, skb, daddr, len, dev, saddr);
  
  ip_rt_put(rt); 

  ip_send_check(skb->ip_hdr);

  /* change ICMP header */
  icmp2 = (struct icmphdr *) (((char *)skb->ip_hdr) + (skb->ip_hdr->ihl << 2));
  memcpy (icmp2, icmp, len - sizeof(struct iphdr));
  icmp2->type     = ICMP_ECHOREPLY;
  icmp2->checksum = 0;

  /* Calculate ICMP checksum */
  csum = 0; 
  sp = (__u16 *) icmp2;
  for (i = 0; i < (len - sizeof(struct iphdr)) >> 1; i++, sp++)
    csum += *sp;
  while (csum >> 16)
    csum  = (csum & 0xffff) + (csum >> 16);
  icmp2->checksum = ~csum;

  /* And send the packet... */
  dev_queue_xmit(skb, dev, 0);

  dev_kfree_skb(skb, FREE_WRITE);
}

void send_icmp_error(struct device *dev, struct iphdr *ip, int type, int code)
{
  /* send ICMP error */
  struct sk_buff skb; /* fake skb */
  memset(&skb, 0, sizeof(skb));

#ifdef SF_DEBUG
  printk("send_icmp_error\n");
#endif

  skb.ip_hdr   = ip;
  skb.pkt_type = PACKET_HOST;
  skb.len      = ntohs(ip->tot_len) - (ip->ihl << 2);

  icmp_send(&skb, type, code, 0, dev);
}

