
/*
 * ----------------------------------------------------------------------
 * sf Firewall Software -- a TCP/IP packet filter for Linux
 * Copyright (C) 1996 Robert Muchsel and Roland Schmid
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Please address all correspondence concerning the software to 
 * firewall-bugs@switch.ch.
 * ----------------------------------------------------------------------
 */

#include <stdio.h>
#include <stdlib.h>
#include <netdb.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <fcntl.h>
#include <strings.h>
#include <netinet/in.h>
#include <arpa/inet.h>

struct hash {
  struct hash *next;
  int addr;
  char *name;
};

#define HASH_SIZE 199
struct hash *hashtable[HASH_SIZE];
#define HASH_KEY(x) (((unsigned) x) % HASH_SIZE)

void getname(int addr, char **name, int quiet)
{
  struct hostent *h_ent;
  unsigned key = HASH_KEY(addr);
  struct hash *h = hashtable[key];

  if (h == NULL)
    h = hashtable[key] = malloc(sizeof(struct hash));
  else {
    while ((h->addr != addr) && (h->next != NULL))
      h = h->next;

    if (h->addr == addr) {
      if (h->name != NULL)
        *name = h->name; 
      return;
    }

    h->next = malloc(sizeof(struct hash));
    h = h->next;
  }
  if (h == NULL) {
    fprintf(stderr, "malloc failed\n"); 
    return;
  }

  h->next = NULL;
  h->name = NULL;
  h->addr = addr;

  if (!quiet) {
    fprintf(stderr, "\r[%s]", *name); fflush(stderr);
  }
  h_ent = gethostbyaddr((char *) &addr, sizeof(struct in_addr), AF_INET);
  if (!quiet) {
    fprintf(stderr, "\r                 \r"); fflush(stderr);
  }
  if ((h_ent != NULL) && (h_ent->h_name != NULL) && (h_ent->h_name[0])) {
    h->name = strdup(h_ent->h_name);
    *name = h->name;
  }
}

int main(int argc, char **argv)
{
  FILE *f;
  char c, s[255], arrow[5];
  char logname[255] = "/var/log/firewall";
  struct stat stats;
  int tail = 0, onlyblock = 0, forever = 0, lines = 0, written = 0,
      tcp = 0, udp = 0, icmp = 0, quiet = 0;

  memset(hashtable, 0, sizeof(hashtable));

  while ((c = getopt(argc, argv, "f:tbcqTUI")) != EOF) {
    switch (c) {
      case 'f': strcpy(logname, optarg); 
      break;

      case 't': tail = 1;
      break;
       
      case 'b': onlyblock = 1;
      break;
       
      case 'c': forever = 1;
      break;

      case 'T': tcp = 1;
      break;

      case 'U': udp = 1;
      break;

      case 'I': icmp = 1;
      break;

      case 'q': quiet = 1;
      break;

      case '?': fprintf(stderr, "usage: %s [-tbcqTUI] [-f firewall_log_file]\n", argv[0]);
                return -1;
      break;
    }
  } 

  if (optind < argc) {
     fprintf(stderr, "usage: %s [-tbcqTUI] [-f firewall_log_file]\n", argv[0]);
     return -1;
  }

  if (!icmp && !tcp && !udp)
    tcp = icmp = udp = 1;

  if (strcmp(logname,"-") == 0)
    f = stdin;
  else
    f = fopen(logname, "r");
  if (f == NULL) {
    fprintf(stderr, "cannot open firewall log file '%s'\n", logname);
    return -1;
  }

  if (fstat(fileno(f), &stats) < 0) {
    fprintf(stderr, "stat error\n");
    return -1;
  }

  if (tail) {
    int filepos = stats.st_size > 1500 ? stats.st_size-1500 : 0;

    printf("Seeking to end of file (%i)...\n", filepos);
    fseek(f, filepos, SEEK_SET);
    fgets(s, sizeof(s), f);
  }

  /* Use single TCP connection for queries */
  sethostent(1);

  while (forever || (ftell(f) < stats.st_size)) {
    char s0[255], s1[255], s2[255], s3[255],
         sip[255], dip[255], sport[255], dport[255];
    char *scip, *dcip;
    int snip, dnip;

    if (fstat(fileno(f), &stats) < 0) {
      fprintf(stderr,"stat error\n");
      return -1;
    }

    if (stats.st_size > ftell(f)) {
      memset(s, 0, sizeof(s));
      fgets(s, sizeof(s), f);
      lines++;

      if (strlen(s) < 16)
        continue;

      /* Parse string */
      memset(sip, 0, sizeof(sip));
      memset(dip, 0, sizeof(dip));
      s0[0] = s1[0] = s2[0] = s3[0] = sport[0] = dport[0] = 0;
      
      sscanf(&s[16], "%s %s %[^ \n] %[^0-9\n]%[.0-9]%[^>\n]%[>]%[.0-9]%[^\n]", 
             s0, s1, s2, s3, sip, sport, arrow, dip, dport);
      if (strcmp(s0, "last") != 0) {	
        if ((strcmp(s2, "block") == 0) || (!onlyblock &&
            ((strcmp(s2, "accept") == 0) ||
             (strncmp(s2, "reject", 6) == 0)))) {

          if (s3[3] == 0) { s3[3] = ' '; s3[4] = 0; }
	  if (s3[0] == '(') {
	    char junk[255];
            char *c = s3;
	    
	    while (*c && *c != ')')
              c++;
            if (*c) c++;
            if (*c) c++;
            strcpy(junk, c);
            strcpy(s3, junk); 

            if (s3[3] == 0) { s3[3] = ' '; s3[4] = 0; }
	  }
          else
            if (s3[3] != ' ') s3[strlen(s3)-1] = 0;
          s2[6] = 0; if (!s2[5]) s2[5] = ' ';

	  if (((strncmp(s3,"TCP",3) == 0) ||
               (strncmp(s3,"ICMP",4) == 0) ||
               (strncmp(s3,"UDP",3) == 0)) &&
               (!sip[0])) {
            char t[255];

            memset(t, 0, sizeof(t));
            fgets(t, sizeof(t), f);
            lines++;
            sscanf(t, "%*[ ]%[.0-9]%[^>\n]%[>]%[.0-9]%[^\n]", 
                   sip, sport, arrow, dip, dport);
          }

          if ((strncmp(s3,"TCP",3) == 0) && !tcp)
            continue;
          if ((strncmp(s3,"UDP",3) == 0) && !udp)
            continue;
          if ((strncmp(s3,"ICMP",4) == 0) && !icmp)
            continue;

          inet_aton(sip, (struct in_addr *) &snip); scip = sip;
          inet_aton(dip, (struct in_addr *) &dnip); dcip = dip;
          getname(snip, &scip, quiet);
          getname(dnip, &dcip, quiet);
          printf("%c%c:%c%c %s %s %s%s>%s%s\n", s[7], s[8], s[10], s[11],
                 s2, s3, scip, sport, dcip, dport);
          written++;
        }
        else {
          printf("%c%c:%c%c %s", s[7], s[8], s[10], s[11], &s[16]);
          written++;
        }
      }
    }
    else { 
      if (forever)
        sleep(1); 
    }
  }

  if (!written)
    printf("Scanned %i lines - no relevant data found\n", lines);

  endhostent();
  fclose(f);
  return 0;
}
