#!/usr/bin/env python

from gnuradio import gr, eng_notation
from gnuradio import usrp
from gnuradio import audio
from gnuradio import blks
from gnuradio.eng_option import eng_option
from optparse import OptionParser
import math
import sys

from gnuradio.wxgui import stdgui, fftsink
import wx

########################################################
# instantiate one transmit chain for each call

class pipeline(gr.hier_block):
    def __init__(self, fg, filename, lo_freq, audio_rate, if_rate):

        src = gr.file_source (gr.sizeof_float, filename, True)
        fmtx = blks.nbfm_tx (fg, audio_rate, if_rate,
                             max_dev=5e3, tau=75e-6)
        
        # Local oscillator
        lo = gr.sig_source_c (if_rate,        # sample rate
                              gr.GR_SIN_WAVE, # waveform type
                              lo_freq,        #frequency
                              1.0,            # amplitude
                              0)              # DC Offset
        mixer = gr.multiply_cc ()
    
        fg.connect (src, fmtx, (mixer, 0))
        fg.connect (lo, (mixer, 1))

        gr.hier_block.__init__(self, fg, src, mixer)



class fm_tx_graph (stdgui.gui_flow_graph):
    def __init__(self, frame, panel, vbox, argv):
        MAX_CHANNELS = 7
        stdgui.gui_flow_graph.__init__ (self, frame, panel, vbox, argv)

        parser = OptionParser (option_class=eng_option)
        parser.add_option ("-c", "--ddc-freq", type="eng_float", default=29.325e6,
                           help="set Tx ddc frequency to FREQ", metavar="FREQ")
        parser.add_option ("-n", "--nchannels", type="int", default=4,
                           help="number of Tx channels [1,4]")
        (options, args) = parser.parse_args ()

        if options.nchannels < 1 or options.nchannels > MAX_CHANNELS:
            sys.stderr.write ("fm_tx4: nchannels out of range.  Must be in [1,%d]\n" % MAX_CHANNELS)
            sys.exit (1)
        
        print "ddc_freq = %s" % (eng_notation.num_to_str (options.ddc_freq))

        # ----------------------------------------------------------------
        # Set up constants and parameters

        self.dac_rate = 128e6
        self.usrp_interp = 400
        self.usrp_rate = self.dac_rate / self.usrp_interp    # 320 kS/s
        self.sw_interp = 10
        self.audio_rate = self.usrp_rate / self.sw_interp    # 32 kS/s

        sum = gr.add_cc ()

        # Instantiate 4 fm channels
        step = 25e3
        offset = (0 * step, 1 * step, -1 * step, 2 * step, -2 * step, 3 * step, -3 * step)
        for i in range (options.nchannels):
            t = pipeline (self, "audio-%d.dat" % (i % 4), offset[i],
                          self.audio_rate, self.usrp_rate)
            self.connect (t, (sum, i))

        gain = gr.multiply_const_cc (4000)

        # USRP interface
        u = usrp.sink_c (0, self.usrp_interp)
        u.set_tx_freq (0, options.ddc_freq)

        # connect it all
        self.connect (sum, gain)
        self.connect (gain, u)

        # plot an FFT to verify we are sending what we want
        if 1:
            post_mod, fft_win2 = fftsink.make_fft_sink_c (self, panel, "Post Modulation",
                                                          512, self.usrp_rate,
                                                          -20, 80)
            self.connect (sum, post_mod)
            vbox.Add (fft_win2, 1, wx.EXPAND)
            

def main ():
    app = stdgui.stdapp (fm_tx_graph, "Quad FM Tx")
    app.MainLoop ()

if __name__ == '__main__':
    main ()
