;;; patcher.el --- Utility for mailing patch information

;; Copyright (C) 1999-2001 Didier Verna.

;; Author:        Didier Verna <didier@xemacs.org>
;; Maintainer:    Didier Verna <didier@xemacs.org>
;; Created:       Tue Sep 28 18:12:43 1999
;; Last Revision: Wed Apr 25 14:57:40 2001
;; Keywords:      maint

;; This file is part of Patcher.

;; Patcher is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; Patcher is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


;;; Commentary:

;; Contents management by FCM version 0.1.

;; Many thanks to Ben Wing <ben@xemcas.org> for good ideas and code.

;; This package automates the process of building and submitting patches for
;; archive-based projects you're working on. In one or two keystrokes, it
;; prepares a mail with a patch corresponding to the differences between your
;; working version and the archive state, and prepares a skeleton for the
;; ChangeLog entries, that you can fill in and insert into the message before
;; sending.

;; Patcher currently supports the `compose-mail' mail sending interface, as
;; well as the `sendmail' and `message' libraries individually, which is
;; probably redundant. There is also a special support for sending mail from a
;; running Gnus session.

;; Here's a typical usage:

;; 0/ After loading the package, customize the `patcher' group to suit your
;;    needs.
;; 1/ Once you have made modifications to one of your projects and you'd like
;;    to submit them, type `M-x patcher-mail' (see the docstring).  The
;;    message will be prepared and the ChangeLog skeletons built.
;;    Alternately, use `M-x patcher-mail-subproject' to mail (and possibly
;;    commit) changes to particular directories and/or files in the project.
;; 2/ Edit the different ChangeLog buffers that have opened, and fill in the
;;    skeletons.  You can save the ChangeLogs buffers, BUT DON't KILL THEM !!
;; 3/ If you want to commit your changes immediately, type C-c C-p c.  You'll
;;    have the opportunity to edit the message log and commit your changes.
;; 4/ Otherwise, or just after that, type `C-c C-p i' in the message buffer.
;;    The new ChangeLog entries will be inserted just above the patch.  Then,
;;    send it.
;; 5/ That's all folks.

;; Requirements:

;; This package requires a recent `add-log' library (from the xemacs-base
;; package version 1.48 or later), and the `sendmail' library.  The different
;; mail methods will also have specific requirements (especially the Gnus
;; one).  Please, refer to them for further information.

;; Suggestions for further improvements:

;; #### Consider doing a better job of handling overlapping patcher
;; instances.  e.g. use a different extent property for each instance,
;; and keep a usage count for the ChangeLog files so that they're only
;; killed when no instance uses them any more.
;;
;; #### Also add an option to kill, not just bury, the mail message when
;; it's sent.
;;
;; #### When the message is sent, the cvs commit results window should be
;; removed and the corresponding buffer buried.
;;
;; #### Have a control window to clarify the progress of everything.
;; Make it perhaps a small buffer, above the mail message/ChangeLog
;; buffer.  It shows what steps have been completed, what haven't, and
;; what to do.  It should have buttons in it for the various actions.
;; One button is for include-changelogs, one for commit, one for send
;; the mail, and one for execute the commit.  These should be enabled
;; or grayed out appropriately.  It should also have buttons that show
;; the associated changelogs; clicking on a button puts that changelog
;; buffer in the main window below the control window.  By each
;; changelog button should be an indication of whether the changelog
;; has been modified so far.  The control window should stay around as
;; much as it can during the whole process (e.g. use
;; set-buffer-dedicated-p or something), so that it's always there to
;; drive the whole process.  One corollary is that you don't actually
;; have to switch to the mail buffer to (e.g.) execute
;; include-changelogs -- you just click on the control window, and it
;; does it automatically.  also, when you execute include-changelogs,
;; it can issue a warning if not all changelogs have been visited, and
;; prompt you to make sure you want to continue.  Similarly, you can
;; run include-changelogs more than once even if it succeeded the
;; first time (e.g. for some reason you didn't like the result and
;; deleted it), but it will prompt "are you sure?".  there could also
;; be an "undo include-changelogs", if you make a mistake after doing
;; include-changelogs and realize you want to go back and fix the
;; problem and do include-changelogs again.


;;; Code:

(require 'add-log)
(require 'sendmail)

(defgroup patcher nil
  "Patch mailing utilities.")


;; Projects description =====================================================

(defgroup patcher-default nil
  "Default settings for Patcher projects."
  :group 'patcher)

(defcustom patcher-default-mail-method 'compose-mail
  "*Default method used by Patcher to prepare a mail (a symbol).
Currently, there are four built-in methods: 'compose-mail \(the default),
'sendmail, 'message, 'gnus and 'fake.  Please refer to the corresponding
`patcher-mail-*' function for a description of each method.  You can also
define your own method, say `foo'.  In that case, you *must* provide a
function named `patcher-mail-foo' which takes two arguments: a project
descriptor and a string containing the subject of the message.  This
function must prepare a mail buffer.  If you want to do this, please see
how it's done for the built-in methods."
  :group 'patcher-default
  :type '(choice (const :value compose-mail)
		 (const :value sendmail)
		 (const :value message)
		 (const :value gnus)
		 (const :value fake)
		 (symbol :tag "other")))

(defcustom patcher-default-diff-command "cvs -q diff -u %f"
  "*Default method used by Patcher to generate a patch \(a string).
A %f occurring in this string will be replaced by the files affected by
the patch.  These files can be specified by using `patcher-mail-subproject'
instead of `patcher-mail' to prepare the patch.  Otherwise, the %f will
simply be removed."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-commit-command "cvs commit -F %s %f"
  "*Default method used by Patcher to commit a patch \(a string).
This command must contain a %s which will be replaced with the name of a
file containing the commit message log.
A %f occurring in the command will be replaced with the files affected by
the patch.  These files can be specified by using `patcher-mail-subproject'
instead of `patcher-mail' to prepare the patch.  Otherwise, the %f will
simply be removed."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-confirm-commits t
  "*Whether Patcher asks for a confirmation before doing a commit."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-committed-notice
  "NOTE: This patch has been committed."
  "*Notice added to a Patcher mail when the patch is committed before sending."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-init-log-message 'subject
  "*How to initialize the commit log message \(a symbol).
The values currently supported are:
- 'subject (the default): use the subject of the mail.
- 'change-logs: use the ChangeLog entries.
- nil: don't initialize the log message.

If at commit time, the log message is empty, \"(none)\" will be used."
  :group 'patcher-default
  :type '(choice (const :value subject)
		 (const :value change-logs)
		 (const :value nil :tag "nothing")))

(defcustom patcher-default-edit-log-message t
  "*Whether Patcher lets you edit the commit log message.
If nil, Patcher will directly use the initialization value \(see
`patcher-default-init-log-message')."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-mail-prologue ""
  "*A string to insert at the beginning of every Patcher mail."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-kill-source-files-after-sending t
  "*Whether to kill source files after sending the mail.
This is effective only when sources files have not been killed already (see
the variable `patcher-default-kill-source-files-after-diffing').

That feature is not implemented yet."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-kill-change-logs-after-sending t
  "*Whether to kill the ChangeLog files after sending the mail."
  :group 'patcher-default
  :type 'boolean)

(defcustom patcher-default-kill-source-files-after-diffing t
  "*Whether to kill source files after building the ChangeLog skeletons.
These files are loaded temporarily by `patch-to-change-log'.  If this
variable is non nil, `patch-to-change-log' will be instructed to remove
them when they are not needed anymore.
See also the variable `patcher-default-kill-source-files-after-sending'."
  :group 'patcher-default
  :type 'boolean)

(defconst patcher-default-inheritance nil
  ;; Defining this const avoids coding a special case for the :inheritance
  ;; project option in the accessor functions.
  )

(defcustom patcher-default-to-address "xemacs-patches@xemacs.org"
  "*Default email address to use when sending a Patcher mail (a string).
This variable is used by all mail methods except the 'gnus one \(see
`patcher-default-mail-method').  If not defined, it is prompted for."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-default-gnus-group ""
  "*Default Gnus group to use when sending a Patcher mail (a string).
This variable is used only in the 'gnus mail method \(see
`patcher-default-mail-method').  The mail sending process will behave as if
you had type `C-u a' in the group buffer on that Gnus group.  If not
defined, it is prompted for."
  :group 'patcher-default
  :type 'string)

(defcustom patcher-projects '()
  "*List of project descriptors used by `patcher-mail'.
Each project descriptor looks like (NAME DIR OPTIONS...):
- NAME is the project's name (it serves to identify the project),
- DIR  is the top level directory where the project's sources live,

The remainder of the project descriptor is composed of \"project options\"
\(keywords and associated values).  Keywords correspond to the variables
`patcher-default-*'.  When Patcher needs a keyword value, it tries to find
at different places:
- first, it looks for the keyword in the project options.
- if that fails, it tries to find it in the project options of the projects
  appearing in the `:inheritance' option of the current project.
- if that fails, it falls back to the corresponding `patcher-default-*'
variable.

As an exception, the `:inheritance' keyword does not have a corresponding
`patcher-default-inheritance' variable."
  :group 'patcher
  :type '(repeat
	  (group (string :tag "Project")
		 (directory :tag "Project directory")
		 (repeat :inline t :tag "Options"
			 (choice :inline t :value (:mail-method compose-mail)
				 (list :inline t :tag "Mail method"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :mail-method)
				       (choice (const :value compose-mail)
					       (const :value sendmail)
					       (const :value message)
					       (const :value gnus)
					       (const :value fake)
					       (symbol :tag "other")))
				 (list :inline t :tag "Diff command"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :diff-command)
				       (string :tag "Value"))
				 (list :inline t :tag "Commit command"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :commit-command)
				       (string :tag "Value"))
				 (list :inline t :tag "Confirm commits"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :confirm-commits)
				       (boolean :tag "Value"))
				 (list :inline t :tag "Committed-notice"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :committed-notice)
				       (string :tag "Value"))
				 (list :inline t :tag "Init log message"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :init-log-message)
				       (choice (const :value subject)
					       (const :value change-logs)))
				 (list :inline t :tag "Edit log message"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :edit-log-message)
				       (boolean :tag "Value"))
				 (list :inline t :tag "Mail Prologue"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :mail-prologue)
				       (string :tag "Value"))
				 (list :inline t
				       :tag "Kill source files after sending"
				       :format "%{%t%}: %v"
				       (const :tag ""
					      :value
					      :kill-source-files-after-sending)
				       (boolean :tag "Value"))
				 (list :inline t
				       :tag "Kill ChangeLogs after sending"
				       :format "%{%t%}: %v"
				       (const :tag ""
					      :value
					      :kill-change-logs-after-sending)
				       (boolean :tag "Value"))
				 (list :inline t
				       :tag "Kill source files after diffing"
				       :format "%{%t%}: %v"
				       (const :tag ""
					      :value
					      :kill-source-files-after-diffing)
				       (boolean :tag "Value"))
				 (list :inline t :tag "To Address"
				       :format "%{%t%}: %v"
				       (const  :tag "" :value :to-address)
				       (string :tag "Value"))
				 (list :inline t :tag "Gnus Group"
				       :format "%{%t%}: %v"
				       (const  :tag "" :value :gnus-group)
				       (string :tag "Value"))
				 (list :inline t :tag "Inheritance"
				       :format "%{%t%}: %v"
				       (const :tag "" :value :inheritance)
				       (repeat :tag "From"
					       (string :tag "Project")))
				 (list :inline t :tag "Other"
				       symbol
				       sexp))
			 )))
  )

;; Project descriptors Accessors:
(defsubst patcher-project-name      (project) (nth 0 project))
(defsubst patcher-project-directory (project) (nth 1 project))

(defun patcher-project-option-1 (project option)
  ;; Try to find an option in the project descriptor, otherwise, try in each
  ;; project from the project's inheritance list.
  (let ((options (cddr project)))
    (or (member option options)
	(let ((projs (cadr (member :inheritance options)))
	      proj value)
	  (when projs
	    (while (and (not value) (setq proj (pop projs)))
	      (setq value (patcher-project-option-1
			   (assoc proj patcher-projects)
			   option))))
	  value))
    ))

(defun patcher-project-option (project option)
  ;; Returns either a project's option, or the patcher-default-* value.
  (let ((opt (patcher-project-option-1 project option)))
      (if opt
	  (cadr opt)
	(symbol-value
	 (intern-soft
	  (concat "patcher-default-" (substring (symbol-name option) 1))))
	)))


;; Version management =======================================================

;; $Format: "(defconst patcher-prcs-major-version \"$ProjectMajorVersion$\")"$
(defconst patcher-prcs-major-version "version-2-2")
;; $Format: "(defconst patcher-prcs-minor-version \"$ProjectMinorVersion$\")"$
(defconst patcher-prcs-minor-version "1")
(defconst patcher-version
  (let ((level patcher-prcs-minor-version)
	major minor status)
    (string-match "\\(branch\\|version\\)-\\([0-9]+\\)-\\([0-9]+\\)"
		  patcher-prcs-major-version)
    (setq major (match-string 2 patcher-prcs-major-version)
	  minor (match-string 3 patcher-prcs-major-version)
	  status (match-string 1 patcher-prcs-major-version))
    (cond ((string= status "version")
	   (setq level (int-to-string (1- (string-to-int level))))
	   (if (string-equal level "0")
	       (concat major "." minor)
	     (concat major "." minor "." level)))
	  ((string= status "branch")
	   (concat major "." minor "-b" level)))
    ))

;;;###autoload
(defun patcher-version ()
  "Show the current version of Patcher."
  (interactive)
  (message "Patcher version %s" patcher-version))


;; Internal variables =======================================================

;; These variables get proper values in each mail buffer.

(make-variable-buffer-local
 (defvar patcher-project nil
   ;; Patcher project related to the current patch.
   ))

(make-variable-buffer-local
 (defvar patcher-files nil
   ;; Files and/or directories concerned by the current patch.
   ;; This is set by `patcher-mail-subproject'.
   ))

(make-variable-buffer-local
 (defvar patcher-change-logs nil
   ;; List of ChangeLog file buffers concerned by the current patch.
   ))

(make-variable-buffer-local
 (defvar patcher-patch-marker nil
   ;; Marker indicating the beginning of the patch in the mail buffer.
   ))

(make-variable-buffer-local
 (defvar patcher-commit-output-buffer nil
   ;; Buffer containing the output of the commit command.
   ))

(make-variable-buffer-local
 (defvar patcher-pre-commit-window-config nil
   ;; Window configuration when we're in the mail buffer, just prior to
   ;; beginning a commit operation, so we can get back to it at the
   ;; appropriate times after prompting for log messages, displaying
   ;; subprocess output, etc.
   ))

(make-variable-buffer-local
 (defvar patcher-logmsg-file-name
   ;; Name of the temporary file where the log message is stored.
   ))


;; Internal functions =======================================================

(defun patcher-file-relative-name (file dir)
  ;; Construct a filename relative to DIR.
  (file-relative-name (expand-file-name file (expand-file-name dir))
		      (expand-file-name dir)))

(defun patcher-offer-save-buffers (buffers)
  ;; Offer to save some buffers.
  (map-y-or-n-p
   (lambda (buffer)
     (and (buffer-modified-p buffer)
	  (not (buffer-base-buffer buffer))
	  (buffer-file-name buffer)
	  (format "Save file %s? "
		  (buffer-file-name buffer))))
   (lambda (buffer)
     (save-excursion
       (set-buffer buffer)
       (condition-case ()
	   (save-buffer)
	 (error nil))))
   buffers
   '("buffer" "buffers" "save")))

(defun patcher-insert-change-logs-1 (mail-buffer)
  ;; Insert ChangeLog entries in the current buffer at the current position.
  ;; ChangeLog entries are those corresponding to the patch being edited in
  ;; MAIL_BUFFER.
  (let* ((project (symbol-value-in-buffer 'patcher-project mail-buffer))
	 (directory (patcher-project-directory project))
	 (change-logs (symbol-value-in-buffer 'patcher-change-logs
					      mail-buffer)))
    (dolist (change-log change-logs)
      (map-extents
       #'(lambda (extent unused)
	   ;; Force forward slashes (for native Windows). -- APA
	   (let ((directory-sep-char ?/))
	     (message "%s %s" (extent-start-position extent)
		      (extent-end-position extent))
	     (sit-for 1)
	     (insert (format "%s addition:\n\n"
			     (file-relative-name (buffer-file-name change-log)
						 directory)))
	     (insert (extent-string extent))
	     ;; Be sure to map all extents.
	     nil))
       change-log nil nil nil nil 'patcher))
    ))

(defun patcher-construct-command (command files)
  ;; replace the %f with the specified files (if any), or append.
  (or files (setq files ""))
  (setq files (replace-in-string files "\\\\" "/"))
  (cond ((string-match "%f" command)
	 (replace-in-string command "%f" files))
	((> (length files) 0)
	 (concat command " " files))
	(t command)))

(defun patcher-after-send (unused)
  ;; Do some cleanup after sending the mail.
  (when (patcher-project-option patcher-project
				:kill-change-logs-after-sending)
    (patcher-offer-save-buffers patcher-change-logs)
    (mapcar #'kill-buffer patcher-change-logs))
  ;; #### Implement kill-source-files-after-sending here.
  (when patcher-pre-commit-window-config
    (set-window-configuration patcher-pre-commit-window-config))
  (when patcher-commit-output-buffer
    (bury-buffer patcher-commit-output-buffer)))


;; Patcher LogMsg mode ======================================================

(defcustom patcher-logmsg-mode-hook nil
  "*Hook to run after setting up Patcher-Logmsg mode."
  :group 'patcher
  :type 'hook)

(make-variable-buffer-local
 (defvar patcher-logmsg-mail-buffer
   ;; Mail buffer where the relevant Patcher message is being edited.
   ;; This variable gets a proper value in the logmsg buffers.
   ))

(defun patcher-logmsg-do-it (arg)
  "Function used to commit a change reported by `patcher-mail'.
When called interactively, use a prefix to override the default commit
command for this project."
  (interactive "P")
  (let* ((project (symbol-value-in-buffer 'patcher-project
					  patcher-logmsg-mail-buffer))
	 (command (patcher-project-option project :commit-command))
	 (confirm-commits (patcher-project-option project :confirm-commits))
	 (change-logs (symbol-value-in-buffer 'patcher-change-logs
					      patcher-logmsg-mail-buffer))
	 (pre-commit-window-config
	  (symbol-value-in-buffer 'patcher-pre-commit-window-config
				  patcher-logmsg-mail-buffer))
	 (logmsg-file-name
	  (symbol-value-in-buffer 'patcher-logmsg-file-name
				  patcher-logmsg-mail-buffer))
	 (output-buffer (get-buffer-create "*Patcher-Commit-Output*")))
    (patcher-offer-save-buffers change-logs)
    (let* ((directory (patcher-project-directory project))
	   (files
	    (let ((f (symbol-value-in-buffer
		      'patcher-files
		      patcher-logmsg-mail-buffer)))
	      (if (not f)
		  ""
		(mapconcat
		 #'identity
		 (cons f (mapcar
			  #'(lambda (buf)
			      (patcher-file-relative-name
			       (buffer-file-name buf)
			       directory))
			  change-logs)) " ")))))
      (when arg
	(setq command (read-shell-command "Commit command: " command)))
      (setq command (patcher-construct-command
		     (replace-in-string command "%s"
					logmsg-file-name t)
		     files)))
    (when (or (not confirm-commits)
	      (save-window-excursion
		(let ((runbuf (get-buffer-create "*Patcher-Commit-Command*")))
		  (erase-buffer runbuf)
		  (insert-string (format "Command to run:\n\n%s" command)
				 runbuf)
		  (display-buffer runbuf)
		  (y-or-n-p "Run commit command? "))))
      (write-region (point-min) (point-max) logmsg-file-name  nil 'silent)
      (erase-buffer output-buffer)
      (shell-command command output-buffer)
      (with-current-buffer patcher-logmsg-mail-buffer
	;; Note that this sets a buffer-local variable in the mail buffer,
	;; not our own buffer.
	(if (buffer-live-p output-buffer)
	    (setq patcher-commit-output-buffer output-buffer))
	(save-excursion
	  (goto-char (point-min))
	  (when (search-forward mail-header-separator nil t)
	    (forward-line 1)
	    (insert (concat (patcher-project-option patcher-project
						    :committed-notice)
			    "\n")))))
      (kill-buffer (current-buffer))
      (set-window-configuration pre-commit-window-config)
      (if (buffer-live-p output-buffer)
	  (display-buffer output-buffer)))
    ))

(defvar patcher-logmsg-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map [(control c) (control c)] 'patcher-logmsg-do-it)
    map))

(defun patcher-logmsg-mode ()
  "Sets up Patcher-LogMsg major mode.
Used for editing the log message for `patcher-commit-change'.  To commit
the change, use \\<patcher-logmsg-mode-map>\\[patcher-logmsg-do-it].
You're not supposed to use this, unless you know what you're doing."
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'patcher-logmsg)
  (setq mode-name "Patcher-LogMsg")
  (use-local-map patcher-logmsg-mode-map)
  (run-hooks 'patcher-logmsg-mode-hook))


;; Patcher FakeMail mode ====================================================

(defun patcher-fakemail-send ()
  "Pretend to send a fake Patcher mail.
Only perform the usual cleanup after real Patcher mails are sent."
  (interactive)
  (patcher-after-send nil)
  (kill-buffer (current-buffer)))

(defvar patcher-fakemail-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map [(control c) (control c)] 'patcher-fakemail-send)
    map))

(defun patcher-fakemail-mode ()
  "Sets up Patcher-FakeMail major mode.
Used for editing a fake Patcher mail.  "
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'patcher-fakemail)
  (setq mode-name "Patcher-FakeMail")
  (use-local-map patcher-fakemail-mode-map)
  (run-hooks 'patcher-logmsg-mode-hook))


;; Patcher minor-mode =======================================================

(defcustom patcher-minor-mode-string " Patch"
  "*String to use in the modeline when Patcher minor mode is active."
  :group 'patcher
  :type 'string)

(defcustom patcher-minor-mode-hook nil
  "*Hooks to run after setting up Patcher minor mode."
  :group 'patcher
  :type 'hook)

(defun patcher-insert-change-logs ()
  "Insert annotated ChangeLog entries, after `patcher-mail' has been run.
Run this command after you've run `patcher-mail' and then annotated the
ChangeLog skeletons that were created."
  (interactive)
  (save-excursion
    (goto-char patcher-patch-marker)
    (patcher-insert-change-logs-1 (current-buffer))))

(defun patcher-commit-change (arg)
  "Prepare to (and possibly) commit a change to a project's repository.
The change is the one that is announced in the mail buffer.

When called interactively, use a prefix (ARG) to override the commit
command to use."
  (interactive "P")
  (let* ((init-log-message
	  (patcher-project-option patcher-project :init-log-message))
	 (edit-log-message
	  (patcher-project-option patcher-project :edit-log-message))
	 (logmsg-buffer (generate-new-buffer "*Patcher-Log-Message*"))
	 (mail-buffer (current-buffer)))
    (setq patcher-pre-commit-window-config (current-window-configuration))
    (pop-to-buffer logmsg-buffer)
    (patcher-logmsg-mode)
    (setq patcher-logmsg-mail-buffer mail-buffer)
    (set-symbol-value-in-buffer
     'patcher-logmsg-file-name
     (replace-in-string
      (make-temp-name (expand-file-name "patch" (temp-directory)))
      "\\\\" "/")
     mail-buffer)
    (cond ((eq init-log-message 'subject)
	   (insert (with-current-buffer mail-buffer
		     (save-excursion
		       (goto-char (point-min))
		       (and (re-search-forward
			     "^Subject: \\(\\[PATCH\\]\\s-*\\)?\\(.*\\)$"
			     nil t)
			    (concat (match-string 2) "\n"))))))
	  ((eq init-log-message 'change-logs)
	   (patcher-insert-change-logs-1 mail-buffer))
	  (t
	   (insert "(none)")))
    (if edit-log-message
	(message (substitute-command-keys
		  "Edit the log message, \
and press \\[patcher-logmsg-do-it] when done."))
      (patcher-logmsg-do-it arg))
    ))

(defvar patcher-minor-mode-map
  (let ((map (make-sparse-keymap 'patcher-minor-mode-map)))
    (define-key map [(control c) (control p) i] 'patcher-insert-change-logs)
    (define-key map [(control c) (control p) c] 'patcher-commit-change)
    (define-key map [(control c) (control p) v] 'patcher-version)
    map)
  ;; Patcher minor mode keymap.
  )

(make-variable-buffer-local
 (defvar patcher-minor-mode nil))

(defun patcher-minor-mode (arg)
  "Toggles Patcher minor mode.
Used for mails prepared with `patcher-mail'.  You're not supposed to use
this, unless you know what you're doing."
  (interactive "*P")
  (setq patcher-minor-mode
	(if (null arg) (not patcher-minor-mode)
	  (> (prefix-numeric-value arg) 0)))
  (run-hooks 'patcher-minor-mode-hook))

(add-minor-mode
 'patcher-minor-mode patcher-minor-mode-string patcher-minor-mode-map)


;; Mail preparation routines =================================================

(defun patcher-mail-compose-mail (project subject)
  "Prepare a patch-related mail with the `compose-mail' function.
See also the `mail-user-agent' variable.

This function uses the `:to-address' project option to determine the email
address for sending the message.  Otherwise, the address is prompted for."
  (compose-mail (or (patcher-project-option project :to-address)
		    (read-string "To address: "))
		subject nil nil nil nil '(patcher-after-send)))

(defun patcher-mail-sendmail (project subject)
  "Prepare a patch-related mail with the `mail' function.
This method requires the `sendmail' library.

This function uses the `:to-address' project option to determine the email
address for sending the message.  Otherwise, the address is prompted for."
  (require 'sendmail)
  (mail nil (or (patcher-project-option project :to-address)
		(read-string "To address: "))
		subject nil nil nil '(patcher-after-send)))


(defun patcher-mail-message (project subject)
  "Prepare a patch-related mail with the `message-mail' function.
This method requires the `message' library.

This function uses the `:to-address' project option to determine the email
address for sending the message.  Otherwise, the address is prompted for."
  (require 'message)
  (message-mail (or (patcher-project-option project :to-address)
		    (read-string "To address: "))
		    subject nil nil nil nil '(patcher-after-send)))

(defun patcher-mail-gnus (project subject)
  "Prepare a patch-related mail with the `gnus-post-news' function.
Don't worry, this function can also send mails ;-).  This method
requires that you have Gnus *running* in your XEmacs session.

This function uses the `:gnus-group' project option to determine the Gnus
group to use \(as if you had typed `C-u a' on that group in the Group
buffer).  Otherwise, the group is prompted for."
  (require 'gnus-util)
  (unless (gnus-alive-p)
    (error "You should have Gnus running in this XEmacs session"))
  ;; this binding is necessary to let message-mode hooks perform correctly
  (let ((gnus-newsgroup-name (or (patcher-project-option project :gnus-group)
				 (read-string "Gnus group name: "))))
    (gnus-post-news 'post gnus-newsgroup-name)
    (message-goto-subject)
    (insert subject)
    (message-goto-body)
    (push '(patcher-after-send) message-exit-actions)))

(defun patcher-mail-fake (project subject)
  "Prepare a patch-related fake mail.
Use this function if you want to do all that Patcher can do, apart from
sending a real mail. This function generates a fake mail buffer which acts
as a standard Patcher mail buffer, apart from the fact that when you type
\\<patcher-fakemail-mode-map>\\[patcher-fakemail-send] in it, it doesn't
really send a mail, but just clean things up."
  (let ((buffer (generate-new-buffer "*Patcher Fake Mail*")))
    (switch-to-buffer buffer)
    (patcher-fakemail-mode)
    ))

(defun patcher-mail-1 (project subject files &optional override)
  ;; Perform the real job of preparing the mail buffer.
  (let ((command   (patcher-project-option project :diff-command))
	(directory (patcher-project-directory project)))
    (funcall (intern (concat "patcher-mail-"
			     (symbol-name
			      (patcher-project-option project :mail-method))))
	     project (concat "[PATCH] " subject))
    (patcher-minor-mode t)
    (setq patcher-project project)
    (setq patcher-files files)
    (cd directory)
    (and (patcher-project-option project :mail-prologue)
	 (insert "\n" (patcher-project-option project :mail-prologue)))
    (save-excursion
      (insert "\n\n")
      (setq patcher-patch-marker (point-marker))
      (when override
	(setq command (read-shell-command "Diff command: " command)))
      (setq command (patcher-construct-command command files))
      (message "Generating the diff ...")
      (insert (format "%s Patch (%s):\n\n"
		      (patcher-project-name project) command))
      (sit-for 0) ;; Let XEmacs redisplay the message buffer
      (shell-command command 'here)
      (message "Generating the ChangeLog skeletons ...")
      (sit-for 0) ;; Let XEmacs redisplay the message buffer
      (narrow-to-region (point) (mark t))
      (setq patcher-change-logs
	    (patch-to-change-log directory
				 :keep-source-files
				 (not (patcher-project-option
				       project
				       :kill-source-files-after-diffing))
				 :extent-property 'patcher
				 :extent-property-value (current-buffer)))
      (widen)
      (message (substitute-command-keys
		"Please annotate the ChangeLogs, \
and run \\[patcher-insert-change-logs] from this buffer when done."))
      )))

(defvar patcher-projects-history nil
  ;; History used for prompting patcher projects.
  )

;;;###autoload
(defun patcher-mail-subproject (project subject files &optional arg)
  "Prepare a mail about a patch to apply on part of a project.
PROJECT is the name of the project (see the variable `patcher-projects').
SUBJECT is the subject of the mail.
FILES is a string listing one or more files, possibly with wild cards --
  essentially, part of a command line to be interpolated into the `diff'
  and maybe the `commit' commands issued by Patcher.

When called interactively, use a prefix (ARG) to override the value of
the diff command to use for this project.  If you want to work on a
subset of the project (e.g. some files, subdirectories etc), use the
function `patcher-mail-subproject' preferably.

When you use this command instead of `patcher-mail', any commits issued
from the mail buffer (using \\<patcher-minor-mode-map>\\[patcher-commit-change]) will automatically include the associated ChangeLogs,
as well as the file(s) specified as part of this command.

Please note that you can have multiple occurrences of a Patcher mail at
the same time, but not more than one at a time on the same project unless
you use `patcher-mail-subproject' and the sections of the project don't
overlap."
  (interactive
   (let* ((prj (assoc (completing-read "Project: " patcher-projects
				       nil t nil 'patcher-projects-history)
		      patcher-projects))
	  (sbj (read-string "Subject: "))
	  (dir (patcher-project-directory prj))
	  (fils (let ((default-directory (file-name-as-directory dir)))
		  ;; Supply the current buffer's file as the default, since
		  ;; often we just finished working on a file and want to
		  ;; submit it as a patch.
		  (let* ((default-file (and (buffer-file-name)
					    (patcher-file-relative-name
					     (buffer-file-name)
					     dir)))
			 (default-file
			   ;; If the file is not actually underneath the
			   ;; project, then don't suggest it as a possibility.
			   (and default-file
				(if (string-match "^\\.\\.$\\|^\\.\\.[/\\]"
						  default-file)
				    nil default-file))))
		    (read-shell-command
		     (if default-file
			 (format "Files (%s): " default-file)
		       "Files: ")
		     nil nil default-file)))))
     (list prj sbj fils current-prefix-arg)))
  (patcher-mail-1 project subject files (and arg (interactive-p))))

;;;###autoload
(defun patcher-mail (project subject &optional arg)
  "Prepare a mail about a patch to apply on a project.
PROJECT is the name of the project (see the variable `patcher-projects').
SUBJECT is the subject of the mail.

When called interactively, use a prefix (ARG) to override the value of
the diff command to use for this project.  If you want to work on a
subset of the project (e.g. some files, subdirectories etc), use the
function `patcher-mail-subproject' preferably.

Please note that you can have multiple occurrences of a Patcher mail at
the same time, but not more than one at a time on the same project unless
you use `patcher-mail-subproject' and the sections of the project don't
overlap."
  (interactive
   (let* ((prj (assoc (completing-read "Project: " patcher-projects
				       nil t nil 'patcher-projects-history)
		      patcher-projects))
	  (sbj (read-string "Subject: ")))
     (list prj sbj current-prefix-arg)))
  (patcher-mail-1 project subject nil (and arg (interactive-p))))

(provide 'patcher)

;;; patcher.el ends here
