/*

	compile with:

		cc -o hfsow hfsow2.0.c -lm [-lc_s]

-------------------------------------------------------------------------------


	hfsow2.0

	(Sow a heightfield with grass)

	Copyright (C) 1992 by Lawrence K. Coffin
	All rights reserved.


	Parts of this program are from "Numerical Recipes ic C" by William H.
	Press, Brian P. Flannery, Saul A. Teukolsky, and William T. Vetterling.

   This software may be freely copied, modified, and redistributed
	provided that this copyright notice is preserved on all copies.
 
   You may not distribute this software, in whole or in part, as part of
	any commercial product without the express consent of the authors.
  
   There is no warranty or other guarantee of fitness of this software
	for any purpose.  It is provided solely "as is".


   This program reads in a file in the Rayshade 4.0 heightfield format and
	randomly places grass objects as generated by "sawrs" across the
	surface.  The plant objects are recursively bounded by "grid 1 1 1"
	objects.

   Useage:

	hfsow [options] heightfield

		heightfield file is the name of the height field file.

   Options:

	-h		print out help message

	-b  blades	Blades in plant range from "blades" to
				2*"blades" (default: 10)
	-e  scale	Enlarge plants by 'scale' in all directions
					(default: 1.0)
	-m  minimum	Minimum altitude below which no plants will be placed
						(default: -1000)
	-n  num		Number of plants in output file (default: 10)
	-o  filename	Name of output file (default: allgrass.obj)
	-s  x y z	X, y, z scales for the heightfield (default: 10)
	-u  unique	Number of \"unique\" plants that will be generated.
				Each one will be in its own .obj file.
				All actual plants will be random copies
				of these unique plants.  (default: 2)


	The -s option should be set to the scale factor used in the scene file
	to scale the heightfield as if it were scaled before any other
	transformations are done to it.

	Ie: "heightfield test.hf  scale 20 20 20"

	would take "-s 20 20 20"

   The points for the plants are stored in a structure called "list" of type
	"listtype".  This type consistis of the number "number" of objects
	int the array and a pointer "*bounds" which points to the objects of
	type "boundstype".  These objects are used to store the x, y, z points
	of the plants, as well as the x and y center points of the bounding
	boxes the points.  Boundstype also has a field "picked" which is used
	in picking the nearest neighbors of an object.  There are also two
	fields "left" and "right" which point to the two objects that are
	bounded by the current object, these fields are NULL when the data
	stored in the xcent, ycent, and zcent fields are the coordinates of
	the plants.

	When the program starts up, memory is allocated for "list".  The number
	of plants is placed in the "number" field, and the memory for
	bounds["number"] is allocated.

	The x and y coordinates of the plants are ranomly picked then the
	altitude at that coordinate is read from the heightfield.  If the
	altitude is below "minimum (-m)" then the point is discarded, otherwise
	the point is stored in the array list->bounds.  After the total number
	of points (plants) have been generated, the points are sorted by x
	value ("sortbyx()").  Then the plants are sorted by nearest neighbors
	("sortobjs()").  Inside "sortobjs()", a new "bounds" array is
	created of size = (list->number/2 + odd) where "odd" = list->number%2,
	or "1" if "number" is odd, "0" if even.  Then, starting with the first
	object in the original "bounds" array, the distance between it and the
	following objects is calculated, keeping track of the minimum distance
	and the corresponding object.  This checking continues until the x
	distance of the current object being checked is greater than the minimum
	distance.  Then the first object in the new bounds array becomes this
	matching pairs "parent" and the left pointer is set to point to the
	first object in the pair, and the right is set to point to the second.
	the xcent and ycent of this new bounds object is set to the middle of
	the two points from the paired objects.  This continues until all the
	objects in the first bounds array have been paired; if the number of
	objects in the first bounds array was odd, then the last object in the
	array (which couldn't be paired up) is copied into the last object of
	the new array.  This new array is then sorted by x value and the
	sortobjs() routine is called again.  This continues until the number
	of objects in list->bounds is "1", which will be the root of the tree,
	or the main bounding box, which points to the two bounding boxes one
	level lower, which point to ..., which eventually point to the objects
	that contain the original data points.

	The routine "printobjs()" is then called which simply travels down
	the tree left, then right, until it hits an object whose left pointer
	is NULL (which signifies an original data point), and a new plant is
	printed out, translated to the point stored in the current object.
	Before each call to printobjs(), the line "grid 1 1 1" is printed out.
	After each call returns, the line "end" is printed out.  This performs
	the proper recursive engridding.



Example:

*--------------------*
|                    |
|                    |
|  *                 |
|        *      *    |
|                    |
|                    |	Original heightfield with plant points
|                    |
|       *            |
|                    |
| *            *     |
|                    |
|                    |
|       *            |
|                    |
|                    |
|           *        |
|                    |
|   *                |
|                *   |
|                    |
*--------------------*


*--------------------*
|                    |
| --                 |
|| *|   --     --    |
||  |  | *|   | *|   |
||  |  |  |   |  |   |
||  |  |  |   |  |   |	After one call to sortbyx() and sortobjs()
||  |  |  |   |  |   |
||  |  |* |   |  |   |
||  |   --    |  |   |
||* |         |* |   |
| --           --    |
|   -----            |
|  |    *|           |
|  |     |           |
|  |     |  ------   |
|  |     | |*     |  |
|  |     | |      |  |
|  |*    | |      |  |
|   -----  |     *|  |
|           ------   |
*--------------------*

*--------------------*
|                    |
| --------           |
|| *|   --|    --    |
||  |  | *|   | *|   |
||  |  |  |   |  |   |
||  |  |  |   |  |   |	After two calls to sortbyx() and sortobjs()
||  |  |  |   |  |   |
||  |  |* |   |  |   |
||  |   --|   |  |   |
||* |     |   |* |   |
| --------     --    |
|   --------------   |
|  |    *|        |  |
|  |     |        |  |
|  |     |  ------|  |
|  |     | |*     |  |
|  |     | |      |  |
|  |*    | |      |  |
|  |-----  |     *|  |
|   --------------   |
*--------------------*

*--------------------*
|                    |
| ----------------   |
|| *|   --|    --||  |
||  |  | *|   | *||  |
||  |  |  |   |  ||  |
||  |  |  |   |  ||  |	After all calls to sortbyx() and sortobjs()
||  |  |  |   |  ||  |
||  |  |* |   |  ||  |
||  |   --|   |  ||  |
||* |     |   |* ||  |
||--------------- |  |
||  --------------|  |
|| |    *|        |  |
|| |     |        |  |
|| |     |  ------|  |
|| |     | |*     |  |
|| |     | |      |  |
|| |*    | |      |  |
|| |-----  |     *|  |
| ----------------   |
*--------------------*


  Possible improvements:

	1)  Fix the sortbyx() routine to use a faster sort algorithm.

	2)  Right now the sortobjs() routine starts at the first element in
the array and finds its closest neighbor, then it moves to the next unpaired
element and finds its closest neighbor.  The routine does not take into account
the fact that the second object may be closer to a third object than to the
first object. ie:

*----------*
|          |
| *        |
| p1       |
|          |
|          |
|     *    |
|     p2 * |
|        p3|
*----------*

	currently sortobjs() will group p1 and p2 together when it would be
better if p2 and p3 were grouped together.  If anybody has an ideas on how to
implement this, I would appreciate their suggestions.

	3)  It might be better, and would probably take care of #2, if the
plants were sorted from by looking at all the plants at first, working toward
the individual plants, rather than starting at the individual plant level and
working up to the main bounding box.



	Please send bugs, comments, etc. to:

	Larry Coffin

		lcoffin@clciris.chem.umr.edu 

*/




#include <stdio.h>
#include <stdlib.h>
#include <math.h>

float alt(float **,int,int,int,int);
void free_matrix(float**,int,int,int,int);
float **matrix(int,int,int,int);
void nrerror(char);


/* structure for holding objects and closest pairs */

typedef struct _bounds {
	float xcent, ycent, zcent;
	int picked;
	struct _bounds *left, *right;
} boundstype;


/* structure for holding the array and eventually the tree
   of data points and closest pairs
*/

typedef struct _list {
	int number;
	boundstype *bounds;
} listtype;


listtype *list;  /* list of datapoints and closest pairs */


int sortbyx()
{
	boundstype temp;
	int front, min, i;
	extern listtype *list;

	for (front = 0; front < list->number - 1; front++){
		min = front;
		for (i = front+1; i < list->number; i++){
			if (list->bounds[i].xcent < list->bounds[min].xcent){
				min = i;
			}
		}
		temp = list->bounds[front];
		list->bounds[front] = list->bounds[min];
		list->bounds[min] = temp;
	}


}


int sortobjs()
{
	int upper, lower, odd, current, best, numbounds, i;
	float dx, dy, mindx, mindy;
	float dist, mindist;
	extern listtype *list;
	boundstype *tempbounds;

	if (list->number == 1){
		return;
	}

	tempbounds = list->bounds;
	numbounds = list->number;

	odd = (list->number)%2;
	list->number = (list->number)/2 + odd;


	list->bounds = (boundstype *)malloc((list->number)*sizeof(boundstype));

	/* intialize the array */
	for (i = 0; i < list->number; i++){
		list->bounds[i].xcent = 0;
		list->bounds[i].ycent = 0;
		list->bounds[i].zcent = 0;
		list->bounds[i].picked = 0;
		list->bounds[i].left = NULL;
		list->bounds[i].right = NULL;
	}

	current = 0;
	
	for (upper = 0; upper < (list->number) - odd; upper++){
		while(tempbounds[current].picked && current < numbounds - odd){
			current++;
		}
		lower = current+1;
		while(tempbounds[lower].picked && lower < numbounds - odd){
			lower++;
		}
		if(lower > numbounds || current > numbounds){
			fprintf(stderr,"current or lower is greater than numbounds\n");
		}
		best = lower;
		dx = (tempbounds[lower].xcent) - (tempbounds[current].xcent);
		dy = (tempbounds[lower].ycent) - (tempbounds[current].ycent);
		mindist = sqrt(dx*dx + dy*dy);
		
		for(lower++ ; lower < numbounds; lower++){
			if (tempbounds[lower].picked == 0){
				dx = (tempbounds[lower].xcent) - (tempbounds[current].xcent);
				dy = (tempbounds[lower].ycent) - (tempbounds[current].ycent);
				dist = sqrt(dx*dx + dy*dy);
				if(dx > mindist){
					break;
				}
				else if (dist < mindist){
					best = lower;
					mindist = dist;
					mindx = dx;
					mindy = dy;
				}
			}
		}
		list->bounds[upper].picked = 0;
		list->bounds[upper].left = &(tempbounds[current]);
		list->bounds[upper].right = &(tempbounds[best]);
		list->bounds[upper].xcent = tempbounds[current].xcent + mindx/2;
		list->bounds[upper].ycent = tempbounds[current].ycent + mindy/2;

		tempbounds[current].picked = 1;
		tempbounds[best].picked = 1;
	}
	if (odd){
		for(current = 0; current < numbounds; current++){
			if(tempbounds[current].picked == 0){
				break;
			}
		}
		list->bounds[upper].left = tempbounds[current].left;
		list->bounds[upper].right = tempbounds[current].right;
		list->bounds[upper].picked = 0;
		list->bounds[upper].xcent = tempbounds[current].xcent;
		list->bounds[upper].ycent = tempbounds[current].ycent;
		list->bounds[upper].zcent = tempbounds[current].zcent;
	}
	sortbyx();
	sortobjs();

}

printobjs(FILE *outf,boundstype *bounds, int xscale, int yscale, int zscale,int unique)
{
	if(bounds->left == NULL){
		fprintf(outf,"   object grassgreen grasstuft%i\n",(unique*rand()/RAND_MAX)+1);
		fprintf(outf,"\tscale scalenum scalenum scalenum translate %.3f %.3f %.3f\n",bounds->xcent,bounds->ycent,bounds->zcent);
		fflush(outf);
		free(bounds);
		return;
	}

	fprintf(outf,"grid 1 1 1\n");
	printobjs(outf,bounds->left,xscale,yscale,zscale,unique);
/*	fprintf(outf,"end\n");
	fprintf(outf,"grid 1 1 1\n");
*/	printobjs(outf,bounds->right,xscale,yscale,zscale,unique);
	fprintf(outf,"end\n");
	free(bounds);
}


main(int argc, char *argv[])
{
	int i, j;
	float x, y;
	int size;
	float **al;
	FILE *hf, *outf;
	char coords[100], *buf, *outfile, sawrs[40];
	int xscale, yscale, zscale;
	float altn, minalt;
	extern char *optarg;
	extern int optind;
	int c, blades;
	int help, number, unique;
	float pscale;
	extern listtype *list;

	help = 0;
	number = 10;
	unique = 2;
	xscale = yscale = zscale = 10;
	minalt = -1000.0;
	pscale = 1.0;
	outfile = "allgrass.obj";
	blades = 10;

	while((c = getopt(argc, argv, "hb:e:m:n:o:s:u:")) != EOF)
		switch(c) {
		case 'h': help = 1;
			break;
		case 'b': blades = atoi(optarg);
			break;
		case 'e': pscale = atof(optarg);
			break;
		case 'm': minalt = atof(optarg);
			break;
		case 'n': number = atoi(optarg);
			break;
		case 'o': outfile = optarg;
			break;
		case 's': xscale = atoi(optarg);
			yscale = atoi(argv[optind++]);
			zscale = atoi(argv[optind++]);
			break;
		case 'u': unique = atoi(optarg);
			break;
		case '?':
			help = 1;
			break;
	}
	if(argv[optind] == NULL && !(help)) {
		fprintf(stderr,"\nI need a file to open!!\n");
		help = 1;
	}
	else if(!help) {
		if((hf = fopen(argv[optind], "r")) == NULL){
			fprintf(stderr,"Error opening %s.\n",argv[optind]);
			exit(1);
		}
	}
	if((outf = fopen(outfile, "w")) == NULL){
		fprintf(stderr,"Error opening %s\n",optarg);
		exit(1);
	}

        if(help){
		fprintf(stderr,"   Useage:\n\n");

		fprintf(stderr,"\thfsow <options> <heightfield file>\n\n");


		fprintf(stderr,"\t\t<heightfield file> is the name of the height field file.\n\n");

		fprintf(stderr,"   Options:\n\n");

		fprintf(stderr,"\t-h\t\tprint out help message\n\n");

		fprintf(stderr,"\t-b  blades\tBlades in plant range from \"blades\" to\n");
		fprintf(stderr,"\t\t\t\t2x\"blades\" (default: 10)\n");
		fprintf(stderr,"\t-e  scale\tEnlarge plants by 'scale' in all directions\n");
		fprintf(stderr,"\t\t\t\t\t(default: 1.0)\n");
		fprintf(stderr,"\t-m  minimum	Minimum altitude below which no plants will be placed\n");
		fprintf(stderr,"\t\t\t\t\t(default: -1000)\n");
		fprintf(stderr,"\t-n  num\t\tNumber of plants in output file (default: 10)\n");
		fprintf(stderr,"\t-o  filename\tName of output file (default: allgrass.obj)\n");
		fprintf(stderr,"\t-s  x y z\tX, y, z scales for the heightfield (default: 10)\n");
		fprintf(stderr,"\t-u  unique\tNumber of \"unique\" plants that will be generated.\n");
		fprintf(stderr,"\t\t\t\tEach one will be in its own .obj file.\n");
		fprintf(stderr,"\t\t\t\tAll actual plants will be random copies\n");
		fprintf(stderr,"\t\t\t\tof these unique plants.  (default: 2)\n\n");
		exit(1);
        }
	
		
	srand(time(0));

	fprintf(stderr,"Creating unique plants\n");
	fprintf(outf,"#define scalenum %f\n\n", pscale);
	for(i = 1; i <= unique; i++){
		sprintf(sawrs,"sawrs %i %i 15 > grass%i.obj",8+(12*rand()/RAND_MAX),blades+(blades*rand()/RAND_MAX),i);
		system(sawrs);
		fprintf(outf,"/* %s */\n",sawrs);
		fprintf(outf,"#undef grasstuft\n");
		fprintf(outf,"#define grasstuft grasstuft%i\n",i);
		fprintf(outf,"#include \"%s/grass%i.obj\"\n\n",getcwd(buf,100),i);
	}


/*
 *  read size of heightfield
 */
	fprintf(stderr,"Opening heightfield file.\n");
	fread(&size,sizeof(int),1,hf);

/*
 * Allocate memory the array for the heightfield data and read the data
 */

	al = matrix(0, size, 0, size);

	for(i=0; i <= size - 1; i++){
		for(j=0; j <= size - 1; j++){
			fread(&al[j][i],sizeof(float),1,hf);
		}
	}
	fclose(hf);



/*
 * Pick points and print out a plant at the proper altitude
 */

	list = (listtype *)malloc(sizeof(listtype));

	list->number = number;


	list->bounds = (boundstype *)malloc(list->number*sizeof(boundstype));

	/* initailize the array */
	for (i = 0; i < number; i++){
		list->bounds[i].xcent = 0;
		list->bounds[i].ycent = 0;
		list->bounds[i].zcent = 0;
		list->bounds[i].picked = 0;
		list->bounds[i].left = NULL;
		list->bounds[i].right = NULL;
	}

	for(i = 0; i < number; i++){
		x = (float)rand()/(float)RAND_MAX;
		y = (float)rand()/(float)RAND_MAX;
		altn = alt(al,size,500,(int)(x*500.),(int)(y*500.));

		if(altn*zscale > minalt){
			list->bounds[i].xcent = x*(float)xscale;
			list->bounds[i].ycent = y*(float)yscale;
			list->bounds[i].zcent = altn*(float)zscale;
		}
		else {
			i--;
		}
	}

	free_matrix(al,0,size,0,size);

	sortbyx();

	sortobjs();

	fprintf(stderr,"Printing out grass positions.\n");

	fprintf(outf,"name bunch_o_grass\n");
	printobjs(outf,(list->bounds),xscale,yscale,zscale,unique);
	close(outf);

}

/*
 *  This procedure does not work for all "scale" values
 *	300 - 1000 work
 */

float alt(float **al,int size,int scale, int x, int y)
{
	float alt1, alt2, alt3, alt4, alt5, alt6, altn;
	float pntspace, fscale, fsize;
	int x1, x2, y1, y2, xn, yn;
	int i1, i2, j1, j2;

	pntspace = ((float)scale/(float)(size-1));

	i1 = (int)(x/pntspace);
	i2 = i1 + 1;
	j1 = (int)(y/pntspace);
	j2 = j1 + 1;



	x1 = (int)(i1*pntspace);
	x2 = (int)(i2*pntspace);
	y1 = (int)(j1*pntspace);
	y2 = (int)(j2*pntspace);
	xn = x;
	yn = y;

	alt1 = al[i1][j1];
	alt2 = al[i2][j1];
	alt3 = al[i1][j2];
	alt4 = al[i2][j2];
/*
 * Here's the magic formula!!
 */

	if( ((x-x1)*(x-x1) + (y-y1)*(y-y1)) < ((x-x2)*(x-x2) + (y-y2)*(y-y2)) ){
		altn = ((alt2 - alt1)*(x - x1) + (alt3 - alt1)*(y - y1)
			+ pntspace*alt1)/pntspace;

		return altn;
	}
	else {
		altn = ((alt3 - alt4)*(x2 - x) + (alt2 - alt4)*(y2 - y)
			+ pntspace*alt4)/pntspace;

		return altn;
	}	
}

/*
 * The following are from "Numerical Recipes in C"
 */

void nrerror(error_text)
char error_text[];
{
	fprintf(stderr,"Numerical Recipes run-time error...\n");
	fprintf(stderr,"%s\n",error_text);
	fprintf(stderr,"...now exiting to system...\n");
	exit(1);
}

float **matrix(nrl,nrh,ncl,nch)
int nrl,nrh,ncl,nch;
{
	int i;
	float **m;

	m=(float **) malloc((unsigned) (nrh-nrl+1)*sizeof(float*));
	if (!m) nrerror("allocation failure 1 in matrix()");
	m -= nrl;

	for(i=nrl;i<=nrh;i++) {
		m[i]=(float *) malloc((unsigned) (nch-ncl+1)*sizeof(float));
		if (!m[i]) nrerror("allocation failure 2 in matrix()");
		m[i] -= ncl;
	}
	return m;
}

void free_matrix(m,nrl,nrh,ncl,nch)
float **m;
int nrl,nrh,ncl,nch;
{
	int i;

	for(i=nrh;i>=nrl;i--) free((char*) (m[i]+ncl));
	free((char*) (m+nrl));
}

/* the end */
