
/*
 *  view.c            -Brian Tierney,  LBL
 *  for use with segal
 *  these routines are for displaying the image
 */

#include "segal.h"

/* #define DEBUG */
/* #define SHOW_COLORS  */

/* these need to be global */
view_win_objects *view_win;
extern pixedit_win_objects *edit_win;
extern segal_win_objects *segal_win;

typedef struct box {
    int      x, y;   /* location of upper corner of the box */
    int      size;   /* size of box */
}         BOX_INFO;

BOX_INFO box_info;


/************************************************************/
void
view_win_init(owner)
    Xv_opaque owner;
{
    void      make_cms_colormap();

    view_win = view_win_objects_initialize(NULL, owner);

    make_cms_colormap();
    (void) xv_set(view_win->canvas, WIN_CMS_NAME, "segal", NULL);
    if (verbose)
	fprintf(stderr, "setting colors.. \n");
    (void) xv_set(view_win->canvas, WIN_CMS_DATA, &cms_data, NULL);

#ifdef SHOW_COLORS
    fprintf(stderr, "after canvas creation \n");
    for (i = 0; i < NGRAY * 2; i++)
	fprintf(stderr, "color table: red = %d, green = %d, blue = %d \n",
		cms_data.red[i], cms_data.green[i], cms_data.blue[i]);
#endif

    /* set pointers to colormap */
    colors = (u_long *) xv_get(view_win->canvas, WIN_X_COLOR_INDICES);
    standout = colors[NGRAY * 2];

#ifdef SHOW_COLORS
    for (i = 0; i < NGRAY * 2; i++)
	fprintf(stderr, " color %d = %d \n", i, colors[i]);
#endif
}

/**************************************************************/
void
view_setup()
{
    Rect      srect, vrect;	/* frame rect structure for segal win and
				 * view win */

    /* set size */
/*  don't need to set canvas size
    (void) xv_set(view_win->canvas, XV_WIDTH, segal.cols + 1, NULL);
    (void) xv_set(view_win->canvas, XV_HEIGHT, segal.rows + 1, NULL);
*/
    (void) xv_set(view_win->win, XV_WIDTH, segal.cols + 1, NULL);
    (void) xv_set(view_win->win, XV_HEIGHT, segal.rows + 1, NULL);

    if((int) xv_get(view_win->win, XV_SHOW, NULL) == TRUE)
	return;

    (void) xv_set(view_win->win, XV_SHOW, TRUE, NULL);

    /*
     * set location: NOTE: to set the location of a frame using
     * 'frame_set_rect'  (at least under the twm window manager) XV_SHOW must
     * be TRUE first
     */

    /* set view window location relative to segal window */
    frame_get_rect(segal_win->win, &srect);

    if (srect.r_top > 0 && srect.r_left > 0) { /* only if segal window
						  is on screen */
	frame_get_rect(view_win->win, &vrect);
	vrect.r_top = srect.r_top;
	vrect.r_left = srect.r_left + srect.r_width + 2;
	frame_set_rect(view_win->win, &vrect);
    }
}

/*************************************************************
 * Repaint callback function for `canvas'.
 */
void
image_repaint_proc()
{
    int bsize; /* size of box drawn on image */

    if (xv_get(view_win->win, XV_SHOW, NULL) == FALSE)
	return;

    if (segal.display_type == 0 && image != NULL)
	XPutImage(display, view_xid, gc, image, 0, 0, 0, 0,
		  image->width, image->height);
    if (segal.display_type == 1 && mask_image != NULL)
	XPutImage(display, view_xid, gc, mask_image, 0, 0, 0, 0,
		  mask_image->width, mask_image->height);
    if (segal.display_type == 2 && blend_image != NULL)
	XPutImage(display, view_xid, gc, blend_image, 0, 0, 0, 0,
		  blend_image->width, blend_image->height);

    if (zoom_info.x >= 0 && zoom_info.y >= 0 && zoom_info.mag > 0) {
	bsize = zoom_info.size / zoom_info.mag;
	XSetForeground(display, gc, standout);
	XDrawRectangle(display, view_xid, gc, zoom_info.x, zoom_info.y,
		       bsize, bsize);

	/* save info on where rectangle was drawn */
	box_info.x = zoom_info.x;
	box_info.y = zoom_info.y;
	box_info.size = bsize;
    }
}

/*****************************************************************/
void
image_subregion_repaint_proc()
{
    int sx, sy, bsize;

    if (xv_get(view_win->win, XV_SHOW, NULL) == FALSE)
	return;

    sx = box_info.x - 3;
    sy = box_info.y - 3;
    if ( sx < 0)
	sx = 0;
    if ( sy < 0)
	sy = 0;
    bsize = box_info.size + 6;

    if (segal.display_type == 0 && image != NULL)
	XPutImage(display, view_xid, gc, image, sx, sy, sx, sy, bsize, bsize);
    if (segal.display_type == 1 && mask_image != NULL)
	XPutImage(display, view_xid, gc, mask_image, sx, sy,
		  sx, sy, bsize, bsize);
    if (segal.display_type == 2 && blend_image != NULL)
	XPutImage(display, view_xid, gc, blend_image, sx, sy,
		  sx, sy, bsize, bsize);
}

/*****************************************************************/
Notify_value
disp_event_proc(win, event, arg, type)
    Xv_Window win;
    Event    *event;
    Notify_arg arg;
    Notify_event_type type;
{
    static int counter = 0;
    void      draw_box(), draw_location(), get_stats(), call_zoom();

#ifdef DEBUG
    fprintf(stderr, "hview: disp_event_proc: event %d\n", event_id(event));
#endif

    /* if polygon mode */
    if (segal.poly_flag == 1) {
	if (event_is_down(event)) {   /* down events only */
	    switch (event_id(event)) {
	    case MS_RIGHT:
	    case MS_LEFT:
	    case MS_MIDDLE:
	    case LOC_DRAG:
		cross_proc(event);
		break;
	    }
	}
	    return notify_next_event_func(win, (Notify_event) event, 
					  arg, type);
    }
/*
  mouse events:
    any button: move zoom box
    shift and control, any button: show gray value 
    meta, any button:   show object statistics
*/

  /* meta events */
    if	(event_meta_is_down(event)) {
	if (event_is_down(event)) {   /* down events only */
	    switch (event_id(event)) {
	    case MS_RIGHT:
	    case MS_LEFT:
	    case MS_MIDDLE:
		get_stats(event_x(event), event_y(event));
		break;
	    }
 	}
	return notify_next_event_func(win, (Notify_event) event, arg, type);
    }

    /* cntrl and shift events :  display gray value at cursor */
    if	(event_ctrl_is_down(event) || event_shift_is_down(event)) {
	switch (event_id(event)) {
	case MS_RIGHT:
	case MS_LEFT:
	case MS_MIDDLE:
	case LOC_DRAG:
	    if (event_is_down(event))
		draw_location(event_x(event), event_y(event), 1);
	    else
		draw_location(event_x(event), event_y(event), 0);
	    break;
	}
	return notify_next_event_func(win, (Notify_event) event, arg, type);
    }

    /* other events */
    switch (event_id(event)) {
    case MS_RIGHT:
    case MS_LEFT:
    case MS_MIDDLE:
	if (event_is_down(event))
	    draw_box(event);
	if (event_is_up(event))
	    call_zoom(event);
	break;
    case LOC_DRAG:
	if (++counter % 5)	/* ignore most motion events */
	    break;
	if (event_is_down(event))
	    draw_box(event);
	break;
    default:
	break;
    }
    return notify_next_event_func(win, (Notify_event) event, arg, type);
}

/**************************************************************/
void
draw_box(event)   /* draws box locating the zoom window on the image */
    Event    *event;
{
    void      image_repaint_proc();
    void image_subregion_repaint_proc();
    int       xpos, ypos, sx, sy;	/* sx and sy upper corner of zoom
					 * image */
    int       line_len;

    if (xv_get(edit_win->win, XV_SHOW, NULL) == FALSE)	/* only do if edit
							 * window is open */
	return;

    xpos = event_x(event);
    ypos = event_y(event);

    line_len = zoom_info.size / zoom_info.mag;

    /* redraw image */
    image_subregion_repaint_proc();

    sx = xpos - (line_len / 2);
    sy = ypos - (line_len / 2);
    if (sx < 0)
	sx = 0;
    if (sy < 0)
	sy = 0;
    if (sx + line_len > image->width)
	sx = image->width - line_len;
    if (sy + line_len > image->height)
	sy = image->height - line_len;

#ifdef DEBUG
    fprintf(stderr, " drawing box from %d,%d to %d,%d \n",
	    sx, sy, sx + line_len, sy + line_len);
#endif

    XSetForeground(display, gc, standout);
    XDrawRectangle(display, view_xid, gc, sx, sy, line_len, line_len);

    /* save info on where rectangle was drawn */
    box_info.x = sx;
    box_info.y = sy;
    box_info.size = line_len;
}

/**************************************************************/
void
call_zoom(event)   /* set up information for zoom */
    Event    *event;
{
    int       cx, cy, dist_to_center;
    void      zoom();

    if (xv_get(edit_win->win, XV_SHOW, NULL) == FALSE)
	return;

    set_watch_cursor();

    dist_to_center = zoom_info.size / zoom_info.mag / 2;

    cx = event_x(event);	/* cx,cy = center of zoom window */
    cy = event_y(event);
    if (cx < dist_to_center)
	cx = dist_to_center;
    if (cy < dist_to_center)
	cy = dist_to_center;
    if (cx + dist_to_center >= image->width)
	cx = image->width - dist_to_center - 1;
    if (cy + dist_to_center >= image->height)
	cy = image->height - dist_to_center - 1;

    zoom_info.cx = cx;
    zoom_info.cy = cy;
    zoom_info.x = cx - (zoom_info.size / 2 / zoom_info.mag);
    zoom_info.y = cy - (zoom_info.size / 2 / zoom_info.mag);

    zoom();

    edit_repaint_proc();

    unset_watch_cursor();
}

/******************************************************/
void
draw_location(x, y, funct)
    int       x, y, funct;
{
    char      dstr[20];
    int       width, height;

    if (x >= segal.cols || y >= segal.rows || x < 0 || y < 0)
	return;

    /* will have problems if default font changes */
    height = 16;

    if (funct) {
	sprintf(dstr, "(%d,%d) = %d ", x, y, himage.data[y][x]);
	width = 7 * strlen(dstr);

	XSetForeground(display, gc, WhitePixel(display, screen));
	XClearArea(display, view_xid, 0, 0, width, height, False);
	XDrawString(display, view_xid, gc, 2, height - 4, dstr, strlen(dstr));
    } else {
	if (segal.display_type == 0)
	    XPutImage(display, view_xid, gc, image, 0, 0, 0, 0, 140, height);
	else if (segal.display_type == 1)
	    XPutImage(display, view_xid, gc, mask_image,
		      0, 0, 0, 0, 140, height);
	else
	    XPutImage(display, view_xid, gc, blend_image,
		      0, 0, 0, 0, 140, height);
    }
}

/**************************************************************/
void
map_image_to_lut(buf, newbuf, bufsize)  
 /* used to map the image to the color look-up-table based
    on the value of NGRAY */

    u_char   *buf;		/* image */
    u_char   *newbuf;
    int       bufsize;
{
    register int       i;
    u_char    j;
    float     scale = NGRAY./ 257.;

    for (i = 0; i < bufsize; i++) {
	/* scale to 0 to NGRAY */
	j = (u_char) (buf[i] * scale);
	if (j >= NGRAY)
	    j = NGRAY - 1;
	/* map to lookup table */
	newbuf[i] = (u_char) colors[j];
    }
}

/**************************************************************************/
void
make_cms_colormap()
{
    int       i, j;
    int       color_size = NGRAY * 2 + 1;	/* 1 standout color */
    u_char    red[NGRAY * 2 + 1], green[NGRAY * 2 + 1], blue[NGRAY * 2 + 1];
    double    gammaval;
    int       r, g, b;
    int       gh = 0;		/* HSV values */
    double    gs = 1., gv = 1.;
    extern int overlay_hue;	/* command line arg */

    gammaval = (float) xv_get(segal_win->gamma_val, PANEL_VALUE, NULL) / 10.;
    if (verbose)
	fprintf(stderr, "setting colormap: gamma value of %.1f \n", gammaval);

    for (i = 0; i < NGRAY; i++) {
	if (gammaval == 1.0)
	    j = (int) ((i * 257.) / NGRAY.);
	else
	    j = (int) (257. * pow((double) i / NGRAY., 1.0 / gammaval));
	if (j > 255)
	    j = 255;
	red[i] = blue[i] = green[i] = (u_char) j;
    }

    for (; i < NGRAY * 2; i++) { /* blend colors */
#ifdef OLD_RGB
/*     old version:  purple */
       red[i] = blue[i] = (u_char) 255;
	j = (int) (257. * pow((double) (i- NGRAY) / NGRAY., 1.0 / 3.0));
	if (j > 255)
	    j = 255;
	green[i] = (u_char) j;
#else
	/* hsv model seems to work better */
	gh = overlay_hue;	/* greenish */
	gs = .8;
	gv = .5 + (.5 * (float) (i - NGRAY) / NGRAY.);	/* .5 to 1.0 */
	HSV_to_RGB((short) gh, gs, gv, &r, &g, &b);
	red[i] = (u_char) r;
	blue[i] = (u_char) b;
	green[i] = (u_char) g;
#endif
    }

#define RED_STANDOUT

#ifdef GREEN_STANDOUT
    /* standout color = green */
    green[NGRAY * 2] = 255;
    red[NGRAY * 2] = blue[NGRAY * 2] = 0;
#endif
#ifdef BLUE_STANDOUT
    /* standout color = light blue */
    blue[NGRAY * 2] = green[NGRAY * 2] =  255;
    red[NGRAY * 2] = 0;
#endif
#ifdef YELLOW_STANDOUT
    /* standout color = yellow */
    red[NGRAY * 2] = green[NGRAY * 2] = 255;
    blue[NGRAY * 2] = 0;
#endif
#ifdef RED_STANDOUT
    /* standout color = red */
    red[NGRAY * 2] = 255;
    blue[NGRAY * 2] = green[NGRAY * 2] = 0;
#endif

    cms_data.type = XV_DYNAMIC_CMS;
    cms_data.size = color_size;
    cms_data.rgb_count = color_size;
    cms_data.index = 0;
    cms_data.red = red;
    cms_data.green = green;
    cms_data.blue = blue;

#ifdef SHOW_COLORS
    fprintf(stderr, "after setting colors... \n");
    for (i = 0; i < NGRAY * 2; i++)
	fprintf(stderr, "color table: red = %d, green = %d, blue = %d \n",
		cms_data.red[i], cms_data.green[i], cms_data.blue[i]);
#endif
}

/**************************************************/
void
change_colormap(item, event)  /* procedure called when adjusting contrast */
    Panel_item item;
    Event    *event;
{
    void      make_cms_colormap();

    if ((int) xv_get(view_win->win, XV_SHOW, NULL) == FALSE)
	return;

    make_cms_colormap();

    (void) xv_set(view_win->canvas, WIN_CMS_DATA, &cms_data, NULL);
    (void) xv_set(edit_win->canvas, WIN_CMS_DATA, &cms_data, NULL);
}

/*********************************************************/

HSV_to_RGB(hue, sat, val, red, green, blue)
/* written by David Robertson, LBL
 *
 * HSV_to_RGB - converts a color specified by hue, saturation and intensity
 * to r,g,b values.
 * Hue should be in the range 0-360 (0 is red); saturation and intensity
 * should both be in [0,1].
 */

    short     hue;
    double    sat, val;
    int      *red, *green, *blue;

{
    register double r, g, b;
    int       i;
    register double f, nhue;
    double    p, q, t;

    val *= 255.0;
    if (hue == 360)
	nhue = 0.0;
    else
	nhue = (double) hue / 60.0;
    i = (int) nhue;		/* Get integer part of nhue */
    f = nhue - (double) i;	/* And fractional part */
    p = val * (1.0 - sat);
    q = val * (1.0 - sat * f);
    t = val * (1.0 - sat * (1.0 - f));
    switch (i) {
    case 0:
	r = val;
	g = t;
	b = p;
	break;
    case 1:
	r = q;
	g = val;
	b = p;
	break;
    case 2:
	r = p;
	g = val;
	b = t;
	break;
    case 3:
	r = p;
	g = q;
	b = val;
	break;
    case 4:
	r = t;
	g = p;
	b = val;
	break;
    case 5:
	r = val;
	g = p;
	b = q;
    }

    *red = (int) (r + 0.5);
    *green = (int) (g + 0.5);
    *blue = (int) (b + 0.5);

}
