/*	Copyright (c) 1989 Michael Landy

Disclaimer:  No guarantees of performance accompany this software,
nor is any responsibility assumed on the part of the authors.  All the
software has been tested extensively and every effort has been made to
insure its reliability.   */

/* stretchpix.c - stretch or compress the range of gray-levels
 *		  in a byte-formatted frame.
 *
 * usage: stretchpix [ exp1 [ boundary [ exp2 ]]] <oldframe >newframe
 *
 * defaults:  exp1:  2., boundary: .5, exp2: 1/exp1.
 *
 * to load: cc -o stretchpix stretchpix.c -lhips -lm
 *
 * Yoav Cohen 2/19/82
 */

/*
 * exponentiation bug fixed - WEJohnston 9/89
 *
 *  added support for short images and
 *  modified to use look-up table for byte and short images:
 *     Brian Tierney, LBL 10/90
 */

#include <hipl_format.h>
#include <math.h>
#include <sys/types.h>
#include <stdio.h>

char     *Progname;

u_char   *pic;
short    *spic;

#define MAXSHORT 65536

/* #define DEBUG */

u_char    byte_lut[256];
short     short_lut[MAXSHORT];

/* scaling values */
double    expt1, expt2, d;
float     mval;			/* max value in a short image */

main(argc, argv)
    int       argc;
    char     *argv[];
{
    struct header hd;
    int       form;

    Progname = strsave(*argv);
    if (argv[argc - 1][0] == '-')
	argc--;
    expt1 = 2.;
    d = .5;
    if (argc > 1)
	expt1 = atof(argv[1]);
    if (argc > 2)
	d = atof(argv[2]);
    if (d <= 0 || d >= 1)
	perr("boundary must be between 0 and 1");
    expt2 = 1. / expt1;
    if (argc > 3)
	expt2 = atof(argv[3]);

    read_header(&hd);
    form = hd.pixel_format;
    if (form == PFBYTE) {
	hd.bits_per_pixel = 8;
	pic = (u_char *) halloc(hd.rows * hd.cols, sizeof(char));
    } else if (form == PFSHORT) {
	hd.bits_per_pixel = 16;
	spic = (short *) halloc(hd.rows * hd.cols, sizeof(short));
    } else
	perr("input format must be byte or short format");

    update_header(&hd, argc, argv);
    write_header(&hd);

    if (form == PFBYTE) {
	make_byte_lut();
	bstretchpix(hd.num_frame, hd.rows, hd.cols);
    } else if (form == PFSHORT) {
	sstretchpix(hd.num_frame, hd.rows, hd.cols);
    }
    return (0);
}

/*******************************************************************/
make_byte_lut()
{
    int       i, j;
    int       d1;
    double    a, s1, s2, s3, dtmp;

    d1 = d * 255;
    s1 = pow(255. * d, (1. - expt1));
    s2 = pow(255. * (1. - d), 1. - expt2);
    s3 = 255. * d;

    byte_lut[0] = 0;

    for (i = 1; i < 256; i++) {
	dtmp = (double) i;
	if (dtmp == s3)
	    a = 0.;
	else if (dtmp <= d1)
	    a = (u_char) (s1 * pow(dtmp, expt1));
	else if (dtmp - s3 < 0)
	    a = s3;
	else
	    a = s3 + pow(dtmp - s3, expt2) * s2;
	j = (u_char) (a + 0.5);

	byte_lut[i] = j;
#ifdef DEBUG
	fprintf(stderr, " %d will be %d \n", i, j);
#endif
    }
}

/*******************************************************************/
bstretchpix(fr, r, c)
    int       fr, r, c;
{
    int       j, i, rc;

    rc = r * c;
    for (j = 0; j < fr; j++) {
	if (pread(0, pic, rc * sizeof(u_char)) != rc * sizeof(u_char))
	    perr("error during read");

	for (i = 0; i < rc; i++)
	    pic[i] = byte_lut[pic[i]];

	if (write(1, pic, rc * sizeof(char)) != rc * sizeof(char))
	    perr("error during write");
    }
    return (0);
}

/*******************************************************************/
sstretchpix(fr, r, c)
    int       fr, r, c;
{
    int       j, i, rc;

    rc = r * c;
    for (j = 0; j < fr; j++) {
	if (pread(0, spic, rc * sizeof(short)) != rc * sizeof(short))
	    perr("error during read");

	get_max(r * c);
	make_short_lut();

	for (i = 0; i < rc; i++) {
	    if (spic[i] >= 0)
		spic[i] = short_lut[spic[i]];
	    else {
		fprintf(stderr, "Warning: negative value being set to zero. \n");
		spic[i] = 0;
	    }
	}

	if (write(1, spic, rc * sizeof(short)) != rc * sizeof(short))
	    perr("error during write");

    }
    return (0);
}

/*****************************************************************/
get_max(size)
    int       size;
{
    register int i;

    mval = 0.;

    for (i = 0; i < size; i++) {
	if (spic[i] > mval)
	    mval = spic[i];
    }

    fprintf(stderr, "Maximum value is: %.3f \n", mval);
}

/*******************************************************************/
make_short_lut()
{
    int       i, j;
    int       d1;
    double    a, s1, s2, s3, dtmp;

    d1 = d * mval;
    s1 = pow(mval * d, (1. - expt1));
    s2 = pow(mval * (1. - d), 1. - expt2);
    s3 = mval * d;

    short_lut[0] = 0;

    for (i = 1; i <= mval; i++) {
	dtmp = (double) i;
	if (dtmp == s3)
	    a = 0.;
	else if (dtmp <= d1)
	    a = (u_char) (s1 * pow(dtmp, expt1));
	else if (dtmp - s3 < 0)
	    a = i;
	else
	    a = s3 + pow(dtmp - s3, expt2) * s2;
	j = (u_char) (a + 0.5);

	short_lut[i] = j;
#ifdef DEBUG
	fprintf(stderr, " %d will be %d \n", i, j);
#endif
    }
}
