/* 
 * Mach Operating System
 * Copyright (c) 1990 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	ufs_inode.c,v $
 * Revision 2.3  90/10/01  14:06:25  rwd
 * 	Convert to new inode code from XUX26/from BSD4.3-Reno.
 * 	[90/09/30            rwd]
 * 
 * Revision 2.2  90/09/08  00:21:59  rwd
 * 	Fix indirect code.
 * 	[90/09/04            rwd]
 * 	First checkin
 * 	[90/08/31  14:02:39  rwd]
 * 
 */
/*
 *	File:	./ufs_inode.c
 *	Author:	Joseph S. Barrera III, Randall W. Dean
 *
 *	Copyright (c) 1990 Joseph S. Barrera III, Randall W. Dean
 */

#include <mach.h>
#include <fnode.h>
#include <errno.h>
#include <ux_user.h>
#include <ufs_fops.h>
#include <device/device_types.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <machine/param.h>

extern port_t device_server_port;
extern struct unode *unode_alloc();

struct unode *un_head = 0;

struct unode *
un_by_memory_object(memory_object)
	mach_port_t memory_object;
{
	return (struct unode *)memory_object;
}

/*
 * Open an unode.
 */
int
openi(ud, ino, unp)
	struct udev *ud;
	ino_t ino;
	struct unode **unp;
{
	int error;
	register struct dinode *dp;
	int index;
	struct unode *un;

	/*
	 * First, see if there's an appropriate active unode already.
	 */
	for (un = un_head; un; un = un->un_next) {
		if (ud == un->un_ud && ino == un->un_ino) {
			FOP_INCR(&un->un_fn);
			*unp = un;
			return 0;
		}
	}

	/*
	 * Read block containing inode from disk,
	 * and pull the inode out of it.
	 */
	index = itoo(&ud->ud_fs, ino);
	dp = (struct dinode*)
	   DISK_BLOCK_ADDR( ud, fsbtodb( &ud->ud_fs, itod(&ud->ud_fs, ino)));
	un = unode_alloc();
	un->i_ic = &dp[index].di_ic;
	un->un_fn.fn_fs = &ud->ud_fnfs; /* XXX refcount */
	un->un_ud = ud; /* xxx refcount + redundant */
	un->un_ino = ino;
	un->un_next = un_head;
	un_head = un;
	FOP_INCR(&un->un_fn);
	*unp = un;
	return 0;
}

sbmap(un, file_block, disk_block_p)
register struct unode	*un;
register daddr_t	file_block;
daddr_t			*disk_block_p;	/* out */
{
	int		level;
	int		idx;
	daddr_t		ind_block_num;
	kern_return_t	rc;
	struct udev	*ud;

	/*
	 * Index structure of an inode:
	 *
	 * i_db[0..NDADDR-1]	hold block numbers for blocks
	 *			0..NDADDR-1
	 *
	 * i_ib[0]		index block 0 is the single indirect
	 *			block
	 *			holds block numbers for blocks
	 *			NDADDR .. NDADDR + NINDIR(fs)-1
	 *
	 * i_ib[1]		index block 1 is the double indirect
	 *			block
	 *			holds block numbers for INDEX blocks
	 *			for blocks
	 *			NDADDR + NINDIR(fs) ..
	 *			NDADDR + NINDIR(fs) + NINDIR(fs)**2 - 1
	 *
	 * i_ib[2]		index block 2 is the triple indirect
	 *			block
	 *			holds block numbers for double-indirect
	 *			blocks for blocks
	 *			NDADDR + NINDIR(fs) + NINDIR(fs)**2 ..
	 *			NDADDR + NINDIR(fs) + NINDIR(fs)**2
	 *				+ NINDIR(fs)**3 - 1
	 */

	if (file_block < NDADDR) {
	    /* Direct block. */
	    *disk_block_p = un->i_db[file_block];
	    return (0);
	}

	file_block -= NDADDR;

	/*
	 * nindir[0] = NINDIR
	 * nindir[1] = NINDIR**2
	 * nindir[2] = NINDIR**3
	 *	etc
	 */

	ud = un->un_ud;
	for (level = 0; level < NIADDR; level++) {
	    if (file_block < ud->ud_nindir[level])
		break;
	    file_block -= ud->ud_nindir[level];
	}
	if (level == NIADDR) {
	    /* Block number too high */
	    return (EIO);
	}

	ind_block_num = un->i_ib[level];

	for (; level >= 0; level--) {
	    kern_return_t error;

	    if (ind_block_num == 0) {
		*disk_block_p = 0;	/* missing */
		return (0);
	    }

	    if (level > 0) {
		idx = file_block / ud->ud_nindir[level-1];
		file_block %= ud->ud_nindir[level-1];
	    }
	    else
		idx = file_block;

	    ind_block_num =
	       ((daddr_t*) DISK_BLOCK_ADDR( ud, fsbtodb( &ud->ud_fs,
							ind_block_num )))[idx];
	}

	*disk_block_p = ind_block_num;

	return (0);
}

int
read_fs(fn, fs)
	struct fnode *fn;
	struct fs *fs;
{
	int	error;
	vm_offset_t	buf;
	vm_size_t	buf_size;

	error = bdev_device_read(fn, 0, (recnum_t) SBLOCK, SBSIZE,
				 (char **) &buf, &buf_size);
	if (error) {
		return error;
	}

	if (SBSIZE != buf_size) {
		printf("read_fs: SBSIZE=%d, buf_size=%d\n", SBSIZE, buf_size);
		return EINVAL;
	}

	bcopy(buf, fs, sizeof(struct fs));
	(void) vm_deallocate(mach_task_self(), buf, buf_size);

	if (fs->fs_magic != FS_MAGIC ||
	    fs->fs_bsize > MAXBSIZE ||
	    fs->fs_bsize < sizeof(struct fs)) {
		return EINVAL;
	}
	/* don't read cylinder groups - we aren't modifying anything */
	return 0;
}

static void
vm_sync() { return; }

#define	SINGLE	0	/* index of single indirect block */
#define	DOUBLE	1	/* index of double indirect block */
#define	TRIPLE	2	/* index of triple indirect block */
/*
 * Truncate the inode ip to at most length size.  Free affected disk
 * blocks -- the blocks of the file are removed in reverse order.
 *
 * NB: triple indirect blocks are untested.
 */
itrunc(oip, length, flags)
	register struct unode *oip;
	u_long length;
	int flags;
{
	register daddr_t lastblock;
	daddr_t bn, lbn, lastiblock[NIADDR];
	register struct fs *fs;
	register struct unode *ip;
	int offset, osize, size, level;
	long count, nblocks, blocksreleased = 0;
	register int i;
	int aflags, error, allerror;
	struct unode tip;

	/* vnode_pager_setsize(ITOV(oip), length); */
	if (oip->i_size <= length) {
#if 0				/* XXX is there a POE equivalent to this? */
		error = iupdat(oip, &time, &time, 1);
		return (error); */
#else /* not 0 */
		return 0;
#endif /* 0 */
	}
	/*
	 * Calculate index into inode's block list of
	 * last direct and indirect blocks (if any)
	 * which we want to keep.  Lastblock is -1 when
	 * the file is truncated to 0.
	 */
	fs = &oip->un_ud->ud_fs;
	lastblock = lblkno(fs, length + fs->fs_bsize - 1) - 1;
	lastiblock[SINGLE] = lastblock - NDADDR;
	lastiblock[DOUBLE] = lastiblock[SINGLE] - NINDIR(fs);
	lastiblock[TRIPLE] = lastiblock[DOUBLE] - NINDIR(fs) * NINDIR(fs);
	nblocks = btodb(fs->fs_bsize);
	/*
	 * Update the size of the file. If the file is not being
	 * truncated to a block boundry, the contents of the
	 * partial block following the end of the file must be
	 * zero'ed in case it ever become accessable again because
	 * of subsequent file growth.
	 */
	osize = oip->i_size;
	offset = blkoff(fs, length);
	if (offset == 0) {
		oip->i_size = length;
	} else {
	   	int diskblkno;
		lbn = lblkno(fs, length);
		aflags = B_CLRBUF;
		if (flags & IO_SYNC)
			aflags |= B_SYNC;
#ifdef QUOTA
		if (error = getinoquota(oip))
			return (error);
#endif
		if (error = balloc(oip, lbn, offset, &diskblkno, aflags))
			return (error);
		oip->i_size = length;
		size = blksize(fs, oip, lbn);
		/* (void) vnode_pager_uncache(ITOV(oip)); */ /*XXX*/
		bzero(DISK_BLOCK_ADDR( oip->un_ud, diskblkno ) + offset,
		      (unsigned)(size - offset));
		/* allocbuf(bp, size); */
		if (flags & IO_SYNC) {
		   vm_sync( DISK_BLOCK_ADDR( oip->un_ud, diskblkno ), size );
		}
	}
	/*
	 * Update file and block pointers
	 * on disk before we start freeing blocks.
	 * If we crash before free'ing blocks below,
	 * the blocks will be returned to the free list.
	 * lastiblock values are also normalized to -1
	 * for calls to indirtrunc below.
	 */
	tip = *oip;
	tip.i_size = osize;
	for (level = TRIPLE; level >= SINGLE; level--)
		if (lastiblock[level] < 0) {
			oip->i_ib[level] = 0;
			lastiblock[level] = -1;
		}
	for (i = NDADDR - 1; i > lastblock; i--)
		oip->i_db[i] = 0;
#if 0 /* XXX Is there some equivalent operation in POE? */
	vinvalbuf(ITOV(oip), (length > 0));
	allerror = iupdat(oip, &time, &time, MNT_WAIT);
#endif

	/*
	 * Indirect blocks first.
	 */
	ip = &tip;
	for (level = TRIPLE; level >= SINGLE; level--) {
		bn = ip->i_ib[level];
		if (bn != 0) {
			error = indirtrunc(ip, bn, lastiblock[level], level,
				&count);
			if (error)
				allerror = error;
			blocksreleased += count;
			if (lastiblock[level] < 0) {
				ip->i_ib[level] = 0;
				blkfree(ip, bn, (off_t)fs->fs_bsize);
				blocksreleased += nblocks;
			}
		}
		if (lastiblock[level] >= 0)
			goto done;
	}

	/*
	 * All whole direct blocks or frags.
	 */
	for (i = NDADDR - 1; i > lastblock; i--) {
		register off_t bsize;

		bn = ip->i_db[i];
		if (bn == 0)
			continue;
		ip->i_db[i] = 0;
		bsize = (off_t)blksize(fs, ip, i);
		blkfree(ip, bn, bsize);
		blocksreleased += btodb(bsize);
	}
	if (lastblock < 0)
		goto done;

	/*
	 * Finally, look for a change in size of the
	 * last direct block; release any frags.
	 */
	bn = ip->i_db[lastblock];
	if (bn != 0) {
		off_t oldspace, newspace;

		/*
		 * Calculate amount of space we're giving
		 * back as old block size minus new block size.
		 */
		oldspace = blksize(fs, ip, lastblock);
		ip->i_size = length;
		newspace = blksize(fs, ip, lastblock);
		if (newspace == 0)
			panic("itrunc: newspace");
		if (oldspace - newspace > 0) {
			/*
			 * Block number of space to be free'd is
			 * the old block # plus the number of frags
			 * required for the storage we're keeping.
			 */
			bn += numfrags(fs, newspace);
			blkfree(ip, bn, oldspace - newspace);
			blocksreleased += btodb(oldspace - newspace);
		}
	}
done:
/* BEGIN PARANOIA */
	for (level = SINGLE; level <= TRIPLE; level++)
		if (ip->i_ib[level] != oip->i_ib[level])
			panic("itrunc1");
	for (i = 0; i < NDADDR; i++)
		if (ip->i_db[i] != oip->i_db[i])
			panic("itrunc2");
/* END PARANOIA */
	oip->i_blocks -= blocksreleased;
	if (oip->i_blocks < 0)			/* sanity */
		oip->i_blocks = 0;
#ifdef QUOTA
	if (!getinoquota(oip))
		(void) chkdq(oip, -blocksreleased, NOCRED, 0);
#endif
	return (allerror);
}

/*
 * Release blocks associated with the inode ip and
 * stored in the indirect block bn.  Blocks are free'd
 * in LIFO order up to (but not including) lastbn.  If
 * level is greater than SINGLE, the block is an indirect
 * block and recursive calls to indirtrunc must be used to
 * cleanse other indirect blocks.
 *
 * NB: triple indirect blocks are untested.
 */
indirtrunc(ip, bn, lastbn, level, countp)
	register struct unode *ip;
	daddr_t bn, lastbn;
	int level;
	long *countp;
{
	register int i;
	register struct fs *fs = &ip->un_ud->ud_fs;
	register daddr_t *bap;
	daddr_t nb, last;
	long blkcount, factor;
	int nblocks, blocksreleased = 0;
	int error, allerror = 0;

	/*
	 * Calculate index in current block of last
	 * block to be kept.  -1 indicates the entire
	 * block so we need not calculate the index.
	 */
	factor = 1;
	for (i = SINGLE; i < level; i++)
		factor *= NINDIR(fs);
	last = lastbn;
	if (lastbn > 0)
		last /= factor;
	nblocks = btodb(fs->fs_bsize);
	/*
	 * Get buffer of block pointers, zero those 
	 * entries corresponding to blocks to be free'd,
	 * and update on disk copy first.
	 */
	bap = (daddr_t*) DISK_BLOCK_ADDR( ip->un_ud, fsbtodb( fs, bn));
	/* lock_free_list( fs ); */	/* XXX need to make this work. */

	/*
	 * Recursively free totally unused blocks.
	 */
	for (i = NINDIR(fs) - 1; i > last; i--) {
		nb = bap[i];
		if (nb == 0)
			continue;
		if (level > SINGLE) {
			error = indirtrunc(ip, nb, (daddr_t)-1, level - 1,
				&blkcount);
			if (error)
				allerror = error;
			blocksreleased += blkcount;
		}
		blkfree(ip, nb, (off_t)fs->fs_bsize);
		blocksreleased += nblocks;
	}

	/*
	 * Recursively free last partial block.
	 */
	if (level > SINGLE && lastbn >= 0) {
		last = lastbn % factor;
		nb = bap[i];
		if (nb != 0) {
			error = indirtrunc(ip, nb, last, level - 1, &blkcount);
			if (error)
				allerror = error;
			blocksreleased += blkcount;
		}
	}
	*countp = blocksreleased;
	return (allerror);
}
