/*
 * WMAIL -	MicroWalt Extended Mail Agent.
 *		This is the MicroWalt Mail Agent; which is derived
 *		from  the  "Mini-Mail" written by Peter S. Housel.
 *		This version  has a better  user-interface, and it
 *		also "knows" about things like forwarding, replies
 *		etc. Overall, it looks like the Mail (uppercase M)
 *		on our local DYNIX(tm) (==BSD) system...
 *		The paging-code (for paging letters on the screen)
 *		was taken from "more.c", by Brandon Allbery.
 *
 *		I N T E R A C T I O N    M O D U L E
 *
 * Author:	Fred van Kempen, MicroWalt Corporation
 *
 * Revisions:
 *		11/07/89 FvK	Edited a little for the new MSS.
 *		11/10/89 FvK	Fixed the overall security-bug.
 *		11/16/89 RW	Fixed some interact() bugs. E.g.:
 *				d and p works from the beginning,
 *				+ is the real opposite of - now.
 *		12/04/89 FvK	Removed unneeded variables.
 *		12/16/89 FvK	Cleanup.
 *		02/17/90 Fvk	Cleaned for release.
 *		04/10/90 FvK	Added "Reply" status report if failed.
 *		04/28/90 FvK	Changed all "continue" to "break" statements
 *				in interact().
 *
 * To Do:
 *		- Builtin escapes (~i and friends)
 */
#include <sys/types.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <stdio.h>
#include "wmail.h"


/*
 * Send a reply to a message.
 */
static int do_reply(let)
LETTER *let;
{
  char *who[2];

  who[0] = let->sender;
  who[1] = (char *)NULL;
  sprintf(subject, "Re: %s", let->subject);

  printf("\n*** From: %s %-24.24s\n", sender, xtime());
  printf("*** To: %s\n", who[0]);
  printf("*** Subject: %s\n\n", subject);

  if (deliver(1, who) == -1) {
	fprintf(stderr, "%s: mail delivery for REPLY failed.\n", progname);
  }
}


/*
 * Execute a shell (or a command only)
 */
static void do_shell(command)
char *command;
{
  int status, pid;
  char *shell;

  if ((shell = getenv("SHELL")) == (char *)NULL) shell = SHELL;

  if ((pid = fork()) == 0) {
	setuid(old_uid);	/* security! */
	setgid(old_gid);
	umask(oldmask);

	execl(shell, shell, "-c", command, (char *)NULL);
	fprintf(stderr, "%s: cannot exec shell", progname);
	exit(127);
  } else {
	  if (pid > 0) wait(&status);
	    else fprintf(stderr, "%s: could not fork", progname);
  }
}


/*
 * Goto a specific letter.
 */
static LETTER *goto_letter(num)
register int num;
{
  register LETTER *let;

  let = firstlet;
  while (let != NIL_LET) {
	if (let->seqno == num) return(let);
	let = let->next;
  }
  return(NIL_LET);
}


/*
 * Skip the header of the current message.
 * Do this by just updating the 'boxfp' pointer...
 */
static off_t skiphead()
{
  char skbuf[1024];
  off_t count;

  count = (off_t) 0;
  while (fgets(skbuf, sizeof(skbuf), boxfp) != (char *)NULL) {
	count += (off_t) strlen(skbuf);
      	if (skbuf[0] == '\n') break;  /* end of header */
  }
  return(count);
}


/*
 * Save the current letter to a disk-file.
 * Do this as a child process to make sure we
 * do not violate system security!
 */
void savelet(let, savefile, withhead)
LETTER *let;
char *savefile;
int withhead;
{
  off_t curr, limit, oldpos;
  register char *bp;
  FILE *savefp;
  int c, temp;

  bp = savefile;
  while (*bp && *bp!='\n') bp++;
  *bp = '\0';

  /* Check if we may write to that file. */
  if (allowed(savefile, 02) == FALSE) {
	fprintf(stderr, "%s: cannot append to savefile \"%s\"\n",
							progname, savefile);
	return;
  }

  /* We may, go ahead! */
  if ((savefp = fopen(savefile, "a")) == (FILE *)NULL) {
	fprintf(stderr, "%s: cannot append to savefile \"%s\"\n",
							progname, savefile);
	return;
  }

  temp = umask(oldmask);	/* set previous umask() */

  oldpos = ftell(boxfp);
  fseek(boxfp, (curr = let->location), SEEK_SET);
  limit = (let->next != NIL_LET) ? let->next->location : -1L;

  if (withhead == 0) curr += skiphead();	/* skip the message header */
  while(curr != limit && (c = fgetc(boxfp)) != EOF) {
	fputc(c, savefp);
      	++curr;
  }
  fflush(savefp);
  fseek(boxfp, oldpos, SEEK_SET);

  if ((ferror(savefp) != 0) | (fclose(savefp) != 0)) {
	fprintf(stderr, "%s: savefile write error:", progname);
  }

  umask(temp);		/* set previous umask() */

  chown(savefile, old_uid, old_gid);
}


/*
 * Give a list of possible Interact Commands.
 */
static void do_help()
{
  printf("\n   ** W-MAIL Commands **\n\n");
  printf("?\tThis help\n");
  printf("!\tShell Command Escape\n");
  printf("-\tPrevious letter\n");
  printf("+\tNext letter\n");
  printf("<ENTER>\tShow current letter, goto next one\n");
  printf("d\tDelete current letter\n");
  printf("i\tDisplay a summary of letters\n");
  printf("p\tPrint a letter again\n");
  printf("q\tQuit, update mailbox\n");
  printf("r\tReply to the current letter\n");
  printf("s\tSave current letter\n");
  printf("t\tType a letter, no paging\n");
  printf("w\tSave letter without header\n");
  printf("x\tExit, do not update mailbox\n");
  printf("\n");
}


/*
 * Give a summary of letters./
 */
static void summary(nextlet)
int nextlet;
{
  register LETTER *let;
  register int i;
  register char *sp;

  let = firstlet;
  printf(" No.  Sender           Date            Subject\n");
  printf(" ----------------------------------------------");
  printf("--------------------------------\n");

  for (i=0; i<numlet; i++) {
	sp = let->date;
	if (strchr(sp, ',')) sp += 5;	/* xtime() or ctime() ?? */
	  else sp += 4;
	printf("%c%c %-3.3d  %-8.8s  %-15.15s  \"%.40s\"\n",
		(nextlet == let->seqno) ? '>': ' ',
		(let->status == DELETED) ? '*': ' ',
 		let->seqno, basepath(let->sender), sp, let->subject);
        let = let->next;
  }
  printf("\n");
}


/*
 * Interactively read the mail-box.
 */
void interact()
{
  char input[512];			/* user input line */
  LETTER *let, *templet;		/* current and next letter */
  int interrupted = FALSE;		/* SIGINT hit during letter print */
  char *savefile;
  int temp, nextlet;

  if (firstlet == NIL_LET) {
	printf("No mail for %s.\n", sender);
      	return;
  }

  printf("W-MAIL %s.  Type ? for Help.\n", Version);
  printf("\"%s\": %d message(s)\n\n", mailbox, numlet);

  let = firstlet;		/* Set first letter. */

  summary(1);			/* Show a summary of all the letters. */

  while(TRUE) {
	nextlet = let->seqno;

       	if (!quitmode) {
		interrupted = setjmp(printjump);
		signal(SIGINT, onint);
       	}

      	if (interrupted == TRUE) printf("\n");
      	printf(PROMPT, let->seqno);
      	fflush(stdout);

      	if (fgets(input, sizeof(input), stdin) == (char *)NULL) break;

      	if (!quitmode) signal(SIGINT, SIG_IGN);

	/*
	 * Look at the first character of the command line.
	 * This should be improved!
	 */
      	switch(input[0]) {
		case '!':	/* Shell escape. */
			do_shell(&input[1]);
			break;
		case '?':	/* Type some help. */
			do_help();
			break;
		case '-':	/* Show previous letter. */
			if (let->prev != NIL_LET) let = let->prev;
			  else printf("Top of mailbox\n");
			break;
		case '+':	/* Show next letter. */
			if (let->next != NIL_LET) let = let->next;
			  else printf("At EOF\n");
			break;
		case '\n':	/* Show current letter and move to next. */
			if (!interrupted) {
				if (let->status != DELETED)
						let->status = READ;
		    		printlet(let, stdout);
			}
			if (let->next == NIL_LET) {
				printf(PROMPT, let->seqno);
				printf("At EOF\n");
			} else let = let->next;
			break;
		case 'd':	/* Delete current letter. */
			let->status = DELETED;
			if (let->next != NIL_LET) let = let->next;
			needupdate = TRUE;
			break;
		case 'i':	/* Show letter summary */
			summary(nextlet);
			break;
		case 'p':	/* Print a letter. */
			if (!interrupted) printlet(let, stdout);
			break;
		case 'q':	/* Update mailbox and quit. */
			return;
		case 'r':	/* Reply to a message. */
			do_reply(let);
			break;
		case 's':	/* Save message to disk. */
			savefile = &input[1];
			if (*savefile != '\0') {
			  	while (*savefile==' ' || *savefile=='\t')
								savefile++; 
			} else savefile = SAVEFILE;
			savelet(let, savefile, TRUE);
			break;
		case 't':	/* Type (no paging) current letter. */
			temp = printmode;
			printmode = TRUE;
			if (!interrupted) printlet(let, stdout);
			printmode = temp;
			break;
		case 'w':	/* Write (without header) message to disk. */
			savefile = &input[1];
			if (*savefile != '\0') {
			  	while (*savefile==' ' || *savefile=='\t')
								savefile++; 
			} else savefile = SAVEFILE;
			savelet(let, savefile, FALSE);
			break;
		case 'x':	/* Abort, do not update mailbox. */
			exit(0);
		default:
			if (isdigit(input[0])) {
				templet = goto_letter(atoi(input));
				if (templet != NIL_LET) {
					let = templet;
					printlet(let, stdout);
			   	} else printf("Illegal message-number\n");
			} else printf("Illegal command\n");
			break;
  	}
  }   
}
