/* Copyright (c) 1999, 2000 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS ADV.T/STD.T LIBRARY EXTENSION
 *  MULTIFORM.T				
 *  version 2.0
 *
 *	The multiform object is an openable AND a surface. Things can be placed both
 *	inside and on a multiform. 
 *
 *	Because the multiform has properties of both types of object it displays
 *	messages for contents on its surface, and inside contents.
 *
 *  More detailed information concerning the scope functions can be 
 *  obtained from the manual provided with Scope.t.
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.2.6 or later
 *      + Should be #included after ADV.T and STD.T 
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      + This file replaces the deepverb validXoList() and validXo()
 *        methods. Other replacements include isVisible() and several
 *        methods of:
 *              * thing
 *              * surface
 *              * container
 *              * openable
 *              * transparentItem
 *      + Listing functions have been modified to handle "on" and "in"
 *      + This file incorporates SCOPE.T 2.0 which is an alternative 
 *        scoping mechanism for TADS ADVT. Scope.t should NOT be 
 *        #included in your source.
 *      + Sense.t is NOT compatible with this implementation of
 *        multiform, nor is any other accessibility-related extension
 *        that modifies the validXoList() / validXo() deepverb methods.
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
 *  REVISION HISTORY
 *
 *		21-Jun-99:	Creation.
 */
 
#define __MULTIFORM_MODULE_

#define SCOPE_OBJTREE  1
#define SCOPE_VISIBLE   2
#define SCOPE_REACHABLE 3

scope: function;
scopeCeiling: function;
scopeList: function;
addToScope: function;
isScopedef: function;
inScope: function;

/*
 *  multiform: object
 *
 *  This class can be *prefixed* to a class list for an object to 
 *  allow it to combine the ldesc() and doSearch() methods for the
 *  openable, container, and surface classes. 
 *
 *  multiform class objects always display their 'surface' contents
 *  first, then their 'inside' contents, regardless of the order of
 *  the class list.
 */  
class multiform: object
    ldesc =
    {
        if (isclass(self,surface))
            inherited surface.ldesc;
        if (isclass(self,openable))
            inherited openable.ldesc;
        else if (isclass(self,container))
            inherited container.ldesc;
    }
    doSearch(actor) =
    {
       if (isclass(self,surface))
            inherited surface.doSearch;
       if (isclass(self,container))
            inherited container.doSearch;
    }
;

/*
 *	scope: function( vantage, limit, level )
 *
 *	Function returns a list of all objects that are valid for a given vantage
 *	and scope level. 
 *
 *	First it determines the scopeCeiling for a given vantage and scope level. 
 *	This becomes the first entry in the list. 
 *
 *  limit can be useful in cases where the author wishes to limit the
 *  scope to within a select location, even though the scope level might
 *  extend beyond the location.
 *
 *	Next it calls scopeList with the location, vantage, scope ceiling, and 
 *	scope level and adds the return to the list. 
 *
 *	Next each object in global.floatingList is checked using scopeList and 
 *	the results are added to the list.
 *
 *	Finally, the special vantage.scopelist is added to scope.
 */
scope: function( vantage, limit, level )
{
    local c, o, loc, ceiling, lst := [];
 
    ceiling := scopeCeiling( vantage, limit, level );
        
    loc := ceiling;

    lst += ceiling;
    lst += scopeList( loc, vantage, ceiling, level );

    /*
     *  Each floatingItem object is evaluated. If it has the same
     *  scope ceiling as the vantage or its scope ceiling is the
     *  vantage then it is added to scope along with any appropriate
     *  contents.
     */
	c := global.floatingList;
	o := car( c );
	while( o )
	{
		loc := o;
		ceiling := scopeCeiling( o, limit, level );
		if ( ceiling = lst[1] or ceiling = vantage )
		{
			lst += o;
			lst += scopeList( loc, o, ceiling, level );
		}
		
		c := cdr( c );
		o := car( c );
	}
	
	if ( level <> SCOPE_OBJTREE )
		lst := addToScope( lst, vantage );
    
    return lst;
}

/* 
 *	scopeCeiling: function( vantage, limit, level )
 *
 *	Function returns the scopeCeiling for a given vantage and scope level. The
 *	scopeCeiling is the highest level location that the vantage has access to
 *	for the given scope level.
 *
 *  limit acts to constrain the function a given scope ceiling "height",
 *  but the function will always return a valid scope ceiling for a
 *  valid location.
 *
 */
scopeCeiling: function( vantage, limit, level )
{
    local loc;
    
    if (vantage = limit )
    	return vantage;
    	
    if (vantage.location)
    	loc := vantage.location;
    else
    	loc := vantage;
    while( loc and loc.location ) 
   	{
   	    if ( loc = limit )
   	        break;
   	        
   	    if ( level = SCOPE_VISIBLE 
   	    and not isScopedef( loc, level ))
            break;
            
        if ( level = SCOPE_REACHABLE 
        and not isScopedef( loc, level ))
            break;
            
        loc := loc.location;
    }
    return loc;
}

/*
 *  scopeList: function( loc, vantage, ceiling, level )
 *
 *  Function returns a list of all valid objects for a given vantage, scope
 *	ceiling and scope level. It recurses down each object in the location's
 *	contents.
 */
scopeList: function( loc, vantage, ceiling, level )
{
    local ret := [];
    local i, lst, len;

    /* don't look in "nil" objects */
    if (loc = nil)
        return ret;

	 /* 
	  * We don't want items on the surface of the scope ceiling in
	  * our scope list.
	  *
	  * check each location's contents to see if any are on the 
	  * surface. If they are then add them to scope and recurse
	  * down into their contents. Remove any objects 'on surface'
	  * from lst, so that we don't pick them up again when we 
	  * are dealing with 'inside'.
	  */
	lst := loc.contents;
	if (loc <> ceiling)
	{
	    local tmp := [];
	    len := length(lst);
	    for (i := 1 ; i <= len ; ++i)
	    {
            /* recurse into this object */
	        if (lst[i].isOnSurface)
	        {
      		    ret += lst[i];
       		    tmp += lst[i];
      		    ret += scopeList( lst[i], vantage, ceiling, level );
	        }
	    }
	 	lst -= tmp;
	}
	/*
	 *	The location is added to the list if,
	 *		a. it is the vantage or the scope ceiling
	 *		b. it meets the requirements of a scope definition for the given
	 *		   scope level.
	 */
	if (loc = vantage 
	or  loc = ceiling
	or  isScopedef( loc, level ))
    {
        len := length(lst);
        ret += lst;
        for (i := 1 ; i <= len ; ++i)
        {
            /* recurse into this object */
            ret += scopeList( lst[i], vantage, ceiling, level );
        }
    }

    return ret;
}

/*
 *  addToScope: function( scope, vantage )
 *
 *  Function adds special vantage scopelist objects to the scope. This function
 *	provides an alternative mechanism to the normal library's "bubble-up/filter-
 *	down" accessibility mechanisms. An object in the vantage's scopelist will be
 *	added to scope regardless of the verb. This mechanism can be used the same
 *	way validActor is used, except that the vantage need not be the actor.
 *
 *	Function ensures that the object to be added isn't already part of scope. If
 *	it is not then it will be added. 
 */
addToScope: function( scope, vantage )
{
    local i, len := length(vantage.scopelist);
    
    if (len)
    {
        for (i := 1; i <= len; ++i)
        {
            if (not find(scope, vantage.scopelist[i]))
            {
                scope += vantage.scopelist[i];
            }
        }
    }
    return scope;
}

/*
 *  isScopedef: function( loc, level )
 *
 *  Function returns true if the location fits the specified scope
 *  level definition; otherwise it returns nil. This allows scope-
 *  related definitions to be maintained and modified in one central
 *  function.
 */
isScopedef: function( loc, level )
{
    if (level = SCOPE_OBJTREE)
        return true;
        
    if (level = SCOPE_VISIBLE
    and (not isclass(loc, openable)
        or (isclass(loc, openable) and loc.isopen)
        or loc.contentsVisible))
            return true;

	if (level = SCOPE_REACHABLE
    and (not isclass(loc, openable)
        or (isclass(loc, openable) and loc.isopen)))
            return true;
    
    return nil;
}

/*
 *  inScope: function( vantage, target, limit, level )
 *
 *  Function returns true if the target is within the scope list
 *  produced for the given scope level for the vantage; otherwise
 *  it returns nil.
 */
inScope: function( vantage, target, limit, level )
{	
	local value := scope( vantage, limit, level );
	if ( find( value, target ) ) 
	    return true;
	return nil;
}

/*----------------------------------------------------------------------
 *  CHANGES TO ADV.T
 *--------------------------------------------------------------------*/

modify thing
	/*
	 *	scopelist is a special list that can be modified in the author's code to
	 *	add objects to scope outside of the normal "bubble-up/filter-down"
	 *	scoping mechanism. Objects added to this list will be in scope
	 *	regardless of the scope level and therefore available for all actions
	 *	which use scope to determine their validXoList() and validXo() methods
	 *	and should therefore be used only for very exceptional cases.
	 *
	 *	scopelist should never include the object itself or anything ordinarily
	 *	within the scope of the object, as these are handled by normal scoping 
	 *	processes and will already be part of scope.
	 */
    scopelist = []
	/*
 	 *	Modifying verifyRemove() to 'bubble up' verGrab() check, but only as
 	 *	high as thescope ceiling. This allows us to grab things when we're
 	 *	inside closed locations, such as enterables.
 	 */
    replace verifyRemove(actor) =
    {
        /*
         *   Check with each container to make sure that the container
         *   doesn't object to the object's removal.
         */
        local loc, ceiling;

		ceiling := scopeCeiling( actor, nil, SCOPE_REACHABLE );

        loc := self.location;
        while (loc)
        {
        	if (loc = ceiling)
      			break;
            if (loc <> actor)
                loc.verGrab(self);
            loc := loc.location;
        }
    }
    /*
     *	isVisible() now incorporates scope mechanism.
     */
    replace isVisible(vantage) =
    {
        return( inScope( vantage, self, nil, SCOPE_VISIBLE ) );
    }
    isOnSurface = nil       // the object default's to 'inside'
    /*
     *  containerList
     *
     *  This method returns a list of all content objects with 
     *  isOnSurface = nil. If the list is empty the method returns
     *  nil.
     */
    containerList =
    {
    	local i, list := [];
    	
    	for (i := 1; i <= length(self.contents); ++i)
    	{
    		if (self.contents[i].isListed)
    		{
    			if (not self.contents[i].isOnSurface)
    				list += self.contents[i];
     		}
    	}
    	if (length(list))
    	    return list;
    	else
    	    return nil;
    }
    /*
     *  surfaceList
     *
     *  This method returns a list of all content objects with 
     *  isOnSurface = true. If the list is empty the method returns
     *  nil.
     */
    surfaceList =
    {
    	local i, list := [];
    	
    	for (i := 1; i <= length(self.contents); ++i)
    	{
    		if (self.contents[i].isListed)
    		{
    			if (self.contents[i].isOnSurface)
    				list += self.contents[i];
    		}
    	}
    	if (length(list))
    	    return list;
    	else
    	    return nil;
    }
    replace verDoPutIn(actor, io) =
    {
        if (io = nil)
            return;

        if (self.location = io and !self.isOnSurface)
        {
            caps(); self.thedesc; " is already in "; io.thedesc; "! ";
        }
        else if (io = self)
        {
            "%You% can't put "; self.thedesc; " in <<self.itselfdesc>>! ";
        }
        else if (io.isIn(self))
            self.circularMessage(io);
        else
            self.verifyRemove(actor);
    }
    replace verDoPutOn(actor, io) =
    {
        if (io = nil)
            return;

        if (self.location = io and self.isOnSurface)
        {
            caps(); self.thedesc; " is already on "; io.thedesc; "! ";
        }
        else if (io = self)
        {
            "%You% can't put "; self.thedesc; " on <<self.itselfdesc>>! ";
        }
        else if (io.isIn(self))
            self.circularMessage(io);
        else
            self.verifyRemove(actor);
    }
    /*
     *  doPutOn(actor, io)
     *
     *  This modification sets self.isOnSurface to true.
     */
    replace doPutOn(actor, io) =
    {
        self.moveInto(io);
        self.isOnSurface := true;
        "Done. ";
    }
    /*
     *  moveInto(obj)
     *  
     *  This modification sets the object's isOnSurfact attribute
     *  to nil.
     */
    replace moveInto(obj) =
    {
        local loc;

        /*
         *   For the object containing me, and its container, and so forth,
         *   tell it via a Grab message that I'm going away.
         */
        loc := self.location;
        while (loc)
        {
            loc.Grab(self);
            loc := loc.location;
        }
        
        if (self.location)
            self.location.contents := self.location.contents - self;
        self.location := obj;
        if (obj)
            obj.contents := obj.contents + self;
        self.isOnSurface := nil;
    }
;

/*
 *  surface: item
 *
 *  The following modifications incorporate the surfaceList()
 *  method for display purposes.
 */
modify surface
    replace ldesc =
    {
        if (self.surfaceList)
        {
            "On "; self.thedesc; " %you% see%s% "; 
            listcont(self.surfaceList); ". ";
        }
        else
        {
            "There's nothing on "; self.thedesc; ". ";
        }
    }
    replace doSearch(actor) =
    {
        if (self.surfaceList)
            "On <<self.thedesc>> %you% see%s% 
            <<listcont(self.surfaceList)>>. ";
        else
            "There's nothing on <<self.thedesc>>. ";
    }
;

/*
 *  container: item
 *  
 *  The following modifications incorporate the containerList()
 *  method for display and bulk computation purposes.
 */
modify container
    replace ldesc =
    {
        if (self.contentsVisible 
        and self.containerList)
        {
            "In "; self.thedesc; " %you% see%s% "; 
            listcont(self.containerList); ". ";
        }
        else
        {
            "There's nothing in "; self.thedesc; ". ";
        }
    }
    replace ioPutIn(actor, dobj) =
    {
        local cont := self.containerList;
        
        if (cont)
        {
        	if (length(cont)=0) cont := [];
        }
        else
        	cont := [];
        	
        if (addbulk(cont) + dobj.bulk > self.maxbulk)
        {
            "%You% can't fit <<dobj.thatdesc>> in "; self.thedesc; ". ";
        }
        else
        {
            dobj.doPutIn(actor, self);
        }
    }
    replace doSearch(actor) =
    {
        if (self.contentsVisible 
        and self.containerList)
            "In <<self.thedesc>> %you% see%s% 
            <<listcont(self.containerList)>>. ";
        else
            "There's nothing in <<self.thedesc>>. ";
    }
;

/*
 *  openable: container
 *  
 *  The following modifications incorporate the containerList()
 *  method for display purposes.
 */
modify openable
    replace doOpen(actor) =
    {
        if (self.containerList)
        {
            "Opening "; self.thedesc; " reveals "; 
            listcont(self.containerList); ". ";
        }
        else
            "Opened. ";
        self.isopen := true;
    }
;


modify transparentItem
    replace verGrab(obj) =
    {
        if (self.isopenable and not self.isopen)
        {
            if (not(obj.isOnSurface and find(self.contents, obj) <> 0))
            	"%You% will have to open << self.thedesc >> first. ";
        }
    }
;

 /*
  *  listcont: function(obj)
  *
  *  This function performs exactly as ADV.T listcont(obj). The
  *  only difference is that it accepts a list or an object.
  *
  *  This function displays the contents of an object/list, separated
  *  by commas.  The thedesc properties of the contents are used.
  *  It is up to the caller to provide the introduction to the list
  *  (usually something to the effect of "The box contains" is
  *  displayed before calling listXcont) and finishing the
  *  sentence (usually by displaying a period).  An object is listed
  *  only if its isListed property is true.  If there are
  *  multiple indistinguishable items in the list, the items are
  *  listed only once (with the number of the items).
  */
replace listcont: function(obj)
{
    local i, count, tot, list, cur, disptot, prefix_count;

	if (datatype(obj) = 2 )
    	list := obj.contents;
    else
    	list := obj;

    tot := length(list);
    count := 0;
    disptot := itemcnt(list);
    for (i := 1 ; i <= tot ; ++i)
    {
        cur := list[i];
        if (cur.isListed)
        {
            /* presume there is only one such object */
            prefix_count := 1;            

            /*
             *   if this is one of more than one equivalent items, list it
             *   only if it's the first one, and show the number of such
             *   items along with the first one 
             */
            if (cur.isEquivalent)
            {
                local before, after;
                local j;
                local sc;

                sc := firstsc(cur);
                for (before := after := 0, j := 1 ; j <= tot ; ++j)
                {
                    if (isIndistinguishable(cur, list[j]))
                    {
                        if (j < i)
                        {
                            /*
                             *   note that objects precede this one, and
                             *   then look no further, since we're just
                             *   going to skip this item anyway
                             */
                            ++before;
                            break;
                        }
                        else
                            ++after;
                    }
                }
                
                /*
                 *  if there are multiple such objects, and this is 
                 *  the first such object, list it with the count 
                 *  prefixed; if there are multiple and this isn't 
                 *  the first one, skip it; otherwise, go on as 
                 *  normal 
                 */
                if (before = 0)
                    prefix_count := after;
                else
                    continue;
            }

            if (count > 0)
            {
                if (count+1 < disptot)
                    ", ";
                else if (count = 1)
                    " and ";
                else
                    ", and ";
            }  

            /* list the object, along with the number of such items */
            if (prefix_count = 1)
                cur.adesc;
            else
            {
                sayPrefixCount(prefix_count); " ";
                cur.pluraldesc;
            }  

            /* show any additional information about the item */
            if (cur.isworn)
                " (being worn)";
            if (cur.islamp and cur.islit)
                " (providing light)";
            count := count + 1;
        }
    }
}
;

/*
 *  showcontcont: function(obj)
 *  
 *  showcontcont:  list the contents of the object, plus the contents
 *  of an fixeditem's contained by the object.  A complete sentence 
 *  is shown. This is an internal routine used by listcontcont and 
 *  listfixedcontcont.
 */
replace showcontcont: function(obj)
{
    if (itemcnt(obj.contents))
    {
        if (obj.issurface)
        {
            if (not obj.isqsurface 
            and obj.surfaceList)
            {
                "Sitting on "; obj.thedesc;" is "; 
                listcont(obj.surfaceList);
                ". ";
            }
        }
        if (obj.contentsVisible and not obj.isqcontainer 
        and obj.containerList)
        {
            caps();
            obj.thedesc; " seem";
            if (!obj.isThem) "s";
            " to contain ";
            listcont(obj.containerList);
            ". ";
        }
    }
    if (obj.contentsVisible and not obj.isqcontainer 
    and obj.containerList)
        listfixedcontcont(obj.containerList);
}
;

replace listfixedcontcont: function(obj)
{
    local list, i, tot, thisobj;

	if (datatype(obj) = 2 )
    	list := obj.contents;
    else
    	list := obj;

    tot := length(list);
    i := 1;
    while (i <= tot)
    {
        thisobj := list[i];
        if (thisobj.isfixed and thisobj.contentsVisible and
             not thisobj.isqcontainer)
            showcontcont(thisobj);
        i := i + 1;
    }
}
;

/*
 *	Modifying deepverb to use the new SCOPING algorithms.
 */
modify deepverb
    replace validDo(actor, obj, seqno) =
    {
        return inScope( actor, obj, nil, SCOPE_REACHABLE );
    }
    replace validDoList(actor, prep, iobj) =
    {
        return scope( actor, nil, SCOPE_VISIBLE );
    }
    replace validIo(actor, obj, seqno) =
    {
        return inScope( actor, obj, nil, SCOPE_REACHABLE );
    }
;

/*
 *	Modifying inspectVerb to use the new SCOPING algorithms.
 */
modify inspectVerb
    replace validDo(actor, obj, seqno) =
    {
        return inScope( actor, obj, nil, SCOPE_VISIBLE );
    }
;
