//
//  XTDirectoryHelper.m
//  TadsTerp
//
//  Created by Rune Berg on 31/10/14.
//  Copyright (c) 2014 Rune Berg. All rights reserved.
//

#import "XTFileNameHelper.h"
#import "XTFileUtils.h"
#import "XTPrefs.h"
#import "XTLogger.h"


@interface XTFileNameHelper ()

@property XTPrefs *prefs;
@property NSURL *lastUsedGameDirectoryRegardless;

@end


@implementation XTFileNameHelper

static XTFileNameHelper *singletonInstance;

static XTLogger* logger;

+ (void)initialize
{
	logger = [XTLogger loggerForClass:[XTFileNameHelper class]];
	singletonInstance = [XTFileNameHelper new];
	singletonInstance.prefs = [XTPrefs prefs];
}

+ (id)helper
{
	return singletonInstance;
}

- (NSURL*)findDefaultDirectoryForFileType:(XTadsFileNameDialogFileType)fileType
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	switch (fileType) {
		case XTadsFileNameDialogFileTypeGame:
			res = [self findDefaultGamesDirectory];
			break;
		case XTadsFileNameDialogFileTypeSavedGame:
			res = [self findDefaultSavesDirectory];
			break;
		case XTadsFileNameDialogFileTypeTranscript:
			res = [self findDefaultTranscriptsDirectory];
			break;
		case XTadsFileNameDialogFileTypeCommandScript:
			res = [self findDefaultCommandScriptsDirectory];
			break;
		case XTadsFileNameDialogFileTypeGeneral:
			// no default dir for this
			break;
		default:
			XT_WARN_1(@"unknown fileType %d", fileType);
			break;
	}
	
	return res;
}

- (void)noteUsedDirectory:(NSURL *)fileUrl
			  forFileType:(XTadsFileNameDialogFileType)fileType
{
	XT_DEF_SELNAME;
	
	switch (fileType) {
		case XTadsFileNameDialogFileTypeGame:
			[self noteUsedGamesDirectory:fileUrl];
			break;
		case XTadsFileNameDialogFileTypeSavedGame:
			[self noteUsedSavesDirectory:fileUrl];
			break;
		case XTadsFileNameDialogFileTypeTranscript:
			[self noteUsedTranscriptsDirectory:fileUrl];
			break;
		case XTadsFileNameDialogFileTypeCommandScript:
			[self noteUsedCommandScriptsDirectory:fileUrl];
			break;
		case XTadsFileNameDialogFileTypeGeneral:
			// nothing to note for this
			break;
		default:
			XT_WARN_1(@"unknown fileType %d", fileType);
			break;
	}
}

- (NSURL *)findDefaultGamesDirectory
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	NSString *dirModeValue = self.prefs.gamesDirectoryMode.value;
	if ([XTPREFS_DIRECTORY_MODE_SPECIFIC isEqualToString:dirModeValue]) {
		res = self.prefs.gamesDirectoryWhenSpecific.value;
	} else if ([XTPREFS_DIRECTORY_MODE_NONE isEqualToString:dirModeValue]) {
		res = nil;
	} else if ([XTPREFS_DIRECTORY_MODE_LAST_SELECTED isEqualToString:dirModeValue]) {
		res = self.prefs.gamesDirectoryLastUsed.value;
	} else {
		XT_WARN_1(@"unexpected dir mode %@", dirModeValue);
	}
	
	XT_TRACE_1(@"-> \"%@\"", res);
	
	return res;
}

- (void)noteUsedGamesDirectory:(NSURL *)gameFileUrl
{
	XT_DEF_SELNAME;
	XT_TRACE_1(@"%@", gameFileUrl);
	
	self.lastUsedGameDirectoryRegardless = [XTFileUtils existingDirectoryUrlFor:gameFileUrl];
	self.prefs.gamesDirectoryLastUsed.value = self.lastUsedGameDirectoryRegardless;
}

- (NSURL *)findDefaultSavesDirectory
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	NSString *dirModeValue = self.prefs.savesDirectoryMode.value;
	if ([XTPREFS_DIRECTORY_MODE_SPECIFIC isEqualToString:dirModeValue]) {
		res = self.prefs.savesDirectoryWhenSpecific.value;
	} else if ([XTPREFS_DIRECTORY_MODE_NONE isEqualToString:dirModeValue]) {
		res = nil;
	} else if ([XTPREFS_DIRECTORY_MODE_LAST_SELECTED isEqualToString:dirModeValue]) {
		res = self.prefs.savesDirectoryLastUsed.value;
	} else if ([XTPREFS_DIRECTORY_MODE_CURRENT_GAMEFILE isEqualToString:dirModeValue]) {
		res = self.lastUsedGameDirectoryRegardless;
	} else {
		XT_WARN_1(@"unexpected dir mode %@", dirModeValue);
	}
	
	XT_TRACE_1(@"-> \"%@\"", res);
	
	return res;
}

- (void)noteUsedSavesDirectory:(NSURL *)saveFileUrl
{
	self.prefs.savesDirectoryLastUsed.value = [XTFileUtils existingDirectoryUrlFor:saveFileUrl];
}

- (NSURL *)findDefaultTranscriptsDirectory
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;

	NSString *dirModeValue = self.prefs.transcriptsDirectoryMode.value;
	if ([XTPREFS_DIRECTORY_MODE_SPECIFIC isEqualToString:dirModeValue]) {
		res = self.prefs.transcriptsDirectoryWhenSpecific.value;
	} else if ([XTPREFS_DIRECTORY_MODE_NONE isEqualToString:dirModeValue]) {
		res = nil;
	} else if ([XTPREFS_DIRECTORY_MODE_LAST_SELECTED isEqualToString:dirModeValue]) {
		res = self.prefs.transcriptsDirectoryLastUsed.value;
	} else if ([XTPREFS_DIRECTORY_MODE_CURRENT_GAMEFILE isEqualToString:dirModeValue]) {
		res = self.lastUsedGameDirectoryRegardless;
	} else {
		XT_WARN_1(@"unexpected dir mode %@", dirModeValue);
	}

	XT_TRACE_1(@"-> \"%@\"", res);
	
	return res;
}

- (void)noteUsedTranscriptsDirectory:(NSURL *)transcriptFileUrl
{
	self.prefs.transcriptsDirectoryLastUsed.value = [XTFileUtils existingDirectoryUrlFor:transcriptFileUrl];
}

- (NSURL *)findDefaultCommandScriptsDirectory
{
	XT_DEF_SELNAME;
	
	NSURL *res = nil;
	
	NSString *dirModeValue = self.prefs.commandScriptsDirectoryMode.value;
	if ([XTPREFS_DIRECTORY_MODE_SPECIFIC isEqualToString:dirModeValue]) {
		res = self.prefs.commandScriptsDirectoryWhenSpecific.value;
	} else if ([XTPREFS_DIRECTORY_MODE_NONE isEqualToString:dirModeValue]) {
		res = nil;
	} else if ([XTPREFS_DIRECTORY_MODE_LAST_SELECTED isEqualToString:dirModeValue]) {
		res = self.prefs.commandScriptsDirectoryLastUsed.value;
	} else if ([XTPREFS_DIRECTORY_MODE_CURRENT_GAMEFILE isEqualToString:dirModeValue]) {
		res = self.lastUsedGameDirectoryRegardless;
	} else {
		XT_WARN_1(@"unexpected dir mode %@", dirModeValue);
	}
	
	XT_TRACE_1(@"-> \"%@\"", res);
	
	return res;
}

- (void)noteUsedCommandScriptsDirectory:(NSURL *)commandScriptFileUrl
{
	self.prefs.commandScriptsDirectoryLastUsed.value = [XTFileUtils existingDirectoryUrlFor:commandScriptFileUrl];
}

- (NSString*)findDefaultFileBasenameForGameFileUrl:(NSURL *)gameFileUrl
										  fileType:(XTadsFileNameDialogFileType)fileType
{
	XT_DEF_SELNAME;
	
	NSString *res = nil;
	
	switch (fileType) {
		case XTadsFileNameDialogFileTypeSavedGame:
			res = [self makeDefaultFileBasenameForGameFileUrl:gameFileUrl fileNameMode:self.prefs.savesFileNameMode.value];
			break;
		case XTadsFileNameDialogFileTypeTranscript:
			res = [self makeDefaultFileBasenameForGameFileUrl:gameFileUrl fileNameMode:self.prefs.transcriptsFileNameMode.value];
			break;
		case XTadsFileNameDialogFileTypeCommandScript:
			res = [self makeDefaultFileBasenameForGameFileUrl:gameFileUrl fileNameMode:self.prefs.commandScriptsFileNameMode.value];
			break;
		case XTadsFileNameDialogFileTypeGeneral:
			// no default basename for this
			break;
		default:
			XT_WARN_1(@"unknown fileType %d", fileType);
			break;
	}
	
	return res;
}

- (NSString *)makeDefaultFileBasenameForGameFileUrl:(NSURL *)gameFileUrl
									   fileNameMode:(NSString *)fileNameMode
{
	XT_DEF_SELNAME;
	
	NSString *res = nil;
	
	if ([XTPREFS_FILE_NAME_MODE_DATE_TIME isEqualToString:fileNameMode]) {
		NSURL *gameFileUrlMinusExtension = [gameFileUrl URLByDeletingPathExtension];
		NSString *gameFileBasename = [gameFileUrlMinusExtension lastPathComponent];
		NSDateFormatter *dateFormatter = [NSDateFormatter new];
		[dateFormatter setDateFormat:@"yyyyMMdd_HHmmss"];
		NSDate *now = [NSDate date];
		NSString *nowString = [dateFormatter stringFromDate:now];
		res = [NSString stringWithFormat:@"%@-%@", gameFileBasename, nowString];

	} else if ([XTPREFS_FILE_NAME_MODE_GAME_FILE isEqualToString:fileNameMode]) {
		NSURL *gameFileUrlMinusExtension = [gameFileUrl URLByDeletingPathExtension];
		NSString *gameFileBasename = [gameFileUrlMinusExtension lastPathComponent];
		res = [NSString stringWithFormat:@"%@", gameFileBasename];

	} else if ([XTPREFS_FILE_NAME_MODE_UNTITLED isEqualToString:fileNameMode]) {
		res = @"untitled";
		
	} else {
		XT_WARN_1(@"unknown fileNameMode %d", fileNameMode);
	}
	
	return res;
}

@end
